//===- MCAsmInfoDarwin.cpp - Darwin asm properties ------------------------===//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//
//
// This file defines target asm properties related what form asm statements
// should take in general on Darwin-based targets
//
//===----------------------------------------------------------------------===//

#include "llvm/MC/MCAsmInfoDarwin.h"
#include "llvm/BinaryFormat/MachO.h"
#include "llvm/MC/MCDirectives.h"
#include "llvm/MC/MCSectionMachO.h"

using namespace llvm;

bool MCAsmInfoDarwin::isSectionAtomizableBySymbols(
    const MCSection &Section) {
  const MCSectionMachO &SMO = static_cast<const MCSectionMachO &>(Section);

  // Sections holding 1 byte strings are atomized based on the data they
  // contain.
  // Sections holding 2 byte strings require symbols in order to be atomized.
  // There is no dedicated section for 4 byte strings.
  if (SMO.getType() == MachO::S_CSTRING_LITERALS)
    return false;

  if (SMO.getSegmentName() == "__DATA" && SMO.getName() == "__cfstring")
    return false;

  if (SMO.getSegmentName() == "__DATA" && SMO.getName() == "__objc_classrefs")
    return false;

  switch (SMO.getType()) {
  default:
    return true;

  // These sections are atomized at the element boundaries without using
  // symbols.
  case MachO::S_4BYTE_LITERALS:
  case MachO::S_8BYTE_LITERALS:
  case MachO::S_16BYTE_LITERALS:
  case MachO::S_LITERAL_POINTERS:
  case MachO::S_NON_LAZY_SYMBOL_POINTERS:
  case MachO::S_LAZY_SYMBOL_POINTERS:
  case MachO::S_THREAD_LOCAL_VARIABLE_POINTERS:
  case MachO::S_MOD_INIT_FUNC_POINTERS:
  case MachO::S_MOD_TERM_FUNC_POINTERS:
  case MachO::S_INTERPOSING:
    return false;
  }
}

MCAsmInfoDarwin::MCAsmInfoDarwin() {
  // Common settings for all Darwin targets.
  // Syntax:
  LinkerPrivateGlobalPrefix = "l";
  HasSingleParameterDotFile = false;
  HasSubsectionsViaSymbols = true;

  AlignmentIsInBytes = false;
  COMMDirectiveAlignmentIsInBytes = false;
  LCOMMDirectiveAlignmentType = LCOMM::Log2Alignment;
  InlineAsmStart = " InlineAsm Start";
  InlineAsmEnd = " InlineAsm End";

  // Directives:
  HasWeakDefCanBeHiddenDirective = true;
  WeakRefDirective = "\t.weak_reference ";
  ZeroDirective = "\t.space\t";  // ".space N" emits N zeros.

  HiddenVisibilityAttr = MCSA_PrivateExtern;
  HiddenDeclarationVisibilityAttr = MCSA_Invalid;

  // Doesn't support protected visibility.
  ProtectedVisibilityAttr = MCSA_Invalid;

  HasDotTypeDotSizeDirective = false;
  HasNoDeadStrip = true;

  DwarfUsesRelocationsAcrossSections = false;
  SetDirectiveSuppressesReloc = true;
}
