/*******************************************************************************
 * Copyright (C) 2018 OTK Software
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package com.otk.application;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.geom.Dimension2D;
import java.awt.geom.Rectangle2D.Double;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.ObjectStreamClass;
import java.io.OutputStream;
import java.io.Serializable;
import java.util.Arrays;
import java.util.List;

import javax.swing.ImageIcon;

import com.otk.application.error.StandardError;
import com.otk.application.error.UnexpectedError;
import com.otk.application.image.filter.Retinex;
import com.otk.application.image.filter.Temperature;
import com.otk.application.image.printer.PrintingManager;
import com.otk.application.util.DoubleDimension;
import com.otk.application.util.ImageUtils;
import com.otk.application.util.MathUtils;
import com.otk.application.util.PageOrientation;
import com.otk.application.util.draw.FaceFrame;
import com.otk.application.util.draw.Mosaic;
import com.otk.application.util.draw.Mosaic.IMosaicDescriptor;

/**
 * This class holds the image and all the settings that can be adjusted in order
 * to get a printable mosaic of standard identity photographs.
 * 
 * It offers methods allowing to preview, check the conformity and output the
 * prints.
 * 
 * @author olitank
 *
 */
public class IdPhotoProject implements Serializable {

	public static void main(String[] args) {
		IdPhotoUI.RENDERER.openObjectFrame(new IdPhotoProject());
	}

	private static final long serialVersionUID = 1L;

	/**
	 * The maximum width/height of the imported picture (resized if bigger).
	 */
	private static final int MAX_PHOTOGRAPH_SIZE = 3000;

	/**
	 * The maximum width/height of the cropped (face) imported picture (resized if
	 * bigger).
	 */
	private static final int MAX_CROPPED_PHOTOGRAPH_SIZE = 1000;

	/**
	 * The box size of the optimized (for processing speed) version of the imported
	 * picture.
	 */
	private static final int FAST_PROCESSING_PHOTOGRAPH_SIZE = 500;

	/**
	 * Whether the usage statistics should be sent or not.
	 */
	private static boolean usageStatisticsShared = true;

	/**
	 * The imported (or obtained from a webcam) picture.
	 */
	private ImageIcon photograph = new ImageIcon(ImageUtils.NULL_IMAGE);

	/**
	 * The standard size of an identity photograph (depends on the country).
	 */
	private DoubleDimension standardCellSizeMillimeters = new DoubleDimension(35.0, 45.0);

	/**
	 * The ratio used to compute the face frame size.
	 */
	private double faceFrameHeightRatio = 0.6;

	/**
	 * The ratio used to compute the face frame vertical position.
	 */
	private double faceFrameVerticalOffsetRatio = 0.0;

	/**
	 * The ratio used to compute the face frame horizontal position.
	 */
	private double faceFrameHorizontalOffsetRatio = 0.0;

	/**
	 * The ratio used to compute the face overlay vertical position inside the face
	 * frame.
	 */
	private double faceFramePositioningOverlayVerticalOffsetRatio = 0.0;

	/**
	 * The ratio used to compute the face overlay height relatively to the face
	 * frame height.
	 */
	private double faceFramePositioningOverlayHeightRatio = 0.7;

	/**
	 * The ratio used to compute the face overlay width relatively to the face frame
	 * width.
	 */
	private double faceFramePositioningOverlayWidthRatio = 0.9;

	/**
	 * The output page orientation.
	 */
	private PageOrientation pageOrientation = PageOrientation.Landscape;

	/**
	 * The spacing of the cells in the output mosaic of identity photographs.
	 */
	private double mosaicCellSpacingRatio = 0.02;

	/**
	 * The number of columns the output mosaic of identity photographs.
	 */
	private int mosaicColumnCount = 4;

	/**
	 * The number of rows the output mosaic of identity photographs.
	 */
	private int mosaicRowCount = 2;

	/**
	 * The selected printer name.
	 */
	private String printerName = (PrintingManager.get().getAvailablePrinterNames().size() > 0)
			? PrintingManager.get().getAvailablePrinterNames().get(0)
			: "";

	/**
	 * The left margin of the output page.
	 */
	private double leftMarginMillimiters = 2;

	/**
	 * The right margin of the output page.
	 */
	private double rightMarginMillimiters = 2;

	/**
	 * The top margin of the output page.
	 */
	private double topMarginMillimiters = 2;

	/**
	 * The bottom margin of the output page.
	 */
	private double bottomMarginMillimiters = 2;

	/**
	 * The output page width.
	 */
	private double paperWidthMillimeters = 105;

	/**
	 * The output page height.
	 */
	private double paperHeightMillimeters = 148;

	/**
	 * The amount of brightness effect used to improve the picture.
	 */
	private int brightness = 0;

	/**
	 * The amount of contrast effect used to improve the picture.
	 */
	private int contrast = 0;

	/**
	 * The amount of 'background enhancement' effect used to improve the picture.
	 */
	private int backgroundEnhancementAmount = 0;

	/**
	 * The amount of temperature effect used to improve the picture.
	 */
	private int temperature = 0;

	/**
	 * A version of the {@link #photograph} image scaled-down for fast processing.
	 */
	private transient BufferedImage fastProcessingPhotographImageVersion;

	/**
	 * A version of the {@link #photograph} image preserving the quality.
	 */
	private transient BufferedImage qualityProcessingPhotographImageVersion;

	/**
	 * @return A scaled-down version of the {@link #photograph} image with the face
	 *         frame overlay drawn on it.
	 */
	public BufferedImage getPhotographWithFaceFrame() {
		BufferedImage result;
		if (fastProcessingPhotographImageVersion == null) {
			result = new BufferedImage(800, 600, ImageUtils.getAdaptedBufferedImageType());
		} else {
			result = ImageUtils.copy(fastProcessingPhotographImageVersion);
		}
		Graphics2D g = result.createGraphics();
		getFaceFrame().paint(g, result.getWidth(), result.getHeight());
		g.dispose();
		return result;
	}

	/**
	 * @return The value of {@link #usageStatisticsShared}.
	 */
	public static boolean isUsageStatisticsShared() {
		return IdPhotoProject.usageStatisticsShared;
	}

	/**
	 * Updates the value of {@link #usageStatisticsShared}.
	 * 
	 * @param usageStatisticsShared
	 *            The new value of {@link #usageStatisticsShared}.
	 */
	public static void setUsageStatisticsShared(boolean usageStatisticsShared) {
		IdPhotoProject.usageStatisticsShared = usageStatisticsShared;
	}

	/**
	 * @return The value of {@link #standardCellSizeMillimeters}.
	 */
	public DoubleDimension getStandardCellSizeMillimeters() {
		return standardCellSizeMillimeters;
	}

	/**
	 * Updates the value of {@link #standardCellSizeMillimeters}.
	 * 
	 * @param standardCellSizeMillimeters
	 *            The new value of {@link #standardCellSizeMillimeters}.
	 */
	public void setStandardCellSizeMillimeters(DoubleDimension standardCellSizeMillimeters) {
		this.standardCellSizeMillimeters = standardCellSizeMillimeters;
	}

	/**
	 * @return The value of {@link #faceFramePositioningOverlayVerticalOffsetRatio}.
	 */
	public double getFaceFramePositioningOverlayVerticalOffsetRatio() {
		return faceFramePositioningOverlayVerticalOffsetRatio;
	}

	/**
	 * Updates the value of {@link #faceFramePositioningOverlayVerticalOffsetRatio}.
	 * 
	 * @param faceFramePositioningOverlayVerticalOffsetRatio
	 *            The new value of
	 *            {@link #faceFramePositioningOverlayVerticalOffsetRatio}.
	 */
	public void setFaceFramePositioningOverlayVerticalOffsetRatio(
			double faceFramePositioningOverlayVerticalOffsetRatio) {
		this.faceFramePositioningOverlayVerticalOffsetRatio = faceFramePositioningOverlayVerticalOffsetRatio;
	}

	/**
	 * @return The value of {@link #faceFramePositioningOverlayHeightRatio}.
	 */
	public double getFaceFramePositioningOverlayHeightRatio() {
		return faceFramePositioningOverlayHeightRatio;
	}

	/**
	 * Updates the value of {@link #faceFramePositioningOverlayHeightRatio}.
	 * 
	 * @param faceFramePositioningOverlayHeightRatio
	 *            The new value of {@link #faceFramePositioningOverlayHeightRatio}.
	 */
	public void setFaceFramePositioningOverlayHeightRatio(double faceFramePositioningOverlayHeightRatio) {
		this.faceFramePositioningOverlayHeightRatio = faceFramePositioningOverlayHeightRatio;
	}

	/**
	 * @return The value of {@link #faceFramePositioningOverlayWidthRatio}.
	 */
	public double getFaceFramePositioningOverlayWidthRatio() {
		return faceFramePositioningOverlayWidthRatio;
	}

	/**
	 * Updates the value of {@link #faceFramePositioningOverlayWidthRatio}.
	 * 
	 * @param faceFramePositioningOverlayWidthRatio
	 *            The new value of {@link #faceFramePositioningOverlayWidthRatio}.
	 */
	public void setFaceFramePositioningOverlayWidthRatio(double faceFramePositioningOverlayWidthRatio) {
		this.faceFramePositioningOverlayWidthRatio = faceFramePositioningOverlayWidthRatio;
	}

	/**
	 * @param image
	 *            The image to process.
	 * @return An image with the temperature effect applied according to the value
	 *         of {@link #temperature}.
	 */
	private BufferedImage changeTemperature(BufferedImage image) {
		if (temperature == 0) {
			return image;
		}
		Temperature filter = new Temperature();
		filter.temperatureChange = temperature;
		image = ImageUtils.filter(image, filter);
		return image;
	}

	/**
	 * @param image
	 *            The image to process.
	 * @return An image with the 'background enhancement' effect applied according
	 *         to the value of {@link #backgroundEnhancementAmount}. Note: the
	 *         Retinex algorithm is used to create this effect.
	 */
	private BufferedImage enhanceBackground(BufferedImage image, boolean cropped) {
		if (backgroundEnhancementAmount == 0) {
			return image;
		}
		Retinex retinex = new Retinex();
		retinex.amount = backgroundEnhancementAmount;
		retinex.dynamic = 100;
		retinex.level = "High";
		retinex.scale = 16;
		retinex.scaleDivision = 8;
		image = ImageUtils.filter(image, retinex);
		return image;
	}

	/**
	 * @return The specified face frame object.
	 */
	private FaceFrame getFaceFrame() {
		FaceFrame result = new FaceFrame();
		result.setWidthOverHeightRatio(
				standardCellSizeMillimeters.getWidth() / standardCellSizeMillimeters.getHeight());
		result.setHeightRatio(faceFrameHeightRatio);
		result.setVerticalOffsetRatio(-faceFrameVerticalOffsetRatio);
		result.setHorizontalOffsetRatio(faceFrameHorizontalOffsetRatio);
		result.setPositioningOverlayYellowSquareVisible(false);
		result.setPositioningOverlayRulersVisible(true);
		result.setPositioningOverlayWidthRatio(faceFramePositioningOverlayWidthRatio);
		result.setPositioningOverlayHeightRatio(faceFramePositioningOverlayHeightRatio);
		result.setPositioningOverlayVerticalOffsetRatio(faceFramePositioningOverlayVerticalOffsetRatio);
		return result;
	}

	/**
	 * @return The image object attached to the value of {@link #photograph}.
	 */
	public Image getRawPhotograph() {
		return this.photograph.getImage();
	}

	/**
	 * Updates the value of {@link #photograph} from the given image.
	 * 
	 * @param image
	 *            The image that will be attached to the new value of
	 *            {@link #photograph}. It may be scaled-down according to
	 *            {@link #MAX_PHOTOGRAPH_SIZE}.
	 */
	public void setPhotograph(Image image) {
		if (image == null) {
			throw new IllegalArgumentException("Image not provided!");
		}
		if ((image.getWidth(null) > MAX_PHOTOGRAPH_SIZE) || (image.getHeight(null) > MAX_PHOTOGRAPH_SIZE)) {
			image = ImageUtils.scalePreservingRatio(image, MAX_PHOTOGRAPH_SIZE, MAX_PHOTOGRAPH_SIZE, true, true);
		}
		this.photograph = new ImageIcon(image);
		updateAllPhotographImageVersions();
	}

	/**
	 * Updates the value of {@link #photograph} from the given image file.
	 * 
	 * @param file
	 *            The file containing the image that will be attached to the new
	 *            value of {@link #photograph}. The image may be scaled-down
	 *            according to {@link #MAX_PHOTOGRAPH_SIZE}.
	 */
	public void setPhotographFromFile(File file) throws IOException {
		BufferedImage image = ImageUtils.loadAdaptedImage(file);
		if ((image.getWidth() > MAX_PHOTOGRAPH_SIZE) || (image.getHeight() > MAX_PHOTOGRAPH_SIZE)) {
			image = ImageUtils.scalePreservingRatio(image, MAX_PHOTOGRAPH_SIZE, MAX_PHOTOGRAPH_SIZE, true, true);
		}
		this.photograph = new ImageIcon(image);
		updateAllPhotographImageVersions();
	}

	/**
	 * @return The value of {@link #brightness}.
	 */
	public int getBrightness() {
		return brightness;
	}

	/**
	 * Updates the value of {@link #brightness}.
	 * 
	 * @param brightness
	 *            The new value of {@link #brightness}.
	 */
	public void setBrightness(int brightness) {
		this.brightness = brightness;
	}

	/**
	 * @return The value of {@link #contrast}.
	 */
	public int getContrast() {
		return contrast;
	}

	/**
	 * Updates the value of {@link #contrast}.
	 * 
	 * @param contrast
	 *            The new value of {@link #contrast}.
	 */
	public void setContrast(int contrast) {
		this.contrast = contrast;
	}

	/**
	 * @return The value of {@link #temperature}.
	 */
	public int getTemperature() {
		return temperature;
	}

	/**
	 * Updates the value of {@link #temperature}.
	 * 
	 * @param temperature
	 *            The new value of {@link #temperature}.
	 */
	public void setTemperature(int temperature) {
		this.temperature = temperature;
	}

	/**
	 * @return The value of {@link #backgroundEnhancementAmount}.
	 */
	public int getBackgroundEnhancementAmount() {
		return backgroundEnhancementAmount;
	}

	/**
	 * Updates the value of {@link #backgroundEnhancementAmount}.
	 * 
	 * @param backgroundEnhancementAmount
	 *            The new value of {@link #backgroundEnhancementAmount}.
	 */
	public void setBackgroundEnhancementAmount(int backgroundEnhancementAmount) {
		this.backgroundEnhancementAmount = backgroundEnhancementAmount;
	}

	/**
	 * @return The value of {@link #paperWidthMillimeters}.
	 */
	public double getPaperWidthMillimeters() {
		return paperWidthMillimeters;
	}

	/**
	 * Updates the value of {@link #paperWidthMillimeters}.
	 * 
	 * @param paperWidthMillimeters
	 *            The new value of {@link #paperWidthMillimeters}.
	 */
	public void setPaperWidthMillimeters(double paperWidthMillimeters) {
		this.paperWidthMillimeters = paperWidthMillimeters;
	}

	/**
	 * @return The value of {@link #paperHeightMillimeters}.
	 */
	public double getPaperHeightMillimeters() {
		return paperHeightMillimeters;
	}

	/**
	 * Updates the value of {@link #paperHeightMillimeters}.
	 * 
	 * @param paperHeightMillimeters
	 *            The new value of {@link #paperHeightMillimeters}.
	 */
	public void setPaperHeightMillimeters(double paperHeightMillimeters) {
		this.paperHeightMillimeters = paperHeightMillimeters;
	}

	/**
	 * @return The value of {@link #leftMarginMillimiters}.
	 */
	public double getLeftMarginMillimiters() {
		return leftMarginMillimiters;
	}

	/**
	 * Updates the value of {@link #leftMarginMillimiters}.
	 * 
	 * @param leftMarginMillimiters
	 *            The new value of {@link #leftMarginMillimiters}.
	 */
	public void setLeftMarginMillimiters(double leftMarginMillimiters) {
		this.leftMarginMillimiters = leftMarginMillimiters;
	}

	/**
	 * @return The value of {@link #rightMarginMillimiters}.
	 */
	public double getRightMarginMillimiters() {
		return rightMarginMillimiters;
	}

	/**
	 * Updates the value of {@link #rightMarginMillimiters}.
	 * 
	 * @param rightMarginMillimiters
	 *            The new value of {@link #rightMarginMillimiters}.
	 */
	public void setRightMarginMillimiters(double rightMarginMillimiters) {
		this.rightMarginMillimiters = rightMarginMillimiters;
	}

	/**
	 * @return The value of {@link #topMarginMillimiters}.
	 */
	public double getTopMarginMillimiters() {
		return topMarginMillimiters;
	}

	/**
	 * Updates the value of {@link #topMarginMillimiters}.
	 * 
	 * @param topMarginMillimiters
	 *            The new value of {@link #topMarginMillimiters}.
	 */
	public void setTopMarginMillimiters(double topMarginMillimiters) {
		this.topMarginMillimiters = topMarginMillimiters;
	}

	/**
	 * @return The value of {@link #bottomMarginMillimiters}.
	 */
	public double getBottomMarginMillimiters() {
		return bottomMarginMillimiters;
	}

	/**
	 * Updates the value of {@link #bottomMarginMillimiters}.
	 * 
	 * @param bottomMarginMillimiters
	 *            The new value of {@link #bottomMarginMillimiters}.
	 */
	public void setBottomMarginMillimiters(double bottomMarginMillimiters) {
		this.bottomMarginMillimiters = bottomMarginMillimiters;
	}

	/**
	 * @return The value of {@link #faceFrameHeightRatio}.
	 */
	public double getFaceFrameHeightRatio() {
		return faceFrameHeightRatio;
	}

	/**
	 * Updates the value of {@link #faceFrameHeightRatio}.
	 * 
	 * @param faceFrameHeightRatio
	 *            The new value of {@link #faceFrameHeightRatio}.
	 */
	public void setFaceFrameHeightRatio(double faceFrameHeightRatio) {
		this.faceFrameHeightRatio = faceFrameHeightRatio;
	}

	/**
	 * @return The value of {@link #faceFrameHeightRatio}.
	 */
	public int getFaceFrameHeightPercentage() {
		return (int) Math.round(faceFrameHeightRatio * 100);
	}

	/**
	 * Updates the value of {@link #faceFrameHeightRatio}.
	 * 
	 * @param faceFrameHeightRatio
	 *            The new value of {@link #faceFrameHeightRatio}.
	 */
	public void setFaceFrameHeightPercentage(int faceFrameHeightPercentage) {
		this.faceFrameHeightRatio = faceFrameHeightPercentage / 100.0;
	}

	/**
	 * @return The value of {@link #faceFrameVerticalOffsetRatio}.
	 */
	public double getFaceFrameVerticalOffsetRatio() {
		return faceFrameVerticalOffsetRatio;
	}

	/**
	 * Updates the value of {@link #faceFrameVerticalOffsetRatio}.
	 * 
	 * @param faceFrameVerticalOffsetRatio
	 *            The new value of {@link #faceFrameVerticalOffsetRatio}.
	 */
	public void setFaceFrameVerticalOffsetRatio(double faceFrameVerticalOffsetRatio) {
		this.faceFrameVerticalOffsetRatio = faceFrameVerticalOffsetRatio;
	}

	public void addFaceFrameVerticalOffsetRatio(double faceFrameVerticalOffsetRatio) {
		this.faceFrameVerticalOffsetRatio += faceFrameVerticalOffsetRatio;
	}

	/**
	 * @return The value of {@link #faceFrameHorizontalOffsetRatio}.
	 */
	public double getFaceFrameHorizontalOffsetRatio() {
		return faceFrameHorizontalOffsetRatio;
	}

	/**
	 * Updates the value of {@link #faceFrameHorizontalOffsetRatio}.
	 * 
	 * @param faceFrameHorizontalOffsetRatio
	 *            The new value of {@link #faceFrameHorizontalOffsetRatio}.
	 */
	public void setFaceFrameHorizontalOffsetRatio(double faceFrameHorizontalOffsetRatio) {
		this.faceFrameHorizontalOffsetRatio = faceFrameHorizontalOffsetRatio;
	}

	/**
	 * Adds the given value to the current value of
	 * {@link #faceFrameHorizontalOffsetRatio}.
	 * 
	 * @param faceFrameHorizontalOffsetRatio
	 *            The {@link #faceFrameHorizontalOffsetRatio} offset value.
	 */
	public void addFaceFrameHorizontalOffsetRatio(double faceFrameHorizontalOffsetRatio) {
		this.faceFrameHorizontalOffsetRatio += faceFrameHorizontalOffsetRatio;
	}

	/**
	 * @return The value of {@link #pageOrientation}.
	 */
	public PageOrientation getPageOrientation() {
		return pageOrientation;
	}

	/**
	 * Updates the value of {@link #pageOrientation}.
	 * 
	 * @param pageOrientation
	 *            The new value of {@link #pageOrientation}.
	 */
	public void setPageOrientation(PageOrientation pageOrientation) {
		this.pageOrientation = pageOrientation;
	}

	/**
	 * @return The value of {@link #mosaicCellSpacingRatio}.
	 */
	public double getMosaicCellSpacingRatio() {
		return mosaicCellSpacingRatio;
	}

	/**
	 * Updates the value of {@link #mosaicCellSpacingRatio}.
	 * 
	 * @param mosaicCellSpacingRatio
	 *            The new value of {@link #mosaicCellSpacingRatio}.
	 */
	public void setMosaicCellSpacingRatio(double mosaicCellSpacingRatio) {
		this.mosaicCellSpacingRatio = mosaicCellSpacingRatio;
	}

	/**
	 * @return The value of {@link #mosaicColumnCount}.
	 */
	public int getMosaicColumnCount() {
		return mosaicColumnCount;
	}

	/**
	 * Updates the value of {@link #mosaicColumnCount}.
	 * 
	 * @param mosaicColumnCount
	 *            The new value of {@link #mosaicColumnCount}.
	 */
	public void setMosaicColumnCount(int mosaicColumnCount) {
		this.mosaicColumnCount = mosaicColumnCount;
	}

	/**
	 * @return The value of {@link #mosaicRowCount}.
	 */
	public int getMosaicRowCount() {
		return mosaicRowCount;
	}

	/**
	 * Updates the value of {@link #mosaicRowCount}.
	 * 
	 * @param mosaicRowCount
	 *            The new value of {@link #mosaicRowCount}.
	 */
	public void setMosaicRowCount(int mosaicRowCount) {
		this.mosaicRowCount = mosaicRowCount;
	}

	/**
	 * @return The value of {@link #printerName}.
	 */
	public String getPrinterName() {
		return printerName;
	}

	/**
	 * Updates the value of {@link #printerName}.
	 * 
	 * @param printerName
	 *            The new value of {@link #printerName}.
	 */
	public void setPrinterName(String printerName) {
		this.printerName = printerName;
	}

	/**
	 * @return The names of the available printers.
	 */
	public List<String> getPrinterNames() {
		return getPrintingManager().getAvailablePrinterNames();
	}

	/**
	 * @return A preview image of the identity photographs mosaic output
	 *         (scaled-down).
	 */
	public Image getPrintPreview() {
		if (fastProcessingPhotographImageVersion == null) {
			return null;
		}
		BufferedImage result = buildResultImage(fastProcessingPhotographImageVersion);
		result = getPrintingManager().printToMemory(result, PrintingManager.BAD_QUALITY_MEMORY_PRINT_DPI);
		result = PageOrientation.revertOrientation(result, pageOrientation);
		result = ImageUtils.addRealisticPaperEffect(result);
		return result;
	}

	/**
	 * @return A preview image of the identity photographs mosaic output.
	 */
	public Image getDetailedPrintPreview() {
		if (qualityProcessingPhotographImageVersion == null) {
			throw new IllegalStateException("There is no picture");
		}
		BufferedImage result = buildResultImage(qualityProcessingPhotographImageVersion);
		result = getPrintingManager().printToMemory(result, PrintingManager.GOOD_QUALITY_MEMORY_PRINT_DPI);
		result = PageOrientation.revertOrientation(result, pageOrientation);
		result = ImageUtils.addRealisticPaperEffect(result);
		return result;
	}

	/**
	 * @return An image attesting the conformity or the non-conformity of the output
	 *         identity photographs dimensions.
	 */
	public Image getCellSizeConformityImage() {
		BufferedImage result = new BufferedImage((int) standardCellSizeMillimeters.getWidth(),
				(int) standardCellSizeMillimeters.getHeight(), ImageUtils.getAdaptedBufferedImageType());
		Graphics2D g2d = result.createGraphics();
		ImageUtils.drawLineBorder(g2d, Color.BLACK, 1, result.getWidth(), result.getHeight());
		getFaceFrame().paintPositioningOverlayOnCroppedImage(g2d, new Dimension(result.getWidth(), result.getHeight()));
		g2d.dispose();
		result = Mosaic.generate(getMosaicDescriptor(result));
		result = PageOrientation.updateOrientation(result, pageOrientation);
		result = getPrintingManager().printToMemory(result, PrintingManager.BAD_QUALITY_MEMORY_PRINT_DPI);
		result = PageOrientation.revertOrientation(result, pageOrientation);
		result = ImageUtils.addRealisticPaperEffect(result);
		BufferedImage noticeImage;
		if (isSizeConform()) {
			noticeImage = ImageUtils.getTextImage("The cell size is conform!",
					TheConstants.readableFont.deriveFont(50f), Color.GREEN);
		} else {
			Dimension2D currentSizeMillimeters = getCellSizeMillimeters();
			if ((currentSizeMillimeters.getWidth()
					* currentSizeMillimeters.getHeight()) > (standardCellSizeMillimeters.getWidth()
							* standardCellSizeMillimeters.getHeight())) {
				noticeImage = ImageUtils.getTextImage("The cell size is not conform!\n            (too big)",
						TheConstants.readableFont.deriveFont(50f), Color.RED);
			} else {
				noticeImage = ImageUtils.getTextImage("The cell size is not conform!\n            (too small)",
						TheConstants.readableFont.deriveFont(50f), Color.RED);
			}
		}
		noticeImage = ImageUtils.scaleToWidth(noticeImage, result.getWidth());
		result = ImageUtils.joinImages(Arrays.asList(result, noticeImage), false);
		return result;
	}

	/**
	 * @return The printing manager (used to print or get a print preview).
	 */
	private PrintingManager getPrintingManager() {
		PrintingManager.get().setPrinterName(printerName);
		if (pageOrientation == PageOrientation.Portrait) {
			PrintingManager.get().setTopMarginMillimiters(topMarginMillimiters);
			PrintingManager.get().setLeftMarginMillimiters(leftMarginMillimiters);
			PrintingManager.get().setBottomMarginMillimiters(bottomMarginMillimiters);
			PrintingManager.get().setRightMarginMillimiters(rightMarginMillimiters);
		} else if (pageOrientation == PageOrientation.Landscape) {
			PrintingManager.get().setTopMarginMillimiters(leftMarginMillimiters);
			PrintingManager.get().setLeftMarginMillimiters(bottomMarginMillimiters);
			PrintingManager.get().setBottomMarginMillimiters(rightMarginMillimiters);
			PrintingManager.get().setRightMarginMillimiters(topMarginMillimiters);
		} else {
			throw new StandardError("Illegal printing page orientation: '" + pageOrientation + "'");
		}
		PrintingManager.get().setPaperWidthMillimeters(paperWidthMillimeters);
		PrintingManager.get().setPaperHeightMillimeters(paperHeightMillimeters);
		return PrintingManager.get();
	}

	/**
	 * @return Whether the dimensions of the output identity photographs are
	 *         correct.
	 */
	private boolean isSizeConform() {
		Dimension2D outputPhotoSize = getCellSizeMillimeters();
		if (Math.abs(outputPhotoSize.getWidth() - standardCellSizeMillimeters.getWidth()) > 1.0) {
			return false;
		}
		if (Math.abs(outputPhotoSize.getHeight() - standardCellSizeMillimeters.getHeight()) > 1.0) {
			return false;
		}
		return true;
	}

	/**
	 * @return The actual size of an output identity photograph cell.
	 */
	public DoubleDimension getCellSizeMillimeters() {
		Dimension sampleCellSize = new Dimension((int) Math.round(standardCellSizeMillimeters.getWidth()),
				(int) Math.round(standardCellSizeMillimeters.getHeight()));
		BufferedImage sampleCellImage = new BufferedImage(sampleCellSize.width, sampleCellSize.height,
				ImageUtils.getAdaptedBufferedImageType());
		IMosaicDescriptor sampleMosaicDescriptor = getMosaicDescriptor(sampleCellImage);
		Dimension sampleMosaicSize = Mosaic.getMosaicSize(sampleMosaicDescriptor);
		int anyDpi = PrintingManager.PRINT_DPI;
		double pixelsToDots = predictMosaicPixelsToDotsFactor(sampleMosaicSize, anyDpi);
		int photoWidthDots = MathUtils.round(sampleCellSize.getWidth() * pixelsToDots);
		int photoHeightDots = MathUtils.round(sampleCellSize.getHeight() * pixelsToDots);
		final double photoWidthMillimeters = MathUtils.dotsToMillimeters(photoWidthDots, anyDpi);
		final double photoHeightMillimeters = MathUtils.dotsToMillimeters(photoHeightDots, anyDpi);
		return new DoubleDimension(photoWidthMillimeters, photoHeightMillimeters);
	}

	/**
	 * @param mosaicSize
	 * @param dpi
	 * @return A ratio allowing to infer (by multiplying) a numbers of dots on the
	 *         output page from a number of pixels on the mosaic image.
	 */
	private double predictMosaicPixelsToDotsFactor(Dimension mosaicSize, int dpi) {
		BufferedImage sampleImage = new BufferedImage(mosaicSize.width, mosaicSize.height,
				ImageUtils.getAdaptedBufferedImageType());
		sampleImage = PageOrientation.updateOrientation(sampleImage, getPageOrientation());
		mosaicSize = new Dimension(sampleImage.getWidth(), sampleImage.getHeight());
		Double scaledBoundsDots = getPrintingManager().getImageScaledBoundsInsidePrintableArea(mosaicSize.width,
				mosaicSize.height, dpi);
		return (double) scaledBoundsDots.width / (double) mosaicSize.width;
	}

	/**
	 * @param photographImage
	 *            A version of the {@link #photograph} image.
	 * @return The output identity photographs mosaic image.
	 */
	private BufferedImage buildResultImage(Image photographImage) {
		BufferedImage result = ImageUtils.getBufferedImage(photographImage);
		result = getFaceFrame().crop(result);
		if ((result.getWidth() > MAX_CROPPED_PHOTOGRAPH_SIZE) || (result.getHeight() > MAX_CROPPED_PHOTOGRAPH_SIZE)) {
			result = ImageUtils.scalePreservingRatio(result, MAX_CROPPED_PHOTOGRAPH_SIZE, MAX_CROPPED_PHOTOGRAPH_SIZE,
					true, true);
		}
		result = ImageUtils.changeBrightnessAndContrast(result, brightness, contrast, Color.GRAY);
		result = changeTemperature(result);
		result = enhanceBackground(result, true);
		result = Mosaic.generate(getMosaicDescriptor(result));
		result = PageOrientation.updateOrientation(result, pageOrientation);
		return result;
	}

	/**
	 * @param cellImage
	 *            The image that is meant to be in each cell of the mosaic.
	 * @return An object describing the output mosaic according the given cell
	 *         image.
	 */
	private IMosaicDescriptor getMosaicDescriptor(final BufferedImage cellImage) {
		return new IMosaicDescriptor() {

			@Override
			public boolean isCellVisible(int col, int row) {
				return true;
			}

			@Override
			public int getRowCount() {
				return mosaicRowCount;
			}

			@Override
			public Image getInputImage() {
				return cellImage;
			}

			@Override
			public int getColumnCount() {
				return mosaicColumnCount;
			}

			@Override
			public double getCellSpacingRatio() {
				return mosaicCellSpacingRatio;
			}

			@Override
			public int getCellRotationDegrees() {
				return 0;
			}
		};
	}

	/**
	 * Prints the identity photographs mosaic.
	 */
	public void print() {
		if (printerName == null) {
			throw new IllegalStateException("Cannot print: The printer was not selected");
		}
		if (ImageUtils.isNullImage(ImageUtils.copy(photograph.getImage()))) {
			throw new IllegalStateException("There is no picture!");
		}
		BufferedImage imageToPrint = buildResultImage(qualityProcessingPhotographImageVersion);
		getPrintingManager().print(imageToPrint);
	}

	/**
	 * Exports the identity photographs mosaic to a PDF file.
	 * 
	 * @param file
	 *            The destination PDF file.
	 */
	public void exportPDF(File file) throws IOException {
		if (ImageUtils.isNullImage(ImageUtils.copy(photograph.getImage()))) {
			throw new IllegalStateException("There is no picture!");
		}
		BufferedImage imageToPrint = buildResultImage(qualityProcessingPhotographImageVersion);
		getPrintingManager().printToPDF(imageToPrint, file);
	}

	/**
	 * Overrides this class default serialization.
	 * 
	 * @param oos
	 *            The object output stream used to serialize an instance.
	 * @throws IOException
	 *             If an exception is thrown by the given ObjectOutputStream.
	 */
	private void writeObject(ObjectOutputStream oos) throws IOException {
		oos.defaultWriteObject();
		oos.writeObject(new Boolean(usageStatisticsShared));
	}

	/**
	 * Overrides this class default deserialization.
	 * 
	 * @param ois
	 *            The object input stream used to deserialize an instance.
	 * @throws IOException
	 *             If an exception is thrown by the given ObjectOutputStream.
	 */
	private void readObject(ObjectInputStream ois) throws ClassNotFoundException, IOException {
		ois.defaultReadObject();
		try {
			usageStatisticsShared = (Boolean) ois.readObject();
		} catch (Throwable ignore) {
		}
	}

	/**
	 * Saves the current instance state to the given output stream.
	 * 
	 * @param out
	 *            The output stream that will receive the current instance data.
	 */
	public void save(OutputStream out) {
		try {
			ObjectOutputStream oos = new ObjectOutputStream(out);
			oos.writeObject(this);
		} catch (Throwable t) {
			throw new UnexpectedError("Failed to serialize object: " + t.toString());
		}
	}

	/**
	 * Loads the current instance state from the given input stream.
	 * 
	 * @param out
	 *            The input stream that will provide the current instance data.
	 */
	public void load(InputStream in) {
		IdPhotoProject deserialized;
		try {
			@SuppressWarnings("resource")
			ObjectInputStream ois = new BackwardCompatibileObjectInputStream(in);
			deserialized = (IdPhotoProject) ois.readObject();
		} catch (Throwable t) {
			throw new UnexpectedError("Failed to deserialize object: " + t.toString());
		}
		this.photograph = deserialized.photograph;
		if (this.photograph == null) {
			this.photograph = new ImageIcon(ImageUtils.NULL_IMAGE);
		}
		updateAllPhotographImageVersions();
		this.faceFrameHeightRatio = deserialized.faceFrameHeightRatio;
		this.faceFrameHorizontalOffsetRatio = deserialized.faceFrameHorizontalOffsetRatio;
		this.faceFrameVerticalOffsetRatio = deserialized.faceFrameVerticalOffsetRatio;
		this.faceFramePositioningOverlayVerticalOffsetRatio = deserialized.faceFramePositioningOverlayVerticalOffsetRatio;
		this.faceFramePositioningOverlayHeightRatio = deserialized.faceFramePositioningOverlayHeightRatio;
		this.faceFramePositioningOverlayWidthRatio = deserialized.faceFramePositioningOverlayWidthRatio;
		this.mosaicCellSpacingRatio = deserialized.mosaicCellSpacingRatio;
		this.mosaicColumnCount = deserialized.mosaicColumnCount;
		this.mosaicRowCount = deserialized.mosaicRowCount;
		this.pageOrientation = deserialized.pageOrientation;
		this.printerName = deserialized.printerName;
		this.leftMarginMillimiters = deserialized.leftMarginMillimiters;
		this.rightMarginMillimiters = deserialized.rightMarginMillimiters;
		this.bottomMarginMillimiters = deserialized.bottomMarginMillimiters;
		this.topMarginMillimiters = deserialized.topMarginMillimiters;
		this.paperWidthMillimeters = deserialized.paperWidthMillimeters;
		this.paperHeightMillimeters = deserialized.paperHeightMillimeters;
		this.brightness = deserialized.brightness;
		this.contrast = deserialized.contrast;
		this.backgroundEnhancementAmount = deserialized.backgroundEnhancementAmount;
		this.temperature = deserialized.temperature;
		this.standardCellSizeMillimeters = deserialized.standardCellSizeMillimeters;
	}

	/**
	 * Updates the photograph image versions
	 * ({@link #fastProcessingPhotographImageVersion},
	 * {@link #qualityProcessingPhotographImageVersion}, ...)..
	 */
	private void updateAllPhotographImageVersions() {
		if (ImageUtils.isNullImage(ImageUtils.copy(photograph.getImage()))) {
			this.fastProcessingPhotographImageVersion = null;
			this.qualityProcessingPhotographImageVersion = null;
		} else {
			this.fastProcessingPhotographImageVersion = ImageUtils.scalePreservingRatio(photograph.getImage(),
					FAST_PROCESSING_PHOTOGRAPH_SIZE, FAST_PROCESSING_PHOTOGRAPH_SIZE, true, true);
			this.qualityProcessingPhotographImageVersion = ImageUtils.getBufferedImage(photograph.getImage());
		}
	}

	/**
	 * This class is used by the {@link IdPhotoProject#load(InputStream)} method to
	 * guarantee the backward compatibility of the serialized files.
	 * 
	 * @author olitank
	 *
	 */
	private class BackwardCompatibileObjectInputStream extends ObjectInputStream {

		public BackwardCompatibileObjectInputStream(InputStream in) throws IOException {
			super(in);
		}

		@Override
		protected ObjectStreamClass readClassDescriptor() throws IOException, ClassNotFoundException {
			ObjectStreamClass resultClassDescriptor = super.readClassDescriptor();
			if (resultClassDescriptor.getName().endsWith(".PageOrientation")) {
				resultClassDescriptor = ObjectStreamClass.lookup(PageOrientation.class);
			}
			if (resultClassDescriptor.getName().endsWith(".IdPhotoProject")) {
				resultClassDescriptor = ObjectStreamClass.lookup(IdPhotoProject.class);
			}
			if (resultClassDescriptor.getName().endsWith(".DoubleDimension")) {
				resultClassDescriptor = ObjectStreamClass.lookup(DoubleDimension.class);
			}
			return resultClassDescriptor;
		}
	}

}
