/*******************************************************************************
 * Copyright (C) 2018 OTK Software
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package com.otk.application.error;

import com.otk.application.util.MiscUtils;

/**
 * This class allows to add debug assertions in the code that can be
 * activated/deactivated at runtime by setting a system property.
 * 
 * @author olitank
 *
 */
public class Assertion {

	/**
	 * Key of the system property used to activate/deactivate all the assertions.
	 */
	public static final String ACTIVATION_PROPERTY_KEY = Assertion.class.getName() + ".active";

	/**
	 * Checks the given predicate: an exception is thrown only and only if the
	 * predicate is false.
	 * 
	 * @param predicate
	 *            the predicate to evaluate.
	 */
	public static void check(Predicate predicate) {
		if (!isActive()) {
			return;
		}
		if (!predicate.isTrue()) {
			throw new UnexpectedError("Assertion failure at:\n" + MiscUtils.getPrintedStackTrace(new Exception()));
		}
		try {
			predicate.doesNotFail();
		} catch (Throwable t) {
			throw new UnexpectedError("Assertion failure at:\n" + MiscUtils.getPrintedStackTrace(t));
		}
	}

	/**
	 * @return Whether the assertions are active (can throw exceptions) or not.
	 */
	protected static boolean isActive() {
		return System.getProperty(ACTIVATION_PROPERTY_KEY, "false").equals("true");
	}

	/**
	 * This represent an assertion predicate.
	 * 
	 * @author olitank
	 *
	 */
	public static abstract class Predicate {
		/**
		 * @return true in order to validate the assertion.
		 */
		protected boolean isTrue() {
			return true;
		}

		/**
		 * Throws an exception to indicate that the assertion is not valid.
		 * 
		 * @throws Throwable
		 *             In order to invalidate the assertion.
		 */
		protected void doesNotFail() throws Throwable {
		}
	}

}
