/*******************************************************************************
 * Copyright (C) 2018 OTK Software
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package com.otk.application.image.camera;

import java.awt.image.BufferedImage;
import java.awt.image.BufferedImageOp;
import java.awt.image.ColorModel;
import java.awt.image.RescaleOp;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;

import com.otk.application.error.UnexpectedError;
import com.otk.application.util.MiscUtils;

/**
 * This class abstract the actual library/API used to communicate with
 * {@link Camera} devices.
 * 
 * @author olitank
 *
 */
public abstract class AbstractCameraDriver {

	protected Camera camera;
	protected ExecutorService withoutFreezingExecutor = Executors.newCachedThreadPool(new ThreadFactory() {
		@Override
		public Thread newThread(Runnable r) {
			Thread result = new Thread(r,
					MiscUtils.formatThreadName(AbstractCameraDriver.this.getClass(), "TimeOutMonitor"));
			result.setDaemon(true);
			return result;
		}
	});

	public AbstractCameraDriver(Camera camera) {
		this.camera = camera;
	}

	public abstract String getName();

	public abstract List<String> listDeviceLocalNamesWhileCameraNotInitialized();

	public abstract List<FrameFormat> getVideoFormatsWhileCameraInitializedAndNotActive();

	public abstract void doSetUp() throws Exception;

	public abstract void startCapture();

	public abstract void stopCapture();

	public abstract void doCleanUp() throws Exception;

	public abstract void doWaitForCompleteInterruption();

	protected void withoutFreezing(final Runnable runnable) {
		if (!MiscUtils.runWithTimeout(new Runnable() {
			@Override
			public void run() {
				runnable.run();
			}
		}, camera.getFreezeTimeoutMilliSeconds(), withoutFreezingExecutor)) {
			throw new UnexpectedError("Time out error. The software execution may become unstable");
		}
	}

	public void checkNewDevice() {
	}

	protected BufferedImageOp getDeviceImagePrecorrectionOp(BufferedImage deviceImage) {
		if ((camera.getContrastCompensation() == 0) && (camera.getBrightnessCompensation() == 0)) {
			return null;
		}
		float scaleFactor = (float) Math.pow(3.0, ((double) camera.getContrastCompensation()) / 100.0);
		float offset = camera.getBrightnessCompensation();

		ColorModel colorModel = deviceImage.getColorModel();
		int componentCount = colorModel.getNumComponents();
		boolean hasAlpha = colorModel.hasAlpha();

		float[] scaleFactors = new float[componentCount];
		float[] offsets = new float[componentCount];

		for (int i = 0; i < colorModel.getNumComponents(); i++) {
			scaleFactors[i] = scaleFactor;
			offsets[i] = offset;
		}
		if (hasAlpha) {
			scaleFactors[componentCount - 1] = 1f;
			offsets[componentCount - 1] = 255f;
		}

		return new RescaleOp(scaleFactors, offsets, null);
	}

	protected void settingsUpdated() {
	}

	@Override
	public String toString() {
		Class<?> theClass = getClass();
		while (theClass.isAnonymousClass()) {
			theClass = theClass.getSuperclass();
		}
		return theClass.getSimpleName() + " [" + getName() + "]";
	}

}
