/*******************************************************************************
 * Copyright (C) 2018 OTK Software
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package com.otk.application.image.camera;

import java.awt.Image;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.swing.JOptionPane;
import com.otk.application.error.UnexpectedError;
import com.otk.application.image.video.MJPEGVideoSequence;
import com.otk.application.util.Accessor;
import com.otk.application.util.CommandExecutor;
import com.otk.application.util.CommandLineInterface;
import com.otk.application.util.ImageUtils;
import com.otk.application.util.Listener;

/**
 * {@link Camera} driver based on the ffmpeg command.
 * 
 * TYPICAL COMMANDS USED:
 * <ul>
 * 
 * <li>List devices: v4l2-ctl --list-devices</li>
 * 
 * <li>List encoding/resolutions: ffmpeg -f video4linux2 -list_formats all -i
 * /dev/video0</li>
 * 
 * <li>Streaming: ffmpeg -f video4linux2 -s 320x240 -i /dev/video0 -f mjpeg - |
 * ffplay -</li>
 * 
 * </ul>
 * 
 * @author olitank
 *
 */
public class FFMpegDriver extends AbstractCommandLineBasedCameraDriver {

	public static void main(String[] args) {
		Camera camera = new Camera() {
			@Override
			public List<AbstractCameraDriver> createDrivers() {
				return Collections.singletonList(new FFMpegDriver(this));
			}
		};
		List<String> devices = camera.listAvailableDevices(null);
		camera.setDeviceFullName(devices.get(0));
		List<FrameFormat> formats = camera.getVideoFormats();
		camera.setVideoFormat(formats.get(0));
		camera.start();
		JOptionPane.showMessageDialog(null, camera.getCurrentImage());
		camera.stop();
	}

	private BufferedImage lastImage;

	public FFMpegDriver(Camera camera) {
		super(camera);
	}

	@Override
	public String getName() {
		return "Video4Linux";
	}

	@Override
	protected String cleanLogs(String outputLogs) {
		return outputLogs;
	}

	@Override
	protected CameraControlInterface<Void> getCompatibilityCheckingCommandInterface() {
		return new CameraControlInterface<Void>(new File("ffmpeg")) {
			{
				setCommandOptions("-version");
			}

			@Override
			protected Void doCommunicate(Process process) {
				waitSafelyFor(process, getSafeReadingTimeoutMilliseconds());
				if (process.exitValue() != 0) {
					throw new UnexpectedError("Invalid status code");
				}
				return null;
			}
		};
	}

	@Override
	protected CameraControlInterface<List<String>> getDeviceListingCommandInterface() {
		return new CameraControlInterface<List<String>>(new File("v4l2-ctl")) {
			{
				setCommandOptions("--list-devices");
			}

			@Override
			protected List<String> doCommunicate(Process process) {
				acquireSafelyAvailableOutputData();
				waitSafelyFor(process, getSafeReadingTimeoutMilliseconds());
				if (process.exitValue() != 0) {
					throw new UnexpectedError("Invalid status code");
				}
				List<String> result = new ArrayList<String>();
				String output = new String(getOutputBuffer().toByteArray());
				output = output.trim();
				if (output.length() > 0) {
					Pattern pattern = Pattern.compile("(.*):\\s+(.*)");
					Matcher macther = pattern.matcher(output);
					while (macther.find()) {
						if (macther.groupCount() != 2) {
							throw new UnexpectedError("Invalid output");
						}
						String deviceName = macther.group(1);
						String deviceId = macther.group(2);
						if (deviceName.contains("v4l2loopback")) {
							continue;
						}
						result.add(deviceId);
					}
				}
				return result;
			}
		};
	}

	@Override
	protected CommandLineInterface<List<FrameFormat>> getVideoFormatListingCommandInterface(String localDeviceName) {
		return new CameraControlInterface<List<FrameFormat>>(new File("ffmpeg")) {
			{
				setCommandOptions(
						"-f video4linux2 -list_formats all -i " + CommandExecutor.quoteArgument(localDeviceName));
			}

			@Override
			protected List<FrameFormat> doCommunicate(Process process) {

				acquireSafelyAvailableErrorData();
				waitSafelyFor(process, getSafeReadingTimeoutMilliseconds());

				String output = new String(getErrorBuffer().toByteArray());
				if (!output.contains("Immediate exit requested")) {
					throw new UnexpectedError("Invalid output");
				}

				SortedSet<FrameFormat> result = new TreeSet<FrameFormat>();
				output = output.trim();
				if (output.length() > 0) {
					Pattern pattern = Pattern.compile("([^ ][0-9]+)x([0-9]+[$ ])");
					Matcher macther = pattern.matcher(output);
					while (macther.find()) {
						if (macther.groupCount() != 2) {
							throw new UnexpectedError("Invalid output");
						}
						int width = Integer.valueOf(macther.group(1).trim());
						int height = Integer.valueOf(macther.group(2).trim());
						result.add(new FrameFormat(width, height));
					}
				}
				return new ArrayList<FrameFormat>(result);
			}
		};
	}

	@Override
	protected CameraControlInterface<Void> getLiveViewCommandInterface(String deviceLocalName, FrameFormat format,
			final Listener<BufferedImage> deviceImageListener, final Accessor<Boolean> interruptionRequested) {
		return new CameraControlInterface<Void>(new File("ffmpeg")) {
			{
				setCommandOptions("-f video4linux2 -s " + format.width + "x" + format.height + " -i "
						+ CommandExecutor.quoteArgument(deviceLocalName) + " -f mjpeg -");
			}

			@Override
			protected Void doCommunicate(Process process) {
				if (-1 == acquireSafelyErrorDataUntil(process, "Press [q] to stop, [?] for help\n".getBytes())) {
					camera.handleRunErrorAndStop(new UnexpectedError("Invalid output"));
					return null;
				}
				try {
					MJPEGVideoSequence sequence = new MJPEGVideoSequence(
							new MJPEGVideoSequence.IMJPEGPStreamProvider() {
								@Override
								public InputStream acquireInputStream() {
									return process.getInputStream();
								}

								@Override
								public void releaseInputStream(InputStream inputStream) {
								}

								@Override
								public float getActualFps() {
									return 0;
								}

							}, 0f);
					Iterator<Image> frameIterator = sequence.createFrameIterator();
					while (true) {
						if (interruptionRequested.get()) {
							process.getOutputStream().write("q".getBytes());
							process.getOutputStream().flush();
							acquireSafelyAvailableOutputData();
							waitSafelyFor(process, getSafeReadingTimeoutMilliseconds());
							break;
						}
						if (!frameIterator.hasNext()) {
							throw new UnexpectedError("Failed to get the next frame");
						}
						lastImage = ImageUtils.getBufferedImage(frameIterator.next());
						deviceImageListener.handle(lastImage);
					}

				} catch (Throwable t) {
					camera.handleRunErrorAndStop(t);
				} finally {
					lastImage = null;
				}
				return null;
			}
		};
	}

	@Override
	protected boolean isErrorDiagnosticEnabled() {
		return true;
	}

	@Override
	public boolean isActive() {
		return true;
	}

	@Override
	public void setActive(boolean b) {
		throw new UnsupportedOperationException();
	}

}
