/*******************************************************************************
 * Copyright (C) 2018 OTK Software
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package com.otk.application.image.camera;

import java.awt.image.BufferedImage;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.swing.JOptionPane;

import org.openimaj.image.ImageUtilities;
import org.openimaj.image.MBFImage;
import org.openimaj.video.capture.Device;
import org.openimaj.video.capture.VideoCapture;

import com.otk.application.error.AbstractApplicationError;
import com.otk.application.error.StandardError;
import com.otk.application.error.UnexpectedError;
import com.otk.application.util.AbstractSingleThreadModule;
import com.otk.application.util.MiscUtils;

/**
 * {@link Camera} driver based on the OpenIMAJ library.
 * 
 * @author olitank
 *
 */
public class OpenIMAJDriver extends WebcamDriver {

	public static void main(String[] args) {
		Camera camera = new Camera() {
			@Override
			public List<AbstractCameraDriver> createDrivers() {
				return Collections.singletonList(new OpenIMAJDriver(this));
			}
		};
		List<String> devices = camera.listAvailableDevices(null);
		camera.setDeviceFullName(devices.get(devices.size() - 1));
		camera.setVideoFormat(camera.getVideoFormats().get(0));
		camera.start();
		JOptionPane.showMessageDialog(null, camera.getCurrentImage());
		camera.stop();
	}

	private static final int THREAD_PRIORITY = Thread.NORM_PRIORITY;
	private static final SingleThreadedDeviceAccess SINGLE_THREADED_DEVICE_ACCESS = new SingleThreadedDeviceAccess();

	private Device device;
	private VideoCapture videoCapture;
	private List<FrameFormat> deviceSupportedFormats = new ArrayList<FrameFormat>();
	private List<Device> videoDeviceList;

	protected boolean verbose = false;

	public OpenIMAJDriver(Camera camera) {
		super(camera);
	}

	@Override
	public void doSetUp() throws Exception {
		withoutFreezing(new Runnable() {
			@Override
			public void run() {
				if (verbose) {
					camera.logInfo("Initializing " + OpenIMAJDriver.this);
				}
				SINGLE_THREADED_DEVICE_ACCESS.setCurrentDriver(OpenIMAJDriver.this);
				SINGLE_THREADED_DEVICE_ACCESS.updateVideoDeviceList();
				for (Device candidateDevice : videoDeviceList) {
					if (verbose) {
						camera.logInfo("Found camera device: '" + candidateDevice.getNameStr() + "'");
					}
					if (camera.getLocalDeviceName().equals(candidateDevice.getNameStr())) {
						device = candidateDevice;
					}
				}
				if (device == null) {
					throw new StandardError("Camera device not found: '" + camera.getDeviceFullName() + "'");
				}
				updateDeviceSupportedFormats();
			}
		});
	}

	private void updateDeviceSupportedFormats() {
		deviceSupportedFormats = FrameFormat.COMMON;

	}

	@Override
	public void startCapture(FrameFormat format) {
		if (verbose) {
			camera.logInfo("Starting " + OpenIMAJDriver.this + " capture");
		}
		withoutFreezing(new Runnable() {
			@Override
			public void run() {
				try {
					SINGLE_THREADED_DEVICE_ACCESS.createVideoCapture(format);
				} catch (Exception e) {
					throw new StandardError("Camera video capture startup error: " + e.toString(), e);
				}
			}
		});
	}

	@Override
	public void stopCapture() {
		if (verbose) {
			camera.logInfo("Stopping " + OpenIMAJDriver.this + " capture");
		}
		withoutFreezing(new Runnable() {
			@Override
			public void run() {
				SINGLE_THREADED_DEVICE_ACCESS.destroyVideoCapture();
			}
		});
	}

	@Override
	public void doWaitForCompleteInterruption() {
		SINGLE_THREADED_DEVICE_ACCESS.waitForVideoCaptureTermination();
	}

	@Override
	public void doCleanUp() throws Exception {
		if (verbose) {
			camera.logInfo("Finalizing " + OpenIMAJDriver.this);
		}
		device = null;
		deviceSupportedFormats = null;
		videoCapture = null;
		SINGLE_THREADED_DEVICE_ACCESS.setCurrentDriver(null);
	}

	@Override
	public List<String> listDeviceLocalNamesWhileCameraNotInitialized() {
		if (verbose) {
			camera.logInfo("Listing devices of " + OpenIMAJDriver.this);
		}
		SINGLE_THREADED_DEVICE_ACCESS.setCurrentDriver(this);
		try {
			SINGLE_THREADED_DEVICE_ACCESS.updateVideoDeviceList();
			List<String> result = new ArrayList<String>();
			for (Device candidateDevice : videoDeviceList) {
				result.add(candidateDevice.getNameStr());
			}
			return result;
		} finally {
			SINGLE_THREADED_DEVICE_ACCESS.setCurrentDriver(null);
		}
	}

	@Override
	public List<FrameFormat> getVideoFormatsWhileCameraInitializedAndNotActive() {
		return deviceSupportedFormats;
	}

	private static class SingleThreadedDeviceAccess extends AbstractSingleThreadModule<String> {

		private OpenIMAJDriver currentDriver;

		public SingleThreadedDeviceAccess() {
			super(THREAD_PRIORITY);
		}

		private String DEVICE_LIST_UPDATE_REQUEST = "DEVICE_LIST_UPDATE_REQUEST";
		private String VIDEO_CAPTURE_CREATION_REQUEST = "VIDEO_CAPTURE_CREATION_REQUEST";
		private String VIDEO_CAPTURE_DESTRUCTION_REQUEST = "VIDEO_CAPTURE_DESTRUCTION_REQUEST";
		private String VIDEO_CAPTURE_TERMINATION_WAIT_REQUEST = "VIDEO_CAPTURE_TERMINATION_WAIT_REQUEST";
		private String FRAME_GRAB_REQUEST = "FRAME_GRAB_REQUEST";

		private Thread frameGrabbingRequestor;
		private boolean videoCaptureOpen = false;
		private FrameFormat captureRequestFormat;

		public void setCurrentDriver(OpenIMAJDriver currentDriver) {
			this.currentDriver = currentDriver;
		}

		@Override
		protected void executeFunction(String request) throws Throwable {
			if (request.equals(DEVICE_LIST_UPDATE_REQUEST)) {
				try {
					currentDriver.videoDeviceList = VideoCapture.getVideoDevices();
				} catch (Throwable t) {
					currentDriver.videoDeviceList = Collections.emptyList();
					throw t;
				}
			} else if (request.equals(VIDEO_CAPTURE_CREATION_REQUEST)) {
				currentDriver.videoCapture = new VideoCapture(captureRequestFormat.getWidth(),
						captureRequestFormat.getHeight(), currentDriver.device);
				videoCaptureOpen = true;
				frameGrabbingRequestor = new Thread(
						MiscUtils.formatThreadName(OpenIMAJDriver.class, "Frame Grabbing Requestor")) {
					@Override
					public void run() {
						while (!isInterrupted()) {
							try {
								callFunction(FRAME_GRAB_REQUEST);
							} catch (InvocationTargetException e) {
								currentDriver.camera.handleRunErrorAndStop(e.getTargetException());
							}
						}
					}
				};
				frameGrabbingRequestor.setDaemon(true);
				frameGrabbingRequestor.setPriority(THREAD_PRIORITY);
				frameGrabbingRequestor.start();
			} else if (request.equals(VIDEO_CAPTURE_DESTRUCTION_REQUEST)) {
				videoCaptureOpen = false;
				frameGrabbingRequestor.interrupt();
				currentDriver.videoCapture.stopCapture();
				currentDriver.videoCapture.close();
			} else if (request.equals(VIDEO_CAPTURE_TERMINATION_WAIT_REQUEST)) {
				while (frameGrabbingRequestor.isAlive()) {
					frameGrabbingRequestor.interrupt();
					MiscUtils.relieveCPU();
				}
				videoCaptureOpen = false;
				frameGrabbingRequestor.interrupt();
				currentDriver.videoCapture.stopCapture();
				currentDriver.videoCapture.close();
			} else if (request.equals(FRAME_GRAB_REQUEST)) {
				if (videoCaptureOpen) {
					try {
						MBFImage frame;
						try {
							frame = currentDriver.videoCapture.getNextFrame();
						} catch (Throwable t) {
							frame = currentDriver.videoCapture.getNextFrame();
						}
						BufferedImage image = ImageUtilities.createBufferedImageForDisplay(frame);
						image = currentDriver.adaptDeviceImage(image);
						currentDriver.camera.handleNewImageFromDevice(image);
					} catch (Throwable t) {
						final Throwable finalError;
						if (t.toString().contains("Timed out waiting for next frame")) {
							finalError = new StandardError(
									t.getMessage() + " \n(This issue may be caused by an invalid resolution)");
						} else {
							finalError = t;
						}
						AbstractApplicationError.rethrow(finalError);
					}
				}
			} else {
				throw new UnexpectedError();
			}
		}

		public void updateVideoDeviceList() {
			try {
				callFunction(DEVICE_LIST_UPDATE_REQUEST);
			} catch (InvocationTargetException e) {
				throw new UnexpectedError(e.getTargetException());
			}
		}

		public void createVideoCapture(FrameFormat format) {
			try {
				captureRequestFormat = format;
				callFunction(VIDEO_CAPTURE_CREATION_REQUEST);
			} catch (InvocationTargetException e) {
				throw new UnexpectedError(e.getTargetException());
			}
		}

		public void destroyVideoCapture() {
			try {
				callFunction(VIDEO_CAPTURE_DESTRUCTION_REQUEST);
			} catch (InvocationTargetException e) {
				throw new UnexpectedError(e.getTargetException());
			}
		}

		public void waitForVideoCaptureTermination() {
			try {
				callFunction(VIDEO_CAPTURE_TERMINATION_WAIT_REQUEST);
			} catch (InvocationTargetException e) {
				throw new UnexpectedError(e.getTargetException());
			}
		}
	}

}
