/*******************************************************************************
 * Copyright (C) 2018 OTK Software
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package com.otk.application.image.filter;

import java.awt.Color;
import java.awt.Point;
import java.awt.image.BufferedImage;

import com.otk.application.util.ImageUtils;

public abstract class AbstractSimpleFilter extends AbstractFilter {

	protected abstract int getNewPixelColorComponentValue(char colorComponent, BufferedImage srcImage,
			Point pixelPosition);

	protected abstract boolean isPixelAlterable(BufferedImage srcImage, Point pixelPosition);

	protected OptimizationCache optimizationCache = new OptimizationCache();

	@Override
	public FilteringContext doExecute(FilteringContext context) {
		BufferedImage srcImage = context.getImage();
		BufferedImage destImage = new BufferedImage(srcImage.getWidth(), srcImage.getHeight(),
				ImageUtils.getAdaptedBufferedImageType());
		for (int i = 0; i < srcImage.getWidth(); i++) {
			for (int j = 0; j < srcImage.getHeight(); j++) {
				Point pixelPosition = new Point(i, j);
				if (isPixelAlterable(srcImage, pixelPosition)) {
					Color destColor = optimizationCache.get(srcImage, pixelPosition);
					if (destColor == null) {
						destColor = getNewPixelColor(srcImage, pixelPosition);
						optimizationCache.put(srcImage, pixelPosition, destColor);
					}
					destImage.setRGB(i, j, destColor.getRGB());
				} else {
					destImage.setRGB(i, j, srcImage.getRGB(i, j));
				}
			}
		}
		return context.withImage(destImage);
	}

	protected Color getNewPixelColor(BufferedImage srcImage, Point pixelPosition) {
		int destRed = getNewPixelColorComponentValue('r', srcImage, pixelPosition);
		int destGreen = getNewPixelColorComponentValue('g', srcImage, pixelPosition);
		int destBlue = getNewPixelColorComponentValue('b', srcImage, pixelPosition);
		destRed = Math.min(255, Math.max(0, destRed));
		destGreen = Math.min(255, Math.max(0, destGreen));
		destBlue = Math.min(255, Math.max(0, destBlue));
		Color srcColor = new Color(srcImage.getRGB(pixelPosition.x, pixelPosition.y), true);
		return new Color(destRed, destGreen, destBlue, srcColor.getAlpha());
	}

	protected Object getOptimizationCacheKey(BufferedImage srcImage, Point pixelPosition) {
		return null;
	}

	protected class OptimizationCache extends AbstractOptimizationCache {

		public synchronized Color get(BufferedImage srcImage, Point pixelPosition) {
			Object key = getOptimizationCacheKey(srcImage, pixelPosition);
			if (key == null) {
				return null;
			}
			return get(key);
		}

		public synchronized void put(BufferedImage srcImage, Point pixelPosition, Color newPixelColor) {
			Object key = getOptimizationCacheKey(srcImage, pixelPosition);
			if (key == null) {
				return;
			}
			put(key, newPixelColor);
		}

	}

}
