/*******************************************************************************
 * Copyright (C) 2018 OTK Software
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package com.otk.application.image.video;

import java.awt.Image;
import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Iterator;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import javax.imageio.ImageIO;
import javax.imageio.ImageReader;
import javax.imageio.stream.ImageInputStream;

import com.otk.application.error.UnexpectedError;
import com.otk.application.image.camera.NOCamDriver;
import com.otk.application.util.ImageUtils;
import com.otk.application.util.MiscUtils;

/**
 * Video streaming class able to process MPEG data.
 * 
 * @author olitank
 *
 */
public class MJPEGVideoSequence implements IVideoSequence {

	public static void main(String[] args) throws Throwable {
		ByteArrayOutputStream output = new ByteArrayOutputStream();
		for (int i = 0; i < 100; i++) {
			ImageIO.write(NOCamDriver.NO_CAMERA_IMAGE, "jpg", output);
			ImageIO.write(ImageUtils.NULL_IMAGE, "jpg", output);
		}
		MJPEGVideoSequence sequence = new MJPEGVideoSequence(new IMJPEGPStreamProvider() {

			@Override
			public void releaseInputStream(InputStream inputStream) {
			}

			@Override
			public InputStream acquireInputStream() {
				return new ByteArrayInputStream(output.toByteArray());
			}

			@Override
			public float getActualFps() {
				return 0;
			}
		}, 10f);
		Iterator<Image> it = sequence.createFrameIterator();
		if (it.next().getWidth(null) != NOCamDriver.NO_CAMERA_IMAGE.getWidth()) {
			throw new UnexpectedError();
		}
		if (it.next().getWidth(null) != ImageUtils.NULL_IMAGE.getWidth()) {
			throw new UnexpectedError();
		}
	}

	private IMJPEGPStreamProvider streamProvider;
	private float requiredFps;

	public MJPEGVideoSequence(IMJPEGPStreamProvider source, float requiredFps) {
		this.streamProvider = source;
		this.requiredFps = requiredFps;
	}

	@Override
	public float getFps() {
		return requiredFps;
	}

	@Override
	public Iterator<Image> createFrameIterator() {
		return new MJPEGFrameIterator(this);
	}

	@Override
	public void disposeFrameIterator(Iterator<Image> frameIterator) {
		((MJPEGFrameIterator) frameIterator).dispose();
	}

	public static interface IMJPEGPStreamProvider {

		InputStream acquireInputStream();

		void releaseInputStream(InputStream inputStream);

		float getActualFps();

	}

	protected static class MJPEGFrameIterator implements Iterator<Image> {

		private static final int STREAM_READ_TIMEOUT = 5000;
		private static final byte[] JPEG_END = new byte[] { (byte) 0xFF, (byte) 0xD9 };

		protected MJPEGVideoSequence sequence;
		protected InputStream input;
		protected int skippedImageReaderErrors = 0;
		protected ExecutorService safeReadingExecutor = Executors.newSingleThreadExecutor(new ThreadFactory() {
			@Override
			public Thread newThread(Runnable r) {
				Thread result = new Thread(r, MiscUtils.formatThreadName(MJPEGVideoSequence.class, "StreamReader"));
				result.setDaemon(true);
				return result;
			}
		});;
		protected BufferedImage nextImage;
		protected long lastImageByteCount = -1;
		protected float alteredFps;

		public MJPEGFrameIterator(MJPEGVideoSequence sequence) {
			this.sequence = sequence;
			this.input = sequence.streamProvider.acquireInputStream();
			alteredFps = sequence.requiredFps;
			prepareNext();
		}

		protected void prepareNext() {
			try {
				final ByteArrayOutputStream imageDataBuffer = new ByteArrayOutputStream();
				while (isNextImageSkippingRequired()) {
					if (!readQuicklyMostOfNextImageBytes(imageDataBuffer)) {
						nextImage = null;
						return;
					}
				}
				if (!readQuicklyMostOfNextImageBytes(imageDataBuffer)) {
					nextImage = null;
					return;
				}
				if (!readNextImageBytes(imageDataBuffer)) {
					nextImage = null;
					return;
				}
				try {
					nextImage = loadNextImage(imageDataBuffer);
					if (nextImage == null) {
						throw new UnexpectedError();
					}
					skippedImageReaderErrors = 0;
				} catch (Throwable t) {
					skippedImageReaderErrors++;
					if (skippedImageReaderErrors == 10) {
						throw new UnexpectedError("Persistent image reader error detected: " + t, t);
					}
					if (-1 == MiscUtils.readUntil(safeReadingExecutor, input, MiscUtils.getNullOutputStream(), 5000,
							null, JPEG_END)) {
						throw new UnexpectedError("Invalid MJPEG format detected");
					}
					prepareNext();
				}
			} catch (Throwable t) {
				nextImage = null;
				throw new UnexpectedError(t);
			} finally {
				if (nextImage == null) {
					dispose();
				}
			}
		}

		protected boolean isNextImageSkippingRequired() {
			float actualFps = sequence.streamProvider.getActualFps();
			float FPS_ERROR_TOLERANCE = 5.0f;
			if ((actualFps <= 0) || (actualFps >= (sequence.requiredFps - FPS_ERROR_TOLERANCE))) {
				return false;
			}
			if (alteredFps < actualFps) {
				alteredFps += (sequence.requiredFps - alteredFps) / 2.0f;
				return true;
			} else {
				alteredFps += (0 - alteredFps) / 2.0f;
				return false;
			}
		}

		protected BufferedImage loadNextImage(ByteArrayOutputStream imageDataBuffer) throws IOException {
			ImageInputStream imageStream = ImageIO
					.createImageInputStream(new ByteArrayInputStream(imageDataBuffer.toByteArray()));
			Iterator<ImageReader> readerIterator = ImageIO.getImageReaders(imageStream);
			ImageReader reader = readerIterator.next();
			reader.setInput(imageStream);
			nextImage = reader.read(0);
			if (nextImage != null) {
				lastImageByteCount = imageStream.getStreamPosition();
			}
			return nextImage;
		}

		protected boolean readNextImageBytes(ByteArrayOutputStream imageDataBuffer) throws IOException {
			return -1 != MiscUtils.readUntil(safeReadingExecutor, input, imageDataBuffer, STREAM_READ_TIMEOUT, null,
					JPEG_END);
		}

		protected boolean readQuicklyMostOfNextImageBytes(ByteArrayOutputStream imageDataBuffer)
				throws TimeoutException {
			if (lastImageByteCount == -1) {
				return true;
			}
			return readQuicklyNextImageBytes(imageDataBuffer, Math.round(lastImageByteCount * 0.9));
		}

		protected boolean skipRoughlyNextImageBytes(ByteArrayOutputStream imageDataBuffer) throws TimeoutException {
			if (lastImageByteCount == -1) {
				return true;
			}
			return readQuicklyNextImageBytes(imageDataBuffer, lastImageByteCount);
		}

		protected boolean readQuicklyNextImageBytes(ByteArrayOutputStream imageDataBuffer, final long bytesToRead)
				throws TimeoutException {
			final boolean[] eof = new boolean[] { false };
			if (!MiscUtils.runWithTimeout(new Runnable() {

				byte[] readBuffer = new byte[1024];

				@Override
				public void run() {
					long remainingBytesToRead = bytesToRead;
					while (remainingBytesToRead > 0) {
						try {
							int bytesRead;
							if (remainingBytesToRead >= readBuffer.length) {
								bytesRead = input.read(readBuffer);
								if (bytesRead == -1) {
									eof[0] = true;
									break;
								}
								imageDataBuffer.write(readBuffer);
							} else {
								bytesRead = input.read(readBuffer, 0, (int) remainingBytesToRead);
								if (bytesRead == -1) {
									eof[0] = true;
									break;
								}
								imageDataBuffer.write(readBuffer, 0, (int) remainingBytesToRead);
							}
							remainingBytesToRead -= bytesRead;
						} catch (IOException e) {
							throw new UnexpectedError(e);
						}
					}
				}
			}, STREAM_READ_TIMEOUT, safeReadingExecutor)) {
				throw new TimeoutException();
			}
			if (eof[0]) {
				nextImage = null;
				return false;
			}
			return true;
		}

		@Override
		public Image next() {
			Image result = nextImage;
			prepareNext();
			return result;
		}

		@Override
		public boolean hasNext() {
			return nextImage != null;
		}

		public void dispose() {
			safeReadingExecutor.shutdown();
			try {
				safeReadingExecutor.awaitTermination(STREAM_READ_TIMEOUT * 2, TimeUnit.MILLISECONDS);
			} catch (InterruptedException e) {
				throw new UnexpectedError(e);
			}
			sequence.streamProvider.releaseInputStream(input);
		}
	}

}
