/*******************************************************************************
 * Copyright (C) 2018 OTK Software
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package com.otk.application.util;

import java.lang.reflect.InvocationTargetException;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import com.otk.application.error.UnexpectedError;

public abstract class AbstractSingleThreadModule<R> {

	private Lock requestMutex = new ReentrantLock(true);
	private Thread theThread;
	private boolean threadSleeping = false;
	private boolean disposeRequested = false;
	private boolean disposed = false;

	private R activeRequest;
	private Throwable lastRequestError;

	protected abstract void executeFunction(R request) throws Throwable;

	protected AbstractSingleThreadModule(ThreadFactory threadFactory) {
		theThread = threadFactory.newThread(new Runnable() {
			@Override
			public void run() {
				process();
			}
		});
		theThread.start();
		while (!threadSleeping) {
			MiscUtils.relieveCPU();
		}
	}

	protected AbstractSingleThreadModule(int threadPriority) {
		this(new ThreadFactory() {
			@Override
			public Thread newThread(Runnable r) {
				Thread result = new Thread(r,
						MiscUtils.formatThreadName(AbstractSingleThreadModule.class, "FunctionExecutor"));
				result.setDaemon(true);
				result.setPriority(threadPriority);
				return result;
			}
		});
	}

	@Override
	protected void finalize() throws Throwable {
		if (!disposed) {
			dispose();
		}
		super.finalize();
	}

	public void dispose() {
		requestMutex.lock();
		try {
			if (disposed) {
				throw new UnexpectedError();
			}
			disposeRequested = true;
			wakeUpTheThread();
			while (theThread.isAlive()) {
				MiscUtils.relieveCPU();
			}
			disposed = true;
		} finally {
			requestMutex.unlock();
		}
	}

	public boolean isDisposed() {
		return disposed;
	}

	protected void callFunction(R request) throws InvocationTargetException {
		requestMutex.lock();
		try {
			if (disposed) {
				throw new UnexpectedError();
			}
			activeRequest = request;
			wakeUpTheThread();
			while (activeRequest != null) {
				if (Thread.currentThread().isInterrupted()) {
					theThread.interrupt();
				}
				if (!theThread.isAlive()) {
					break;
				}
				MiscUtils.relieveCPU();
			}
			while (!threadSleeping) {
				if (Thread.currentThread().isInterrupted()) {
					theThread.interrupt();
				}
				if (!theThread.isAlive()) {
					break;
				}
				MiscUtils.relieveCPU();
			}
			if (lastRequestError != null) {
				throw new InvocationTargetException(lastRequestError);
			}
		} finally {
			requestMutex.unlock();
		}
	}

	private void process() {
		while (!disposeRequested) {
			if (activeRequest != null) {
				try {
					executeFunction(activeRequest);
					lastRequestError = null;
				} catch (Throwable t) {
					lastRequestError = t;
				} finally {
					activeRequest = null;
				}
			} else {
				sleepFromTheThread();
			}
		}
	}

	private void sleepFromTheThread() {
		threadSleeping = true;
		try {
			MiscUtils.sleepSafely(Long.MAX_VALUE);
			MiscUtils.clearCurrentThreadInterruptedStatus();
		} finally {
			threadSleeping = false;
		}
	}

	private void wakeUpTheThread() {
		theThread.interrupt();
	}

	public void waitForTermination() throws InterruptedException {
		theThread.join();
	}

}
