/*******************************************************************************
 * Copyright (C) 2018 OTK Software
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package com.otk.application.util;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintStream;

import com.otk.application.error.StandardError;
import com.otk.application.error.UnexpectedError;

public class Command {

	private String commandLine;
	private boolean wait = true;
	private OutputStream out;
	private OutputStream err;
	private File workingDir;
	private OutputStream logOut;
	private OutputStream logErr;
	private Process process;

	public Command(String commandLine) {
		this.commandLine = commandLine;
	}

	public Command() {
	}

	public String getCommandLine() {
		return commandLine;
	}

	public void setCommandLine(String commandLine) {
		this.commandLine = commandLine;
	}

	public boolean isWait() {
		return wait;
	}

	public void setWait(boolean wait) {
		this.wait = wait;
	}

	public OutputStream getOut() {
		return out;
	}

	public void setOut(OutputStream out) {
		this.out = out;
	}

	public OutputStream getErr() {
		return err;
	}

	public void setErr(OutputStream err) {
		this.err = err;
	}

	public OutputStream getLogOut() {
		return logOut;
	}

	public void setLogOut(OutputStream logStream) {
		this.logOut = logStream;
	}

	public OutputStream getLogErr() {
		return logErr;
	}

	public void setLogErr(OutputStream logStream) {
		this.logErr = logStream;
	}

	public File getWorkingDir() {
		return workingDir;
	}

	public void setWorkingDir(File workingDir) {
		this.workingDir = workingDir;
	}

	public Process getLaunchedProcess() {
		return process;
	}

	public static Process run(final String commandLine, boolean wait, final OutputStream outReceiver,
			final OutputStream errReceiver, File workingDir) {
		Command cmd = new Command(commandLine);
		cmd.setWait(wait);
		cmd.setOut(outReceiver);
		cmd.setErr(errReceiver);
		cmd.setLogOut(Log.getStream(Log.Level.INFO));
		cmd.setLogErr(Log.getStream(Log.Level.ERROR));
		cmd.setWorkingDir(workingDir);
		cmd.run();
		return cmd.getLaunchedProcess();
	}

	public void run() {
		if (isOutputlogged()) {
			logOut("Running command:\n" + commandLine);
		}
		try {
			process = Runtime.getRuntime().exec(commandLine, null, workingDir);
		} catch (Exception e1) {
			throw new StandardError("Command execution error: " + e1.toString(), e1);
		}

		final Thread outputRedirector = redirectStream(process.getInputStream(), out, logOut, "Output");
		final Thread errorRedirector = redirectStream(process.getErrorStream(), err, logErr, "Error");
		Thread redirectionsCleaner = new Thread(
				MiscUtils.formatThreadName(MiscUtils.class, "Redirections Cleaner for " + getCommandDescription())) {
			@Override
			public void run() {
				postClean(outputRedirector, errorRedirector);
			}
		};
		redirectionsCleaner.start();
		if (!wait) {
			return;
		}
		waitProcessEnd(redirectionsCleaner);
	}

	private void waitProcessEnd(Thread cleaner) {
		try {
			cleaner.join();
		} catch (InterruptedException e) {
			Thread.currentThread().interrupt();
		}

		if (Thread.currentThread().isInterrupted()) {
			process.destroy();
			if (isErrorlogged()) {
				logErr("Command execution interrupted");
			}
			return;
		} else {
			if (isOutputlogged()) {
				logOut("Command execution terminated");
			}
			return;
		}
	}

	private void postClean(Thread outputRedirector, Thread errorRedirector) {
		try {
			process.waitFor();
		} catch (Exception e) {
			throw new UnexpectedError(e);
		}
		MiscUtils.sleepSafely(1000);
		for (Thread thread : new Thread[] { outputRedirector, errorRedirector }) {
			if (thread.isAlive()) {
				thread.interrupt();
				while (thread.isAlive()) {
					MiscUtils.relieveCPU();
				}
			}
		}
		try {
			process.getOutputStream().close();
			process.getInputStream().close();
			process.getErrorStream().close();
		} catch (IOException e) {
			throw new AssertionError(e);
		}
	}

	private Thread redirectStream(InputStream srcStream, OutputStream dstStream, OutputStream dstLogStream,
			String streamName) {
		String reason = streamName + " Stream Consumption for " + getCommandDescription();
		if ((dstStream == null) && (dstLogStream == null)) {
			return new Thread("Null");
		}
		if (dstStream == null) {
			dstStream = dstLogStream;
			dstLogStream = null;
		}
		if (dstLogStream != null) {
			dstStream = MiscUtils.unifyOutputStreams(dstStream, dstLogStream);
		}
		return MiscUtils.redirectStream(srcStream, dstStream, reason);
	}

	private String getCommandDescription() {
		return "Command: " + TextUtils.truncateNicely(commandLine, 50);
	}

	public boolean isOutputlogged() {
		return logOut != null;
	}

	public boolean isErrorlogged() {
		return logErr != null;
	}

	private void logOut(String string) {
		PrintStream printStream = new PrintStream(logOut);
		printStream.println(string);
		printStream.close();
	}

	private void logErr(String string) {
		PrintStream printStream = new PrintStream(logErr);
		printStream.println(string);
		printStream.close();
	}

	public static Process run(final String command, boolean wait, final OutputStream outReceiver,
			final OutputStream errReceiver) {
		return run(command, wait, outReceiver, errReceiver, null);
	}

}
