/*******************************************************************************
 * Copyright (C) 2018 OTK Software
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package com.otk.application.util;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.Vector;

import com.otk.application.error.StandardError;
import com.otk.application.error.UnexpectedError;

public class CommandExecutor {

	private String commandLine;
	private OutputStream processOutputRedirectedTo;
	private OutputStream processErrorRedirectedTo;
	private File workingDir;
	private Map<String, String> env;
	private Process process;
	private Thread outputRedirector;
	private Thread errorRedirector;

	public CommandExecutor(String commandLine) {
		this.commandLine = commandLine;
	}

	public CommandExecutor() {
	}

	public String getCommandLine() {
		return commandLine;
	}

	public void setCommandLine(String commandLine) {
		this.commandLine = commandLine;
	}

	public OutputStream getProcessOutputRedirectedTo() {
		return processOutputRedirectedTo;
	}

	public void setProcessOutputRedirectedTo(OutputStream out) {
		this.processOutputRedirectedTo = out;
	}

	public OutputStream getProcessErrorRedirectedTo() {
		return processErrorRedirectedTo;
	}

	public void setProcessErrorRedirectedTo(OutputStream err) {
		this.processErrorRedirectedTo = err;
	}

	public File getWorkingDir() {
		return workingDir;
	}

	public void setWorkingDir(File workingDir) {
		this.workingDir = workingDir;
	}

	public Map<String, String> getEnv() {
		return env;
	}

	public void setEnv(Map<String, String> env) {
		this.env = env;
	}

	public Process getLaunchedProcess() {
		return process;
	}

	public String[] getEnvAsArray() {
		if (env == null) {
			return null;
		}
		List<String> envList = new ArrayList<String>();
		for (Map.Entry<String, String> entry : env.entrySet()) {
			envList.add(entry.getKey() + "=" + entry.getValue());
		}
		return envList.toArray(new String[envList.size()]);
	}

	public void startProcess() {
		try {
			String[] args = splitArguments(commandLine);
			if(args.length == 0) {
				throw new StandardError("Executable file not specified");
			}
			String[] envArray = getEnvAsArray();
			process = Runtime.getRuntime().exec(args , envArray, workingDir);
		} catch (Exception e1) {
			throw new StandardError("Command execution error: " + e1.toString(), e1);
		}
		outputRedirector = getProcessStreamRedirector(process.getInputStream(), processOutputRedirectedTo, "Output");
		errorRedirector = getProcessStreamRedirector(process.getErrorStream(), processErrorRedirectedTo, "Error");
	}

	public boolean isProcessAlive() {
		if (process != null) {
			if (process.isAlive()) {
				return true;
			}
		}
		return false;
	}

	public void waitForProcessEnd() throws InterruptedException {
		process.waitFor();
	}

	public void killProcess() {
		process.destroy();
	}

	public void disconnectProcess() {
		for (Thread thread : new Thread[] { outputRedirector, errorRedirector }) {
			while (thread.isAlive()) {
				thread.interrupt();
				MiscUtils.relieveCPU();
			}
		}
		try {
			process.getOutputStream().close();
		} catch (IOException ignore) {
		}
		try {
			process.getInputStream().close();
		} catch (IOException ignore) {
		}
		try {
			process.getErrorStream().close();
		} catch (IOException ignore) {
		}
	}

	private Thread getProcessStreamRedirector(InputStream processStream, OutputStream destinationStream,
			String processStreamName) {
		if (destinationStream == null) {
			return new Thread("No redirection");
		} else {
			String reason = processStreamName + " stream consumption for " + getCommandDescription();
			return MiscUtils.redirectStream(processStream, destinationStream, reason);
		}
	}

	private String getCommandDescription() {
		return "Command: " + TextUtils.truncateNicely(commandLine, 50);
	}

	@Override
	public String toString() {
		return "CommandExecutor [commandLine=" + commandLine + "]";
	}

	public static Process run(final String commandLine, boolean wait, final OutputStream outReceiver,
			final OutputStream errReceiver, File workingDir) {
		CommandExecutor cmd = new CommandExecutor(commandLine);
		cmd.setProcessOutputRedirectedTo(outReceiver);
		cmd.setProcessErrorRedirectedTo(errReceiver);
		cmd.setWorkingDir(workingDir);
		cmd.startProcess();
		if (wait) {
			try {
				cmd.waitForProcessEnd();
			} catch (InterruptedException e) {
				cmd.killProcess();
			}
			cmd.disconnectProcess();
		} else {
			new Thread(MiscUtils.formatThreadName(CommandExecutor.class, "ProcessMonitor")) {
				@Override
				public void run() {
					try {
						cmd.waitForProcessEnd();
					} catch (InterruptedException e) {
						throw new UnexpectedError(e);
					}
					cmd.disconnectProcess();
				}
			}.start();
		}
		return cmd.getLaunchedProcess();
	}

	public static Process run(final String command, boolean wait, final OutputStream outReceiver,
			final OutputStream errReceiver) {
		return run(command, wait, outReceiver, errReceiver, null);
	}

	public static void run(String command, boolean wait) {
		run(command, wait, System.out, System.err);
	}

	public static String quoteArgument(String argument) {
		String[] argumentSplitByQuotes = argument.split("\"");
		if (argumentSplitByQuotes.length == 1) {
			return "\"" + argument + "\"";
		} else {
			StringBuilder result = new StringBuilder();
			for (int i = 0; i < argumentSplitByQuotes.length; i++) {
				if (i > 0) {
					result.append("'\"'");
				}
				String elementOfArgumentSplitByQuotes = argumentSplitByQuotes[i];
				result.append(quoteArgument(elementOfArgumentSplitByQuotes));
			}
			return result.toString();
		}
	}

	public static String[] splitArguments(String s) {
		if ((s == null) || (s.length() == 0)) {
			return new String[0];
		}
		final int normal = 0;
		final int inQuote = 1;
		final int inDoubleQuote = 2;
		int state = normal;
		StringTokenizer tok = new StringTokenizer(s, "\"\' ", true);
		Vector<String> v = new Vector<String>();
		StringBuilder current = new StringBuilder();

		while (tok.hasMoreTokens()) {
			String nextTok = tok.nextToken();
			switch (state) {
			case inQuote:
				if ("\'".equals(nextTok)) {
					state = normal;
				} else {
					current.append(nextTok);
				}
				break;
			case inDoubleQuote:
				if ("\"".equals(nextTok)) {
					state = normal;
				} else {
					current.append(nextTok);
				}
				break;
			default:
				if ("\'".equals(nextTok)) {
					state = inQuote;
				} else if ("\"".equals(nextTok)) {
					state = inDoubleQuote;
				} else if (" ".equals(nextTok)) {
					if (current.length() != 0) {
						v.addElement(current.toString());
						current.setLength(0);
					}
				} else {
					current.append(nextTok);
				}
				break;
			}
		}

		if (current.length() != 0) {
			v.addElement(current.toString());
		}

		if ((state == inQuote) || (state == inDoubleQuote)) {
			throw new StandardError("unbalanced quotes in " + s);
		}

		String[] args = new String[v.size()];
		v.copyInto(args);
		return args;
	}

}
