/*******************************************************************************
 * Copyright (C) 2018 OTK Software
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package com.otk.application.util;

import java.awt.Component;
import java.awt.Dimension;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.imageio.ImageIO;
import javax.swing.BorderFactory;
import javax.swing.JFileChooser;
import javax.swing.filechooser.FileFilter;

import com.otk.application.TheConstants;
import com.otk.application.error.UnexpectedError;

import xy.reflect.ui.util.component.ImagePanel;

public class FileUtils {

	private static File lastDirectory;

	public static byte[] requireBytes(File file) {
		try {
			return FileUtils.readBinary(file);
		} catch (Exception e) {
			throw new UnexpectedError(e);
		}
	}

	public static File createTempDirectory() throws Exception {
		File baseDir = new File(System.getProperty("java.io.tmpdir"));
		String baseName = System.currentTimeMillis() + "-";
		int TEMP_DIR_ATTEMPTS = 10000;
		for (int counter = 0; counter < TEMP_DIR_ATTEMPTS; counter++) {
			File tempDir = new File(baseDir, baseName + counter);
			if (tempDir.mkdir()) {
				return tempDir;
			}
		}
		throw new Exception("Failed to create directory within " + TEMP_DIR_ATTEMPTS + " attempts (tried " + baseName
				+ "0 to " + baseName + (TEMP_DIR_ATTEMPTS - 1) + ')');
	}

	public static String read(File file) throws Exception {
		return new String(readBinary(file));
	}

	public static String read(InputStream in) throws Exception {
		return new String(readBinary(in));
	}

	public static byte[] readBinary(InputStream in) throws Exception {
		try {
			ByteArrayOutputStream buffer = new ByteArrayOutputStream();
			int nRead;
			byte[] data = new byte[16384];
			while ((nRead = in.read(data, 0, data.length)) != -1) {
				buffer.write(data, 0, nRead);
			}
			buffer.flush();
			return buffer.toByteArray();
		} catch (IOException e) {
			throw new Exception("Error while reading input stream: " + e.getMessage(), e);
		}
	}

	public static byte[] readBinary(File file) throws Exception {
		FileInputStream in = null;
		try {
			in = new FileInputStream(file);
			return readBinary(in);
		} catch (IOException e) {
			throw new Exception("Unable to read file : '" + file.getAbsolutePath() + "': " + e.getMessage(), e);
		} finally {
			if (in != null) {
				try {
					in.close();
				} catch (IOException e) {
				}
			}
		}
	}

	public static void write(File file, String text, boolean append) throws Exception {
		writeBinary(file, text.getBytes(), append);
	}

	public static void writeBinary(File file, byte[] bytes, boolean append) throws Exception {
		FileOutputStream out = null;
		try {
			out = new FileOutputStream(file, append);
			out.write(bytes);
			out.flush();
			out.close();
		} catch (IOException e) {
			throw new Exception("Unable to write file : '" + file.getAbsolutePath() + "': " + e.getMessage(), e);
		} finally {
			if (out != null) {
				try {
					out.close();
				} catch (IOException e) {
				}
			}
		}

	}

	public static void historicizeFile(String filePath, int historySize) {
		for (int i = (historySize - 1); i >= 0; i--) {
			File file;
			if (i == 0) {
				file = new File(filePath);
			} else {
				file = new File(getRotatedFilePath(filePath, i));
			}
			if (file.exists()) {
				try {
					if (i == (historySize - 1)) {
						delete(file);
					} else {
						File nextRotatedFile = new File(getRotatedFilePath(filePath, i + 1));
						try {
							rename(file, nextRotatedFile.getName());
						} catch (Exception e) {
							copy(file, nextRotatedFile);
							delete(file);
						}
					}
				} catch (Exception e) {
					throw new UnexpectedError(e);
				}
			}
		}
	}

	public static String getRotatedFilePath(String filePath, int i) {
		String fileNameExtension = getFileNameExtension(filePath);
		if ((fileNameExtension != null) && (fileNameExtension.length() > 0)) {
			String fileNameWithoutExtension = removeFileNameExtension(filePath);
			return fileNameWithoutExtension + "-" + i + "." + fileNameExtension;
		} else {
			return filePath + "-" + i;
		}
	}

	public static String removeFileNameExtension(String fileName) {
		String extension = getFileNameExtension(fileName);
		if (extension.length() > 0) {
			return fileName.substring(0, fileName.length() - ("." + extension).length());
		} else {
			return fileName;
		}
	}

	public static String getFileNameExtension(String fileName) {
		int lastDotIndex = fileName.lastIndexOf(".");
		if (lastDotIndex == -1) {
			return "";
		} else if (lastDotIndex == 0) {
			return "";
		} else {
			return fileName.substring(lastDotIndex + 1);
		}
	}

	public static String askDirectoryPath(Component parent, String actionTitle, File defaultFile) {
		return askPath(parent, actionTitle, defaultFile, true, null);

	}

	public static String askPath(Component parent, String actionTitle, File defaultFile, boolean directory,
			FileFilter fileFilter) {
		final JFileChooser fileChooser = new JFileChooser();
		if (directory) {
			fileChooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
		}
		if (fileFilter != null) {
			fileChooser.setFileFilter(fileFilter);
		}
		final ImagePanel imagePreview = new ImagePanel();
		imagePreview.setPreservingRatio(true);
		imagePreview.setPreferredSize(new Dimension(100, 100));
		imagePreview.setBorder(BorderFactory.createEtchedBorder());
		fileChooser.addPropertyChangeListener(new PropertyChangeListener() {

			@Override
			public void propertyChange(PropertyChangeEvent evt) {
				imagePreview.setImage(TheConstants.notFoundImage);
				File file = fileChooser.getSelectedFile();
				if (file == null) {
					return;
				}
				if (!file.isFile()) {

					return;
				}
				if (!FileUtils.hasFileNameExtension(file.getName(), ImageUtils.IMAGE_EXTENSIONS)) {
					return;
				}
				try {
					imagePreview.setImage(ImageIO.read(file));
				} catch (Exception e) {
					imagePreview.setImage(TheConstants.notFoundImage);
				}
			}
		});

		fileChooser.setAccessory(imagePreview);
		if (defaultFile != null) {
			fileChooser.setSelectedFile(defaultFile.getAbsoluteFile());
		} else {
			if (lastDirectory != null) {
				fileChooser.setCurrentDirectory(lastDirectory);
			}
		}
		int returnVal = fileChooser.showDialog(parent, actionTitle);
		if (returnVal != JFileChooser.APPROVE_OPTION) {
			return null;
		}
		File file = fileChooser.getSelectedFile();
		String result = file.getPath();
		String relativepath = getRelativePath(file, new File("."));
		if (relativepath != null) {
			result = relativepath;
		}
		result = getOSAgnosticFilePath(result);
		lastDirectory = fileChooser.getCurrentDirectory();
		return result;
	}

	public static String getOSAgnosticFilePath(String path) {
		return path.replace(File.separator, "/");
	}

	public static byte[] requireBinary(File file) {
		try {
			return FileUtils.readBinary(file);
		} catch (Exception e) {
			throw new UnexpectedError(e);
		}
	}

	public static byte[] requireBinary(InputStream in) {
		try {
			return FileUtils.readBinary(in);
		} catch (Exception e) {
			throw new UnexpectedError(e);
		}
	}

	public static void copy(File src, File dst) throws Exception {
		copy(src, dst, true);
	}

	public static void copy(File src, File dst, boolean recusrsively) throws Exception {
		copy(src, dst, recusrsively, null, null);
	}

	public static void copy(File src, File dst, boolean recusrsively, FilenameFilter srcFilesFilter,
			Listener<Pair<File, Exception>> errorHandler) throws Exception {
		try {
			if ((srcFilesFilter != null) && !srcFilesFilter.accept(src.getParentFile(), src.getName())) {
				return;
			}
			if (src.isDirectory()) {
				try {
					createDir(dst);
				} catch (Exception e) {
					if (errorHandler != null) {
						errorHandler.handle(new Pair<File, Exception>(src, e));
					} else {
						throw e;
					}
				}
				if (recusrsively) {
					for (File srcChild : src.listFiles()) {
						copy(srcChild, new File(dst, srcChild.getName()), recusrsively, srcFilesFilter, errorHandler);
					}
				}
			} else if (src.isFile()) {
				try {
					writeBinary(dst, readBinary(src), false);
				} catch (Exception e) {
					if (errorHandler != null) {
						errorHandler.handle(new Pair<File, Exception>(src, e));
					} else {
						throw e;
					}
				}
			} else {
				throw new Exception("File not found: '" + src + "'", null);
			}
		} catch (Exception e) {
			throw new Exception("Unable to copy resource: '" + src.getAbsolutePath() + "' > '" + dst.getAbsolutePath()
					+ "': " + e.getMessage(), e);
		}
	}

	public static void createDir(File dir) throws Exception {
		if (dir.isDirectory()) {
			return;
		}
		final boolean success;
		try {
			success = dir.mkdir();
		} catch (Exception e) {
			throw new Exception("Failed to create directory: '" + dir.getAbsolutePath() + "': " + e.getMessage(), e);
		}
		if (!success) {
			throw new Exception("Unable to create directory: '" + dir.getAbsolutePath() + "'", null);
		}
	}

	public static File relativizeFile(File ancestor, File file) {
		try {
			ancestor = ancestor.getCanonicalFile();
			file = file.getCanonicalFile();
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
		if (!isAncestor(ancestor, file)) {
			return null;
		}
		String relativePath = file.getPath().substring(ancestor.getPath().length(), file.getPath().length());
		if (relativePath.startsWith("/") || relativePath.startsWith("\\")) {
			relativePath = relativePath.substring(1);
		}
		return new File(relativePath);
	}

	public static String getRelativePath(File child, File ancestor) {
		File relativeFile = relativizeFile(ancestor, child);
		if (relativeFile == null) {
			return null;
		}
		return relativeFile.getPath();
	}

	public static boolean canonicallyEquals(File file1, File file2) {
		try {
			return file1.getCanonicalFile().equals(file2.getCanonicalFile());
		} catch (IOException e) {
			throw new RuntimeException(e.toString(), e);
		}
	}

	public static File getCanonicalParent(File file) {
		try {
			return file.getCanonicalFile().getParentFile();
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
	}

	public static void delete(File file) throws Exception {
		delete(file, null, null);
	}

	public static void delete(File file, FilenameFilter filter, Listener<Pair<File, Exception>> errorHandler)
			throws Exception {
		if (file.isDirectory()) {
			for (File childFile : file.listFiles(filter)) {
				delete(childFile, filter, errorHandler);
			}
			if (file.listFiles().length > 0) {
				return;
			}
		}
		boolean success;
		try {
			success = file.delete();
			if (!success) {
				throw new Exception("System error");
			}
		} catch (Exception e) {
			e = new Exception("Failed to delete resource: '" + file.getAbsolutePath() + "': " + e.getMessage(), e);
			if (errorHandler != null) {
				errorHandler.handle(new Pair<File, Exception>(file, e));
			} else {
				throw e;
			}
		}
	}

	public static void rename(File file, String destFileName) throws Exception {
		try {
			if (new File(destFileName).getParent() != null) {
				throw new Exception("Destination file name is not is not a local name: '" + destFileName + "'");
			}
			File destFile = new File(file.getParent(), destFileName);
			boolean success = file.renameTo(destFile);
			if (!success) {
				throw new Exception("System error");
			}
		} catch (Exception e) {
			throw new Exception("Failed to rename resource: '" + file.getAbsolutePath() + "' to '" + destFileName
					+ "': " + e.getMessage(), e);
		}
	}

	public static boolean hasFileNameExtension(String fileName, String[] extensions) {
		for (String ext : extensions) {
			if (ext.toLowerCase().equals(getFileNameExtension(fileName).toLowerCase())) {
				return true;
			}
		}
		return false;
	}

	public static boolean isAncestor(File ancestor, File file) {
		File mayBeAncestor = getCanonicalParent(file);
		while (true) {
			if (mayBeAncestor == null) {
				return false;
			}
			if (canonicallyEquals(mayBeAncestor, ancestor)) {
				return true;
			}
			mayBeAncestor = getCanonicalParent(mayBeAncestor);
		}
	}

	public static void createFile(File file) throws Exception {
		try {
			if (!file.isFile()) {
				if (!file.createNewFile()) {
					throw new UnexpectedError("System error");
				}
			}
		} catch (IOException e) {
			throw new Exception("Failed to create the file '" + file + "': " + e.toString(), e);
		}

	}

	public static File createTemporaryFile() throws Exception {
		try {
			File tmpFile = File.createTempFile(FileUtils.class.getName() + ".", ".tmp");
			tmpFile.deleteOnExit();
			return tmpFile;
		} catch (IOException e) {
			throw new Exception("Failed to create temporary file: " + e.getMessage(), e);
		}
	}

	public static void withFileRestoredInCaseOfError(File file, Runnable runnable) {
		File backupFile = null;
		try {
			backupFile = FileUtils.createTemporaryFile();
			FileUtils.copy(file, backupFile);
			runnable.run();
		} catch (Exception e) {
			if (backupFile != null) {
				try {
					FileUtils.copy(backupFile, file);
				} catch (Exception e1) {
					Log.error(e1.toString());
				}
			}
			throw new UnexpectedError(e);
		} finally {
			if (backupFile != null) {
				try {
					FileUtils.delete(backupFile);
				} catch (Exception e1) {
					Log.error(e1.toString());
				}
			}
		}
	}

	public static List<File> grep(Pattern regex, File fileOrDir, boolean recursive,
			FilenameFilter directoryContentFilter) throws Exception {
		List<File> result = new ArrayList<File>();
		if (fileOrDir.isDirectory()) {
			for (File child : fileOrDir.listFiles(directoryContentFilter)) {
				result.addAll(grep(regex, child, recursive, directoryContentFilter));
			}
		} else if (fileOrDir.isFile()) {
			String fileContent = read(fileOrDir);
			Matcher matcher = regex.matcher(fileContent);
			if (matcher.find()) {
				result.add(fileOrDir);
			}
		}
		return result;
	}

}
