/*******************************************************************************
 * Copyright (C) 2018 OTK Software
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package com.otk.application.util;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Cursor;
import java.awt.Desktop;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.GraphicsEnvironment;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.KeyboardFocusManager;
import java.awt.Rectangle;
import java.awt.Toolkit;
import java.awt.Window;
import java.awt.event.AWTEventListener;
import java.awt.event.AWTEventListenerProxy;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.ContainerEvent;
import java.awt.event.ContainerListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.StringJoiner;
import java.util.Vector;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.swing.JComponent;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSplitPane;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;

import com.otk.application.error.Assertion;
import com.otk.application.error.ErrorManager;
import com.otk.application.error.StandardError;
import com.otk.application.error.UnexpectedError;

public class MiscUtils {

	public static void clearCurrentThreadInterruptedStatus() {
		Thread.interrupted();
	}

	public static JPanel flowInLayout(Component c, int gridBagConstraintsAnchor) {
		return flowInLayout(Collections.singletonList(c), gridBagConstraintsAnchor);
	}

	public static JPanel flowInLayout(List<Component> components, int gridBagConstraintsAnchor) {
		JPanel result = new JPanel();
		result.setLayout(new GridBagLayout());
		GridBagConstraints contraints = new GridBagConstraints();
		contraints.gridx = 0;
		contraints.gridy = 0;
		contraints.weightx = 1;
		contraints.weighty = 1;
		contraints.anchor = gridBagConstraintsAnchor;
		for (Component c : components) {
			result.add(c, contraints);
		}
		return result;
	}

	public static String getPrintedStackTrace(Throwable e) {
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		e.printStackTrace(new PrintStream(out));
		return out.toString();
	}

	public static <C> C getLast(List<C> list) {
		return list.get(list.size() - 1);
	}

	public static Properties toSortedProperties(String text) {
		Properties result = getSortedProperties();
		try {
			result.load(new ByteArrayInputStream(text.getBytes()));
		} catch (IOException e) {
			throw new UnexpectedError(e);
		}
		return result;
	}

	public static Properties getSortedProperties() {
		return new Properties() {
			private static final long serialVersionUID = 1L;

			@Override
			public synchronized Enumeration<Object> keys() {
				Enumeration<Object> keysEnum = super.keys();
				Vector<Object> keyList = new Vector<Object>();
				while (keysEnum.hasMoreElements()) {
					keyList.add(keysEnum.nextElement().toString());
				}
				Collections.sort(keyList, new Comparator<Object>() {
					@Override
					public int compare(Object o1, Object o2) {
						return o1.toString().compareTo(o2.toString());
					}
				});
				return keyList.elements();
			}
		};
	}

	public static OutputStream getNullOutputStream() {
		return new OutputStream() {
			@Override
			public void write(int b) throws IOException {
			}
		};
	}

	public static OutputStream unifyOutputStreams(final OutputStream... outputStreams) {
		return new OutputStream() {

			@Override
			public void write(int b) throws IOException {
				for (OutputStream out : outputStreams) {
					out.write(b);
				}
			}

			@Override
			public void write(byte[] b) throws IOException {
				for (OutputStream out : outputStreams) {
					out.write(b);
				}
			}

			@Override
			public void write(byte[] b, int off, int len) throws IOException {
				for (OutputStream out : outputStreams) {
					out.write(b, off, len);
				}
			}

			@Override
			public void flush() throws IOException {
				for (OutputStream out : outputStreams) {
					out.flush();
				}
			}

			@Override
			public void close() throws IOException {
				for (OutputStream out : outputStreams) {
					out.close();
				}
			}
		};
	}

	public static void showBusyCursor(Component c, boolean b) {
		Window window = getWindow(c);
		if (window == null) {
			return;
		}
		if (b) {
			window.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
		} else {
			window.setCursor(Cursor.getDefaultCursor());
		}
	}

	public static void showBusyCusrorWhile(final Component c, final Runnable runnable) {
		Window window = getWindow(c);
		if (window == null) {
			runnable.run();
		} else {
			showBusyCursor(c, true);
			try {
				runnable.run();
			} finally {
				showBusyCursor(c, false);
			}
		}
	}

	public static Window getWindow(Component c) {
		if (c instanceof Window) {
			return (Window) c;
		} else {
			return SwingUtilities.getWindowAncestor(c);
		}

	}

	public static void showBusyDialogWhile(final Component ownerComponent, final Runnable runnable, String title) {
		final JLabel busyLabel = new JLabel();
		busyLabel.setHorizontalAlignment(SwingConstants.CENTER);
		busyLabel.setText("Please wait...");
		busyLabel.setVerticalTextPosition(SwingConstants.TOP);
		busyLabel.setHorizontalTextPosition(SwingConstants.CENTER);
		final JDialog dialog = new JDialog(
				(ownerComponent == null) ? null : SwingUtilities.getWindowAncestor(ownerComponent));
		dialog.setTitle(title);
		dialog.getContentPane().setLayout(new BorderLayout());
		dialog.getContentPane().add(busyLabel);
		dialog.setLocationRelativeTo(null);
		dialog.pack();
		final Throwable[] error = new Throwable[1];
		final Thread thread = new Thread(title) {
			@Override
			public void run() {
				try {
					runnable.run();
				} catch (Throwable t) {
					error[0] = t;
				} finally {
					dialog.dispose();
				}
			}
		};
		dialog.addWindowListener(new WindowAdapter() {
			@Override
			public void windowClosing(WindowEvent e) {
				thread.interrupt();
			}
		});
		thread.start();
		try {
			Thread.sleep(1000);
		} catch (InterruptedException e) {
			throw new UnexpectedError(e);
		}
		dialog.setVisible(true);
	}

	public static Window getFocusedWindow() {
		return KeyboardFocusManager.getCurrentKeyboardFocusManager().getFocusedWindow();
	}

	public static void relieveCPU() {
		sleepSafely(10);
	}

	public static void sleepSafely(long durationMilliseconds) {
		try {
			Thread.sleep(durationMilliseconds);
		} catch (InterruptedException e) {
			Thread.currentThread().interrupt();
		}
	}

	public static Date now() {
		return new Date();
	}

	public static String formatThreadName(Class<?> class1, String role) {
		String result = class1.getSimpleName();
		long threadNumber = Counter.next(result);
		result += " " + role + " " + threadNumber;
		return result;
	}

	public static void loadFieldValuesLine(String line, String[] fieldNames, Object obj) {
		for (String paramName : fieldNames) {
			Pattern pattern = Pattern.compile("(?:" + paramName + "=([^;]+)" + ")");
			Matcher matcher = pattern.matcher(line);
			if (!matcher.find()) {
				continue;
			}
			try {
				Field field = obj.getClass().getField(paramName);
				String stringValue = matcher.group(1);
				stringValue = stringValue.replace(";;", ";");
				stringValue = TextUtils.unescapeNewLine(stringValue);
				Object value = deserialize(stringValue, field);
				setFieldValue(obj, paramName, value);
			} catch (Exception e) {
				throw new StandardError(obj.getClass().getName() + " deserialization error from string '" + line + "'"
						+ ": Parameter '" + paramName + "': " + e.toString(), e);
			}
		}
	}

	public static Object deserialize(String stringValue, Field field) throws Exception {
		Class<?> fieldType = field.getType();
		if (fieldType.isPrimitive()) {
			if (fieldType == int.class) {
				fieldType = Integer.class;
			} else if (fieldType == long.class) {
				fieldType = Long.class;
			} else if (fieldType == double.class) {
				fieldType = Double.class;
			} else if (fieldType == boolean.class) {
				fieldType = Boolean.class;
			} else if (fieldType == char.class) {
				fieldType = Character.class;
			} else {
				throw new Exception("Unhandled type: " + fieldType);
			}
		}
		Object value;
		if (fieldType == Character.class) {
			stringValue = stringValue.trim();
			if (stringValue.length() != 1) {
				throw new Exception("Invalid value: '" + stringValue + "'. 1 character is expected");
			}
			value = stringValue.charAt(0);
		} else if (fieldType == Color.class) {
			value = ImageUtils.stringToColor(stringValue);
		} else {
			value = fieldType.getConstructor(new Class[] { String.class }).newInstance(stringValue);
		}
		return value;
	}

	public static String serialize(Object value) {
		if (value instanceof Color) {
			return ImageUtils.toString((Color) value);
		}
		return value.toString();
	}

	public static String saveFieldValuesLine(String[] fieldNames, Object obj) {
		StringBuilder result = new StringBuilder();
		for (String paramName : fieldNames) {
			if (Arrays.asList(fieldNames).indexOf(paramName) > 0) {
				result.append("; ");
			}
			String paramValue = serialize(MiscUtils.getFieldValue(obj, paramName));
			paramValue = TextUtils.escapeNewLines(paramValue);
			paramValue = paramValue.replace(";", ";;");
			result.append(paramName + "=" + paramValue);
		}
		return result.toString();
	}

	public static void setFieldValue(final Object obj, final String fieldName, Object value) {
		Assertion.check(new Assertion.Predicate() {
			@Override
			protected boolean isTrue() {
				return Arrays.asList(getPublicFieldNames(obj.getClass())).contains(fieldName);
			}
		});
		Field field;
		try {
			field = obj.getClass().getField(fieldName);
			field.set(obj, value);
		} catch (Exception e) {
			throw new UnexpectedError(e);
		}
	}

	public static Object getFieldValue(final Object obj, final String fieldName) {
		return getFieldValue(obj.getClass(), fieldName, obj);

	}

	public static Object getFieldValue(final Class<? extends Object> objClass, final String fieldName, Object obj) {
		Assertion.check(new Assertion.Predicate() {
			@Override
			protected boolean isTrue() {
				return Arrays.asList(getPublicFieldNames(objClass)).contains(fieldName);
			}
		});
		try {
			return objClass.getField(fieldName).get(obj);
		} catch (Exception e) {
			throw new UnexpectedError(e);
		}
	}

	public static String[] getPublicFieldNames(Class<? extends Object> class1) {
		List<String> result = new ArrayList<String>();
		for (Field field : class1.getFields()) {
			result.add(field.getName());
		}
		return result.toArray(new String[result.size()]);
	}

	public static Object getInaccesibleFieldValue(Class<?> clazz, Object obj, String fieldName) {
		try {
			Field field = clazz.getDeclaredField(fieldName);
			field.setAccessible(true);
			return field.get(obj);
		} catch (Exception e) {
			throw new UnexpectedError(e);
		}
	}

	public static void setInaccesibleFieldValue(Class<?> clazz, Object obj, String fieldName, Object value) {
		try {
			Field field = clazz.getDeclaredField(fieldName);
			field.setAccessible(true);
			field.set(obj, value);
		} catch (Exception e) {
			throw new UnexpectedError(e);
		}
	}

	public static Object invokeInaccessibleMethod(Class<?> clazz, Object obj, String methodName,
			Class<?>[] parameterTypes, Object[] args) {
		try {
			Method method = clazz.getDeclaredMethod(methodName, parameterTypes);
			method.setAccessible(true);
			return method.invoke(obj, args);
		} catch (Exception e) {
			throw new UnexpectedError(e);
		}
	}

	public static Object getInaccesibleFieldValue(Object obj, String fieldName) {
		return getInaccesibleFieldValue(obj.getClass(), obj, fieldName);
	}

	public static void setInaccesibleFieldValue(Object obj, String fieldName, Object value) {
		setInaccesibleFieldValue(obj.getClass(), obj, fieldName, value);
	}

	public static Object invokeInaccessibleMethod(Object obj, String methodName, Class<?>[] parameterTypes,
			Object[] args) {
		return invokeInaccessibleMethod(obj.getClass(), obj, methodName, parameterTypes, args);
	}

	public static int secondsElapsedSince(Date date) {
		Date now = now();
		long diff = now.getTime() - date.getTime();
		return MathUtils.round((double) diff / (1000));
	}

	public static <BASE, C extends BASE> List<BASE> convertCollection(Collection<C> ts) {
		List<BASE> result = new ArrayList<BASE>();
		for (C t : ts) {
			result.add((BASE) t);
		}
		return result;
	}

	@SuppressWarnings("unchecked")
	public static <BASE, C extends BASE> List<C> convertCollectionUnsafely(Collection<BASE> bs) {
		List<C> result = new ArrayList<C>();
		for (BASE b : bs) {
			result.add((C) b);
		}
		return result;
	}

	public static void fillBoxLayout(Component c) {
		c.setMaximumSize(new Dimension(Integer.MAX_VALUE, Integer.MAX_VALUE));
	}

	public static Rectangle getSmallestBoxWithRatio(Rectangle rectangle, double ratioWidth, double ratioHeight) {
		Rectangle result = new Rectangle(rectangle.x, rectangle.y, rectangle.width, rectangle.height);
		double ratio = ratioWidth / ratioHeight;
		double resultRatio = (double) result.width / (double) result.height;
		if (ratio > resultRatio) {
			int newWidth = MathUtils.round(result.height * ratio);
			result.x -= MathUtils.round((newWidth - result.width) / 2.0);
			result.width = newWidth;
		} else {
			int newHeight = MathUtils.round(result.width / ratio);
			result.y -= MathUtils.round((newHeight - result.height) / 2.0);
			result.height = newHeight;
		}
		return result;
	}

	public static void changeLayoutContraints(Component c, Object newConstraints) {
		Container container = c.getParent();
		int index = indexOf(container, c);
		container.remove(c);
		container.add(c, newConstraints, index);
	}

	public static int indexOf(Container container, Component component) {
		for (int i = 0; i < container.getComponentCount(); i++) {
			Component candidate = container.getComponent(i);
			if (component == candidate) {
				return i;
			}
		}
		return -1;
	}

	public static int removeAWTEventListener(AWTEventListener listener) {
		final List<AWTEventListener> listenersToRemove = new ArrayList<AWTEventListener>();
		for (AWTEventListener l : Toolkit.getDefaultToolkit().getAWTEventListeners()) {
			if (l == listener) {
				listenersToRemove.add(l);
			} else if (l instanceof AWTEventListenerProxy) {
				final AWTEventListenerProxy proxyListener = (AWTEventListenerProxy) l;
				if (proxyListener.getListener() == listener) {
					listenersToRemove.add(proxyListener);
				}
			}
		}
		for (AWTEventListener l : listenersToRemove) {
			Toolkit.getDefaultToolkit().removeAWTEventListener(l);
		}
		return listenersToRemove.size();
	}

	public static void disableComponentTree(JComponent c, final boolean revert) {
		String CONTAINER_LISTENER_KEY = MiscUtils.class.getName() + ".disableComponentTree.CONTAINER_LISTENER_KEY";
		String LAST_STATE_KEY = MiscUtils.class.getName() + ".disableComponentTree.LAST_STATE_KEY";
		Boolean lastState = (Boolean) c.getClientProperty(LAST_STATE_KEY);
		if (revert) {
			if (lastState == null) {
				return;
			}
			if (lastState) {
				c.setEnabled(true);
			}
			c.putClientProperty(LAST_STATE_KEY, null);
			ContainerListener containerListener = (ContainerListener) c.getClientProperty(CONTAINER_LISTENER_KEY);
			c.removeContainerListener(containerListener);
		} else {
			if (lastState != null) {
				return;
			}
			c.putClientProperty(LAST_STATE_KEY, c.isEnabled());
			c.setEnabled(false);
			ContainerListener containerListener = new ContainerListener() {

				@Override
				public void componentRemoved(ContainerEvent e) {
				}

				@Override
				public void componentAdded(ContainerEvent e) {
					Component child = e.getChild();
					if (!(child instanceof JComponent)) {
						return;
					}
					disableComponentTree((JComponent) child, revert);
				}
			};
			c.addContainerListener(containerListener);
			c.putClientProperty(CONTAINER_LISTENER_KEY, containerListener);
		}
		for (Component child : c.getComponents()) {
			if (!(child instanceof JComponent)) {
				continue;
			}
			disableComponentTree((JComponent) child, revert);
		}
	}

	public static String identifierToCaption(String id) {
		StringBuilder result = new StringBuilder();
		int i = 0;
		char lastC = 0;
		for (char c : id.toCharArray()) {
			if (i == 0) {
				result.append(Character.toUpperCase(c));
			} else if (Character.isUpperCase(c) && !Character.isUpperCase(lastC)) {
				result.append(" " + c);
			} else if (Character.isDigit(c) && !Character.isDigit(lastC)) {
				result.append(" " + c);
			} else if (!Character.isLetterOrDigit(c) && Character.isLetterOrDigit(lastC)) {
				result.append(" " + c);
			} else {
				result.append(c);
			}
			lastC = c;
			i++;
		}
		return result.toString();
	}

	public static Rectangle getWindowMaximumBounds() {
		return GraphicsEnvironment.getLocalGraphicsEnvironment().getMaximumWindowBounds();
	}

	public static boolean runWithTimeout(final Runnable runnable, long timeoutMilliseconds, ExecutorService executor) {
		Runnable runOnInterruptionTimeout = new Runnable() {
			@Override
			public void run() {
				throw getTaskCancellationFailureException("Hang detected");
			}
		};
		return runWithTimeout(runnable, timeoutMilliseconds, 5 * 1000, runOnInterruptionTimeout, executor);
	}

	public static CancellationException getTaskCancellationFailureException(String message) {
		return new CancellationException(message);
	}

	public static TimeoutException getTaskTimeoutException(String message) {
		return new TimeoutException(message);
	}

	public static boolean runWithTimeout(final Runnable runnable, long timeoutMillisecondsToInterrupt,
			final long interruptionTimeoutMillisecondsToDestroy, final Runnable runOnInterruptionTimeout,
			final ExecutorService executor) {
		final boolean[] status = new boolean[1];
		// need to report task completion by hand because
		// future.isDone() always returns true after cancel()
		final boolean[] runnableDone = new boolean[] { false };
		Runnable runnableReportingCompletion = new Runnable() {
			@Override
			public void run() {
				runnable.run();
				runnableDone[0] = true;
			}
		};
		Runnable onSuccess = new Runnable() {
			@Override
			public void run() {
				status[0] = true;
			}
		};
		@SuppressWarnings("rawtypes")
		Listener<Future> onTimeout = new Listener<Future>() {
			@Override
			public void handle(Future future) {
				long timedoutSince = System.currentTimeMillis();
				while (!runnableDone[0]) {
					future.cancel(true);
					if ((System.currentTimeMillis() - timedoutSince) > interruptionTimeoutMillisecondsToDestroy) {
						runOnInterruptionTimeout.run();
						break;
					}
					relieveCPU();
				}
				status[0] = false;
			}
		};
		Listener<Throwable> onError = new Listener<Throwable>() {
			@Override
			public void handle(Throwable thrown) {
				ErrorManager.rethrow(thrown);
			}
		};
		runWithTimeout(runnableReportingCompletion, executor, timeoutMillisecondsToInterrupt, onSuccess, onTimeout,
				onError);
		return status[0];
	}

	@SuppressWarnings("rawtypes")
	public static void runWithTimeout(final Runnable runnable, ExecutorService executor, long timeoutMilliseconds,
			Runnable onSuccess, Listener<Future> onTimeout, Listener<Throwable> onError) {
		boolean defaultExecutorUsed;
		if (executor == null) {
			defaultExecutorUsed = true;
			final Thread callingThread = Thread.currentThread();
			executor = Executors.newSingleThreadExecutor(new ThreadFactory() {
				@Override
				public Thread newThread(Runnable r) {
					Thread result = new Thread(r, MiscUtils.formatThreadName(MiscUtils.class,
							"DefaultTimeOutMonitor(" + callingThread.getName() + ")"));
					return result;
				}
			});
		} else {
			defaultExecutorUsed = false;
		}
		Future<?> future = executor.submit(runnable);
		try {
			future.get(timeoutMilliseconds, TimeUnit.MILLISECONDS);
			if (onSuccess != null) {
				onSuccess.run();
			}
		} catch (InterruptedException e) {
			if (onError != null) {
				onError.handle(e);
			}
		} catch (ExecutionException e) {
			if (onError != null) {
				onError.handle(e.getCause());
			}
		} catch (TimeoutException e) {
			if (onTimeout != null) {
				onTimeout.handle(future);
			}
		} finally {
			if (defaultExecutorUsed) {
				executor.shutdown();
			}
		}
	}

	public static void repaintImmaediately(JComponent c) {
		c.paintImmediately(0, 0, c.getWidth(), c.getHeight());
	}

	public static Window getWindowAncestorOrSelf(Component c) {
		if (c instanceof Window) {
			return (Window) c;
		} else {
			return SwingUtilities.getWindowAncestor(c);
		}
	}

	public static void runImmediatelyInUIThread(Runnable runnable) {
		if (SwingUtilities.isEventDispatchThread()) {
			runnable.run();
		} else {
			try {
				SwingUtilities.invokeAndWait(runnable);
			} catch (InterruptedException e) {
				throw new UnexpectedError(e);
			} catch (InvocationTargetException e) {
				ErrorManager.rethrow(e.getCause(), true);
			}
		}
	}

	public static <T> String stringJoin(T[] array, String separator) {
		return stringJoin(Arrays.asList(array), separator);
	}

	public static String stringJoin(List<?> list, String separator) {
		StringBuilder result = new StringBuilder();
		for (int i = 0; i < list.size(); i++) {
			Object item = list.get(i);
			if (i > 0) {
				result.append(separator);
			}
			if (item == null) {
				result.append("null");
			} else {
				result.append(item.toString());
			}
		}
		return result.toString();
	}

	public static <K, V> List<K> getKeysFromValue(Map<K, V> map, Object value) {
		List<K> result = new ArrayList<K>();
		for (Map.Entry<K, V> entry : map.entrySet()) {
			if (equalsOrBothNull(entry.getValue(), value)) {
				result.add(entry.getKey());
			}
		}
		return result;
	}

	public static boolean equalsOrBothNull(Object o1, Object o2) {
		if (o1 == null) {
			return o2 == null;
		} else {
			return o1.equals(o2);
		}
	}

	public static <E> boolean endsWith(List<E> list, List<E> end) {
		int i = Collections.lastIndexOfSubList(list, end);
		if (i == -1) {
			return false;
		}
		return (i + end.size()) == list.size();
	}

	@SuppressWarnings("rawtypes")
	public static int readUntil(InputStream input, OutputStream out, final long timeoutMilliseconds,
			Listener<Future> onTimeout, byte[]... endings) throws IOException {
		final Thread callingThread = Thread.currentThread();
		ExecutorService executor = Executors.newSingleThreadExecutor(new ThreadFactory() {
			@Override
			public Thread newThread(Runnable r) {
				Thread result = new Thread(r, MiscUtils.formatThreadName(MiscUtils.class,
						"DefaultTimeOutEnabledStreamReader(" + callingThread.getName() + ")"));
				return result;
			}
		});
		try {
			return readUntil(executor, input, out, timeoutMilliseconds, onTimeout, endings);
		} finally {
			executor.shutdownNow();
		}
	}

	@SuppressWarnings("rawtypes")
	public static int readUntil(ExecutorService executor, InputStream input, OutputStream output,
			final long timeoutMilliseconds, Listener<Future> onTimeout, byte[]... endings) throws IOException {
		final List<List<Byte>> endingList = new ArrayList<List<Byte>>();
		final int[] endingmaxLenth = new int[] { 0 };
		for (byte[] ending : endings) {
			endingList.add(Arrays.asList((Byte[]) PrimitiveUtils.primitiveToWrapperArray(ending)));
			endingmaxLenth[0] = Math.max(endingmaxLenth[0], ending.length);
		}
		final int[] encounterEndingIndex = new int[] { -1 };
		final List<Byte> endingComparaisonBuffer = new ArrayList<Byte>(endingmaxLenth[0]);
		final Throwable[] error = new Throwable[1];
		Runnable runnable = new Runnable() {
			@Override
			public void run() {
				int nextByte;
				while (true) {
					if (Thread.currentThread().isInterrupted()) {
						return;
					}
					try {
						nextByte = input.read();
					} catch (IOException e) {
						error[0] = e;
						return;
					}
					if (nextByte == -1) {
						return;
					}
					try {
						output.write(nextByte);
					} catch (IOException e) {
						throw new UnexpectedError(e);
					}
					if (endingmaxLenth[0] > 0) {
						if (endingComparaisonBuffer.size() == endingmaxLenth[0]) {
							endingComparaisonBuffer.remove((int) 0);
						}
						endingComparaisonBuffer.add((byte) nextByte);
						for (int iEnding = 0; iEnding < endingList.size(); iEnding++) {
							List<Byte> ending = endingList.get(iEnding);
							if (endsWith(endingComparaisonBuffer, ending)) {
								encounterEndingIndex[0] = iEnding;
								return;
							}
						}
					}
				}
			}
		};
		Runnable onSuccess = null;
		Listener<Throwable> onError = new Listener<Throwable>() {
			@Override
			public void handle(Throwable thrown) {
				error[0] = thrown;
			}
		};
		runWithTimeout(runnable, executor, timeoutMilliseconds, onSuccess, onTimeout, onError);
		if (error[0] != null) {
			if (error[0] instanceof IOException) {
				throw (IOException) error[0];
			} else {
				ErrorManager.rethrow(error[0]);
			}
		}
		return encounterEndingIndex[0];
	}

	public static boolean isStreamClosedException(IOException e) {
		return e.toString().toLowerCase().contains("stream closed");
	}

	public static String captureThreadDump() {
		Map<?, ?> allThreads = Thread.getAllStackTraces();
		Iterator<?> iterator = allThreads.keySet().iterator();
		StringBuffer stringBuffer = new StringBuffer();
		while (iterator.hasNext()) {
			Thread key = (Thread) iterator.next();
			StackTraceElement[] trace = (StackTraceElement[]) allThreads.get(key);
			stringBuffer.append(key + "\n");
			for (int i = 0; i < trace.length; i++) {
				stringBuffer.append(" " + trace[i] + "\n");
			}
			stringBuffer.append("\n");
		}
		return stringBuffer.toString();
	}

	public static boolean isEventDispatchThreadBusy() {
		return Toolkit.getDefaultToolkit().getSystemEventQueue().peekEvent() != null;
	}

	public static boolean hasOrContainsFocus(Component c) {
		if (c.isFocusOwner()) {
			return true;
		}
		if (c instanceof Container) {
			Container container = (Container) c;
			for (Component childComponent : container.getComponents()) {
				if (hasOrContainsFocus(childComponent)) {
					return true;
				}
			}
		}
		return false;
	}

	public static void forceUpdateConstant(Field field, Object newValue) throws Exception {
		field.setAccessible(true);
		Field modifiersField = Field.class.getDeclaredField("modifiers");
		modifiersField.setAccessible(true);
		modifiersField.setInt(field, field.getModifiers() & ~Modifier.FINAL);
		field.set(null, newValue);
	}

	public static int getStandardCharacterWidth(Component c) {
		Font font = c.getFont();
		if (font == null) {
			font = UIManager.getFont("Panel.font");
		}
		return c.getFontMetrics(font).charWidth('a');
	}

	public static void openWebPage(String url) {
		try {
			Desktop.getDesktop().browse(new URL(url).toURI());
		} catch (Exception e) {
			throw new UnexpectedError("Failed to display the web page '" + url + "': " + e, e);
		}
	}

	public static void setSplitPaneDividerLocationBeforeRealization(final JSplitPane splitPane,
			final double dividerLocationRatio) {
		splitPane.addComponentListener(new ComponentAdapter() {
			@Override
			public void componentResized(ComponentEvent ce) {
				splitPane.removeComponentListener(this);
				splitPane.setDividerLocation(dividerLocationRatio);
			}
		});
		splitPane.setResizeWeight(dividerLocationRatio);
	}

	static Thread redirectStream(final InputStream src, final OutputStream dst, String reason) {
		Thread thread = new Thread(MiscUtils.formatThreadName(MiscUtils.class, "Stream Redirector (" + reason + ")")) {
			public void run() {
				try {
					while (true) {
						if (src.available() > 0) {
							int b = src.read();
							if (b == -1) {
								break;
							}
							dst.write(b);
						} else {
							if (isInterrupted()) {
								if (src.available() == 0) {
									break;
								}
							} else {
								MiscUtils.relieveCPU();
							}
						}
					}
				} catch (IOException e) {
					throw new UnexpectedError(e);
				}
			}
		};
		thread.setPriority(Thread.MIN_PRIORITY);
		thread.start();
		return thread;
	}

	public static InputStream sendHttpPost(URL url, Map<String, String> arguments) throws IOException {
		URLConnection con = url.openConnection();
		HttpURLConnection http = (HttpURLConnection) con;
		http.setRequestMethod("POST");
		http.setDoOutput(true);
		StringJoiner postBuilder = new StringJoiner("&");
		for (Map.Entry<String, String> entry : arguments.entrySet())
			postBuilder.add(
					URLEncoder.encode(entry.getKey(), "UTF-8") + "=" + URLEncoder.encode(entry.getValue(), "UTF-8"));
		byte[] post = postBuilder.toString().getBytes(StandardCharsets.UTF_8);
		int length = post.length;
		http.setFixedLengthStreamingMode(length);
		http.setRequestProperty("Content-Type", "application/x-www-form-urlencoded; charset=UTF-8");
		http.connect();
		OutputStream outputStream = http.getOutputStream();
		outputStream.write(post);
		return http.getInputStream();
	}

}
