/*******************************************************************************
 * Copyright (C) 2018 OTK Software
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 ******************************************************************************/
package com.otk.application.util;

import java.awt.Component;
import java.awt.Font;
import java.awt.GraphicsEnvironment;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.text.DateFormat;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.otk.application.TheConstants;
import com.otk.application.error.UnexpectedError;

public class TextUtils {

	public static final String[] NEW_LINE_SEQUENCES = new String[] { "\r\n", "\n", "\r" };

	public static String captureRegexGroup(String s, String regex) {
		Pattern pattern = Pattern.compile(regex);
		Matcher matcher = pattern.matcher(s);
		if (!matcher.matches()) {
			return null;
		}
		if (matcher.groupCount() == 0) {
			return null;
		}
		return matcher.group(1);
	}

	public static Font getReadableFont(int size) {
		return TheConstants.readableFont.deriveFont((float) size);
	}

	public static String toText(Properties properties) {
		ByteArrayOutputStream out = new ByteArrayOutputStream();
		try {
			properties.store(out, null);
		} catch (IOException e) {
			throw new UnexpectedError(e);
		}
		return out.toString();
	}

	public static String[] splitLines(String s) {
		if (s.length() == 0) {
			return new String[0];
		}
		return s.split(getNewLineRegex(), -1);
	}

	public static String getNewLineRegex() {
		return stringJoin(Arrays.asList(TextUtils.NEW_LINE_SEQUENCES), "|");
	}

	public static String escapeHTML(String string) {
		return escapeHTML(string, false);
	}

	public static String escapeHTML(String string, boolean convertNewLinesToHTML) {
		string = string.replaceAll(getNewLineRegex(), "\n");
		StringBuffer sb = new StringBuffer(string.length());
		int len = string.length();
		char c;

		for (int i = 0; i < len; i++) {
			c = string.charAt(i);
			if (c == '"')
				sb.append("&quot;");
			else if (c == '&')
				sb.append("&amp;");
			else if (c == '<')
				sb.append("&lt;");
			else if (c == '>')
				sb.append("&gt;");
			else if (c == '\n')
				// Handle Newline
				if (convertNewLinesToHTML) {
					sb.append("<br/>");
				} else {
					sb.append(c);
				}
			else {
				int ci = 0xffff & c;
				if (ci < 160)
					// nothing special only 7 Bit
					sb.append(c);
				else {
					// Not 7 Bit use the unicode system
					sb.append("&#");
					sb.append(new Integer(ci).toString());
					sb.append(';');
				}
			}
		}
		return sb.toString();
	}

	public static String truncateNicely(String s, int length) {
		if (s.length() > length) {
			s = s.substring(0, length - 3) + "...";
		}
		return s;
	}

	public static DateFormat getDefaultDateFormat() {
		return DateFormat.getDateTimeInstance(DateFormat.SHORT, DateFormat.MEDIUM);
	}

	public static String escapeNewLines(String s) {
		StringBuilder result = new StringBuilder();
		char lastC = 0;
		for (char currentC : s.toCharArray()) {
			char standardC = standardizeNewLineSequences(lastC, currentC);
			try {
				if (standardC == 0) {
					continue;
				} else if (standardC == '\n') {
					result.append("\\n");
				} else if (standardC == '\\') {
					result.append("\\\\");
				} else {
					result.append(standardC);
				}
			} finally {
				lastC = currentC;
			}
		}
		return result.toString();
	}

	public static char standardizeNewLineSequences(char lastC, char c) {
		for (String newLineSequence : TextUtils.NEW_LINE_SEQUENCES) {
			if (newLineSequence.equals("" + lastC + c)) {
				return 0;
			}
		}
		for (String newLineSequence : TextUtils.NEW_LINE_SEQUENCES) {
			if (newLineSequence.startsWith("" + c)) {
				return '\n';
			}
		}
		return c;
	}

	public static String unescapeNewLine(String s) {
		StringBuilder result = new StringBuilder();
		boolean escaping = false;
		for (char c : s.toCharArray()) {
			if (escaping) {
				if (c == '\\') {
					result.append('\\');
				} else if (c == 'n') {
					result.append('\n');
				} else {
					result.append('\\');
					result.append(c);
				}
				escaping = false;
			} else {
				if (c == '\\') {
					escaping = true;
				} else {
					result.append(c);
				}
			}
		}
		return result.toString();
	}

	public static String stringJoin(Collection<?> col, String delim) {
		StringBuilder sb = new StringBuilder();
		Iterator<?> iter = col.iterator();
		if (iter.hasNext())
			sb.append(iter.next().toString());
		while (iter.hasNext()) {
			sb.append(delim);
			sb.append(iter.next().toString());
		}
		return sb.toString();
	}

	public static String padRight(String s, int len, char c) {
		StringBuilder result = new StringBuilder(s);
		for (int i = s.length(); i < len; i++) {
			result.append(c);
		}
		return result.toString();
	}

	public static String padLeft(String s, int len, char c) {
		StringBuilder result = new StringBuilder(s);
		for (int i = s.length(); i < len; i++) {
			result.insert(0, c);
		}
		return result.toString();
	}

	public static String padBothSides(String s, int len, char c) {
		int padLength = MathUtils.round((len - s.length()) / 2.0);
		s = TextUtils.padLeft(s, padLength + s.length(), ' ');
		s = TextUtils.padRight(s, len, ' ');
		return s;
	}

	public static void changeFont(Component c, Font f) {
		f = f.deriveFont(c.getFont().getSize2D());
		c.setFont(f);
	}

	public static void changeFontSize(Component c, int newSize) {
		c.setFont(c.getFont().deriveFont((float) newSize));
	}

	public static <C> String toString(C[] ts) {
		return Arrays.asList(ts).toString();
	}

	public static String fixNewLines(String string) {
		string = string.replace("\r\n", "\n");
		string = string.replace("\r", "\n");
		string = string.replace("\n", System.getProperty("line.separator", "\n"));
		return string;
	}

	public static String normalizeLineTerminators(String text) {
		return text.replace("\r\n", "\n").replace("\r", "\n");

	}

	public static Font requireFont(InputStream in) {
		try {
			Font result = Font.createFont(Font.PLAIN, in);
			GraphicsEnvironment.getLocalGraphicsEnvironment().registerFont(result);
			return result;
		} catch (Exception e) {
			throw new UnexpectedError(e);
		} finally {
			if (in != null) {
				try {
					in.close();
				} catch (IOException ignore) {
				}
			}
		}
	}

	public static Font requireFont(String filePath) {
		InputStream in = null;
		try {
			File file = new File(filePath);
			in = new FileInputStream(file);
			return requireFont(in);
		} catch (Exception e) {
			throw new UnexpectedError(e);
		} finally {
			if (in != null) {
				try {
					in.close();
				} catch (IOException ignore) {
				}
			}
		}
	}

}
