! See copyright notice in the COPYRIGHT file.
! **************************************************************************** !
!> author: Kannan Masilamani
!! This module contains routine to retrieve auxiliary field variables for 
!! getElement, getPoint, setupIndices and getValOfIndex.
!! Auxilary field variables are:
!!    * density and velocity for fluid 
!!    * species desity and velocity for multispecies
!!    * potential for poisson
!!
?? include 'header/lbm_macros.inc'
?? include 'treelm/source/deriveMacros.inc'
module mus_auxFieldVar_module
  use iso_c_binding, only: c_loc, c_ptr, c_f_pointer

  ! include treelm modules
  use env_module,          only: rk
  use tem_aux_module,      only: tem_abort
  use tem_param_module,    only: rho0, rho0Inv, cs2
  use tem_varSys_module,   only: tem_varSys_type, tem_varSys_op_type
  use tem_time_module,     only: tem_time_type
  use treelmesh_module,    only: treelmesh_type
  use tem_geometry_module, only: tem_CoordOfReal, &
    &                            tem_PosofId, tem_BaryOfId
  use tem_topology_module, only: tem_IdOfCoord
  use tem_topology_module, only: tem_levelOf
  use tem_stencil_module,  only: tem_stencilHeader_type
  use tem_comm_module,     only: tem_commpattern_type, tem_communication_type
  use tem_construction_module, only: tem_levelDesc_type
  use tem_logging_module,  only: logUnit

  use mus_varSys_module,         only: mus_varSys_data_type
  use mus_scheme_type_module,    only: mus_scheme_type
  use mus_scheme_header_module,  only: mus_scheme_header_type
  use mus_derVarPos_type_module, only: mus_derVarPos_type
  use mus_physics_module,        only: mus_convertFac_type
  use mus_auxField_module,       only: mus_proc_calcAuxField
  use mus_source_var_module,     only: mus_source_op_type
  use mus_connectivity_module,   only: mus_getSrcElemPosForIntp

  implicit none
  private

  public :: mus_assign_calcAuxField_ptr
  public :: mus_access_auxFieldVar_forElement
  public :: mus_auxFieldVar_forPoint
  public :: mus_auxFieldVar_fromIndex
  public :: mus_addSrcToAuxField_lbm
  public :: mus_addSrcToAuxField_lbm_incomp
  public :: mus_addSrcToAuxField_dummy

contains

  ! ************************************************************************* !
  !> This routine assign function pointer to compute auxField var
  subroutine mus_assign_calcAuxField_ptr(schemeHeader, calcAuxField)
    ! --------------------------------------------------------------------- !
    !> scheme defnition
    type(mus_scheme_header_type), intent(in) :: schemeHeader
    !> function pointer to assign
    procedure(mus_proc_calcAuxField), pointer, intent(out) :: calcAuxField
    ! --------------------------------------------------------------------- !
    ! --------------------------------------------------------------------- !
    calcAuxField => mus_calcAuxField_dummy
    select case(trim(schemeHeader%kind))
    case ('lbm', 'lbm_nNwtn')
      calcAuxField => mus_calcAuxField_lbm
    case ('lbm_incomp','lbm_incomp_nNwtn')
      calcAuxField => mus_calcAuxField_lbm_incomp
    case ('passive_scalar','poisson', 'poisson_boltzmann_linear', &
      &   'poisson_boltzmann_nonlinear', 'nernst_planck')
      calcAuxField => mus_calcAuxField_zerothMoment
    case ('multi-species_gas', 'multi-species_liquid')
      calcAuxField => mus_calcAuxField_MS
    case ('isotherm_acEq')
      calcAuxField => mus_calcAuxField_lbm_incomp
    case default
      calcAuxField => mus_calcAuxField_dummy
    end select 
  end subroutine mus_assign_calcAuxField_ptr
  ! ************************************************************************* !


  ! ************************************************************************* !
  !> Return the solver aux variable for a given set of elements
  !!
  !! The interface has to comply to the abstract interface
  !! [[tem_varSys_module:tem_varSys_proc_element]].
  !!
?? copy :: get_element_headtxt(mus_access_auxFieldVar_forElement)
    ! --------------------------------------------------------------------- !
    integer :: statePos, iElem, iComp, iLevel
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    ! --------------------------------------------------------------------- !
    call C_F_POINTER( fun%method_Data, fPtr )
    scheme => fPtr%solverData%scheme

    ! res is always AOS layout
    res = 0.0_rk
    do iElem = 1, nElems
      ! if state array is defined level wise then use levelPointer(pos)
      ! to access state array
      statePos = fPtr%solverData%geometry%levelPointer( elemPos(iElem) )
      iLevel = tem_levelOf( tree%treeID( elemPos(iElem) ) )
      do iComp = 1, fun%nComponents
        res( (iElem-1)*fun%nComponents+iComp ) =                         &
          & scheme%auxField( iLevel )%val(                               &
          ! position of this aux variable in the aux array
          & (statePos-1)*varSys%nAuxScalars + fun%auxField_varPos(iComp) )
      end do !iComp
    end do !iElem

  end subroutine mus_access_auxFieldVar_forElement
  ! ************************************************************************* !

  ! ************************************************************************* !
  !> Auxilary field variable for a given set of points using linear 
  !! interpolation. Unlike mus_deriveVar_forPoint which does not consider
  !! ghost and halo elements, this routine considers them because
  !! auxField vars on ghost elements are interpolated and halo elements are 
  !! exchanged.
  !! The interface has to comply to the abstract interface
  !! [[tem_varSys_module:tem_varSys_proc_point]].
  !!
?? copy :: get_point_headtxt(mus_auxFieldVar_forPoint)
    ! --------------------------------------------------------------------- !
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    integer :: statePos, iPnt, iLevel, elemPos
    integer :: coord(4)
    integer, allocatable :: srcElemPos(:)
    real(kind=rk), allocatable :: weights(:), srcRes(:), pntVal(:)
    integer :: iSrc, iComp, nSrcElems
    ! --------------------------------------------------------------------- !
!write(dbgUnit(1),*) 'Derive for point :'//trim(varSys%varname%val(fun%myPos))
    call C_F_POINTER( fun%method_Data, fPtr )
    scheme => fPtr%solverData%scheme
    allocate(srcElemPos(scheme%layout%fStencil%QQ))
    allocate(weights(scheme%layout%fStencil%QQ))
    allocate(srcRes(scheme%layout%fStencil%QQ*fun%nComponents))
    allocate(pntVal(fun%nComponents))

    res = 0.0_rk
    do iPnt = 1, nPnts
      weights = 0.0_rk 
      srcRes = 0.0_rk
      srcElemPos = 0
      coord =  tem_CoordOfReal(tree, point(iPnt,:), tree%global%maxLevel )
      ! returns position of existing element in tree which contains point
      ! 0 if no corresponding node is found,
      ! or the negative of the found ID, if it is a virtual node.
      elemPos = abs(tem_PosofId( tem_IdOfCoord(coord), tree%treeID ))

      ! skip this point if no corresponding element is found
      if (elemPos == 0) cycle

      ! level of existing element 
      iLevel = tem_levelOf( tree%treeID( elemPos ) )

      ! position of element in levelDesc total list
      statePos = fPtr%solverData%geometry%levelPointer( elemPos )

      ! get position of source element position in state array for interpolation
      ! using neighbor connectivity array. 
      ! Also calculate weights for interpolation using distance between the 
      ! point and source element barycenter 
      call mus_getSrcElemPosForIntp(                              &
        & srcElemPos  = srcElemPos,                               &
        & weights     = weights,                                  &
        & nSrcElems   = nSrcElems,                                &
        & point       = point(iPnt,:),                            &
        & elemPos     = statePos,                                 &
        & neigh       = fPtr%solverData%scheme%pdf(iLevel)%neigh, & 
        & baryOfTotal = scheme%levelDesc(iLevel)%baryOfTotal,     &
        & nElems      = scheme%pdf(iLevel)%nSize,                 &
        & nSolve      = scheme%pdf(iLevel)%nElems_solve,          &
        & stencil     = scheme%layout%fStencil,                   &
        & excludeHalo = .false.,                                  &
        & nScalars    = varSys%nScalars                           )

      ! get source element values
      do iSrc = 1, nSrcElems
        ! position of element in levelDesc total list
        statePos = srcElemPos(iSrc)
        do iComp = 1, fun%nComponents
          srcRes( (iSrc-1)*fun%nComponents + iComp )                       &
            & = scheme%auxField( iLevel )%val(                             &
            & (statePos-1)*varSys%nAuxScalars + fun%auxField_varPos(iComp) )
        end do !iComp
      end do !iSrc 

      ! Linear interpolation res = sum(weight_i*phi_i)
      pntVal = 0.0_rk
      do iSrc = 1, nSrcElems
        pntVal(:) = pntVal(:) + weights(iSrc) &
          & * srcRes( (iSrc-1)*fun%nComponents+1 : iSrc*fun%nComponents )
      end do
      res( (iPnt-1)*fun%nComponents+1 : iPnt*fun%nComponents ) = pntVal

    end do !iPnt 
  end subroutine mus_auxFieldVar_forPoint
  ! ************************************************************************* !

  ! ************************************************************************* !
  !> Routine to get the actual value for a given array of indices. 
  !! The indices belong to the grwarray of points storing levelwise in 
  !! Pointdata%pntLvl(iLevel).
  !! Hence this routines takes the indeices as input, can refer to the pointData
  !! and evaluate the variable and returns the values
  subroutine mus_auxFieldVar_fromIndex( fun, varSys, time, iLevel, idx, &
    &                                   idxLen,  nVals,  res            )
    ! -------------------------------------------------------------------------- !
    !> Description of the method to obtain the variables, 
    class(tem_varSys_op_type), intent(in) :: fun

    !> The variable system to obtain the variable from.
    type(tem_varSys_type), intent(in) :: varSys

    !> Point in time at which to evaluate the variable.
    type(tem_time_type), intent(in) :: time

    !> Level on which values are requested
    integer, intent(in) :: iLevel

    !> Index of points in the growing array and variable val array to
    !! return.
    !! Size: n
    integer, intent(in) :: idx(:)

    !> With idx as start index in contiguous memory,
    !! idxLength defines length of each contiguous memory
    !! Size: dependes on number of first index for contiguous array,
    !! but the sum of all idxLen is equal to nVals
    integer, optional, intent(in) :: idxLen(:)

    !> Number of values to obtain for this variable (vectorized access).
    integer, intent(in) :: nVals

    !> Resulting values for the requested variable.
    !!
    !! Dimension: n requested entries x nComponents of this variable
    !! Access: (iElem-1)*fun%nComponents + iComp
    real(kind=rk), intent(out) :: res(:)
    ! -------------------------------------------------------------------------- !
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    integer :: iComp, iSrc, iVal, elemPos, first, last, nSize, loc_level
    integer :: iSrcElems, statePos, nSrcElems
    real(kind=rk), allocatable :: srcRes(:), pntVal(:)
    real(kind=rk) :: weight
    ! -------------------------------------------------------------------------- !

!    write(dbgUnit(4),*) 'get the values of indices for variable ',  &
!      &                  trim(varSys%varname%val(fun%myPos))

    call C_F_POINTER( fun%method_Data, fPtr )
    scheme => fPtr%solverData%scheme
    allocate(srcRes(scheme%layout%fStencil%QQ*fun%nComponents))
    allocate(pntVal(fun%nComponents))

    res = 0.0_rk

    ! distinguish if we have an array of index or we have contingous memory
    ! access where index are always first entries!
    if (present(idxLen)) then
      call tem_abort('Error: idxLen is not supported in get_valOfIndex for ' &
        &          //'state variable')
    end if

    ! Get the state value at the specific point
    do iVal = 1, nVals
      if (idx(iVal)>0) then

        ! elemPos in tree 
        ! elemPos = fPtr%pointData%pntLvl(iLevel)%elemPos%val(idx(iVal))

        first = fPtr%pointData%pntLvl(iLevel)%srcElem%first%val(idx(iVal))
        last = fPtr%pointData%pntLvl(iLevel)%srcElem%last%val(idx(iVal))

        ! get pdf's of source elements
        srcRes = 0.0_rk
        iSrcElems = 0
        do iSrc = first, last
          iSrcElems = iSrcElems + 1

          statePos = fPtr%pointData%pntLvl(iLevel)%srcElem%elemPos%val(iSrc)
          loc_level = fPtr%pointData%pntLvl(iLevel)%pntLevel%val(idx(iVal))

          do iComp = 1, fun%nComponents
            srcRes( (iSrcElems-1)*fun%nComponents + iComp )                  &
              & = scheme%auxField(loc_level)%val(                            &
              & (statePos-1)*varSys%nAuxScalars + fun%auxField_varPos(iComp) )
          end do !iComp
        end do !iSrc 

        ! Linear interpolation res = sum(weight_i*phi_i)
        pntVal = 0.0_rk
        nSrcElems = 0
        do iSrc = first, last
          weight = fPtr%pointData%pntLvl(iLevel)%srcElem%weight%val(iSrc)
          nSrcElems = nSrcElems + 1
          pntVal(:) = pntVal(:) + weight &
            & * srcRes( (nSrcElems-1)*fun%nComponents+1 &
            &         : nSrcElems*fun%nComponents )
        end do

        ! get the state value for each component of this 
        res( (iVal-1)*fun%nComponents+1: iVal*fun%nComponents ) = pntVal
      end if !idx>0
    end do !iVal
    
  end subroutine mus_auxFieldVar_fromIndex 
  ! ************************************************************************* !

  ! ************************************************************************* !
  !> This routine compute auxFields density and velocity for compressible model
  !! for fluid and nGhostFromCoarser elements
  subroutine mus_calcAuxField_lbm(auxField, state, neigh, nSize, nSolve, &
    & iLevel, stencil, varSys, derVarPos)
    ! -------------------------------------------------------------------------- !
    !> output auxField array
    real(kind=rk), intent(inout) :: auxField(:)
    !> input state array
    real(kind=rk), intent(in) :: state(:)
    !> connectivity array
    integer, intent(in) :: neigh(:)
    !> number of elements in the state array
    integer, intent(in) :: nSize
    !> number of elements excluding halos
    integer, intent(in) :: nSolve
    !> current level
    integer, intent(in) :: iLevel
    !> stencil header 
    type(tem_stencilHeader_type), intent(in) :: stencil
    !> variable system definition
    type(tem_varSys_type), intent(in) :: varSys
    !> position of derived quantities in varsys
    type(mus_derVarPos_type), intent(in) :: derVarPos(:)
    ! -------------------------------------------------------------------------- !
    integer :: dens_pos, vel_pos(3), elemOff
    real(kind=rk) :: rho, pdf( stencil%QQ ), inv_rho
    integer :: iElem, iDir, QQ, nScalars
    ! -------------------------------------------------------------------------- !
    dens_pos = varSys%method%val(derVarPos(1)%density)%auxField_varPos(1)
    vel_pos = varSys%method%val(derVarPos(1)%velocity)%auxField_varPos(1:3)
    
    QQ = stencil%QQ
    nScalars = varSys%nScalars

    !NEC$ ivdep
    do iElem = 1, nSolve
      !NEC$ shortloop
      do iDir = 1, QQ
        pdf(iDir) = state(?FETCH?(iDir, 1, iElem, QQ, nScalars, nSize, neigh))
      end do
      
      ! element offset
      elemoff = (iElem-1)*varSys%nAuxScalars

      ! density
      rho = sum( pdf )
      ! store density
      auxField(elemOff+dens_pos) = rho

      ! store velocity
      inv_rho = 1.0_rk/rho
      auxField(elemOff+vel_pos(1)) = sum( pdf * stencil%cxDirRK(1,:) ) * inv_rho 
      auxField(elemOff+vel_pos(2)) = sum( pdf * stencil%cxDirRK(2,:) ) * inv_rho
      auxField(elemOff+vel_pos(3)) = sum( pdf * stencil%cxDirRK(3,:) ) * inv_rho

    end do
  end subroutine mus_calcAuxField_lbm
  ! ************************************************************************* !

  ! ************************************************************************* !
  !> This routine compute auxFields density and velocity for incompressible model
  !! for fluid and nGhostFromCoarser elements
  subroutine mus_calcAuxField_lbm_incomp(auxField, state, neigh, nSize, &
    & nSolve, iLevel, stencil, varSys, derVarPos)
    ! -------------------------------------------------------------------------- !
    !> output auxField array
    real(kind=rk), intent(inout) :: auxField(:)
    !> input state array
    real(kind=rk), intent(in) :: state(:)
    !> connectivity array
    integer, intent(in) :: neigh(:)
    !> number of elements excluding halos
    integer, intent(in) :: nSolve
    !> number of elements in the state array
    integer, intent(in) :: nSize
    !> current level
    integer, intent(in) :: iLevel
    !> stencil header 
    type(tem_stencilHeader_type), intent(in) :: stencil
    !> variable system definition
    type(tem_varSys_type), intent(in) :: varSys
    !> position of derived quantities in varsys
    type(mus_derVarPos_type), intent(in) :: derVarPos(:)
    ! -------------------------------------------------------------------------- !
    integer :: dens_pos, vel_pos(3), elemOff
    real(kind=rk) :: pdf( stencil%QQ )
    integer :: iElem, iDir, QQ, nScalars
    ! -------------------------------------------------------------------------- !
    dens_pos = varSys%method%val(derVarPos(1)%density)%auxField_varPos(1)
    vel_pos = varSys%method%val(derVarPos(1)%velocity)%auxField_varPos(1:3)
    
    QQ = stencil%QQ
    nScalars = varSys%nScalars

    !NEC$ ivdep
    do iElem = 1, nSolve
      !NEC$ shortloop
      do iDir = 1, QQ
        pdf(iDir) = state(?FETCH?(iDir, 1, iElem, QQ, nScalars, nSize, neigh))
      end do
      
      ! element offset
      elemoff = (iElem-1)*varSys%nAuxScalars

      ! store density
      auxField(elemOff+dens_pos) = sum(pdf)

      ! store velocity
      auxField(elemOff+vel_pos(1)) = sum( pdf * stencil%cxDirRK(1,:) ) * rho0Inv 
      auxField(elemOff+vel_pos(2)) = sum( pdf * stencil%cxDirRK(2,:) ) * rho0Inv 
      auxField(elemOff+vel_pos(3)) = sum( pdf * stencil%cxDirRK(3,:) ) * rho0Inv 

    end do
  end subroutine mus_calcAuxField_lbm_incomp
  ! ************************************************************************* !

  ! ************************************************************************* !
  !> This routine compute zeroth moment from state and store in auxField.
  !! use this routine only for models which requires only zeroth-order
  !! moment as auxField
  subroutine mus_calcAuxField_zerothMoment(auxField, state, neigh, nSize, &
    & nSolve, iLevel, stencil, varSys, derVarPos)
    ! -------------------------------------------------------------------------- !
    !> output auxField array
    real(kind=rk), intent(inout) :: auxField(:)
    !> input state array
    real(kind=rk), intent(in) :: state(:)
    !> connectivity array
    integer, intent(in) :: neigh(:)
    !> number of elements excluding halos
    integer, intent(in) :: nSolve
    !> number of elements in the state array
    integer, intent(in) :: nSize
    !> current level
    integer, intent(in) :: iLevel
    !> stencil header 
    type(tem_stencilHeader_type), intent(in) :: stencil
    !> variable system definition
    type(tem_varSys_type), intent(in) :: varSys
    !> position of derived quantities in varsys
    type(mus_derVarPos_type), intent(in) :: derVarPos(:)
    ! -------------------------------------------------------------------------- !
    real(kind=rk) :: pdf( stencil%QQ )
    integer :: iElem, iDir
    ! -------------------------------------------------------------------------- !
    ! safety check
    if (varSys%nAuxScalars /= 1) then
      call tem_abort('CalcAuxField_zeroth moment can be used only when' &
        &          //'nAuxScalars = 1')
    end if

    !NEC$ ivdep
    do iElem = 1, nSolve
      !NEC$ shortloop
      do iDir = 1, stencil%QQ
        pdf(iDir) = state(                                                    &
          & ?FETCH?(iDir, 1, iElem, stencil%QQ, varSys%nScalars, nSize, neigh))
      end do
      
      ! store scalar zeroth moment 
      auxField(iElem) = sum(pdf)
    end do
  end subroutine mus_calcAuxField_zerothMoment
  ! ************************************************************************* !

  ! ************************************************************************* !
  !> This routine compute auxField density and velocity for each species
  !! for multicomponent models. The velocity computed here is only velocity
  !! of transformed PDF. The velocity of original PDF is computed by solving
  !! linear equation system in compute kernel and velocity in auxField is 
  !! updated there
  subroutine mus_calcAuxField_MS(auxField, state, neigh, nSize, &
    & nSolve, iLevel, stencil, varSys, derVarPos)
    ! ------------------------------------------------------------------------ !
    !> output auxField array
    real(kind=rk), intent(inout) :: auxField(:)
    !> input state array
    real(kind=rk), intent(in) :: state(:)
    !> connectivity array
    integer, intent(in) :: neigh(:)
    !> number of elements excluding halos
    integer, intent(in) :: nSolve
    !> number of elements in the state array
    integer, intent(in) :: nSize
    !> current level
    integer, intent(in) :: iLevel
    !> stencil header 
    type(tem_stencilHeader_type), intent(in) :: stencil
    !> variable system definition
    type(tem_varSys_type), intent(in) :: varSys
    !> position of derived quantities in varsys
    type(mus_derVarPos_type), intent(in) :: derVarPos(:)
    ! ------------------------------------------------------------------------ !
    integer :: iElem, iFld, iDir, elemOff, nScalars
    integer :: QQ, nFields, dens_pos, vel_pos(3)
    real(kind=rk) :: pdf( stencil%QQ )
    real(kind=rk) :: rho, inv_rho
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    ! ------------------------------------------------------------------------ !
    call C_F_POINTER( varSys%method%val(1)%method_Data, fPtr )
    scheme => fPtr%solverData%scheme

    nFields = scheme%nFields
    QQ = stencil%QQ
    nScalars = varSys%nScalars

    !NEC$ ivdep
    do iElem = 1, nSolve
      ! element offset
      elemoff = (iElem-1)*varSys%nAuxScalars

      !NEC$ shortloop
      do iFld = 1, scheme%nFields
        do iDir = 1, QQ
          pdf(iDir) = state(?FETCH?(iDir, iFld, iElem, QQ, nScalars, nSize, neigh))
        end do
        ! position of density and velocity of current field in auxField array
        dens_pos = varSys%method%val(derVarPos(iFld)%density)%auxField_varPos(1)
        vel_pos = varSys%method%val(derVarPos(iFld)%velocity)%auxField_varPos(:)

        ! mass density of species
        rho = sum(pdf) 

        ! store density
        auxField(elemOff+dens_pos) = rho 
        
        ! store velocity
        inv_rho = 1.0_rk/rho
        auxField(elemOff+vel_pos(1)) = sum(pdf * stencil%cxDirRK(1, :))* inv_rho
        auxField(elemOff+vel_pos(2)) = sum(pdf * stencil%cxDirRK(2, :))* inv_rho
        auxField(elemOff+vel_pos(3)) = sum(pdf * stencil%cxDirRK(3, :))* inv_rho
      end do !iField
    end do   

  end subroutine mus_calcAuxField_MS
  ! ************************************************************************* !

  ! ************************************************************************* !
  !> Dummy routine for calcAuxField
  subroutine mus_calcAuxField_dummy(auxField, state, neigh, nSize, nSolve, &
    & iLevel, stencil, varSys, derVarPos)
    ! -------------------------------------------------------------------------- !
    !> output auxField array
    real(kind=rk), intent(inout) :: auxField(:)
    !> input state array
    real(kind=rk), intent(in) :: state(:)
    !> connectivity array
    integer, intent(in) :: neigh(:)
    !> number of elements excluding halos
    integer, intent(in) :: nSolve
    !> number of elements in the state array
    integer, intent(in) :: nSize
    !> current level
    integer, intent(in) :: iLevel
    !> stencil header 
    type(tem_stencilHeader_type), intent(in) :: stencil
    !> variable system definition
    type(tem_varSys_type), intent(in) :: varSys
    !> position of derived quantities in varsys
    type(mus_derVarPos_type), intent(in) :: derVarPos(:)
    ! -------------------------------------------------------------------------- !
    call tem_abort('Dummy routine for calcAuxField')
  end subroutine mus_calcAuxField_dummy
  ! ************************************************************************* !

  ! ************************************************************************** !
  subroutine mus_addSrcToAuxField_lbm(fun, auxField, iLevel, time, varSys, &
    &                                 phyConvFac, derVarPos) 
    ! ------------------------------------------------------------------------ !
    !> Description of method to update source
    class(mus_source_op_type), intent(in) :: fun
    !> output auxField array
    real(kind=rk), intent(inout)         :: auxField(:)
    !> current level
    integer, intent(in)                :: iLevel
    !> current timing information
    type(tem_time_type), intent(in)    :: time
    !> variable system definition
    type(tem_varSys_type), intent(in) :: varSys
    !> Physics conversion factor for current level
    type(mus_convertFac_type), intent(in) :: phyConvFac
    !> position of derived quantities in varsys
    type(mus_derVarPos_type), intent(in) :: derVarPos(:)
    ! ------------------------------------------------------------------------ !
    integer :: dens_pos, vel_pos(3)
    real(kind=rk) :: forceTerm(3)
    integer :: iElem, nElems, posInTotal, elemOff
    real(kind=rk) :: forceField(fun%elemLvl(iLevel)%nElems*3), inv_rho
    ! ------------------------------------------------------------------------ !
    ! position of density and velocity field in auxField
    dens_pos = varSys%method%val(derVarPos(1)%density)%auxField_varPos(1)
    vel_pos = varSys%method%val(derVarPos(1)%velocity)%auxField_varPos(1:3)
    ! Number of elements to apply source terms
    nElems = fun%elemLvl(iLevel)%nElems
    ! Get force which is refered in config file either its
    ! spacetime variable or operation variable
    call varSys%method%val(fun%data_varPos)%get_valOfIndex( &
      & varSys  = varSys,                                   &
      & time    = time,                                     &
      & iLevel  = iLevel,                                   &
      & idx     = fun%elemLvl(iLevel)%idx(1:nElems),        &
      & nVals   = nElems,                                   & 
      & res     = forceField                                )

    ! convert physical to lattice
    forceField = forceField / phyConvFac%body_force

!$omp parallel do schedule(static), private( posInTotal, forceTerm, inv_rho, elemOff )
    !NEC$ ivdep
    do iElem = 1, nElems
      posInTotal = fun%elemLvl(iLevel)%posInTotal(iElem)
      ! element offset
      elemoff = (posInTotal-1)*varSys%nAuxScalars
      ! inverse of density
      inv_rho = 1.0_rk/auxField(elemOff+dens_pos) 
      ! forceterm to add to velocity: F/(2*rho)
      forceTerm = forceField((iElem-1)*3+1 : iElem*3)*0.5_rk*inv_rho 
      ! add force to velocity
      auxField(elemOff+vel_pos(1)) = auxField(elemOff+vel_pos(1)) + forceTerm(1)
      auxField(elemOff+vel_pos(2)) = auxField(elemOff+vel_pos(2)) + forceTerm(2)
      auxField(elemOff+vel_pos(3)) = auxField(elemOff+vel_pos(3)) + forceTerm(3)
    end do 

  end subroutine mus_addSrcToAuxField_lbm
  ! ************************************************************************** !

  ! ************************************************************************** !
  subroutine mus_addSrcToAuxField_lbm_incomp(fun, auxField, iLevel, time, &
    & varSys, phyConvFac, derVarPos)
    ! ------------------------------------------------------------------------ !
    !> Description of method to update source
    class(mus_source_op_type), intent(in) :: fun
    !> output auxField array
    real(kind=rk), intent(inout)         :: auxField(:)
    !> current level
    integer, intent(in)                :: iLevel
    !> current timing information
    type(tem_time_type), intent(in)    :: time
    !> variable system definition
    type(tem_varSys_type), intent(in) :: varSys
    !> Physics conversion factor for current level
    type(mus_convertFac_type), intent(in) :: phyConvFac
    !> position of derived quantities in varsys
    type(mus_derVarPos_type), intent(in) :: derVarPos(:)
    ! ------------------------------------------------------------------------ !
    integer :: vel_pos(3), elemOff
    real(kind=rk) :: forceTerm(3)
    integer :: iElem, nElems, posInTotal 
    real(kind=rk) :: forceField(fun%elemLvl(iLevel)%nElems*3)
    ! ------------------------------------------------------------------------ !
    ! position of velocity field in auxField
    vel_pos = varSys%method%val(derVarPos(1)%velocity)%auxField_varPos(1:3)
    ! Number of elements to apply source terms
    nElems = fun%elemLvl(iLevel)%nElems
    ! Get force which is refered in config file either its
    ! spacetime variable or operation variable
    call varSys%method%val(fun%data_varPos)%get_valOfIndex( &
      & varSys  = varSys,                                   &
      & time    = time,                                     &
      & iLevel  = iLevel,                                   &
      & idx     = fun%elemLvl(iLevel)%idx(1:nElems),        &
      & nVals   = nElems,                                   & 
      & res     = forceField                                )

    ! convert physical to lattice
    forceField = forceField / phyConvFac%body_force

!$omp parallel do schedule(static), private( posInTotal, forceTerm, elemOff )
    !NEC$ ivdep
    do iElem = 1, nElems
      posInTotal = fun%elemLvl(iLevel)%posInTotal(iElem)
      ! element offset
      elemoff = (posInTotal-1)*varSys%nAuxScalars
      ! forceterm to add to velocity: F/(2*rho)
      forceTerm = forceField((iElem-1)*3+1 : iElem*3)*0.5_rk*rho0Inv
      ! add force to velocity
      auxField(elemOff+vel_pos(1)) = auxField(elemOff+vel_pos(1)) + forceTerm(1)
      auxField(elemOff+vel_pos(2)) = auxField(elemOff+vel_pos(2)) + forceTerm(2)
      auxField(elemOff+vel_pos(3)) = auxField(elemOff+vel_pos(3)) + forceTerm(3)
    end do 

  end subroutine mus_addSrcToAuxField_lbm_incomp
  ! ************************************************************************** !

  ! ************************************************************************** !
  subroutine mus_addSrcToAuxField_dummy(fun, auxField, iLevel, time, varSys, &
    &                                   phyConvFac, derVarPos)
    ! ------------------------------------------------------------------------ !
    !> Description of method to update source
    class(mus_source_op_type), intent(in) :: fun
    !> output auxField array
    real(kind=rk), intent(inout)         :: auxField(:)
    !> current level
    integer, intent(in)                :: iLevel
    !> current timing information
    type(tem_time_type), intent(in)    :: time
    !> variable system definition
    type(tem_varSys_type), intent(in) :: varSys
    !> Physics conversion factor for current level
    type(mus_convertFac_type), intent(in) :: phyConvFac
    !> position of derived quantities in varsys
    type(mus_derVarPos_type), intent(in) :: derVarPos(:)
    ! ------------------------------------------------------------------------ !
    ! abort only if source is active and function pointer is not assinged
    write(logUnit(6),*) 'WARNING: Dummy routine for addSrcToAuxField'
  end subroutine mus_addSrcToAuxField_dummy
  ! ************************************************************************** !


end module mus_auxFieldVar_module

