! See copyright notice in the COPYRIGHT file.
! **************************************************************************** !
!> author: Kannan Masilamani
!! This module provides the MUSUBI specific functions for calculating
!! macroscopic quantities from the state variables for multispecies liquid model.
!!
!! The depending common interface between MUSUBI and ATELES is defined in the
!! [[tem_derived_module]]. The functionality for accessing a variable from the
!! state
!! and evaluating a lua function are also provided in the 
!! [[tem_derived_module]].
!!
!! Do not use get_Element or get_Point routines to update the state !
!!
?? include 'header/lbm_macros.inc'
?? include 'header/lbm_deriveMacros.inc'
?? include 'header/lbm_interfaceMacros.inc'
?? include 'treelm/source/deriveMacros.inc'
module mus_derQuanMSLiquid_module
  use iso_c_binding, only: c_loc, c_ptr, c_f_pointer

  ! include treelm modules
  use env_module,               only: rk, long_k, globalMaxLevels, labelLen
  use tem_param_module,         only: div1_2, div1_3, div1_54, div1_9, div3_4, &
    &                                 sqrt3, cs2inv, cs2, t2cs2inv, t2cs4inv,  &
    &                                 cs4inv
  use tem_aux_module,           only: tem_abort
  use tem_varSys_module,        only: tem_varSys_type, tem_varSys_op_type,     &
    &                                 tem_varSys_append_derVar,                &
    &                                 tem_varSys_proc_point,                   &
    &                                 tem_varSys_proc_element,                 &
    &                                 tem_varSys_proc_setParams,               &
    &                                 tem_varSys_proc_getParams,               &
    &                                 tem_varSys_proc_setupIndices,            &
    &                                 tem_varSys_proc_getValOfIndex,           &
    &                                 tem_varSys_getPoint_dummy,               &
    &                                 tem_varSys_getElement_dummy,             &
    &                                 tem_varSys_setupIndices_dummy,           &
    &                                 tem_varSys_getValOfIndex_dummy,          &
    &                                 tem_varSys_setParams_dummy,              &
    &                                 tem_varSys_getParams_dummy
  use tem_variable_module,      only: tem_variable_type
  use tem_stencil_module,       only: tem_stencilHeader_type
  use tem_logging_module,       only: logUnit
  use tem_topology_module,      only: tem_levelOf
  use tem_time_module,          only: tem_time_type
  use treelmesh_module,         only: treelmesh_type
  use tem_math_module,          only: invert_matrix
  use tem_spatial_module,       only: tem_spatial_for
  use tem_subTree_type_module,  only: tem_subTree_type, tem_treeIDfrom_subTree
  use tem_debug_module,         only: dbgUnit
  use tem_operation_var_module, only: tem_evalMag_forElement,                &
    &                                 tem_evalMag_forPoint,                  &
    &                                 tem_evalMag_fromIndex,                 &
    &                                 tem_opVar_setupIndices,                &
    &                                 tem_get_new_varSys_data_ptr,           &
    &                                 tem_evalAdd_forElement,                &
    &                                 tem_evalAdd_fromIndex,                 &
    &                                 tem_opVar_setParams, tem_opVar_getParams
  use tem_spacetime_fun_module, only: tem_spacetime_for,       &
    &                                 tem_st_fun_listElem_type
  use tem_tools_module,         only: tem_PositionInSorted
  use tem_dyn_array_module,     only: PositionOfVal
  use tem_grow_array_module,    only: grw_labelarray_type, append
  use tem_stencil_module,       only: tem_stencilHeader_type

  ! include musubi modules
  use mus_source_var_module,     only: mus_source_op_type
  use mus_pdf_module,            only: pdf_data_type
  use mus_scheme_header_module,  only: mus_scheme_header_type
  use mus_scheme_type_module,    only: mus_scheme_type
  use mus_eNRTL_module,          only: mus_calc_thermFactor, &
    &                                  mus_calc_MS_DiffMatrix
  use mus_scheme_layout_module,  only: mus_scheme_layout_type
  use mus_varSys_module,         only: mus_varSys_data_type,        &
    &                                  mus_varSys_solverData_type,  &
    &                                  mus_get_new_solver_ptr,      &
    &                                  mus_deriveVar_ForPoint
  use mus_stateVar_module,       only: mus_accessVar_setupIndices  
  use mus_operation_var_module,  only: mus_opVar_setupIndices 
  use mus_derVarPos_type_module,     only: mus_derVarPos_type
  use mus_physics_module,            only: mus_convertFac_type

  ! include aotus modules
  use aotus_module, only: flu_State

  implicit none

  private

  public :: mus_append_derVar_MSLiquid
  public :: mus_append_derMixVar_MS
  public :: deriveMoleDensityMS
  public :: derivePressureMS
  public :: deriveMoleFluxMS
  public :: deriveDensityMS 

  public :: deriveEquilMSLiquid_FromMacro
  public :: deriveEqMSLiquid_FromState
  public :: deriveVelMSLiquid_FromState 
  public :: deriveMomMSLiquid_FromState
  public :: deriveMomentaMSLiquid_FromState
  public :: deriveVelocitiesMSLiquid_FromState
  public :: deriveAuxMSLiquid_fromState
  public :: deriveAuxMSLiquid_fromState_WTDF
  public :: deriveEquilMSLiquid_fromAux

  public :: momentumFromMacroLSE, momentumFromMacroLSE_WTDF
  ! routines which uses thermodynamic factor and variable diffusivities
  ! public :: deriveVelocityWTDF_MSLiquid_FromState
  !@todo TO IMPLEMENT
!  public :: deriveEquilWTDF_MSLiquid_FromState

  ! source variables
  public :: derive_electricFieldMSLiquid
  public :: derive_gravityFieldMSLiquid
  public :: applySrc_electricFieldMSLiquid
  public :: applySrc_gravityFieldMSLiquid
  public :: derive_electricFieldMSLiquid_WTDF
  public :: derive_gravityFieldMSLiquid_WTDF
  public :: applySrc_electricFieldMSLiquid_WTDF
  public :: applySrc_gravityFieldMSLiquid_WTDF

  public :: applySrc_electricFieldMSLiquid_1order_old
  public :: applySrc_electricFieldMSLiquid_diff
  public :: applySrc_electricFieldMSLiquid_const
  public :: applySrc_electricFieldMSLiquid_2order
  public :: applySrc_electricFieldMSLiquid_2order_new
  public :: applySrc_electricFieldMSLiquid_diff_WTDF

contains


  ! ************************************************************************ !
  !> subroutine to add derive variables for multi-species-liquid
  !! (schemekind = 'multi-species_liquid') to the varsys. 
  !! @todo KM: set proper velocity and equilbrium pointer for bgk_thermodynfac
  subroutine mus_append_derVar_MSLiquid( varSys, solverData, schemeHeader, &
    &                                    stencil, nFields, fldLabel,       &
    &                                    derVarName ) 
    ! -------------------------------------------------------------------- !
    !> global variable system
    type(tem_varSys_type), intent(inout)  :: varSys

    !> Contains pointer to solver data types
    type(mus_varSys_solverData_type), target, intent(in) :: solverData

    !> identifier of the scheme
    type(mus_scheme_header_type), intent(in)  :: schemeHeader

    !> compute stencil defintion
    type(tem_stencilHeader_type), intent(in)       :: stencil

    !> number of fields
    integer, intent(in)                      :: nFields

    !> array of field label prefix. Size=nFields
    character(len=*), intent(in)              :: fldLabel(:)

    !> array of derive physical variables
    type(grw_labelarray_type), intent(inout) :: derVarName
    ! -------------------------------------------------------------------- !
    ! number of derive variables
    integer :: nDerVars, iVar, nComponents, addedPos, iIn
    integer :: iField, iField_2
    logical :: wasAdded
    character(len=labelLen), allocatable ::  input_varname(:)
    character(len=labelLen)  ::  varName
    procedure(tem_varSys_proc_point), pointer :: get_point => NULL()
    procedure(tem_varSys_proc_element), pointer :: get_element => NULL()
    procedure(tem_varSys_proc_setParams), pointer :: set_params => null()
    procedure(tem_varSys_proc_getParams), pointer :: get_params => null()
    procedure(tem_varSys_proc_setupIndices), pointer :: &
      &                                      setup_indices => null()
    procedure(tem_varSys_proc_getValOfIndex), pointer :: &
      &                                       get_valOfIndex => null()
    type(c_ptr) :: method_data  
    character(len=labelLen), allocatable :: derVarName_loc(:)
    ! -------------------------------------------------------------------- !
    nullify(get_point, get_element, set_params, get_params, setup_indices, &
      &     get_valOfIndex)

    nDerVars = 11
    allocate(derVarName_loc(nDerVars))
    derVarName_loc    = [ 'pressure          ', 'equilibrium       ', &
      &                   'kinematic_pressure', 'equilibrium_vel   ', &
      &                   'mole_density      ', 'mole_fraction     ', &
      &                   'mole_flux         ', 'momentum          ', &
      &                   'vel_mag           ', 'charge_density    ', &
      &                   'current_density   '                        ] 

    ! append local derVarName to growing array. 
    ! should be done only once for all fields
    do iVar = 1, nDerVars
      call append(derVarName, derVarName_loc(iVar))
    end do

    ! add variable for each field and add mixture variable later
    do iField = 1, nFields
      do iVar = 1, nDerVars

        ! set default pointers, overwrite if neccessary
        get_element => tem_varSys_getElement_dummy
        get_point => mus_deriveVar_ForPoint
        setup_indices => mus_opVar_setupIndices
        get_valOfIndex => tem_varSys_getValOfIndex_dummy
        method_data  = mus_get_new_solver_ptr(solverData)
        set_params => tem_varSys_setParams_dummy
        get_params => tem_varSys_getParams_dummy

        select case(trim(adjustl(derVarName_loc(iVar))))
!KM!        case ('density')
!KM!          get_element => deriveDensityMS
!KM!          nComponents = 1
!KM!          allocate(input_varname(1))
!KM!          input_varname(1) = 'pdf'
  
        case ('mole_density')
          get_element => deriveMoleDensityMS 
          nComponents = 1
          allocate(input_varname(1))
          input_varname(1) = 'pdf'
        
        case ('charge_density')
          get_element => deriveChargeDensity
          !KM \todo 20170906 Split mixture variables from field variable
          ! and introduce allpdf variable which most multispecies variable
          ! requires
          setup_indices => mus_accessVar_setupIndices
          get_valOfIndex => deriveChargeDensity_fromIndex
          nComponents = 1
          allocate(input_varname(1))
          input_varname(1) = 'pdf'

        case ('current_density')
          if (trim(schemeHeader%relaxation) == 'bgk_withthermodynfac' .or. &
            & trim(schemeHeader%relaxation) == 'mrt_withthermodynfac') then
            !call tem_abort('ERROR: Current_density variable not implemented')
            get_element => deriveCurrentDensity_WTDF
          else
            get_element => deriveCurrentDensity
          end if  
          nComponents = 3
          allocate(input_varname(1))
          input_varname(1) = 'pdf'
 
        case ('mole_fraction')
          get_element => deriveMoleFracMSLiquid 
          nComponents = 1
          allocate(input_varname(1))
          input_varname(1) = 'pdf'
  
!KM!        case ('velocity')
!KM!          if (trim(schemeHeader%relaxation) == 'bgk_withthermodynfac' .or. &
!KM!            & trim(schemeHeader%relaxation) == 'mrt_withthermodynfac') then
!KM!            get_element => deriveVelocityWTDF_MSLiquid 
!KM!          else  
!KM!            get_element => deriveVelocityMSLiquid 
!KM!          end if  
!KM!          nComponents = 3
!KM!          allocate(input_varname(1))
!KM!          input_varname(1) = 'pdf'
!@todo KM to be implemented
!KM!        case ('spc_velocities')
!KM!          if (trim(schemeHeader%relaxation) == 'bgk_withthermodynfac') then
!KM!            get_element => deriveSpcVelocitiesWTDF_MSLiquid 
!KM!          else  
!KM!            get_element => deriveSpcVelocitiesMSLiquid 
!KM!          end if  
!KM!          nComponents = 3*nFields
!KM!          allocate(input_varname(1))
!KM!          input_varname(1) = 'pdf'

        case ('mole_flux')
          ! mole_density * velocity
          get_element => deriveMoleFluxMS 
          nComponents = 3
          allocate(input_varname(2))
          input_varname(1) = 'mole_density'
          input_varname(2) = 'velocity'
  
        case ('pressure')
          get_element => derivePressureMS 
          nComponents = 1
          allocate(input_varname(1))
          input_varname(1) = 'pdf'
  
        case ('kinematic_pressure')
          get_element => deriveKinePressMSLiquid 
          nComponents = 1
          allocate(input_varname(1))
          input_varname(1) = 'pressure'
  
        case ('equilibrium_vel')
          if (trim(schemeHeader%relaxation) == 'bgk_withthermodynfac' .or. &
            & trim(schemeHeader%relaxation) == 'mrt_withthermodynfac') then
            get_element => deriveEquilVelWTDF_MSLiquid 
          else
            get_element => deriveEquilVelMSLiquid 
          end if  
          nComponents = 3
          allocate(input_varname(1))
          input_varname(1) = 'pdf'
  
        case ('equilibrium')
          if (trim(schemeHeader%relaxation) == 'bgk_withthermodynfac' .or. &
            & trim(schemeHeader%relaxation) == 'mrt_withthermodynfac') then
            get_element => deriveEquilWTDF_MSLiquid 
          else
            get_element => deriveEquilMSLiquid 
          end if  
          nComponents = stencil%QQ
          allocate(input_varname(1))
          input_varname(1) = 'pdf'
  
        case ('momentum')
          if (trim(schemeHeader%relaxation) == 'bgk_withthermodynfac' .or. &
            & trim(schemeHeader%relaxation) == 'mrt_withthermodynfac') then
            get_element => deriveMomentumWTDF_MSLiquid 
          else
            get_element => deriveMomentumMSLiquid 
          end if
          nComponents = 3 
          allocate(input_varname(1))
          input_varname(1) = 'pdf'
  
!@todo KM to be implemented
!KM!        case ('spc_momenta')
!KM!          if (trim(schemeHeader%relaxation) == 'bgk_withthermodynfac') then
!KM!            get_element => deriveSpcMomentaWTDF_MSLiquid 
!KM!          else
!KM!            get_element => deriveSpcMomentaMSLiquid 
!KM!          end if
!KM!          nComponents = 3*nFields 
!KM!          allocate(input_varname(1))
!KM!          input_varname(1) = 'pdf'

        case ('vel_mag')
          get_element => tem_evalMag_forElement
          get_point => tem_evalMag_forPoint
          get_valOfIndex => tem_evalMag_fromIndex
          setup_indices => tem_opVar_setupIndices
          method_data = tem_get_new_varSys_data_ptr(method_data)
          nComponents = 1 
          allocate(input_varname(1))
          input_varname(1) = 'velocity'
  
        case default
          write(logUnit(1),*) 'WARNING: Unknown variable: '//&
            &                 trim(derVarName_loc(iVar))
          cycle !go to next variable
        end select
  
        ! update variable names with field label
        varname = trim(fldLabel(iField))//trim(adjustl(derVarName_loc(iVar)))
        do iIn = 1, size(input_varname)
          input_varname(iIn) = trim(fldLabel(iField))//trim(input_varname(iIn))
        end do

        ! append variable to varSys
        call tem_varSys_append_derVar( me             = varSys,         &
          &                            varName        = trim(varname),  &
          &                            nComponents    = nComponents,    &
          &                            input_varname  = input_varname,  &
          &                            method_data    = method_data,    &
          &                            get_point      = get_point,      &
          &                            get_element    = get_element,    &
          &                            set_params     = set_params,     &
          &                            get_params     = get_params,     &
          &                            setup_indices  = setup_indices,  &
          &                            get_valOfIndex = get_valOfIndex, &
          &                            pos            = addedPos,       &
          &                            wasAdded       = wasAdded        )

        if (wasAdded) then
          write(logUnit(10),*) ' Appended variable: '//trim(varname)
        else if (addedpos < 1) then
          write(logUnit(1),*) 'Error: variable '//trim(varname)// &
            &                 ' is not added to variable system'
        end if

        deallocate(input_varname)
      end do !iVar
    end do !iField

  end subroutine mus_append_derVar_MSLiquid
  ! ************************************************************************ !

  ! ************************************************************************ !
  !> Append mixture variables for multicomponent models
  subroutine mus_append_derMixVar_MS( varSys, solverData, nFields, fldLabel, &
    &                                 derVarName ) 
    ! -------------------------------------------------------------------- !
    !> global variable system
    type(tem_varSys_type), intent(inout)                 :: varSys
    !> Contains pointer to solver data types
    type(mus_varSys_solverData_type), target, intent(in) :: solverData
    !> number of fields
    integer, intent(in)                                  :: nFields
    !> array of field label prefix. Size=nFields
    character(len=*), intent(in)                         :: fldLabel(:)
    !> array of derive physical variable
    type(grw_labelarray_type), intent(in)                :: derVarName
    ! -------------------------------------------------------------------- !
    ! number of derive variables
    integer :: iVar, nComponents, addedPos, inPos(nFields)
    integer :: iField
    logical :: wasAdded
    character(len=labelLen) ::  input_varname(nFields)
    character(len=labelLen)  ::  varName
    procedure(tem_varSys_proc_point), pointer :: get_point => NULL()
    procedure(tem_varSys_proc_element), pointer :: get_element => NULL()
    procedure(tem_varSys_proc_setParams), pointer :: set_params => null()
    procedure(tem_varSys_proc_getParams), pointer :: get_params => null()
    procedure(tem_varSys_proc_setupIndices), pointer :: &
      &                                      setup_indices => null()
    procedure(tem_varSys_proc_getValOfIndex), pointer :: &
      &                                       get_valOfIndex => null()
    type(c_ptr) :: method_data  
    ! -------------------------------------------------------------------- !

    ! append mixture variable for every derived species variable
    do iVar = 1, derVarName%nVals
      ! set pointers for mixture. mixture quantity is obtained by summing up
      ! species quantities
      get_element => tem_evalAdd_forElement
      get_point => mus_deriveVar_ForPoint
      setup_indices => tem_opVar_setupIndices
      get_valOfIndex => tem_evalAdd_fromIndex
      method_data = tem_get_new_varSys_data_ptr(method_data)
      set_params => tem_opVar_setParams 
      get_params => tem_opVar_getParams

      varname = trim(adjustl(derVarName%val(iVar)))
      do iField = 1, nFields
        input_varname(iField) = trim(fldlabel(iField))//trim(varname)
        ! position of input variable in varSys
        inPos(iField) = PositionofVal(varSys%varname, input_varname(iField))
      end do

      
      ! input variable not found in varSys. Goto next variable
      if (any(inPos <= 0)) cycle

      ! get nComponents from dependent variable
      nComponents = varSys%method%val(inPos(1))%nComponents  

      ! append variable to varSys
      call tem_varSys_append_derVar(  me             = varSys,         &
        &                             varName        = trim(varname),  &
        &                             nComponents    = nComponents,    &
        &                             input_varname  = input_varname,  &
        &                             method_data    = method_data,    &
        &                             get_point      = get_point,      &
        &                             get_element    = get_element,    &
        &                             set_params     = set_params,     &
        &                             get_params     = get_params,     &
        &                             setup_indices  = setup_indices,  &
        &                             get_valOfIndex = get_valOfIndex, &
        &                             pos            = addedPos,       &
        &                             wasAdded       = wasAdded        )

      if (wasAdded) then
        write(logUnit(10),*) ' Appended variable: '//trim(varname)
      else if (addedpos < 1) then
        write(logUnit(1),*) 'Error: variable '//trim(varname)// &
          &                 ' is not added to variable system'
      end if

    end do !iVar

  end subroutine mus_append_derMixVar_MS
  ! ************************************************************************ !

  ! ************************************************************************ !
  !      Subroutines with common interface for the function pointers         !
  ! ************************************************************************ !

  ! ************************************************************************ !
  !> Calculate the number density of a given element for single species or mixture
  !! from the cptr scheme state vector for liquid mixture.
  !! Number density = density/molecular weight
  !! mixture number density = sum(number_density)
?? copy :: get_element_headtxt(deriveMoleDensityMS)
    ! -------------------------------------------------------------------- !
    integer :: statePos, iElem, iLevel, iDep, iComp, varPos
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    integer :: pdfPos
    real(kind=rk) :: mass_dens, num_dens
    integer :: nSize
    ! -------------------------------------------------------------------- !
    call C_F_POINTER( fun%method_Data, fPtr )
    scheme => fPtr%solverData%scheme

    res = 0.0_rk

    do iElem = 1, nElems
      num_dens = 0.0_rk
      ! if state array is defined level wise then use levelPointer(pos)
      ! to access state array
      statePos = fPtr%solverData%geometry%levelPointer( elemPos(iElem) )
      iLevel = tem_levelOf( tree%treeID( elemPos(iElem) ) )
      nSize = scheme%pdf( iLevel )%nSize

      ! use iDep here to use this routine to compute also mixture density
      ! in multi-species
      do iDep = 1, fun%nInputs
        pdfPos = fun%input_varPos(iDep)

        ! mass density of species
        mass_dens = 0.0_rk
        do iComp = 1, varSys%method%val(pdfPos)%nComponents
          varPos = varSys%method%val(pdfPos)%state_varPos(iComp)
          mass_dens =  mass_dens + scheme%state( iLevel )%val(    &
          ! position of this state variable in the state array
          & ?IDX?(varPos, statePos, varSys%nScalars, nSize ),     &
          & scheme%pdf( iLevel )%nNext )
        end do

        ! number density of species
        num_dens = num_dens + mass_dens                         &
          & * scheme%field(pdfPos)%fieldProp%species%molWeightInv
      end do !iDep

      res( iElem ) = num_dens

    end do ! iElem

  end subroutine deriveMoleDensityMS
  ! ************************************************************************ !


  ! ************************************************************************ !
  !> Calculate charge density of a given element for mixture.
  !! Charge density is mixture quantity to it returns same value for all
  !! species
  !! charge_density = Faraday * \sum_i z_i*density_i/molecularWeight_i
?? copy :: get_element_headtxt(deriveChargeDensity)
    ! -------------------------------------------------------------------- !
    integer :: statePos, iElem, iLevel, iField, iDir, varPos, nFields
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    integer :: pdfPos
    real(kind=rk) :: mass_dens, charge_dens
    integer :: nSize
    ! -------------------------------------------------------------------- !
    call C_F_POINTER( fun%method_Data, fPtr )
    scheme => fPtr%solverData%scheme
    nFields = scheme%nFields

    res = 0.0_rk

    do iElem = 1, nElems
      charge_dens = 0.0_rk
      ! if state array is defined level wise then use levelPointer(pos)
      ! to access state array
      statePos = fPtr%solverData%geometry%levelPointer( elemPos(iElem) )
      iLevel = tem_levelOf( tree%treeID( elemPos(iElem) ) )
      nSize = scheme%pdf( iLevel )%nSize

      do iField = 1, nFields
        pdfPos = scheme%stateVarMap%varPos%val(iField)

        ! mass density of species
        mass_dens = 0.0_rk
        do iDir = 1, scheme%layout%fStencil%QQ
          varPos = varSys%method%val(pdfPos)%state_varPos(iDir)
          mass_dens =  mass_dens + scheme%state( iLevel )%val(    &
          ! position of this state variable in the state array
          & ?IDX?(varPos, statePos, varSys%nScalars, nSize ),     &
          & scheme%pdf( iLevel )%nNext )
        end do

        ! charge density
        charge_dens = charge_dens + mass_dens                     &
          & * scheme%field(iField)%fieldProp%species%molWeightInv &
          & * scheme%field(iField)%fieldProp%species%chargeNr
      end do !iField

      res( iElem ) = charge_dens * scheme%mixture%faradayLB

    end do ! iElem

  end subroutine deriveChargeDensity
  ! ************************************************************************ !


  ! ************************************************************************** !
  !> Calculate charge density from species concentration
  !!
  !! The interface has to comply to the abstract interface
  !! [[tem_varSys_module:tem_varSys_proc_getValOfIndex]].
  !! 
?? copy :: get_valofindex_headtxt(deriveChargeDensity_fromIndex)
    ! ---------------------------------------------------------------------------
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    integer :: iDir, iSrc, iVal, elemPos, first, last, nSize, loc_level
    integer :: nSrcElems, statePos, pdfPos, varPos
    integer :: iField, nFields
    real(kind=rk), allocatable :: srcRes(:), allpdf(:)
    real(kind=rk) :: weight, mass_dens, charge_dens
    ! ---------------------------------------------------------------------------

?? if( DEBUG ) then
    ! check if number of index are the same as number of values asked for
    if (size(idx) /= nVals) then
      call tem_abort(                                                         &
        & 'in deriveChargeDensity_fromIndex: not the same amount of index and'& 
        & //'values are provided, stopping')
    end if
?? endif

    call C_F_POINTER( fun%method_Data, fPtr )
    scheme => fPtr%solverData%scheme
    nFields = scheme%nFields
    allocate(srcRes(scheme%layout%fStencil%QQ*varSys%nScalars))
    allocate(allpdf(varSys%nScalars))

    res = 0.0_rk

    ! distinguish if we have an array of index or we have contingous memory
    ! access where index are always first entries!
    if (present(idxLen)) then
      call tem_abort('Error: idxLen is not supported in get_valOfIndex for ' &
        &          //'state variable')
    else  
      ! Get the state value at the specific point
      do iVal = 1, nVals
        if (idx(iVal)>0) then
  
          ! elemPos in tree 
          ! elemPos = fPtr%pointData%pntLvl(iLevel)%elemPos%val(idx(iVal))
          first = fPtr%pointData%pntLvl(iLevel)%srcElem%first%val(idx(iVal))
          last = fPtr%pointData%pntLvl(iLevel)%srcElem%last%val(idx(iVal))

          ! get pdf's of source elements
          srcRes = 0.0_rk
          nSrcElems = 0
          do iSrc = first, last
            nSrcElems = nSrcElems + 1

            elemPos = fPtr%pointData%pntLvl(iLevel)%srcElem%elemPos%val(iSrc)
            ! position of element in levelDesc total list
            statePos = fPtr%solverData%geometry%levelPointer( elemPos )
            loc_level = tem_levelOf( fPtr%solverData%geometry    &
              &                          %tree%treeID( elemPos ) )
            nSize = scheme%pdf( loc_level )%nSize

            do iField = 1, nFields
              pdfPos = scheme%stateVarMap%varPos%val(iField)
              
              do iDir = 1, scheme%layout%fStencil%QQ
                varPos = varSys%method%val(pdfPos)%state_varPos(iDir)
                srcRes( (nSrcElems-1)*varSys%nScalars + varPos )    &
                  & = scheme%state( loc_level )%val(                &
                  & ?IDX?( varPos,statePos,varSys%nScalars,nSize ), &
                  & scheme%pdf( loc_level )%nNext )
              end do !iComp
            end do !iField 
          end do !iSrc 

          ! Linear interpolation res = sum(weight_i*phi_i)
          allpdf = 0.0_rk
          nSrcElems = 0
          do iSrc = first, last
            weight = fPtr%pointData%pntLvl(iLevel)%srcElem%weight%val(iSrc)
            nSrcElems = nSrcElems + 1
            allpdf(:) = allpdf(:) + weight                &
              & * srcRes( (nSrcElems-1)*varSys%nScalars+1 &
              &         : nSrcElems*varSys%nScalars )
          end do

          charge_dens = 0.0_rk
          do iField = 1, nFields
            pdfPos = scheme%stateVarMap%varPos%val(iField)
    
            ! mass density of species
            mass_dens = 0.0_rk
            do iDir = 1, scheme%layout%fStencil%QQ
              mass_dens =  mass_dens +                                 &
                & allPdf( varSys%method%val(pdfPos)%state_varPos(iDir) )
            end do
    
            ! charge density
            charge_dens = charge_dens + mass_dens                     &
              & * scheme%field(iField)%fieldProp%species%molWeightInv &
              & * scheme%field(iField)%fieldProp%species%chargeNr
          end do !iField

          ! get the state value for each component of this 
          res( iVal ) = charge_dens * scheme%mixture%faradayLB
        end if !idx>0
      end do !iVal
    end if !idx_len  

  end subroutine deriveChargeDensity_fromIndex
  ! ************************************************************************** !


  ! ************************************************************************ !
  !> Current density, J = charge_density*velocity = \rho_e * u
?? copy :: get_element_headtxt(deriveCurrentDensity)
    ! -------------------------------------------------------------------- !
    integer :: statePos, iElem, iComp, iLevel, iField, nFields
    integer :: varPos, QQ
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    real(kind=rk), allocatable :: tmpPDF(:)
    real(kind=rk) :: curr_dens(3), vel(3)
    !mass density of nSpecies
    real(kind=rk) :: mass_dens( varSys%nStateVars )
    !number density of nSpecies 
    real(kind=rk) :: num_dens( varSys%nStateVars )
    !mole fraction
    real(kind=rk) :: moleFraction( varSys%nStateVars )
    real(kind=rk) :: totNum_densInv
    !first moments of nSpecies
    real(kind=rk) :: first_moments( 3, varSys%nStateVars )
    !momentum from linear system of equation
    real(kind=rk) :: momentum( 3, varSys%nStateVars )
    !parameters from solver specific conf
    !field specific info from field table
    real(kind=rk), dimension(varSys%nStateVars) :: phi
    real(kind=rk) :: resi_coeff( varSys%nStateVars, varSys%nStateVars )
    !mixture info
    real(kind=rk) :: omega_diff, omega_fac, paramBInv
    integer :: nSize
    integer :: stateVarMap(varSys%nStateVars)
    ! -------------------------------------------------------------------- !
    call C_F_POINTER( fun%method_Data, fPtr )
    scheme => fPtr%solverData%scheme

    nFields = scheme%nFields
    QQ = scheme%layout%fStencil%QQ
    allocate(tmpPDF(QQ))
    stateVarMap = scheme%stateVarMap%varPos%val(:)

    do iField = 1, nFields
      ! species properties
      ! molecular weight ratio
      phi(iField) = scheme%field(iField)%fieldProp%species%molWeigRatio
      ! resistivity coefficients
      resi_coeff(iField, :) =                                     &
        & scheme%field(iField)%fieldProp%species%resi_coeff(:)
    end do 

    omega_diff = scheme%mixture%omega_diff
    omega_fac = omega_diff * 0.5_rk

    paramBInv = 1.0_rk / scheme%mixture%paramB

    do iElem = 1, nElems
      ! if state array is defined level wise then use levelPointer(pos)
      ! to access state array
      statePos = fPtr%solverData%geometry%levelPointer( elemPos(iElem) )
      iLevel = tem_levelOf( tree%treeID( elemPos(iElem) ) )
      nSize = scheme%pdf( iLevel )%nSize

      do iField = 1, nFields

        do iComp = 1, QQ 
          varPos = varSys%method%val(stateVarMap(iField))%state_varPos(iComp)

          tmpPDF(iComp) = scheme%state( iLevel )%val(                 &
            ! position of this state variable in the state array
            & ?IDX?(varPos, statePos, varSys%nScalars, nSize ),       &
            & scheme%pdf( iLevel )%nNext ) 
        end do

        ! mass density of species
        mass_dens(iField ) = sum( tmpPDF ) 

        ! number density of species   
        num_dens(iField) = mass_dens(iField)                    &
          & * scheme%field(iField)%fieldProp%species%molWeightInv

        ! velocity, first moments
        do iComp = 1, 3
          first_moments(iComp, iField) = sum( tmpPDF *        &
            & scheme%layout%fStencil%cxDirRK(iComp, :) )
        end do  

      end do !iField

      ! total number density Inv
      totNum_densInv = 1.0_rk/sum(num_dens(:))

      ! molefraction
      moleFraction(:) = num_dens(:)*totNum_densInv

      ! momentum of all species
      momentum = momentumFromMacroLSE( moleFraction  = moleFraction,   &
        &                              first_moments = first_moments,  &
        &                              nFields       = nFields,        &
        &                              phi           = phi,            &
        &                              resi_coeff    = resi_coeff,     &
        &                              omega_fac     = omega_fac,      &
        &                              paramBInv     = paramBInv       )

      ! current density= F \sum_i (z_i*n_i*v_i)
      curr_dens = 0.0_rk
      do iField = 1, nFields
        vel(:) = momentum( :, iField ) / mass_dens(iField)
        curr_dens = curr_dens + num_dens(iField) * vel &
          & * scheme%field(iField)%fieldProp%species%chargeNr
      end do !idep

      ! copy the results to the res 
      res( (iElem-1)*fun%nComponents + 1 : iElem*fun%nComponents) &
        & = curr_dens * scheme%mixture%faradayLB
    end do

  end subroutine deriveCurrentDensity
  ! ************************************************************************ !

  ! ************************************************************************ !
  !> Calculate Current density with thermodynamic factor
  !! J = charge_density*velocity = \rho_e * u
?? copy :: get_element_headtxt(deriveCurrentDensity_WTDF)
    ! -------------------------------------------------------------------- !
    integer :: statePos, iElem, iComp, iLevel, iField, nFields
    integer :: varPos, QQ
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    real(kind=rk), allocatable :: tmpPDF(:)
    real(kind=rk) :: curr_dens(3), vel(3)
    !mass density of nSpecies
    real(kind=rk) :: mass_dens( varSys%nStateVars )
    !number density of nSpecies 
    real(kind=rk) :: num_dens( varSys%nStateVars )
    !mole fraction
    real(kind=rk) :: moleFraction( varSys%nStateVars )
    real(kind=rk) :: totNum_densInv, press, temp, phy_moleDens_fac
    !first moments of nSpecies
    real(kind=rk) :: first_moments( 3, varSys%nStateVars )
    !momentum from linear system of equation
    real(kind=rk) :: momentum( 3, varSys%nStateVars )
    !parameters from solver specific conf
    !field specific info from field table
    real(kind=rk), dimension(varSys%nStateVars) :: phi
    real(kind=rk), dimension(varSys%nStateVars, varSys%nStateVars) :: &
      & resi_coeff, thermodynamic_fac, &
      & inv_thermodyn_fac, diff_coeff
    !mixture info
    real(kind=rk) :: omega_diff, omega_fac, paramBInv
    integer :: nSize
    integer :: stateVarMap(varSys%nStateVars)
    ! -------------------------------------------------------------------- !
    call C_F_POINTER( fun%method_Data, fPtr )
    scheme => fPtr%solverData%scheme

    nFields = scheme%nFields
    QQ = scheme%layout%fStencil%QQ
    allocate(tmpPDF(QQ))
    stateVarMap = scheme%stateVarMap%varPos%val(:)

    do iField = 1, nFields
      ! species properties
      ! molecular weight ratio
      phi(iField) = scheme%field(iField)%fieldProp%species%molWeigRatio
      ! resistivity coefficients
      resi_coeff(iField, :) =                                     &
        & scheme%field(iField)%fieldProp%species%resi_coeff(:)
    end do 

    !KM \todo check moleDens for multilevel
    phy_moleDens_fac = fPtr%solverData%physics%moleDens0

    omega_diff = scheme%mixture%omega_diff
    omega_fac = omega_diff * 0.5_rk

    paramBInv = 1.0_rk / scheme%mixture%paramB

    ! temperature
    temp = scheme%mixture%temp0
    ! atmospheric pressure
    press = scheme%mixture%atm_press

    ! initialize thermodynamic factor
    ! and compute it from matthias c-code for each element
    thermodynamic_fac = 0.0_rk
    do iField = 1, nFields
      thermodynamic_fac(iField, iField) = 1.0_rk
    end do

    do iElem = 1, nElems
      ! if state array is defined level wise then use levelPointer(pos)
      ! to access state array
      statePos = fPtr%solverData%geometry%levelPointer( elemPos(iElem) )
      iLevel = tem_levelOf( tree%treeID( elemPos(iElem) ) )
      nSize = scheme%pdf( iLevel )%nSize

      do iField = 1, nFields

        do iComp = 1, QQ 
          varPos = varSys%method%val(stateVarMap(iField))%state_varPos(iComp)

          tmpPDF(iComp) = scheme%state( iLevel )%val(                 &
            ! position of this state variable in the state array
            & ?IDX?(varPos, statePos, varSys%nScalars, nSize ),       &
            & scheme%pdf( iLevel )%nNext ) 
        end do

        ! mass density of species
        mass_dens(iField ) = sum( tmpPDF ) 

        ! number density of species   
        num_dens(iField) = mass_dens(iField)                    &
          & * scheme%field(iField)%fieldProp%species%molWeightInv

        ! velocity, first moments
        do iComp = 1, 3
          first_moments(iComp, iField) = sum( tmpPDF *        &
            & scheme%layout%fStencil%cxDirRK(iComp, :) )
        end do  

      end do !iField

      ! total number density Inv
      totNum_densInv = 1.0_rk/sum(num_dens(:))

      ! molefraction
      moleFraction(:) = num_dens(:)*totNum_densInv

      ! MS-Diff coeff matrix from C++ code
      call mus_calc_MS_DiffMatrix( nFields, temp, press,                 &
        &                          num_dens*phy_moleDens_fac, diff_coeff )

      ! Convert to lattice unit
      resi_coeff = fPtr%solverData%physics%fac(iLevel)%diffusivity/diff_coeff

      ! Thermodynamic factor from C++ code
      call mus_calc_thermFactor( nFields, temp, press, moleFraction, &
        &                        thermodynamic_fac )

      inv_thermodyn_fac = invert_matrix( thermodynamic_fac )

      ! momentum of all species
      momentum = momentumFromMacroLSE_WTDF(                                 &
        &                            moleFraction      = moleFraction,      &
        &                            first_moments     = first_moments,     &
        &                            nFields           = nFields,           &
        &                            inv_thermodyn_fac = inv_thermodyn_fac, &
        &                            phi               = phi,               &
        &                            resi_coeff        = resi_coeff,        &
        &                            omega_fac         = omega_fac,         &
        &                            paramBInv         = paramBInv          )

      ! current density= F \sum_i (z_i*n_i*v_i)
      curr_dens = 0.0_rk
      do iField = 1, nFields
        vel(:) = momentum( :, iField ) / mass_dens(iField)
        curr_dens = curr_dens + num_dens(iField) * vel &
          & * scheme%field(iField)%fieldProp%species%chargeNr
      end do !idep

      ! copy the results to the res 
      res( (iElem-1)*fun%nComponents + 1 : iElem*fun%nComponents) &
        & = curr_dens * scheme%mixture%faradayLB
    end do

  end subroutine deriveCurrentDensity_WTDF
  ! ************************************************************************ !


  ! ************************************************************************ !
  !> Calculate mixture kinematic pressure. 
  !! This routine requires initial total mole density which is defined in
  !! mixture table.
  !! Formula to compute kinematic pressure
  !! \( p = c^2_s (\sum_k \rho_k \phi_k - min_l (m_l) n_0)/\rho_0 \)
  !! here, \( \rho_k \) - species density, \\
  !! \( \phi_k \) - species molecular weight ratio, \\
  !! \( n_0 \) - reference mixture number density,\\
  !! \( \rho_0 \) - reference density.
  !! In tracking,
  !!```lua
  !! variable = {{"kinematic_pressure"}}
  !!```
?? copy :: get_element_headtxt(deriveKinePressMSLiquid)
    ! -------------------------------------------------------------------- !
    integer :: press_pos 
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    real(kind=rk) :: min_molWeight, moleDens0, rho0, const_press
    ! -------------------------------------------------------------------- !
    call C_F_POINTER( fun%method_Data, fPtr )
    scheme => fPtr%solverData%scheme

    ! position of pressue in glob system
    press_pos = fun%input_varPos(1)

    if( varSys%method%val( press_pos )%nInputs /= scheme%nFields) then
      write(logUnit(1),*)' ERROR: Kinematic pressure can be derived only '//   &
        &                'for mixture'
      call tem_abort()
    endif

    ! derive dependent variable, mixture pressue
    call varSys%method%val(press_pos)%get_element( varSys  = varSys,  &
      &                                            elemPos = elemPos, &
      &                                            time    = time,    &
      &                                            tree    = tree,    &
      &                                            nElems  = nElems,  &
      &                                            nDofs   = nDofs,   &
      &                                            res     = res      )

    min_molWeight = minval(scheme%field(:)%fieldProp%species%molWeight)
    moleDens0 = scheme%mixture%moleDens0
    rho0 = scheme%mixture%rho0
    
    const_press = min_molWeight * moleDens0 * cs2 / rho0

    ! p = cs2 (sum_k( rho_k*phi_k ) - min_l (m_l) n_0 ) / rho_0
    res = ( res / rho0 - const_press )

  end subroutine deriveKinePressMSLiquid
  ! ************************************************************************ !


  ! ************************************************************************ !
  !> Calculate species and total pressure both for gas and liquid model 
  !! Note: This routine can also be used to compute species pressure
  !! In case of gas mixture, it is partial pressure where as in
  !! liquid mixture this is not valid. 
  !! Formula to ressure
  !! \( p = c^2_s (\sum_k \rho_k \phi_k ) \)
  !! here, \( \rho_k \) - species density, \\
  !! \( \phi_k \) - species molecular weight ratio, \\
?? copy :: get_element_headtxt(derivePressureMS)
    ! -------------------------------------------------------------------- !
    integer :: statePos, iElem, iLevel, iDep, iComp, varPos
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    integer :: pdfPos
    real(kind=rk) :: mass_dens, press 
    integer :: nSize
    ! -------------------------------------------------------------------- !
    call C_F_POINTER( fun%method_Data, fPtr )
    scheme => fPtr%solverData%scheme

    res = 0.0_rk

    do iElem = 1, nElems
      mass_dens = 0.0_rk
      press = 0.0_rk
      ! if state array is defined level wise then use levelPointer(pos)
      ! to access state array
      statePos = fPtr%solverData%geometry%levelPointer( elemPos(iElem) )
      iLevel = tem_levelOf( tree%treeID( elemPos(iElem) ) )
      nSize = scheme%pdf( iLevel )%nSize

      ! use iDep here to use this routine to compute also mixture density
      ! in multi-species
      do iDep = 1, fun%nInputs
        pdfPos = fun%input_varPos(iDep)

        ! mass density of species
        mass_dens = 0.0_rk
        do iComp = 1, varSys%method%val(pdfPos)%nComponents
          varPos = varSys%method%val(pdfPos)%state_varPos(iComp)
          mass_dens =  mass_dens + scheme%state( iLevel )%val(        &
          ! position of this state variable in the state array
          & ?IDX?(varPos, statePos, varSys%nScalars, nSize ),         &
          & scheme%pdf( iLevel )%nNext ) 
        end do

        ! pressue p_k = cs2 * rho_k * phi_k 
        ! multiply factor cs2 after dep since for mixture pressue
        ! p = cs2 * sum_k(rho_k*phi_k)
        press = press + mass_dens &
          & * scheme%field(pdfPos)%fieldProp%species%molWeigRatio

      end do !iDep

      res( iElem ) = press * cs2

    end do ! iElem

  end subroutine derivePressureMS
  ! ************************************************************************ !


  ! ************************************************************************ !
  !> Calculate the velocity of a given element for single species or mixture
  !! from the cptr scheme state vector for liquid mixture.
  !! Need to solve the system of equations to compute this velocity since
  !! first order moments gives only moments of transformed pdfs. Hence
  !! to obtain the first order moments of actual pdfs we need to solve
  !! system of equation of size = nSpecies for each velocity components
  !! @todo Add equation and reference
?? copy :: get_element_headtxt(deriveVelocityMSLiquid)
    ! -------------------------------------------------------------------- !
    integer :: statePos, iElem, iComp, iLevel, iDep, iField, nFields
    integer :: varPos, QQ
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    integer :: pdfPos
    real(kind=rk), allocatable :: tmpPDF(:)
    real(kind=rk) :: dens, vel( fun%nComponents )
    !mass density of nSpecies
    real(kind=rk) :: mass_dens( varSys%nStateVars )
    !number density of nSpecies 
    real(kind=rk) :: num_dens( varSys%nStateVars )
    !mole fraction
    real(kind=rk) :: moleFraction( varSys%nStateVars )
    real(kind=rk) :: totNum_densInv
    !first moments of nSpecies
    real(kind=rk) :: first_moments( fun%nComponents, varSys%nStateVars )
    !momentum from linear system of equation
    real(kind=rk) :: momentum( fun%nComponents, varSys%nStateVars )
    !parameters from solver specific conf
    !field specific info from field table
    real(kind=rk), dimension(varSys%nStateVars) :: phi
    real(kind=rk) :: resi_coeff( varSys%nStateVars, varSys%nStateVars )
    !mixture info
    real(kind=rk) :: omega_diff, omega_fac, paramBInv
    integer :: nSize
    integer :: stateVarMap(varSys%nStateVars)
    ! -------------------------------------------------------------------- !
    call C_F_POINTER( fun%method_Data, fPtr )
    scheme => fPtr%solverData%scheme

    nFields = scheme%nFields
    QQ = scheme%layout%fStencil%QQ
    allocate(tmpPDF(QQ))
    stateVarMap = scheme%stateVarMap%varPos%val(:)

    do iField = 1, nFields
      ! species properties
      ! molecular weight ratio
      phi(iField) = scheme%field(iField)%fieldProp%species%molWeigRatio
      ! resistivity coefficients
      resi_coeff(iField, :) =                                     &
        & scheme%field(iField)%fieldProp%species%resi_coeff(:)
    end do 

    omega_diff = scheme%mixture%omega_diff
    omega_fac = omega_diff * 0.5_rk

    paramBInv = 1.0_rk / scheme%mixture%paramB

    do iElem = 1, nElems
      ! if state array is defined level wise then use levelPointer(pos)
      ! to access state array
      statePos = fPtr%solverData%geometry%levelPointer( elemPos(iElem) )
      iLevel = tem_levelOf( tree%treeID( elemPos(iElem) ) )
      nSize = scheme%pdf( iLevel )%nSize

      do iField = 1, nFields

        do iComp = 1, QQ 
          varPos = varSys%method%val(stateVarMap(iField))%state_varPos(iComp)

          tmpPDF(iComp) = scheme%state( iLevel )%val(                  &
            ! position of this state variable in the state array
            & ?IDX?(varPos, statePos, varSys%nScalars, nSize ),        &
            & scheme%pdf( iLevel )%nNext ) 
        end do

        ! mass density of species
        mass_dens(iField ) = sum( tmpPDF ) 

        ! number density of species   
        num_dens(iField) = mass_dens(iField)                    &
          & * scheme%field(iField)%fieldProp%species%molWeightInv

        ! velocity, first moments
        do iComp = 1, 3
          first_moments(iComp, iField) = sum( tmpPDF *        &
            & scheme%layout%fStencil%cxDirRK(iComp, :) )
        end do  

      end do !iField

      ! total number density Inv
      totNum_densInv = 1.0_rk/sum(num_dens(:))

      ! molefraction
      moleFraction(:) = num_dens(:)*totNum_densInv

      ! momentum of all species
      momentum = momentumFromMacroLSE( moleFraction  = moleFraction,   &
        &                              first_moments = first_moments,  &
        &                              nFields       = nFields,        &
        &                              phi           = phi,            &
        &                              resi_coeff    = resi_coeff,     &
        &                              omega_fac     = omega_fac,      &
        &                              paramBInv     = paramBInv       )

      ! find required species or mixture velocity   
      dens = 0.0_rk
      vel = 0.0_rk
      ! varDep >1 only for mixture velocity
      do iDep = 1, fun%nInputs
        pdfPos = fun%input_varPos(iDep)
             
        dens = dens + mass_dens( pdfPos )    
        vel(:) = vel(:) + momentum( :, pdfPos )
      end do !idep

      ! copy the results to the res 
      res( (iElem-1)*fun%nComponents + 1 : iElem*fun%nComponents) = vel/dens
    end do

  end subroutine deriveVelocityMSLiquid  
  ! ************************************************************************ !


  ! ************************************************************************ !
  !> Calculate the velocity of a given element for single species or mixture
  !! from the cptr scheme state vector for liquid mixture with thermodynamic
  !! factor and variable diffusivity coeffcients.
  !! Need to solve the system of equations to compute this velocity since
  !! first order moments gives only moments of transformed pdfs. Hence
  !! to obtain the first order moments of actual pdfs we need to solve
  !! system of equation of size = nSpecies for each velocity components
  !! @todo Add equation and reference
?? copy :: get_element_headtxt(deriveVelocityWTDF_MSLiquid)
    ! -------------------------------------------------------------------- !
    integer :: statePos, iElem, iComp, iLevel, iDep, iField, nFields
    integer :: varPos, QQ
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    integer :: pdfPos
    real(kind=rk), allocatable :: tmpPDF(:)
    real(kind=rk) :: dens, vel( fun%nComponents )
    !mass density of nSpecies
    real(kind=rk) :: mass_dens( varSys%nStateVars )
    !number density of nSpecies 
    real(kind=rk) :: num_dens( varSys%nStateVars )
    !mole fraction
    real(kind=rk) :: moleFraction( varSys%nStateVars )
    real(kind=rk) :: totNum_densInv, press, temp, phy_moleDens_fac
    !first moments of nSpecies
    real(kind=rk) :: first_moments( fun%nComponents, varSys%nStateVars )
    !momentum from linear system of equation
    real(kind=rk) :: momentum( fun%nComponents, varSys%nStateVars )
    !parameters from solver specific conf
    !field specific info from field table
    real(kind=rk), dimension(varSys%nStateVars) :: phi
    real(kind=rk), dimension(varSys%nStateVars, varSys%nStateVars) :: &
      & resi_coeff, thermodynamic_fac, &
      & inv_thermodyn_fac, diff_coeff
    !mixture info
    real(kind=rk) :: omega_diff, omega_fac, paramBInv
    integer :: nSize
    integer :: stateVarMap(varSys%nStateVars)
    ! -------------------------------------------------------------------- !
    call C_F_POINTER( fun%method_Data, fPtr )
    scheme => fPtr%solverData%scheme

    nFields = scheme%nFields
    QQ = scheme%layout%fStencil%QQ
    allocate(tmpPDF(QQ))
    stateVarMap = scheme%stateVarMap%varPos%val(:)

    do iField = 1, nFields
      ! species properties
      ! molecular weight ratio
      phi(iField) = scheme%field(iField)%fieldProp%species%molWeigRatio
      ! diffusivity coefficients
      diff_coeff(iField, :) =                                     &
        & scheme%field(iField)%fieldProp%species%diff_coeff(:)
    end do 

    !KM \todo check moleDens for multilevel
    phy_moleDens_fac = fPtr%solverData%physics%moleDens0

    omega_diff = scheme%mixture%omega_diff
    omega_fac = omega_diff * 0.5_rk

    paramBInv = 1.0_rk / scheme%mixture%paramB

    ! temperature
    temp = scheme%mixture%temp0
    ! atmospheric pressure
    press = scheme%mixture%atm_press

    ! initialize thermodynamic factor
    ! and compute it from matthias c-code for each element
    thermodynamic_fac = 0.0_rk
    do iField = 1, nFields
      thermodynamic_fac(iField, iField) = 1.0_rk
    end do

    do iElem = 1, nElems
      ! if state array is defined level wise then use levelPointer(pos)
      ! to access state array
      statePos = fPtr%solverData%geometry%levelPointer( elemPos(iElem) )
      iLevel = tem_levelOf( tree%treeID( elemPos(iElem) ) )
      nSize = scheme%pdf( iLevel )%nSize

      do iField = 1, nFields

        do iComp = 1, QQ 
          varPos = varSys%method%val(stateVarMap(iField))%state_varPos(iComp)

          tmpPDF(iComp) = scheme%state( iLevel )%val(               &
            ! position of this state variable in the state array
            & ?IDX?(varPos, statePos, varSys%nScalars, nSize ),     &
            & scheme%pdf( iLevel )%nNext )
        end do

        ! mass density of species
        mass_dens(iField ) = sum( tmpPDF )

        ! number density of species
        num_dens(iField) = mass_dens(iField)                    &
          & * scheme%field(iField)%fieldProp%species%molWeightInv

        ! velocity, first moments
        do iComp = 1, fun%nComponents
          first_moments(iComp, iField) = sum( tmpPDF * &
            & scheme%layout%fStencil%cxDirRK(iComp, :) )
        end do

      end do !iField

      ! total number density Inv
      totNum_densInv = 1.0_rk/sum(num_dens(:))

      ! molefraction
      moleFraction(:) = num_dens(:)*totNum_densInv

      ! MS-Diff coeff matrix from C++ code
      call mus_calc_MS_DiffMatrix( nFields, temp, press,                 &
        &                          num_dens*phy_moleDens_fac, diff_coeff )

      ! Convert to lattice unit
      resi_coeff = fPtr%solverData%physics%fac(iLevel)%diffusivity/diff_coeff

      ! Thermodynamic factor from C++ code
      call mus_calc_thermFactor( nFields, temp, press, moleFraction,        &
        &                        thermodynamic_fac )

      inv_thermodyn_fac = invert_matrix( thermodynamic_fac )

      ! momentum of all species
      momentum = momentumFromMacroLSE_WTDF(                                 &
        &                            moleFraction      = moleFraction,      &
        &                            first_moments     = first_moments,     &
        &                            nFields           = nFields,           &
        &                            inv_thermodyn_fac = inv_thermodyn_fac, &
        &                            phi               = phi,               &
        &                            resi_coeff        = resi_coeff,        &
        &                            omega_fac         = omega_fac,         &
        &                            paramBInv         = paramBInv          )

      ! find required species or mixture velocity   
      dens = 0.0_rk
      vel = 0.0_rk
      ! varDep >1 only for mixture velocity
      do iDep = 1, fun%nInputs
        pdfPos = fun%input_varPos(iDep)
             
        dens = dens + mass_dens( pdfPos )    
        vel(:) = vel(:) + momentum( :, pdfPos )
      end do !idep

      ! copy the results to the res 
      res( (iElem-1)*fun%nComponents + 1 : iElem*fun%nComponents) = vel/dens
    end do


  end subroutine deriveVelocityWTDF_MSLiquid  
  ! ************************************************************************ !


  ! ************************************************************************ !
  !> Calculate the momentum of a given element for single species or mixture
  !! from the cptr scheme state vector for liquid mixture.
  !! Need to solve the system of equations to compute this momentum since
  !! first order moments gives only moments of transformed pdfs. Hence
  !! to obtain the first order moments of actual pdfs we need to solve
  !! system of equation of size = nSpecies for each velocity components
  !! @todo Add equation and reference
?? copy :: get_element_headtxt(deriveMomentumMSLiquid)
    ! -------------------------------------------------------------------- !
    integer :: statePos, iElem, iComp, iLevel, iDep, iField, nFields
    integer :: varPos, QQ
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    integer :: pdfPos
    real(kind=rk), allocatable :: tmpPDF(:)
    real(kind=rk) :: mom( fun%nComponents )
    !mass density of nSpecies
    real(kind=rk) :: mass_dens( varSys%nStateVars )
    !number density of nSpecies 
    real(kind=rk) :: num_dens( varSys%nStateVars )
    !mole fraction
    real(kind=rk) :: moleFraction( varSys%nStateVars )
    real(kind=rk) :: totNum_densInv
    !first moments of nSpecies
    real(kind=rk) :: first_moments( fun%nComponents, varSys%nStateVars )
    !momentum from linear system of equation
    real(kind=rk) :: momentum( fun%nComponents, varSys%nStateVars )
    !parameters from solver specific conf
    !field specific info from field table
    real(kind=rk), dimension(varSys%nStateVars) :: phi
    real(kind=rk) :: resi_coeff( varSys%nStateVars, varSys%nStateVars )
    !mixture info
    real(kind=rk) :: omega_diff, omega_fac, paramBInv
    integer :: nSize
    integer :: stateVarMap(varSys%nStateVars)
    ! -------------------------------------------------------------------- !
    call C_F_POINTER( fun%method_Data, fPtr )
    scheme => fPtr%solverData%scheme

    nFields = scheme%nFields
    QQ = scheme%layout%fStencil%QQ
    allocate(tmpPDF(QQ))
    stateVarMap = scheme%stateVarMap%varPos%val(:)

    do iField = 1, nFields
      ! species properties
      ! molecular weight ratio
      phi(iField) = scheme%field(iField)%fieldProp%species%molWeigRatio
      ! resistivity coefficients
      resi_coeff(iField, :) =                                     &
        & scheme%field(iField)%fieldProp%species%resi_coeff(:)
    end do 

    omega_diff = scheme%mixture%omega_diff
    omega_fac = omega_diff * 0.5_rk

    paramBInv = 1.0_rk / scheme%mixture%paramB

    do iElem = 1, nElems
      ! if state array is defined level wise then use levelPointer(pos)
      ! to access state array
      statePos = fPtr%solverData%geometry%levelPointer( elemPos(iElem) )
      iLevel = tem_levelOf( tree%treeID( elemPos(iElem) ) )
      nSize = scheme%pdf( iLevel )%nSize

      do iField = 1, nFields
        do iComp = 1, QQ 
          varPos = varSys%method%val(stateVarMap(iField))%state_varPos(iComp)

          tmpPDF(iComp) = scheme%state( iLevel )%val(               &
            ! position of this state variable in the state array
            & ?IDX?(varPos, statePos, varSys%nScalars, nSize ),     &
            & scheme%pdf( iLevel )%nNext ) 
        end do  

        ! mass density of species
        mass_dens(iField ) = sum( tmpPDF ) 

        ! number density of species   
        num_dens(iField) = mass_dens(iField)                    &
          & * scheme%field(iField)%fieldProp%species%molWeightInv

        ! velocity, first moments
        do iComp = 1, fun%nComponents
          first_moments(iComp, iField) = sum( tmpPDF *        &
            & scheme%layout%fStencil%cxDirRK(iComp, :) )
        end do  

      end do !iField

      ! total number density Inv
      totNum_densInv = 1.0_rk/sum(num_dens(:))

      ! molefraction
      moleFraction(:) = num_dens(:)*totNum_densInv

      ! momentum of all species
      momentum = momentumFromMacroLSE( moleFraction  = moleFraction,   &
        &                              first_moments = first_moments,  &
        &                              nFields       = nFields,        &
        &                              phi           = phi,            &
        &                              resi_coeff    = resi_coeff,     &
        &                              omega_fac     = omega_fac,      &
        &                              paramBInv     = paramBInv       )

      ! find required species or mixture velocity   
      mom = 0.0_rk
      ! varDep >1 only for mixture velocity
      do iDep = 1, fun%nInputs
        pdfPos = fun%input_varPos(iDep)
        mom(:) = mom(:) + momentum( :, pdfPos )
      end do !idep

      ! copy the results to the res 
      res( (iElem-1)*fun%nComponents + 1 : iElem*fun%nComponents) = mom 
    end do

  end subroutine deriveMomentumMSLiquid  
  ! ************************************************************************ !


  ! ************************************************************************ !
  !> Calculate the momentum of a given element for single species or mixture
  !! from the cptr scheme state vector for liquid mixture.
  !! Need to solve the system of equations to compute this momentum since
  !! first order moments gives only moments of transformed pdfs. Hence
  !! to obtain the first order moments of actual pdfs we need to solve
  !! system of equation of size = nSpecies for each velocity components
  !! @todo Add equation and reference
?? copy :: get_element_headtxt(deriveMomentumWTDF_MSLiquid)
    ! -------------------------------------------------------------------- !
    integer :: statePos, iElem, iComp, iLevel, iDep, iField, nFields
    integer :: varPos, QQ
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    integer :: pdfPos
    real(kind=rk), allocatable :: tmpPDF(:)
    real(kind=rk) :: mom( fun%nComponents )
    !mass density of nSpecies
    real(kind=rk) :: mass_dens( varSys%nStateVars )
    !number density of nSpecies 
    real(kind=rk) :: num_dens( varSys%nStateVars )
    !mole fraction
    real(kind=rk) :: moleFraction( varSys%nStateVars )
    real(kind=rk) :: totNum_densInv, press, temp, phy_moleDens_fac
    !first moments of nSpecies
    real(kind=rk) :: first_moments( fun%nComponents, varSys%nStateVars )
    !momentum from linear system of equation
    real(kind=rk) :: momentum( fun%nComponents, varSys%nStateVars )
    !parameters from solver specific conf
    !field specific info from field table
    real(kind=rk), dimension(varSys%nStateVars) :: phi
    real(kind=rk), dimension(varSys%nStateVars, varSys%nStateVars) :: &
      & resi_coeff, thermodynamic_fac, &
      & inv_thermodyn_fac, diff_coeff
    !mixture info
    real(kind=rk) :: omega_diff, omega_fac, paramBInv
    integer :: nSize
    integer :: stateVarMap(varSys%nStateVars)
    ! -------------------------------------------------------------------- !
    call C_F_POINTER( fun%method_Data, fPtr )
    scheme => fPtr%solverData%scheme

    nFields = scheme%nFields
    QQ = scheme%layout%fStencil%QQ
    allocate(tmpPDF(QQ))
    stateVarMap = scheme%stateVarMap%varPos%val(:)

    do iField = 1, nFields
      ! species properties
      ! molecular weight ratio
      phi(iField) = scheme%field(iField)%fieldProp%species%molWeigRatio
      ! diffusivity coefficients
      diff_coeff(iField, :) =                                     &
        & scheme%field(iField)%fieldProp%species%diff_coeff(:)
    end do 
    !KM @todo check moleDens for multilevel
    phy_moleDens_fac = fPtr%solverData%physics%moleDens0

    omega_diff = scheme%mixture%omega_diff
    omega_fac = omega_diff * 0.5_rk

    paramBInv = 1.0_rk / scheme%mixture%paramB

    ! temperature
    temp = scheme%mixture%temp0
    ! atmospheric pressure
    press = scheme%mixture%atm_press

    ! initialize thermodynamic factor 
    ! and compute it from matthias c-code for each element
    thermodynamic_fac = 0.0_rk
    do iField = 1, nFields
      thermodynamic_fac(iField, iField) = 1.0_rk
    end do

    do iElem = 1, nElems
      ! if state array is defined level wise then use levelPointer(pos)
      ! to access state array
      statePos = fPtr%solverData%geometry%levelPointer( elemPos(iElem) )
      iLevel = tem_levelOf( tree%treeID( elemPos(iElem) ) )
      nSize = scheme%pdf( iLevel )%nSize

      do iField = 1, nFields

        do iComp = 1, QQ 
          varPos = varSys%method%val(stateVarMap(iField))%state_varPos(iComp)

          tmpPDF(iComp) = scheme%state( iLevel )%val(               &
            ! position of this state variable in the state array
            & ?IDX?(varPos, statePos, varSys%nScalars, nSize ),     &
            & scheme%pdf( iLevel )%nNext ) 
        end do  

        ! mass density of species
        mass_dens(iField ) = sum( tmpPDF ) 

        ! number density of species   
        num_dens(iField) = mass_dens(iField)                    &
          & * scheme%field(iField)%fieldProp%species%molWeightInv

        ! velocity, first moments
        do iComp = 1, fun%nComponents
          first_moments(iComp, iField) = sum( tmpPDF *        &
            & scheme%layout%fStencil%cxDirRK(iComp, :) )
        end do  

      end do !iField

      ! total number density Inv
      totNum_densInv = 1.0_rk/sum(num_dens(:))

      ! molefraction
      moleFraction(:) = num_dens(:)*totNum_densInv

      ! MS-Diff coeff matrix from C++ code
      call mus_calc_MS_DiffMatrix( nFields, temp, press,                 &
        &                          num_dens*phy_moleDens_fac, diff_coeff )

      ! Convert to lattice unit
      resi_coeff = fPtr%solverData%physics%fac(iLevel)%diffusivity/diff_coeff

      ! Thermodynamic factor from C++ code
      call mus_calc_thermFactor( nFields, temp, press, moleFraction,        &
        &                        thermodynamic_fac )

      inv_thermodyn_fac = invert_matrix( thermodynamic_fac )

      ! momentum of all species
      momentum = momentumFromMacroLSE_WTDF(                                 &
        &                            moleFraction      = moleFraction,      &
        &                            first_moments     = first_moments,     &
        &                            nFields           = nFields,           &
        &                            inv_thermodyn_fac = inv_thermodyn_fac, &
        &                            phi               = phi,               &
        &                            resi_coeff        = resi_coeff,        &
        &                            omega_fac         = omega_fac,         &
        &                            paramBInv         = paramBInv          )

      ! find required species or mixture velocity   
      mom = 0.0_rk
      ! varDep >1 only for mixture velocity
      do iDep = 1, fun%nInputs
        pdfPos = fun%input_varPos(iDep)
        mom(:) = mom(:) + momentum( :, pdfPos )
      end do !idep

      ! copy the results to the res 
      res( (iElem-1)*fun%nComponents + 1 : iElem*fun%nComponents) = mom 
    end do

  end subroutine deriveMomentumWTDF_MSLiquid  
  ! ************************************************************************ !


  ! ************************************************************************ !
  !> Equilibrium velocity from state
  !! Must compute species velocity of all species by solving LSE
  !!
?? copy :: get_element_headtxt(deriveEquilVelMSLiquid)
    ! -------------------------------------------------------------------- !
    integer :: statePos, iElem, iComp, iLevel, iField, nFields
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    integer :: pdfPos, varPos, QQ
    real(kind=rk), allocatable :: tmpPDF(:)
    !mass density of nSpecies
    real(kind=rk) :: mass_dens( varSys%nStateVars )
    !number density of nSpecies 
    real(kind=rk) :: num_dens( varSys%nStateVars )
    !mole fraction
    real(kind=rk) :: moleFraction( varSys%nStateVars )
    real(kind=rk) :: totNum_densInv
    !first moments of nSpecies
    real(kind=rk) :: first_moments( 3, varSys%nStateVars )
    !momentum from linear system of equation
    real(kind=rk) :: momentum( 3, varSys%nStateVars )
    real(kind=rk) :: vel( 3, varSys%nStateVars )
    real(kind=rk) :: eqVel(3)
    real(kind=rk), dimension(varSys%nStateVars) :: phi
    real(kind=rk) :: resi_coeff( varSys%nStateVars, varSys%nStateVars )
    !mixture info
    real(kind=rk) :: omega_diff, omega_fac, paramBInv, theta_eq
    integer :: nSize
    integer :: stateVarMap(varSys%nStateVars)
    ! -------------------------------------------------------------------- !
    call C_F_POINTER( fun%method_Data, fPtr )
    scheme => fPtr%solverData%scheme

    nFields = scheme%nFields
    QQ = scheme%layout%fStencil%QQ
    allocate(tmpPDF(QQ))
    stateVarMap = scheme%stateVarMap%varPos%val(:)

    ! equilibrium velocity can be computed only for single species
    pdfPos = fun%input_varPos(1)

    do iField = 1, nFields
      ! species properties
      ! molecular weight ratio
      phi(iField) = scheme%field(iField)%fieldProp%species%molWeigRatio
      ! resistivity coefficients
      resi_coeff(iField, :) =                                     &
        & scheme%field(iField)%fieldProp%species%resi_coeff(:)
    end do 

    omega_diff = scheme%mixture%omega_diff
    omega_fac = omega_diff * 0.5_rk

    paramBInv = 1.0_rk / scheme%mixture%paramB
    theta_eq = scheme%mixture%theta_eq

    do iElem = 1, nElems
      ! if state array is defined level wise then use levelPointer(pos)
      ! to access state array
      statePos = fPtr%solverData%geometry%levelPointer( elemPos(iElem) )
      iLevel = tem_levelOf( tree%treeID( elemPos(iElem) ) )
      nSize = scheme%pdf( iLevel )%nSize

      do iField = 1, nFields

        do iComp = 1, QQ 
          varPos = varSys%method%val(stateVarMap(iField))%state_varPos(iComp)

          tmpPDF(iComp) = scheme%state( iLevel )%val(                &
            ! position of this state variable in the state array
            & ?IDX?(varPos, statePos, varSys%nScalars, nSize ),      &
            & scheme%pdf( iLevel )%nNext ) 
        end do  

        ! mass density of species
        mass_dens(iField ) = sum( tmpPDF ) 

        ! number density of species   
        num_dens(iField) = mass_dens(iField)                    &
          & * scheme%field(iField)%fieldProp%species%molWeightInv

        ! velocity, first moments
        do iComp = 1, 3 
          first_moments(iComp, iField) = sum( tmpPDF *        &
            & scheme%layout%fStencil%cxDirRK(iComp, :) )
        end do  

      end do !iField

      ! total number density Inv
      totNum_densInv = 1.0_rk/sum(num_dens(:))

      ! molefraction
      moleFraction(:) = num_dens(:)*totNum_densInv

       ! momentum of all species
      momentum = momentumFromMacroLSE( moleFraction  = moleFraction,   &
        &                              first_moments = first_moments,  &
        &                              nFields       = nFields,        &
        &                              phi           = phi,            &
        &                              resi_coeff    = resi_coeff,     &
        &                              omega_fac     = omega_fac,      &
        &                              paramBInv     = paramBInv       )

      !velocity of all species
      do iField = 1, nFields
        vel( :, iField) = momentum( :, iField) / mass_dens(iField)
      end do

      eqVel = equilVelFromMacro( iField       = pdfPos,               &
        &                        moleFraction = moleFraction,         &
        &                        velocity     = vel,                  &
        &                        nFields      = nFields,              &
        &                        paramBInv    = paramBInv,            &
        &                        phi          = phi(pdfPos),          &
        &                        resi_coeff   = resi_coeff(pdfPos, :) )

      ! copy the results to the res 
      res( (iElem-1)*fun%nComponents + 1 : iElem*fun%nComponents) = eqVel 
 
    end do 
  end subroutine deriveEquilVelMSLiquid
  ! ************************************************************************ !


  ! ************************************************************************ !
  !> Equilibrium velocity from state with thermodynamic factor
  !! Must compute species velocity of all species by solving LSE
  !!
?? copy :: get_element_headtxt(deriveEquilVelWTDF_MSLiquid)
    ! -------------------------------------------------------------------- !
    integer :: statePos, iElem, iComp, iLevel, iField, nFields
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    integer :: pdfPos, varPos, QQ
    real(kind=rk), allocatable :: tmpPDF(:)
    !mass density of nSpecies
    real(kind=rk) :: mass_dens( varSys%nStateVars )
    !number density of nSpecies 
    real(kind=rk) :: num_dens( varSys%nStateVars )
    !mole fraction
    real(kind=rk) :: moleFraction( varSys%nStateVars )
    real(kind=rk) :: totNum_densInv, press, temp, phy_moleDens_fac
    !first moments of nSpecies
    real(kind=rk) :: first_moments( 3, varSys%nStateVars )
    !momentum from linear system of equation
    real(kind=rk) :: momentum( 3, varSys%nStateVars )
    real(kind=rk) :: vel( 3, varSys%nStateVars )
    real(kind=rk) :: eqVel(3)
    real(kind=rk), dimension(varSys%nStateVars) :: phi
    real(kind=rk), dimension(varSys%nStateVars, varSys%nStateVars) :: &
      & resi_coeff, thermodynamic_fac, &
      & inv_thermodyn_fac, diff_coeff
    !mixture info
    real(kind=rk) :: omega_diff, omega_fac, paramBInv, theta_eq
    integer :: nSize
    integer :: stateVarMap(varSys%nStateVars)
    ! -------------------------------------------------------------------- !
    call C_F_POINTER( fun%method_Data, fPtr )
    scheme => fPtr%solverData%scheme

    nFields = scheme%nFields
    QQ = scheme%layout%fStencil%QQ
    allocate(tmpPDF(QQ))
    stateVarMap = scheme%stateVarMap%varPos%val(:)

    ! equilibrium velocity can be computed only for single species
    pdfPos = fun%input_varPos(1)

    do iField = 1, nFields
      ! species properties
      ! molecular weight ratio
      phi(iField) = scheme%field(iField)%fieldProp%species%molWeigRatio
      ! diffusivity coefficients
      diff_coeff(iField, :) =                                     &
        & scheme%field(iField)%fieldProp%species%diff_coeff(:)
    end do 
    !KM @todo check moleDens for multilevel
    phy_moleDens_fac = fPtr%solverData%physics%moleDens0

    omega_diff = scheme%mixture%omega_diff
    omega_fac = omega_diff * 0.5_rk

    paramBInv = 1.0_rk / scheme%mixture%paramB
    theta_eq = scheme%mixture%theta_eq

    ! temperature
    temp = scheme%mixture%temp0
    ! atmospheric pressure
    press = scheme%mixture%atm_press

    ! initialize thermodynamic factor
    ! and compute from matthias c-code for each element
    thermodynamic_fac = 0.0_rk
    do iField = 1, nFields
      thermodynamic_fac(iField, iField) = 1.0_rk
    end do

    do iElem = 1, nElems
      ! if state array is defined level wise then use levelPointer(pos)
      ! to access state array
      statePos = fPtr%solverData%geometry%levelPointer( elemPos(iElem) )
      iLevel = tem_levelOf( tree%treeID( elemPos(iElem) ) )
      nSize = scheme%pdf( iLevel )%nSize

      do iField = 1, nFields

        do iComp = 1, QQ 
          varPos = varSys%method%val(stateVarMap(iField))%state_varPos(iComp)

          tmpPDF(iComp) = scheme%state( iLevel )%val(               &
            ! position of this state variable in the state array
            & ?IDX?(varPos, statePos, varSys%nScalars, nSize ),     &
            & scheme%pdf( iLevel )%nNext ) 
        end do  

        ! mass density of species
        mass_dens(iField ) = sum( tmpPDF ) 

        ! number density of species   
        num_dens(iField) = mass_dens(iField)                    &
          & * scheme%field(iField)%fieldProp%species%molWeightInv

        ! velocity, first moments
        do iComp = 1, 3 
          first_moments(iComp, iField) = sum( tmpPDF *        &
            & scheme%layout%fStencil%cxDirRK(iComp, :) )
        end do  

      end do !iField

      ! total number density Inv
      totNum_densInv = 1.0_rk/sum(num_dens(:))

      ! molefraction
      moleFraction(:) = num_dens(:)*totNum_densInv

      ! MS-Diff coeff matrix from C++ code
      call mus_calc_MS_DiffMatrix( nFields, temp, press,                 &
        &                          num_dens*phy_moleDens_fac, diff_coeff )

      ! Convert to lattice unit
      resi_coeff = fPtr%solverData%physics%fac(iLevel)%diffusivity/diff_coeff

      ! Thermodynamic factor from C++ code
      call mus_calc_thermFactor( nFields, temp, press, moleFraction,        &
        &                        thermodynamic_fac )

      inv_thermodyn_fac = invert_matrix( thermodynamic_fac )

      ! momentum of all species
      momentum = momentumFromMacroLSE_WTDF(                                 &
        &                            moleFraction      = moleFraction,      &
        &                            first_moments     = first_moments,     &
        &                            nFields           = nFields,           &
        &                            inv_thermodyn_fac = inv_thermodyn_fac, &
        &                            phi               = phi,               &
        &                            resi_coeff        = resi_coeff,        &
        &                            omega_fac         = omega_fac,         &
        &                            paramBInv         = paramBInv          )

      !velocity of all species
      do iField = 1, nFields
        vel( :, iField) = momentum( :, iField) / mass_dens(iField)
      end do

      eqVel = equilVelFromMacroWTDF( iField            = pdfPos,            &
        &                            mass_dens         = mass_dens,         &
        &                            moleFraction      = moleFraction,      &
        &                            velocity          = vel,               &
        &                            nFields           = nFields,           &
        &                            inv_thermodyn_fac = inv_thermodyn_fac, &
        &                            paramBInv         = paramBInv,         &
        &                            phi               = phi,               &
        &                            resi_coeff        = resi_coeff         )

      ! copy the results to the res 
      res( (iElem-1)*fun%nComponents + 1 : iElem*fun%nComponents) = eqVel 
 
    end do 
  end subroutine deriveEquilVelWTDF_MSLiquid
  ! ************************************************************************ !


  ! ************************************************************************ !
  !> Equilibrium from state
?? copy :: get_element_headtxt(deriveEquilMSLiquid)
    ! -------------------------------------------------------------------- !
    integer :: statePos, iElem, iComp, iLevel, iField, nFields
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    integer :: pdfPos, varPos, QQ
    real(kind=rk), allocatable :: tmpPDF(:)
    !mass density of nSpecies
    real(kind=rk) :: mass_dens( varSys%nStateVars )
    !number density of nSpecies 
    real(kind=rk) :: num_dens( varSys%nStateVars )
    !mole fraction
    real(kind=rk) :: moleFraction( varSys%nStateVars )
    real(kind=rk) :: totNum_densInv
    !first moments of nSpecies
    real(kind=rk) :: first_moments( 3, varSys%nStateVars )
    !momentum from linear system of equation
    real(kind=rk) :: momentum( 3, varSys%nStateVars )
    real(kind=rk) :: vel( 3, varSys%nStateVars )
    !parameters from solver specific conf
    !field specific info from field table
    real(kind=rk), dimension(varSys%nStateVars) :: phi
    real(kind=rk) :: resi_coeff( varSys%nStateVars, varSys%nStateVars )
    !mixture info
    real(kind=rk) :: omega_diff, paramBInv, theta_eq, weight0Inv, omega_fac
    real(kind=rk) :: fEq(fun%nComponents)
    integer :: nSize
    integer :: stateVarMap(varSys%nStateVars)
    ! -------------------------------------------------------------------- !
    call C_F_POINTER( fun%method_Data, fPtr )
    scheme => fPtr%solverData%scheme

    nFields = scheme%nFields
    QQ = scheme%layout%fStencil%QQ
    allocate(tmpPDF(QQ))
    stateVarMap = scheme%stateVarMap%varPos%val(:)

    ! equilibrium velocity can be computed only for single species
    pdfPos = fun%input_varPos(1)

    do iField = 1, nFields
      ! species properties
      ! molecular weight ratio
      phi(iField) = scheme%field(iField)%fieldProp%species%molWeigRatio
      ! resistivity coefficients
      resi_coeff(iField, :) =                                     &
        & scheme%field(iField)%fieldProp%species%resi_coeff(:)
    end do 

    omega_diff = scheme%mixture%omega_diff
    omega_fac = omega_diff * 0.5_rk

    paramBInv = 1.0_rk / scheme%mixture%paramB
    theta_eq = scheme%mixture%theta_eq

    weight0Inv = 1.0_rk / scheme%layout%weight( &
      &                              scheme%layout%fStencil%restPosition)
    do iElem = 1, nElems
      ! if state array is defined level wise then use levelPointer(pos)
      ! to access state array
      statePos = fPtr%solverData%geometry%levelPointer( elemPos(iElem) )
      iLevel = tem_levelOf( tree%treeID( elemPos(iElem) ) )
      nSize = scheme%pdf( iLevel )%nSize

      do iField = 1, nFields

        do iComp = 1, QQ 
          varPos = varSys%method%val(stateVarMap(iField))%state_varPos(iComp)

          tmpPDF(iComp) = scheme%state( iLevel )%val(              &
            ! position of this state variable in the state array
            & ?IDX?(varPos, statePos, varSys%nScalars, nSize ),    &
            & scheme%pdf( iLevel )%nNext ) 
        end do  

        ! mass density of species
        mass_dens(iField ) = sum( tmpPDF ) 

        ! number density of species   
        num_dens(iField) = mass_dens(iField)                    &
          & * scheme%field(iField)%fieldProp%species%molWeightInv

        ! velocity, first moments
        do iComp = 1, 3 
          first_moments(iComp, iField) = sum( tmpPDF *        &
            & scheme%layout%fStencil%cxDirRK(iComp, :) )
        end do  

      end do !iField

      ! total number density Inv
      totNum_densInv = 1.0_rk/sum(num_dens(:))

      ! molefraction
      moleFraction(:) = num_dens(:)*totNum_densInv

       ! momentum of all species
      momentum = momentumFromMacroLSE( moleFraction  = moleFraction,   &
        &                              first_moments = first_moments,  &
        &                              nFields       = nFields,        &
        &                              phi           = phi,            &
        &                              resi_coeff    = resi_coeff,     &
        &                              omega_fac     = omega_fac,      &
        &                              paramBInv     = paramBInv       )

      !velocity of all species
      do iField = 1, nFields
        vel( :, iField) = momentum( :, iField) / mass_dens(iField)
      end do

      ! compute equilibrium from macroscopic quantities
      fEq = equilFromMacro( iField       = pdfPos,                &
        &                   mass_dens    = mass_dens,             &
        &                   moleFraction = moleFraction,          &
        &                   velocity     = vel,                   &
        &                   layout       = scheme%layout,         &
        &                   nFields      = nFields,               &
        &                   paramBInv    = paramBInv,             &
        &                   phi          = phi(pdfPos),           &
        &                   resi_coeff   = resi_coeff(pdfPos, :), &
        &                   theta_eq     = theta_eq,              &
        &                   weight0Inv   = weight0Inv             )

      ! copy the results to the res 
      res( (iElem-1)*fun%nComponents + 1 : iElem*fun%nComponents) = fEq 
 
    end do !iElem

   end subroutine deriveEquilMSLiquid
  ! ************************************************************************ !


  ! ************************************************************************ !
  !> Equilibrium from state with thermodynamic factor
?? copy :: get_element_headtxt(deriveEquilWTDF_MSLiquid)
    ! -------------------------------------------------------------------- !
    integer :: statePos, iElem, iComp, iLevel, iField, nFields
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    integer :: pdfPos, varPos, QQ
    real(kind=rk), allocatable :: tmpPDF(:)
    !mass density of nSpecies
    real(kind=rk) :: mass_dens( varSys%nStateVars )
    !number density of nSpecies 
    real(kind=rk) :: num_dens( varSys%nStateVars )
    !mole fraction
    real(kind=rk) :: moleFraction( varSys%nStateVars )
    real(kind=rk) :: totNum_densInv, phy_moleDens_fac
    !first moments of nSpecies
    real(kind=rk) :: first_moments( 3, varSys%nStateVars )
    !momentum from linear system of equation
    real(kind=rk) :: momentum( 3, varSys%nStateVars )
    real(kind=rk) :: vel( 3, varSys%nStateVars )
    !parameters from solver specific conf
    !field specific info from field table
    real(kind=rk), dimension(varSys%nStateVars) :: phi
    real(kind=rk), dimension(varSys%nStateVars, varSys%nStateVars) :: &
      & resi_coeff, thermodynamic_fac, &
      & inv_thermodyn_fac, diff_coeff
    !mixture info
    real(kind=rk) :: omega_diff, paramBInv, theta_eq, weight0Inv, omega_fac
    real(kind=rk) :: press, temp
    real(kind=rk) :: fEq(fun%nComponents)
    integer :: nSize
    integer :: stateVarMap(varSys%nStateVars)
    ! -------------------------------------------------------------------- !
    call C_F_POINTER( fun%method_Data, fPtr )
    scheme => fPtr%solverData%scheme

    nFields = scheme%nFields
    QQ = scheme%layout%fStencil%QQ
    allocate(tmpPDF(QQ))
    stateVarMap = scheme%stateVarMap%varPos%val(:)

    ! equilibrium velocity can be computed only for single species
    pdfPos = fun%input_varPos(1)

    do iField = 1, nFields
      ! species properties
      ! molecular weight ratio
      phi(iField) = scheme%field(iField)%fieldProp%species%molWeigRatio
      ! diffusivity coefficients
      diff_coeff(iField, :) =                                     &
        & scheme%field(iField)%fieldProp%species%diff_coeff(:)
    end do 
    !KM @todo check moleDens for multilevel
    phy_moleDens_fac = fPtr%solverData%physics%moleDens0

    omega_diff = scheme%mixture%omega_diff
    omega_fac = omega_diff * 0.5_rk

    paramBInv = 1.0_rk / scheme%mixture%paramB
    theta_eq = scheme%mixture%theta_eq

    weight0Inv = 1.0_rk / scheme%layout%weight( &
      &                              scheme%layout%fStencil%restPosition)
    ! temperature
    temp = scheme%mixture%temp0
    ! atmospheric pressure
    press = scheme%mixture%atm_press

    ! initialize thermodynamic factor
    ! compute from matthias c-code for each element
    thermodynamic_fac = 0.0_rk
    do iField = 1, nFields
      thermodynamic_fac(iField, iField) = 1.0_rk
    end do

    do iElem = 1, nElems
      ! if state array is defined level wise then use levelPointer(pos)
      ! to access state array
      statePos = fPtr%solverData%geometry%levelPointer( elemPos(iElem) )
      iLevel = tem_levelOf( tree%treeID( elemPos(iElem) ) )
      nSize = scheme%pdf( iLevel )%nSize

      do iField = 1, nFields

        do iComp = 1, QQ 
          varPos = varSys%method%val(stateVarMap(iField))%state_varPos(iComp)

          tmpPDF(iComp) = scheme%state( iLevel )%val(           &
            ! position of this state variable in the state array
            & ?IDX?(varPos, statePos, varSys%nScalars, nSize ), &
            & scheme%pdf( iLevel )%nNext ) 
        end do  

        ! mass density of species
        mass_dens(iField ) = sum( tmpPDF ) 

        ! number density of species   
        num_dens(iField) = mass_dens(iField)                    &
          & * scheme%field(iField)%fieldProp%species%molWeightInv

        ! velocity, first moments
        do iComp = 1, 3 
          first_moments(iComp, iField) = sum( tmpPDF *        &
            & scheme%layout%fStencil%cxDirRK(iComp, :) )
        end do  

      end do !iField

      ! total number density Inv
      totNum_densInv = 1.0_rk/sum(num_dens(:))

      ! molefraction
      moleFraction(:) = num_dens(:)*totNum_densInv

      ! MS-Diff coeff matrix from C++ code
      call mus_calc_MS_DiffMatrix( nFields, temp, press,                 &
        &                          num_dens*phy_moleDens_fac, diff_coeff )

      ! Convert to lattice unit
      resi_coeff = fPtr%solverData%physics%fac(iLevel)%diffusivity/diff_coeff

      ! Thermodynamic factor from C++ code
      call mus_calc_thermFactor( nFields, temp, press, moleFraction,        &
        &                        thermodynamic_fac )

      inv_thermodyn_fac = invert_matrix( thermodynamic_fac )

      ! momentum of all species
      momentum = momentumFromMacroLSE_WTDF(                                 &
        &                            moleFraction      = moleFraction,      &
        &                            first_moments     = first_moments,     &
        &                            nFields           = nFields,           &
        &                            inv_thermodyn_fac = inv_thermodyn_fac, &
        &                            phi               = phi,               &
        &                            resi_coeff        = resi_coeff,        &
        &                            omega_fac         = omega_fac,         &
        &                            paramBInv         = paramBInv          )


      !velocity of all species
      do iField = 1, nFields
        vel( :, iField) = momentum( :, iField) / mass_dens(iField)
      end do

      ! compute equilibrium from macroscopic quantities
      fEq = equilFromMacroWTDF( iField            = pdfPos,             &
        &                       mass_dens         = mass_dens,          &
        &                       moleFraction      = moleFraction,       &
        &                       velocity          = vel,                &
        &                       inv_thermodyn_fac = inv_thermodyn_fac,  &
        &                       layout            = scheme%layout, &
        &                       nFields           = nFields,            &
        &                       paramBInv         = paramBInv,          &
        &                       phi               = phi,                &
        &                       resi_coeff        = resi_coeff,         &
        &                       theta_eq          = theta_eq,           &
        &                       weight0Inv        = weight0Inv          )


      ! copy the results to the res 
      res( (iElem-1)*fun%nComponents + 1 : iElem*fun%nComponents) = fEq 
 
    end do !iElem

   end subroutine deriveEquilWTDF_MSLiquid
  ! ************************************************************************ !


  ! ************************************************************************ !
  !> mole fraction
?? copy :: get_element_headtxt(deriveMoleFracMSLiquid)
    ! -------------------------------------------------------------------- !
    integer :: statePos, iElem, iLevel, iDep, iField, nFields, iComp
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    integer :: varPos, QQ
    real(kind=rk), allocatable :: tmpPDF(:)
    !mass density of nSpecies
    real(kind=rk) :: mass_dens( varSys%nStateVars )
    !number density of nSpecies 
    real(kind=rk) :: num_dens( varSys%nStateVars ), totNum_dens
    !mole fraction
    real(kind=rk) :: moleFraction( varSys%nStateVars ), molFrac_loc
    integer :: nSize
    integer :: stateVarMap(varSys%nStateVars)
    ! -------------------------------------------------------------------- !
    call C_F_POINTER( fun%method_Data, fPtr )
    scheme => fPtr%solverData%scheme

    nFields = scheme%nFields
    QQ = scheme%layout%fStencil%QQ
    allocate(tmpPDF(QQ))
    stateVarMap = scheme%stateVarMap%varPos%val(:)

    res = 0.0_rk

    do iElem = 1, nElems
      ! if state array is defined level wise then use levelPointer(pos)
      ! to access state array
      statePos = fPtr%solverData%geometry%levelPointer( elemPos(iElem) )
      iLevel = tem_levelOf( tree%treeID( elemPos(iElem) ) )
      nSize = scheme%pdf( iLevel )%nSize

      do iField = 1, nFields

        do iComp = 1, QQ 
          varPos = varSys%method%val(stateVarMap(iField))%state_varPos(iComp)

          tmpPDF(iComp) = scheme%state( iLevel )%val(              &
            ! position of this state variable in the state array
            & ?IDX?(varPos, statePos, varSys%nScalars, nSize ),    &
            & scheme%pdf( iLevel )%nNext ) 
        end do  

        ! mass density of species
        mass_dens(iField ) = sum( tmpPDF ) 

        ! number density of species   
        num_dens(iField) = mass_dens(iField)                      &
          & / scheme%field(iField)%fieldProp%species%molWeight
      end do

      ! total number density
      totNum_dens = sum( num_dens(:) )
      ! mole fraction
      moleFraction(:) = num_dens(:) / totNum_dens
 
      molFrac_loc = 0.0_rk
      do iDep = 1, fun%nInputs
        ! get the position of the depending variable in the global varSys
        molFrac_loc = molFrac_loc + moleFraction(fun%input_varPos(iDep))
      end do

      res(iElem) = molFrac_loc
    end do  

  end subroutine deriveMoleFracMSLiquid
  ! ************************************************************************ !


  ! ************************************************************************ !
  !> mole flux = numDens_i*velocity_i
?? copy :: get_element_headtxt(deriveMoleFluxMS)
    ! -------------------------------------------------------------------- !
    integer :: dens_pos, vel_pos, iElem
    real(kind=rk) :: dens_res(nElems)
    ! -------------------------------------------------------------------- !
    
    ! position of mole density in varSys
    dens_pos = fun%input_varPos(1)

    ! derive number density
    call varSys%method%val(dens_pos)%get_element( varSys  = varSys,  &
      &                                           elemPos = elemPos, &
      &                                           time    = time,    &
      &                                           tree    = tree,    &
      &                                           nElems  = nElems,  &
      &                                           nDofs   = nDofs,   &
      &                                           res     = dens_res )

    ! position of velocity in varSys
    vel_pos = fun%input_varPos(2)

    ! derive velocity
    call varSys%method%val(vel_pos)%get_element( varSys  = varSys,  &
      &                                          elemPos = elemPos, &
      &                                          time    = time,    &
      &                                          tree    = tree,    &
      &                                          nElems  = nElems,  &
      &                                          nDofs   = nDofs,   &
      &                                          res     = res      )


    ! copy the results to the res 
    do iElem = 1, nElems
      res( (iElem-1)*fun%nComponents + 1 : iElem*fun%nComponents) =   &
        & res( (iElem-1)*fun%nComponents + 1 : iElem*fun%nComponents) &
        & * dens_res(iElem)
    end do

  end subroutine deriveMoleFluxMS
  ! ************************************************************************ !


  ! ************************************************************************ !
   !> Derive external electrical force variable defined as a source term.
   !! It evaluates spacetime function defined in lua file for force variable
   !! and convert it to state value which is to be added to the state
   !! NOTE: For species source it return Valid only for species variable 
   !! not as mixture variable
?? copy :: get_element_headtxt(derive_electricFieldMSLiquid)
    ! -------------------------------------------------------------------- !
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    real(kind=rk) :: electricField(nElems*3)
    real(kind=rk) :: EF_elem(3)
    integer :: stFun_pos
    integer :: iElem, iDir, iLevel, posInTotal
    integer :: iField, nFields, depField, nScalars, QQ, nInputStates
    !mass density of nSpecies
    real(kind=rk) :: mass_dens( varSys%nStateVars )
    !number density of nSpecies
    real(kind=rk) :: num_dens( varSys%nStateVars )
    real(kind=rk) :: massFrac( varSys%nStateVars )
    real(kind=rk), dimension(varSys%nStateVars) :: chargeNr
    real(kind=rk) :: gasConst, temp0, minMolWeight, mixDiffForce(3)
    real(kind=rk) :: forceTerm(fun%nComponents)
    real(kind=rk), dimension(3, varSys%nStateVars ) :: mixForce,       &
      & diffForce, electricTerm, spcForce
    integer :: nSize
    integer :: stateVarMap(varSys%nStateVars)
    ! -------------------------------------------------------------------- !
    ! convert c pointer to solver type fortran pointer
    call c_f_pointer( fun%method_data, fPtr ) 
    scheme => fPtr%solverData%scheme

    ! last input var is spacetime function
    stFun_pos = fun%input_varPos(fun%nInputs)

    ! Get Electric field which might be defined to vary in space and time
    call varSys%method%val(stfun_pos)%get_element( varSys  = varSys,       &
      &                                            elemPos = elemPos,      &
      &                                            time    = time,         &
      &                                            tree    = tree,         &
      &                                            nElems  = nElems,       &
      &                                            ndofs   = 1,            &
      &                                            res     = electricField )

    ! number of pdf states this source depends on
    ! last input is spacetime function so it is neglected
    nInputStates = fun%nInputs - 1

    ! constant parameter
    nFields = scheme%nFields
    QQ = scheme%layout%fStencil%QQ
    nScalars = varSys%nScalars
    stateVarMap = scheme%stateVarMap%varPos%val(:)

    ! minimum molecular weight
    minMolWeight = minval(scheme%field(:)%fieldProp%species%molWeight)
    ! species charge
    chargeNr(:) = scheme%field(:)%fieldProp%species%chargeNr
    ! gas constant
    gasConst = scheme%mixture%gasConst_R_LB
    ! temperature
    temp0 = scheme%mixture%temp0LB

    ! update source for each element
    do iElem = 1, nElems
      posInTotal = fPtr%solverData%geometry%levelPointer( elemPos(iElem) )
      iLevel = tem_levelOf( tree%treeID( elemPos(iElem) ) )
      nSize = scheme%pdf(iLevel)%nSize

      mass_dens = 0.0_rk
      do iField = 1, nFields
        ! compute field density and first moments
        do iDir = 1, QQ 
          !field density
          ! KM: use nNext to derive source since nNext in the next time steps
          ! becomes nNow which is used by update source
          mass_dens( iField ) = mass_dens( iField )                          &
            & + scheme%state(iLevel)%val(                                    & 
&?FETCH?(iDir,iField,posInTotal,QQ,nScalars,nSize,scheme%pdf(iLevel)%neigh), &
            & scheme%pdf(iLevel)%nNext )
        end do
      end do

      !mass fraction
      massFrac(:) = mass_dens(:)/sum(mass_dens)

      ! number density of all species
      num_dens(:) = mass_dens(:) * scheme%field(:)%fieldProp%species &
        &                                         %molWeightInv

      ! compute external forcing term
      ! d^m_k = w_m*c_m*( n0*min_a(m_a)*F_k + (1/cs2)*(n_k*z_k*F*E) )
      ! F_k is diffsive forcing term

      ! convert physical to lattice
      EF_elem = electricField((iElem-1)*3+1 : iElem*3)  &
        &     * fPtr%solverData%physics%coulomb0        &
        &     / fPtr%solverData%physics%fac(iLevel)%force

      !write(*,*) 'electricField ', EF_elem

      ! electric field term: n_k * z_k * Faraday * electricField
      do iField = 1, nFields
        electricTerm(:, iField) = num_dens(iField) * chargeNr(iField) &
          &                     * scheme%mixture%faradayLB            &
          &                     * EF_elem(:)
      end do
!write(dbgUnit(1),*) 'electricTerm ', electricTerm

      ! total mixture force in diffusive forcing term
      ! totMixForce = sum_l(n_l z_l Faraday E)
      mixDiffForce(1) = sum(electricTerm(1, :))
      mixDiffForce(2) = sum(electricTerm(2, :))
      mixDiffForce(3) = sum(electricTerm(3, :))
!write(dbgUnit(1),*) 'totMixForce ', mixDiffForce 

      ! diffusive forcing term in the Maxwell-Stefan equations
      ! F_k = 1/(nRT) rho_k (z_k Faraday/m_k E) 
      !     - y_k/(nRT) sum_l(rho_l (z_l Faraday/m_l E)
      ! diffForce = (electric term - fraction of totMixForce) *
      ! 1/(nRT) * n0 * min_a(m_a)
      ! Here n and n0 will cancel out so they are not included in computation
      do iField = 1, nFields
        diffForce(:, iField) = minMolWeight * ( electricTerm(:, iField) &
          &                  - massFrac(iField) * mixDiffForce(:) )     &
          &                  / ( gasConst * temp0 ) 
      end do
!write(dbgUnit(1),*) 'diffForce ', diffForce      

      ! forcing term in the mixture for momentum equation
      ! mixForce = (1/cs2)*(rho_k*g + n_k*z_k*F*E)
      mixForce = cs2inv * electricTerm

      ! Force on each species
      ! d^m_k = weight_m*cxDir_m ( min_a(m_a)*n * F_k + n_k*z_k Faraday E )
      spcForce = diffForce + mixForce

      ! last dependency is spacetime function of this source
      forceTerm = 0.0_rk
      do iField = 1, nInputStates
        depField = fun%input_varPos(iField)
        forceTerm = forceTerm + spcForce(:, depField)
      end do

      ! copy the results to the res 
      res( (iElem-1)*fun%nComponents + 1 : iElem*fun%nComponents) =  forceTerm 
!write(dbgUnit(1),*) 'derive forceTerm ', forceTerm           
    end do !iElem 

  end subroutine derive_electricFieldMSLiquid
  ! ************************************************************************ !


  ! ************************************************************************ !
   !> Derive gravity force variable defined as a source term.
   !! It evaluates spacetime function defined in lua file for force variable
   !! and convert it to state value which is to be added to the state
?? copy :: get_element_headtxt(derive_gravityFieldMSLiquid)
    ! -------------------------------------------------------------------- !
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    real(kind=rk) :: gravityField(nElems*3)
    real(kind=rk) :: GF_elem(3)
    integer :: stFun_pos
    integer :: iElem, iLevel, iDir, posInTotal
    integer :: iField, nFields, depField, nScalars, QQ, nInputStates
    !mass density of nSpecies
    real(kind=rk) :: mass_dens( varSys%nStateVars )
    real(kind=rk) :: massFrac( varSys%nStateVars )
    real(kind=rk) :: gasConst, temp0, minMolWeight, mixDiffForce(3)
    real(kind=rk) :: forceTerm(fun%nComponents)
    real(kind=rk), dimension(3, varSys%nStateVars ) :: mixForce,       &
      & diffForce, gravityTerm, spcForce
    integer :: nSize
    integer :: stateVarMap(varSys%nStateVars)
    ! -------------------------------------------------------------------- !
    ! convert c pointer to solver type fortran pointer
    call c_f_pointer( fun%method_data, fPtr ) 
    scheme => fPtr%solverData%scheme

    ! last input var is spacetime function
    stFun_pos = fun%input_varPos(fun%nInputs)

    ! Get Electric field which might be defined to vary in space and time
    call varSys%method%val(stFun_pos)%get_element( varSys  = varSys,      &
      &                                            elemPos = elemPos,     &
      &                                            time    = time,        &
      &                                            tree    = tree,        &
      &                                            nElems  = nElems,      &
      &                                            nDofs   = 1,           &
      &                                            res     = gravityField )


    ! number of pdf states this source depends on
    ! last input is spacetime function so it is neglected
    nInputStates = fun%nInputs - 1
    ! constant parameter
    nFields = scheme%nFields
    QQ = scheme%layout%fStencil%QQ
    nScalars = varSys%nScalars
    stateVarMap = scheme%stateVarMap%varPos%val(:)
    ! minimum molecular weight
    minMolWeight = minval(scheme%field(:)%fieldProp%species%molWeight)
    ! gas constant
    gasConst = scheme%mixture%gasConst_R_LB
    ! temperature
    temp0 = scheme%mixture%temp0LB

    ! update source for each element
    do iElem = 1, nElems
      posInTotal = fPtr%solverData%geometry%levelPointer( elemPos(iElem) )
      iLevel = tem_levelOf( tree%treeID( elemPos(iElem) ) )
      nSize = scheme%pdf(iLevel)%nSize

      mass_dens = 0.0_rk
      do iField = 1, nFields
        ! compute field density and first moments
        do iDir = 1, QQ 
          !field density
          ! KM: use nNext to derive source since nNext in the next time steps
          ! becomes nNow which is used by update source
          mass_dens( iField ) = mass_dens( iField )                                  &
            & + scheme%state(iLevel)%val(                                            &
&?FETCH?(iDir,iField,posInTotal,QQ,nScalars,nSize,scheme%pdf(iLevel)%neigh),         &
            & scheme%pdf(iLevel)%nNext )
        end do
      end do

      !mass fraction
      massFrac(:) = mass_dens(:)/sum(mass_dens)

      ! compute external forcing term
      ! d^m_k = w_m*c_m*( n0*min_a(m_a)*F_k + (1/cs2)*(n_k*z_k*F*E) )
      ! F_k is diffsive forcing term

      ! convert physical to lattice
      GF_elem = gravityField((iElem-1)*3+1 : iElem*3) &
        &     * fPtr%solverData%physics%dtLvl(iLevel)**2 &
        &     / fPtr%solverData%physics%dxLvl(iLevel)

      ! electric field term: rho_k * gravityField
      do iField = 1, nFields
        gravityTerm(:, iField) = mass_dens(iField) * GF_elem 
      end do

      ! total mixture force in diffusive forcing term
      ! totMixForce = sum_l(rho_k * g)
      mixDiffForce(1) = sum(gravityTerm(1, :))
      mixDiffForce(2) = sum(gravityTerm(2, :))
      mixDiffForce(3) = sum(gravityTerm(3, :))

      ! diffusive forcing term in the Maxwell-Stefan equations
      ! F_k = 1/(nRT) rho_k (g) 
      !     - y_k/(nRT) sum_l(rho_l (g)
      ! diffForce = (electric term - fraction of totMixForce) *
      ! 1/(nRT) * n0 * min_a(m_a)
      ! Here n and n0 will cancel out so they are not included in computation
      do iField = 1, nFields
        diffForce(:, iField) = minMolWeight * ( gravityTerm(:, iField) &
          &                  - massFrac(iField) * mixDiffForce(:) )     &
          &                  / ( gasConst * temp0 ) 
      end do

      ! forcing term in the mixture for momentum equation
      ! mixForce = (1/cs2)*(rho_k*g + n_k*z_k*F*E)
      mixForce = cs2inv * gravityTerm

      ! Force on each species
      ! d^m_k = weight_m*cxDir_m ( min_a(m_a)*n * F_k + rho_k * g )
      spcForce = diffForce + mixForce

      forceTerm = 0.0_rk
      do iField = 1, nInputStates
        depField = fun%input_varPos(iField)
        forceTerm = forceTerm + spcForce(:, depField)
      end do

      ! copy the results to the res 
      res( (iElem-1)*fun%nComponents + 1 : iElem*fun%nComponents) =  forceTerm 
    end do !iElem 

  end subroutine derive_gravityFieldMSLiquid
  ! ************************************************************************ !


  ! ************************************************************************ !
   !> Derive external electrical force variable defined as a source term.
   !! It evaluates spacetime function defined in lua file for force variable
   !! and convert it to state value which is to be added to the state
   !! NOTE: For species source it return Valid only for species variable not as mixture variable
?? copy :: get_element_headtxt(derive_electricFieldMSLiquid_WTDF)
    ! -------------------------------------------------------------------- !
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    real(kind=rk) :: electricField(nElems*3)
    real(kind=rk) :: EF_elem(3)
    integer :: stFun_pos
    integer :: iElem, iDir, iLevel, posInTotal
    integer :: iField, nFields, depField, nScalars, QQ, nInputStates
    integer :: iField_2
    !mass density of nSpecies
    real(kind=rk) :: mass_dens( varSys%nStateVars )
    !number density of nSpecies 
    real(kind=rk) :: num_dens( varSys%nStateVars )
    real(kind=rk), dimension(varSys%nStateVars) :: massFrac, moleFrac 
    real(kind=rk), dimension(varSys%nStateVars) :: chargeNr
    real(kind=rk) :: gasConst, temp0, minMolWeight, mixDiffForce(3)
    real(kind=rk) :: forceTerm(fun%nComponents)
    real(kind=rk), dimension(3, varSys%nStateVars ) :: mixForce,       &
      & diffForce, electricTerm, spcForce, diffForce_WTF
    real(kind=rk), dimension(varSys%nStateVars, varSys%nStateVars) :: &
      & resi_coeff, thermodynamic_fac, &
      & inv_thermodyn_fac, diff_coeff
    real(kind=rk) :: press, temp
    real(kind=rk) :: phy_moleDens_fac
    integer :: nSize
    integer :: stateVarMap(varSys%nStateVars)
    ! -------------------------------------------------------------------- !
    ! convert c pointer to solver type fortran pointer
    call c_f_pointer( fun%method_data, fPtr )
    scheme => fPtr%solverData%scheme

    ! last input var is spacetime function
    stFun_pos = fun%input_varPos(fun%nInputs)

    ! Get Electric field which might be defined to vary in space and time
    call varSys%method%val(stfun_pos)%get_element( varSys  = varSys,       &
      &                                            elemPos = elemPos,      &
      &                                            time    = time,         &
      &                                            tree    = tree,         &
      &                                            nElems  = nElems,       &
      &                                            ndofs   = 1,            &
      &                                            res     = electricField )

    ! number of pdf states this source depends on
    ! last input is spacetime function so it is neglected
    nInputStates = fun%nInputs - 1

    ! constant parameter
    nFields = scheme%nFields
    QQ = scheme%layout%fStencil%QQ
    nScalars = varSys%nScalars
    stateVarMap = scheme%stateVarMap%varPos%val(:)
    do iField = 1, nFields
      ! diffusivity coefficients
      diff_coeff(iField, :) =                                     &
        & scheme%field(iField)%fieldProp%species%diff_coeff(:)
    end do
    !KM @todo check moleDens for multilevel
    phy_moleDens_fac = fPtr%solverData%physics%moleDens0
    ! minimum molecular weight
    minMolWeight = minval(scheme%field(:)%fieldProp%species%molWeight)
    ! species charge
    chargeNr(:) = scheme%field(:)%fieldProp%species%chargeNr
    ! gas constant
    gasConst = scheme%mixture%gasConst_R_LB
    ! temperature
    temp0 = scheme%mixture%temp0LB
    ! temperature
    temp = scheme%mixture%temp0
    ! atmospheric pressure
    press = scheme%mixture%atm_press

    ! update source for each element
    do iElem = 1, nElems
      posInTotal = fPtr%solverData%geometry%levelPointer( elemPos(iElem) )
      iLevel = tem_levelOf( tree%treeID( elemPos(iElem) ) )
      nSize = scheme%pdf(iLevel)%nSize

      mass_dens = 0.0_rk
      do iField = 1, nFields
        ! compute field density and first moments
        do iDir = 1, QQ
          !field density
          ! KM: use nNext to derive source since nNext in the next time steps
          ! becomes nNow which is used by update source
          mass_dens( iField ) = mass_dens( iField )                                  &
            & + scheme%state(iLevel)%val(                                            &
&?FETCH?(iDir,iField,posInTotal,QQ,nScalars,nSize,scheme%pdf(iLevel)%neigh),         &
            & scheme%pdf(iLevel)%nNext )
        end do
      end do

      !mass fraction
      massFrac(:) = mass_dens(:)/sum(mass_dens)

      ! number density of all species
      num_dens(:) = mass_dens(:) * scheme%field(:)%fieldProp%species &
        &                                         %molWeightInv

      ! mole fraction
      moleFrac = num_dens/sum(num_dens)

      ! MS-Diff coeff matrix from C++ code
      call mus_calc_MS_DiffMatrix( nFields, temp, press,                       &
        &                        num_dens*phy_moleDens_fac, diff_coeff )
      ! Convert to lattice unit
      resi_coeff = fPtr%solverData%physics%fac(iLevel)%diffusivity/diff_coeff

      ! Thermodynamic factor from C++ code
      call mus_calc_thermFactor( nFields, temp, press, moleFrac,        &
        &                        thermodynamic_fac )

      inv_thermodyn_fac = invert_matrix( thermodynamic_fac )

      ! compute external forcing term
      ! d^m_k = w_m*c_m*( n0*min_a(m_a)*F_k + (1/cs2)*(n_k*z_k*F*E) )
      ! F_k is diffsive forcing term

      ! convert physical to lattice
      EF_elem = electricField((iElem-1)*3+1 : iElem*3) &
        &     * fPtr%solverData%physics%coulomb0       &
        &     / fPtr%solverData%physics%fac(iLevel)%force

      !write(*,*) 'electricField ', EF_elem

      ! electric field term: n_k * z_k * Faraday * electricField
      do iField = 1, nFields
        electricTerm(:, iField) = num_dens(iField) * chargeNr(iField) &
          &                     * scheme%mixture%faradayLB            &
          &                     * EF_elem(:)
      end do
!write(dbgUnit(1),*) 'electricTerm ', electricTerm

      ! total mixture force in diffusive forcing term
      ! totMixForce = sum_l(n_l z_l Faraday E)
      mixDiffForce(1) = sum(electricTerm(1, :))
      mixDiffForce(2) = sum(electricTerm(2, :))
      mixDiffForce(3) = sum(electricTerm(3, :))
!write(dbgUnit(1),*) 'totMixForce ', mixDiffForce

      ! diffusive forcing term in the Maxwell-Stefan equations
      ! F_k = 1/(nRT) rho_k (z_k Faraday/m_k E)
      !     - y_k/(nRT) sum_l(rho_l (z_l Faraday/m_l E)
      ! diffForce = (electric term - fraction of totMixForce) *
      ! 1/(nRT) * n0
      ! Here n and n0 will cancel out so they are not included in computation
      do iField = 1, nFields
        diffForce(:, iField) = ( electricTerm(:, iField) &
          &                  - massFrac(iField) * mixDiffForce(:) )     &
          &                  / ( gasConst * temp0 )
      end do
      ! multiply diffForce with thermFacInv * min_a(m_a)
      ! diffusive driving force term:
      !  min_a(m_a) * c_m* w_m * sum_l(themFacInv_k_l * n * F_l)
      diffForce_WTF = 0.0_rk
      do iField = 1, nFields
        do iField_2 = 1, nFields
          diffForce_WTF(:, iField ) = diffForce_WTF(:, iField)                 &
            &                       + inv_thermodyn_fac(iField, iField_2)      &
            &                       * diffForce(:, iField_2)
        end do
        diffForce_WTF(:, iField ) = minMolWeight * diffForce_WTF(:, iField)
      end do

!write(dbgUnit(1),*) 'diffForce ', diffForce

      ! forcing term in the mixture for momentum equation
      ! mixForce = (1/cs2)*(rho_k*g + n_k*z_k*F*E)
      mixForce = cs2inv * electricTerm

      ! Force on each species
      ! d^m_k = weight_m*cxDir_m ( min_a(m_a)*n * F_k + n_k*z_k Faraday E )
      spcForce = diffForce + mixForce

      ! last dependency is spacetime function of this source
      forceTerm = 0.0_rk
      do iField = 1, nInputStates
        depField = fun%input_varPos(iField)
        forceTerm = forceTerm + spcForce(:, depField)
      end do

      ! copy the results to the res
      res( (iElem-1)*fun%nComponents + 1 : iElem*fun%nComponents) =  forceTerm
!write(dbgUnit(1),*) 'derive forceTerm ', forceTerm
    end do !iElem

  end subroutine derive_electricFieldMSLiquid_WTDF
! ****************************************************************************** !


! ****************************************************************************** !
   !> Derive gravity force variable defined as a source term.
   !! It evaluates spacetime function defined in lua file for force variable
   !! and convert it to state value which is to be added to the state
?? copy :: get_element_headtxt(derive_gravityFieldMSLiquid_WTDF)
    ! -------------------------------------------------------------------- !
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    real(kind=rk) :: gravityField(nElems*3)
    real(kind=rk) :: GF_elem(3)
    integer :: stFun_pos
    integer :: iElem, iLevel, iDir, posInTotal
    integer :: iField, nFields, depField, nScalars, QQ, nInputStates
    integer :: iField_2
    !mass density of nSpecies
    real(kind=rk) :: mass_dens( varSys%nStateVars )
    real(kind=rk) :: num_dens( varSys%nStateVars )
    real(kind=rk), dimension(varSys%nStateVars) :: massFrac, moleFrac 
    real(kind=rk) :: gasConst, temp0, minMolWeight, mixDiffForce(3)
    real(kind=rk) :: forceTerm(fun%nComponents)
    real(kind=rk), dimension(3, varSys%nStateVars ) :: mixForce,       &
      & diffForce, gravityTerm, spcForce, diffForce_WTF
    real(kind=rk), dimension(varSys%nStateVars, varSys%nStateVars) :: &
      & resi_coeff, thermodynamic_fac, &
      & inv_thermodyn_fac, diff_coeff
    real(kind=rk) :: press, temp
    real(kind=rk) :: phy_moleDens_fac
    integer :: nSize
    integer :: stateVarMap(varSys%nStateVars)
    ! -------------------------------------------------------------------- !
    ! convert c pointer to solver type fortran pointer
    call c_f_pointer( fun%method_data, fPtr ) 
    scheme => fPtr%solverData%scheme

    ! last input var is spacetime function
    stFun_pos = fun%input_varPos(fun%nInputs)

    ! Get Electric field which might be defined to vary in space and time
    call varSys%method%val(stFun_pos)%get_element( varSys  = varSys,      &
      &                                            elemPos = elemPos,     &
      &                                            time    = time,        &
      &                                            tree    = tree,        &
      &                                            nElems  = nElems,      &
      &                                            nDofs   = 1,           &
      &                                            res     = gravityField )

    ! number of pdf states this source depends on
    ! last input is spacetime function so it is neglected
    nInputStates = fun%nInputs - 1

    ! constant parameter
    nFields = scheme%nFields
    QQ = scheme%layout%fStencil%QQ
    nScalars = varSys%nScalars
    stateVarMap = scheme%stateVarMap%varPos%val(:)
    do iField = 1, nFields
      ! diffusivity coefficients
      diff_coeff(iField, :) =                                     &
        & scheme%field(iField)%fieldProp%species%diff_coeff(:)
    end do    
    !KM @todo check moleDens for multilevel
    phy_moleDens_fac = fPtr%solverData%physics%moleDens0
    ! minimum molecular weight
    minMolWeight = minval(scheme%field(:)%fieldProp%species%molWeight)
    ! gas constant
    gasConst = scheme%mixture%gasConst_R_LB
    ! temperature
    temp0 = scheme%mixture%temp0LB
    ! temperature
    temp = scheme%mixture%temp0
    ! atmospheric pressure
    press = scheme%mixture%atm_press

    ! update source for each element
    do iElem = 1, nElems
      posInTotal = fPtr%solverData%geometry%levelPointer( elemPos(iElem) )
      iLevel = tem_levelOf( tree%treeID( elemPos(iElem) ) )
      nSize = scheme%pdf(iLevel)%nSize

      mass_dens = 0.0_rk
      do iField = 1, nFields
        ! compute field density and first moments
        do iDir = 1, QQ 
          !field density
          ! KM: use nNext to derive source since nNext in the next time steps
          ! becomes nNow which is used by update source
          mass_dens( iField ) = mass_dens( iField )                                      &
            & + scheme%state(iLevel)%val(                                                &
& ?FETCH?( iDir,iField,posInTotal,QQ,nScalars,nSize,scheme%pdf(iLevel)%neigh ),          &
            & scheme%pdf(iLevel)%nNext )
        end do
      end do  

      !mass fraction
      massFrac(:) = mass_dens(:)/sum(mass_dens)

      ! number density of all species
      num_dens(:) = mass_dens(:) * scheme%field(:)%fieldProp%species &
        &                                         %molWeightInv

      ! mole fraction 
      moleFrac = num_dens/sum(num_dens)

      ! MS-Diff coeff matrix from C++ code
      call mus_calc_MS_DiffMatrix( nFields, temp, press,                       &
        &                        num_dens*phy_moleDens_fac, diff_coeff )
      ! Convert to lattice unit
      resi_coeff = fPtr%solverData%physics%fac(iLevel)%diffusivity/diff_coeff

      ! Thermodynamic factor from C++ code
      call mus_calc_thermFactor( nFields, temp, press, moleFrac,        &
        &                        thermodynamic_fac )

      inv_thermodyn_fac = invert_matrix( thermodynamic_fac )

      ! compute external forcing term
      ! d^m_k = w_m*c_m*( n0*min_a(m_a)*F_k + (1/cs2)*(n_k*z_k*F*E) )
      ! F_k is diffsive forcing term

      ! convert physical to lattice
      GF_elem = gravityField((iElem-1)*3+1 : iElem*3) &
        &     * fPtr%solverData%physics%dtLvl(iLevel)**2 &
        &     / fPtr%solverData%physics%dxLvl(iLevel)

      ! electric field term: rho_k * gravityField
      do iField = 1, nFields
        gravityTerm(:, iField) = mass_dens(iField) * GF_elem 
      end do

      ! total mixture force in diffusive forcing term
      ! totMixForce = sum_l(rho_k * g)
      mixDiffForce(1) = sum(gravityTerm(1, :))
      mixDiffForce(2) = sum(gravityTerm(2, :))
      mixDiffForce(3) = sum(gravityTerm(3, :))

      ! diffusive forcing term in the Maxwell-Stefan equations
      ! F_k = 1/(nRT) rho_k (g) 
      !     - y_k/(nRT) sum_l(rho_l (g)
      ! diffForce = (electric term - fraction of totMixForce) *
      ! 1/(nRT) * n0 * min_a(m_a)
      ! Here n and n0 will cancel out so they are not included in computation
      do iField = 1, nFields
        diffForce(:, iField) = ( gravityTerm(:, iField) &
          &                  - massFrac(iField) * mixDiffForce(:) )     &
          &                  / ( gasConst * temp0 ) 
      end do

      ! multiply diffForce with thermFacInv * min_a(m_a)
      ! diffusive driving force term:
      !  min_a(m_a) * c_m* w_m * sum_l(themFacInv_k_l * n * F_l)
      diffForce_WTF = 0.0_rk
      do iField = 1, nFields
        do iField_2 = 1, nFields
          diffForce_WTF(:, iField ) = diffForce_WTF(:, iField)                 &
            &                       + inv_thermodyn_fac(iField, iField_2)      &
            &                       * diffForce(:, iField_2)
        end do
        diffForce_WTF(:, iField ) = minMolWeight * diffForce_WTF(:, iField)
      end do

      ! forcing term in the mixture for momentum equation
      ! mixForce = (1/cs2)*(rho_k*g + n_k*z_k*F*E)
      mixForce = cs2inv * gravityTerm

      ! Force on each species
      ! d^m_k = weight_m*cxDir_m ( min_a(m_a)*n * F_k + rho_k * g )
      spcForce = diffForce + mixForce

      forceTerm = 0.0_rk
      do iField = 1, nInputStates
        depField = fun%input_varPos(iField)
        forceTerm = forceTerm + spcForce(:, depField)
      end do

      ! copy the results to the res 
      res( (iElem-1)*fun%nComponents + 1 : iElem*fun%nComponents) =  forceTerm 
    end do !iElem 

  end subroutine derive_gravityFieldMSLiquid_WTDF
  ! ************************************************************************ !

  ! ************************************************************************ !
   !> Update state with source variable "electric_field"
   !! Simuilar to derive routine but it updates the state whereas derive 
   !! is used for tracking
   !! @todo species electricField
?? copy :: applySource_header(applySrc_electricFieldMSLiquid_tmp)
    ! -------------------------------------------------------------------- !
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    real(kind=rk) :: electricField(fun%elemLvl(iLevel)%nElems*3)
    real(kind=rk) :: EF_elem(3)
    integer :: iElem, nElems, iDir, posInTotal
    integer :: iField, nFields, depField, nScalars, QQ, nInputStates
    !mass density of nSpecies
    real(kind=rk) :: mass_dens( varSys%nStateVars )
    !number density of nSpecies 
    real(kind=rk) :: num_dens( varSys%nStateVars )
    real(kind=rk) :: massFrac( varSys%nStateVars )
    real(kind=rk) :: molWeigRatio( varSys%nStateVars )
    real(kind=rk) :: molWeight( varSys%nStateVars )
    real(kind=rk) :: avgMolWeight
    real(kind=rk), dimension(varSys%nStateVars) :: chargeNr
    real(kind=rk) :: gasConst, temp0, minMolWeight,  mixDiffForce(3)
    real(kind=rk), dimension(3, varSys%nStateVars ) :: mixForce,       &
      & diffForce, electricTerm, spcForce
    integer :: stateVarMap(varSys%nStateVars)
    real(kind=rk), allocatable :: forceMom(:), force(:)
    real(kind=rk), allocatable :: forceTerm(:,:,:), forceTermGrad(:,:,:)
    integer :: x_plus, x_minus, y_plus, y_minus
    ! -------------------------------------------------------------------- !
!write(dbgUnit(1),*) 'source variable: ', trim(varSys%varname%val(fun%srcTerm_varPos))
    ! convert c pointer to solver type fortran pointer
    call c_f_pointer( varSys%method%val( fun%srcTerm_varPos )%method_data, &
      &               fPtr ) 
    scheme => fPtr%solverData%scheme

    ! Number of elements to apply source terms
    nElems = fun%elemLvl(iLevel)%nElems

    ! Get electrical force which is refered in config file either its
    ! spacetime variable or operation variable
    call varSys%method%val(fun%data_varPos)%get_valOfIndex( &
      & varSys  = varSys,                                   &
      & time    = time,                                     &
      & iLevel  = iLevel,                                   &
      & idx     = fun%elemLvl(iLevel)%idx(1:nElems),        &
      & nVals   = nElems,                                   & 
      & res     = electricField                             )

    ! convert physical to lattice
    electricField = electricField * fPtr%solverData%physics%coulomb0 &
      &           / fPtr%solverData%physics%fac(iLevel)%force

    ! number of pdf states this source depends on
    ! last input is spacetime function so it is neglected
    nInputStates = varSys%method%val(fun%srcTerm_varPos)%nInputs - 1
    ! constant parameter
    nFields = scheme%nFields
    QQ = scheme%layout%fStencil%QQ
    nScalars = varSys%nScalars
    stateVarMap = scheme%stateVarMap%varPos%val(:)
    allocate(forceMom(QQ))
    allocate(force(QQ))
    allocate(forceTerm(nElems, nFields, 3))
    allocate(forceTermGrad(nElems, nFields, 3))
    ! minimum molecular weight
    minMolWeight = minval(scheme%field(:)%fieldProp%species%molWeight)
    ! species charge
    chargeNr(:) = scheme%field(:)%fieldProp%species%chargeNr
    ! gas constant
    gasConst = scheme%mixture%gasConst_R_LB
    ! temperature
    temp0 = scheme%mixture%temp0LB

    ! computer force term 
    do iElem = 1, nElems
!write(dbgUnit(1),*) 'iElem ', iElem    
      ! to access level wise state array
      posInTotal = fun%elemLvl(iLevel)%posInTotal(iElem)
      mass_dens = 0.0_rk
      do iField = 1, nFields
        ! compute field density and first moments
        do iDir = 1, QQ 
          !field density
          mass_dens( iField ) = mass_dens( iField ) + inState(           &
            & ?FETCH?(iDir,iField,posInTotal,QQ,nScalars,nPdfSize,neigh) )
        end do
      end do

      !mass fraction
      massFrac(:) = mass_dens(:)/sum(mass_dens)

      ! number density of all species
      num_dens(:) = mass_dens(:) * scheme%field(:)%fieldProp%species &
        &                                         %molWeightInv
!write(dbgUnit(1),*) 'mass_dens ', mass_dens
!write(dbgUnit(1),*) 'num_dens ', num_dens

      ! compute external forcing term
      ! electric field term: n_k * z_k * Faraday * electricField
      EF_elem = electricField((iElem-1)*3+1 : iElem*3)
      do iField = 1, nFields
        electricTerm(:, iField) = num_dens(iField) * chargeNr(iField) &
          &                    * scheme%mixture%faradayLB            &
          &                    * EF_elem(:) 
      end do

      ! total mixture force in diffusive forcing term
      ! totMixForce = sum_l(n_l z_l Faraday E)
      mixDiffForce(1) = sum(electricTerm(1, :))
      mixDiffForce(2) = sum(electricTerm(2, :))
      mixDiffForce(3) = sum(electricTerm(3, :))
!write(dbgUnit(1),*) 'totMixForce ', mixDiffForce 

      ! diffusive forcing term in the Maxwell-Stefan equations
      ! F_k = 1/(nRT) rho_k (z_k Faraday/m_k E) 
      !     - y_k/(nRT) sum_l(rho_l (z_l Faraday/m_l E)
      ! diffForce = (electric term - fraction of totMixForce) *
      ! 1/(nRT) * n0 * min_a(m_a)
      ! Here n and n0 will cancel out so they are not included in computation
      do iField = 1, nFields
        diffForce(:, iField) = cs2inv * minMolWeight / avgMolWeight &
          &                  * ( electricTerm(:, iField) &
          &                  - massFrac(iField) * mixDiffForce(:) )     !&
        !diffForce(:, iField) = minMolWeight * ( electricTerm(:, iField) &
        !  &                  - massFrac(iField) * mixDiffForce(:) )     &
        !  &                  / ( gasConst * temp0 ) 
      end do
!write(dbgUnit(1),*) 'diffForce ', diffForce      

      ! forcing term in the mixture for momentum equation
      ! mixForce = (1/cs2)*(rho_k*g + n_k*z_k*F*E)
      mixForce = cs2inv * electricTerm
!write(dbgUnit(1),*) 'mixForce ', mixForce

      ! Force on each species
      ! d^m_k = weight_m*cxDir_m ( min_a(m_a)*n * F_k + n_k*z_k Faraday E )
      forceTerm(iElem, :, 1) = (diffForce(1, :) + mixForce(1,:))*cs2
      forceTerm(iElem, :, 2) = (diffForce(2, :) + mixForce(2,:))*cs2
      forceTerm(iElem, :, 3) = (diffForce(3, :) + mixForce(3,:))*cs2

    end do  

    ! compute gradient of force
    do iElem = 1, nElems
      ! to access level wise state array
      posInTotal = fun%elemLvl(iLevel)%posInTotal(iElem)
      ! Get face neighbor to compute gradient of force by finite difference
      x_plus  = ?ElemIDX?(neigh(?NGPOS?(1, posInTotal, nPdfSize)), nScalars, nPdfSize)
      x_minus = ?ElemIDX?(neigh(?NGPOS?(3, posInTotal, nPdfSize)), nScalars, nPdfSize)
      y_plus  = ?ElemIDX?(neigh(?NGPOS?(2, posInTotal, nPdfSize)), nScalars, nPdfSize)
      y_minus = ?ElemIDX?(neigh(?NGPOS?(4, posInTotal, nPdfSize)), nScalars, nPdfSize)
      
      do iField = 1, nFields
        forceTermGrad(iElem, iField, 1) = 0.5_rk*( forceTerm(x_plus,iField,1) &
          &                                       -forceTerm(x_minus,iField,1) )
        forceTermGrad(iElem, iField, 2) = 0.5_rk*( forceTerm(y_plus,iField,2) &
          &                                       -forceTerm(y_minus,iField,2) )
      end do  
!write(dbgUnit(1),*) 'forceTerm field1', forceTerm(iElem, 1, : )      
!write(dbgUnit(1),*) 'forceTerm field2', forceTerm(iElem, 2, : )      
!write(dbgUnit(1),*) 'forceTerm field3', forceTerm(iElem, 3, : )      
!write(dbgUnit(1),*) 'forceTermGrad field1', forceTermGrad(iElem, 1, : )      
!write(dbgUnit(1),*) 'forceTermGrad field2', forceTermGrad(iElem, 2, : )      
!write(dbgUnit(1),*) 'forceTermGrad field3', forceTermGrad(iElem, 3, : )      
    end do  

    do iElem = 1, nElems
      ! to access level wise state array
      posInTotal = fun%elemLvl(iLevel)%posInTotal(iElem)
!write(dbgUnit(1),*) 'iElem ', iElem    
      ! Update souce depends on nInputStates
      ! if nInputStates = 1, it is field source else it is global source
      force=0.0_rk
      forceMom=0.0_rk
      do iField = 1, nInputStates
        depField = varSys%method%val(fun%srcTerm_varPos)%input_varPos(iField)
        forceMom(2:3) = forceTerm(iElem, depField, 1:2)
        forceMom(4:5) = forceTermGrad(iElem, depField, 1:2)
!write(dbgUnit(1),*) depField, 'forceMom: ', forceMom
        force = matmul(scheme%layout%moment%toPDF%A, forceMom)
!write(dbgUnit(1),*)        
!write(dbgUnit(1),*) depField, 'force: ', force
!write(dbgUnit(1),*)        
!write(dbgUnit(1),*)        

        do iDir = 1, QQ
          outState(                                                         &
            & ?SAVE?(iDir,depField,posInTotal,QQ,nScalars,nPdfSize,neigh) ) &
            & = outState(                                                   &
            & ?SAVE?(iDir,depField,posInTotal,QQ,nScalars,nPdfSize,neigh) ) &
            & + force(iDir)
        end do ! iDir
      end do !iField
    end do !iElem 

!write(dbgUnit(1),*)    

  end subroutine applySrc_electricFieldMSLiquid_tmp
  ! ************************************************************************ !

  ! ************************************************************************ !
   !> Update state with source variable "electric_field"
   !! Simuilar to derive routine but it updates the state whereas derive 
   !! is used for tracking
   !! @todo species electricField
?? copy :: applySource_header(applySrc_electricFieldMSLiquid)
    ! -------------------------------------------------------------------- !
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    real(kind=rk) :: electricField(fun%elemLvl(iLevel)%nElems*3)
    real(kind=rk) :: EF_elem(3)
    integer :: iElem, nElems, iDir, posInTotal
    integer :: iField, nFields, depField, nScalars, QQ, nInputStates
    !mass density of nSpecies
    real(kind=rk) :: mass_dens( varSys%nStateVars )
    !number density of nSpecies 
    real(kind=rk) :: num_dens( varSys%nStateVars )
    real(kind=rk) :: massFrac( varSys%nStateVars )
    real(kind=rk) :: molWeigRatio( varSys%nStateVars )
    real(kind=rk) :: molWeight( varSys%nStateVars )
    real(kind=rk) :: avgMolWeight, charge_dens, diffForce_fac
    real(kind=rk), dimension(varSys%nStateVars) :: chargeNr, chargeTerm
    real(kind=rk) :: gasConst, temp0, minMolWeight, forceTerm,  mixDiffForce(3)
    real(kind=rk), dimension(3, varSys%nStateVars ) :: mixForce,       &
      & diffForce, electricTerm, spcForce
    ! -------------------------------------------------------------------- !
!write(dbgUnit(1),*) 'source variable: ', trim(varSys%varname%val(fun%srcTerm_varPos))
    ! convert c pointer to solver type fortran pointer
    call c_f_pointer( varSys%method%val( fun%srcTerm_varPos )%method_data, &
      &               fPtr ) 
    scheme => fPtr%solverData%scheme

    ! Number of elements to apply source terms
    nElems = fun%elemLvl(iLevel)%nElems

    ! Get electrical force which is refered in config file either its
    ! spacetime variable or operation variable
    call varSys%method%val(fun%data_varPos)%get_valOfIndex( &
      & varSys  = varSys,                                   &
      & time    = time,                                     &
      & iLevel  = iLevel,                                   &
      & idx     = fun%elemLvl(iLevel)%idx(1:nElems),        &
      & nVals   = nElems,                                   & 
      & res     = electricField                             )

    ! convert physical to lattice
    electricField = electricField * fPtr%solverData%physics%coulomb0 &
      &           / fPtr%solverData%physics%fac(iLevel)%force

    ! number of pdf states this source depends on
    ! last input is spacetime function so it is neglected
    nInputStates = varSys%method%val(fun%srcTerm_varPos)%nInputs - 1
    ! constant parameter
    nFields = scheme%nFields
    QQ = scheme%layout%fStencil%QQ
    nScalars = varSys%nScalars
    ! minimum molecular weight
    minMolWeight = minval(scheme%field(:)%fieldProp%species%molWeight)
    molWeigRatio = scheme%field(:)%fieldProp%species%molWeigRatio
    molWeight = scheme%field(:)%fieldProp%species%molWeight
!write(dbgUnit(1),*) 'avgMolWeight ', avgMolWeight
!write(dbgUnit(1),*) 'ratio ', molWeight/avgMolWeight
    ! species charge
    chargeNr(:) = scheme%field(:)%fieldProp%species%chargeNr
    ! gas constant
    gasConst = scheme%mixture%gasConst_R_LB
    ! temperature
    temp0 = scheme%mixture%temp0LB
    diffForce_fac = minMolWeight / (gasConst*temp0)

    ! update source for each element
    do iElem = 1, nElems
!write(dbgUnit(1),*) 'iElem ', iElem    
      ! to access level wise state array
      posInTotal = fun%elemLvl(iLevel)%posInTotal(iElem)

      mass_dens = 0.0_rk
      do iField = 1, nFields
        ! compute field density and first moments
        do iDir = 1, QQ 
          !field density
          mass_dens( iField ) = mass_dens( iField ) + inState(           &
            & ?FETCH?(iDir,iField,posInTotal,QQ,nScalars,nPdfSize,neigh) )
        end do
      end do
      !x_plus = neigh()

      !mass fraction
      massFrac(:) = mass_dens(:)/sum(mass_dens)

      ! number density of all species
      num_dens(:) = mass_dens(:) * scheme%field(:)%fieldProp%species &
        &                                         %molWeightInv

      ! mean molecular weight of mixture
      avgMolWeight = sum(mass_dens)/sum(num_dens)
!write(dbgUnit(1),*) 'mass_dens ', mass_dens
!write(dbgUnit(1),*) 'num_dens ', num_dens
!write(dbgUnit(1),*) 'avgMolWeight ', avgMolWeight

      ! compute external forcing term
      ! d^m_k = w_m*c_m*( n0*min_a(m_a)*F_k + (1/cs2)*(n_k*z_k*F*E) )
      ! F_k is diffsive forcing term
      EF_elem = electricField((iElem-1)*3+1 : iElem*3)
!write(dbgUnit(1),*) 'electricField ', EF_elem
      ! n_k * z_k 
      do iField = 1, nFields
        chargeTerm(iField) = num_dens(iField) * chargeNr(iField) &
          &                * scheme%mixture%faradayLB
      end do
      charge_dens = sum(chargeTerm)

      ! electric field term: n_k * z_k * Faraday * electricField
      do iField = 1, nFields
        electricTerm(:, iField) = num_dens(iField) * chargeNr(iField) &
          &                     * scheme%mixture%faradayLB            &
          &                     * EF_elem(:)
      end do
!write(dbgUnit(1),*) 'electricTerm ', electricTerm

      ! total mixture force in diffusive forcing term
      ! totMixForce = sum_l(n_l z_l Faraday E)
      mixDiffForce(1) = sum(electricTerm(1, :))
      mixDiffForce(2) = sum(electricTerm(2, :))
      mixDiffForce(3) = sum(electricTerm(3, :))
!write(dbgUnit(1),*) 'totMixForce ', mixDiffForce 

      ! diffusive forcing term in the Maxwell-Stefan equations
      ! F_k = 1/(nRT) rho_k (z_k Faraday/m_k E) 
      !     - y_k/(nRT) sum_l(rho_l (z_l Faraday/m_l E)
      ! diffForce = (electric term - fraction of totMixForce) *
      ! 1/(nRT) * n0 * min_a(m_a)
      ! Here n and n0 will cancel out so they are not included in computation
      do iField = 1, nFields
        !diffForce(:, iField) = cs2inv * minMolWeight / avgMolWeight &
        !  &                  * ( electricTerm(:, iField) &
        !  &                  - massFrac(iField) * mixDiffForce(:) )     !&
        !diffForce(:, iField) = cs2inv * molWeight(iField) / avgMolWeight &
        !  &                  * ( electricTerm(:, iField) &
        !  &                  - massFrac(iField) * mixDiffForce(:) )     !&
        diffForce(:, iField) = minMolWeight * ( electricTerm(:, iField) &
          &                  - massFrac(iField) * mixDiffForce(:) )     &
          &                  / ( gasConst * temp0 ) 
      end do
!write(dbgUnit(1),*) 'diffForce ', diffForce      

      ! forcing term in the mixture for momentum equation
      ! mixForce = (1/cs2)*(rho_k*g + n_k*z_k*F*E)
      mixForce = cs2inv * electricTerm
!write(dbgUnit(1),*) 'mixForce ', mixForce

      ! Force on each species
      ! d^m_k = weight_m*cxDir_m ( min_a(m_a)*n * F_k + n_k*z_k Faraday E )
      spcForce = diffForce + mixForce
!write(dbgUnit(1),*) 'spcForce ', spcForce

      ! Update souce depends on nInputStates
      ! if nInputStates = 1, it is field source else it is global source
      do iField = 1, nInputStates
        depField = varSys%method%val(fun%srcTerm_varPos)%input_varPos(iField)

        do iDir = 1, QQ
          forceTerm = scheme%layout%fStencil%cxDirRK( 1, iDir ) &
            &         * spcForce(1, depField)                   &
            &       + scheme%layout%fStencil%cxDirRK( 2, iDir ) &
            &         * spcForce(2, depField)                   &
            &       + scheme%layout%fStencil%cxDirRK( 3, iDir ) &
            &         * spcForce(3, depField)


!write(dbgUnit(1),*) 'cxDirRK ', scheme%layout%fStencil%cxDirRK( :, iDir )
!write(dbgUnit(1),*) 'spcForce depField ', spcForce(:, depField)

          outState(                                                         &
            & ?SAVE?(iDir,depField,posInTotal,QQ,nScalars,nPdfSize,neigh) ) &
            & = outState(                                                   &
            & ?SAVE?(iDir,depField,posInTotal,QQ,nScalars,nPdfSize,neigh) ) &
            !& + force(iDir)
            & + scheme%layout%weight( iDir ) * forceTerm
        end do ! iDir
!write(dbgUnit(1),*) depField, 'force: ', force
!write(dbgUnit(1),*) depField, 'forceTerm ', forceTerm
!write(dbgUnit(1),*) depField, 'forceMom rec ', matmul(scheme%layout%moment%toMoments%A, forceTerm)
      end do !iField

    end do !iElem 

!write(dbgUnit(1),*)    

  end subroutine applySrc_electricFieldMSLiquid
  ! ************************************************************************ !

  ! ************************************************************************ !
?? copy :: applySource_header(applySrc_electricFieldMSLiquid_const)
    ! -------------------------------------------------------------------- !
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    real(kind=rk) :: electricField(fun%elemLvl(iLevel)%nElems*3)
    real(kind=rk) :: EF_elem(3)
    integer :: iElem, nElems, iDir, posInTotal
    integer :: iField, nFields, depField, nScalars, QQ, nInputStates
    !mass density of nSpecies
    real(kind=rk) :: mass_dens( varSys%nStateVars )
    !number density of nSpecies 
    real(kind=rk) :: num_dens( varSys%nStateVars )
    real(kind=rk) :: massFrac( varSys%nStateVars )
    real(kind=rk) :: charge_dens
    real(kind=rk), dimension(varSys%nStateVars) :: chargeNr, chargeTerm
    real(kind=rk) :: forceTerm
    real(kind=rk), dimension(3, varSys%nStateVars ) :: spcForce
    ! -------------------------------------------------------------------- !
!write(dbgUnit(1),*) 'source variable: ', trim(varSys%varname%val(fun%srcTerm_varPos))
    ! convert c pointer to solver type fortran pointer
    call c_f_pointer( varSys%method%val( fun%srcTerm_varPos )%method_data, &
      &               fPtr ) 
    scheme => fPtr%solverData%scheme

    ! Number of elements to apply source terms
    nElems = fun%elemLvl(iLevel)%nElems

    ! Get electrical force which is refered in config file either its
    ! spacetime variable or operation variable
    call varSys%method%val(fun%data_varPos)%get_valOfIndex( &
      & varSys  = varSys,                                   &
      & time    = time,                                     &
      & iLevel  = iLevel,                                   &
      & idx     = fun%elemLvl(iLevel)%idx(1:nElems),        &
      & nVals   = nElems,                                   & 
      & res     = electricField                             )

    ! convert physical to lattice
    electricField = electricField * fPtr%solverData%physics%coulomb0 &
      &           / fPtr%solverData%physics%fac(iLevel)%force

    ! number of pdf states this source depends on
    ! last input is spacetime function so it is neglected
    nInputStates = varSys%method%val(fun%srcTerm_varPos)%nInputs - 1
    ! constant parameter
    nFields = scheme%nFields
    QQ = scheme%layout%fStencil%QQ
    nScalars = varSys%nScalars
    ! species charge
    chargeNr(:) = scheme%field(:)%fieldProp%species%chargeNr

    ! update source for each element
    do iElem = 1, nElems
!write(dbgUnit(1),*) 'iElem ', iElem    
      ! to access level wise state array
      posInTotal = fun%elemLvl(iLevel)%posInTotal(iElem)

      mass_dens = 0.0_rk
      do iField = 1, nFields
        ! compute field density and first moments
        do iDir = 1, QQ 
          !field density
          mass_dens( iField ) = mass_dens( iField ) + inState(           &
            & ?FETCH?(iDir,iField,posInTotal,QQ,nScalars,nPdfSize,neigh) )
        end do
      end do
      !x_plus = neigh()

      !mass fraction
      massFrac(:) = mass_dens(:)/sum(mass_dens)

      ! number density of all species
      num_dens(:) = mass_dens(:) * scheme%field(:)%fieldProp%species &
        &                                         %molWeightInv

!write(dbgUnit(1),*) 'mass_dens ', mass_dens
!write(dbgUnit(1),*) 'num_dens ', num_dens

      ! compute external forcing term
      ! d^m_k = w_m*c_m*( n0*min_a(m_a)*F_k + (1/cs2)*(n_k*z_k*F*E) )
      ! F_k is diffsive forcing term
      EF_elem = electricField((iElem-1)*3+1 : iElem*3)
!write(dbgUnit(1),*) 'electricField ', EF_elem
      ! n_k * z_k 
      do iField = 1, nFields
        chargeTerm(iField) = num_dens(iField) * chargeNr(iField) &
          &                * scheme%mixture%faradayLB
      end do
      charge_dens = sum(chargeTerm)

      ! electric field term: n_k * z_k * Faraday * electricField
      do iField = 1, nFields

!        spcForce(:, iField) = cs2inv * chargeTerm(iField) * EF_elem(:)

        spcForce(:, iField) = cs2inv * EF_elem(:) &
          & *  massFrac(iField)*charge_dens

      end do
!write(dbgUnit(1),*) 'electricTerm ', electricTerm


      ! Update souce depends on nInputStates
      ! if nInputStates = 1, it is field source else it is global source
      do iField = 1, nInputStates
        depField = varSys%method%val(fun%srcTerm_varPos)%input_varPos(iField)

        do iDir = 1, QQ
          ! Force on each species
          ! d^m_k = weight_m*cxDir_m ( y_k*\rho_e Faraday E )
          forceTerm = scheme%layout%fStencil%cxDirRK( 1, iDir ) &
            &         * spcForce(1, depField)               &
            &       + scheme%layout%fStencil%cxDirRK( 2, iDir ) &
            &         * spcForce(2, depField)               &
            &       + scheme%layout%fStencil%cxDirRK( 3, iDir ) &
            &         * spcForce(3, depField)


!write(dbgUnit(1),*) 'cxDirRK ', scheme%layout%fStencil%cxDirRK( :, iDir )
!write(dbgUnit(1),*) 'spcForce depField ', spcForce(:, depField)

          outState(                                                         &
            & ?SAVE?(iDir,depField,posInTotal,QQ,nScalars,nPdfSize,neigh) ) &
            & = outState(                                                   &
            & ?SAVE?(iDir,depField,posInTotal,QQ,nScalars,nPdfSize,neigh) ) &
            !& + force(iDir)
            & + scheme%layout%weight( iDir ) * forceTerm
        end do ! iDir
      end do !iField

    end do !iElem 

  end subroutine applySrc_electricFieldMSLiquid_const
  ! ************************************************************************ !

?? copy :: applySource_header(applySrc_electricFieldMSLiquid_diff)
    ! -------------------------------------------------------------------- !
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    real(kind=rk) :: electricField(fun%elemLvl(iLevel)%nElems*3)
    real(kind=rk) :: EF_elem(3)
    integer :: iElem, nElems, iDir, posInTotal
    integer :: iField, nFields, depField, nScalars, QQ, nInputStates
    !mass density of nSpecies
    real(kind=rk) :: mass_dens( varSys%nStateVars )
    !number density of nSpecies 
    real(kind=rk) :: num_dens( varSys%nStateVars )
    real(kind=rk) :: massFrac( varSys%nStateVars )
    real(kind=rk) :: molWeigRatio( varSys%nStateVars )
    real(kind=rk) :: molWeight( varSys%nStateVars )
    real(kind=rk) :: avgMolWeight, charge_dens, diffForce_fac
    real(kind=rk), dimension(varSys%nStateVars) :: chargeNr, chargeTerm
    real(kind=rk) :: gasConst, temp0, minMolWeight, forceTerm,  mixDiffForce(3)
    real(kind=rk), dimension(3, varSys%nStateVars ) :: spcForce
    ! -------------------------------------------------------------------- !
!write(dbgUnit(1),*) 'source variable: ', trim(varSys%varname%val(fun%srcTerm_varPos))
    ! convert c pointer to solver type fortran pointer
    call c_f_pointer( varSys%method%val( fun%srcTerm_varPos )%method_data, &
      &               fPtr ) 
    scheme => fPtr%solverData%scheme

    ! Number of elements to apply source terms
    nElems = fun%elemLvl(iLevel)%nElems

    ! Get electrical force which is refered in config file either its
    ! spacetime variable or operation variable
    call varSys%method%val(fun%data_varPos)%get_valOfIndex( &
      & varSys  = varSys,                                   &
      & time    = time,                                     &
      & iLevel  = iLevel,                                   &
      & idx     = fun%elemLvl(iLevel)%idx(1:nElems),        &
      & nVals   = nElems,                                   & 
      & res     = electricField                             )

    ! convert physical to lattice
    electricField = electricField * fPtr%solverData%physics%coulomb0 &
      &           / fPtr%solverData%physics%fac(iLevel)%force

    ! number of pdf states this source depends on
    ! last input is spacetime function so it is neglected
    nInputStates = varSys%method%val(fun%srcTerm_varPos)%nInputs - 1
    ! constant parameter
    nFields = scheme%nFields
    QQ = scheme%layout%fStencil%QQ
    nScalars = varSys%nScalars
    ! minimum molecular weight
    minMolWeight = minval(scheme%field(:)%fieldProp%species%molWeight)
    molWeigRatio = scheme%field(:)%fieldProp%species%molWeigRatio
    molWeight = scheme%field(:)%fieldProp%species%molWeight
!write(dbgUnit(1),*) 'avgMolWeight ', avgMolWeight
!write(dbgUnit(1),*) 'ratio ', molWeight/avgMolWeight
    ! species charge
    chargeNr(:) = scheme%field(:)%fieldProp%species%chargeNr
    ! gas constant
    gasConst = scheme%mixture%gasConst_R_LB
    ! temperature
    temp0 = scheme%mixture%temp0LB
    diffForce_fac = minMolWeight / (gasConst*temp0)
!write(dbgUnit(1),*) 'diffForce_fac ', diffForce_fac

    ! update source for each element
    do iElem = 1, nElems
!write(dbgUnit(1),*) 'iElem ', iElem    
      ! to access level wise state array
      posInTotal = fun%elemLvl(iLevel)%posInTotal(iElem)

      mass_dens = 0.0_rk
      do iField = 1, nFields
        ! compute field density and first moments
        do iDir = 1, QQ 
          !field density
          mass_dens( iField ) = mass_dens( iField ) + inState(           &
            & ?FETCH?(iDir,iField,posInTotal,QQ,nScalars,nPdfSize,neigh) )
        end do
      end do
      !x_plus = neigh()

      !mass fraction
      massFrac(:) = mass_dens(:)/sum(mass_dens)

      ! number density of all species
      num_dens(:) = mass_dens(:) * scheme%field(:)%fieldProp%species &
        &                                         %molWeightInv

      ! mean molecular weight of mixture
      avgMolWeight = sum(mass_dens)/sum(num_dens)
!write(dbgUnit(1),*) 'mass_dens ', mass_dens
!write(dbgUnit(1),*) 'num_dens ', num_dens
!write(dbgUnit(1),*) 'avgMolWeight ', avgMolWeight

      ! compute external forcing term
      ! d^m_k = w_m*c_m*( n0*min_a(m_a)*F_k + (1/cs2)*(n_k*z_k*F*E) )
      ! F_k is diffsive forcing term
      EF_elem = electricField((iElem-1)*3+1 : iElem*3)
!write(dbgUnit(1),*) 'electricField ', EF_elem
      ! n_k * z_k 
      do iField = 1, nFields
        chargeTerm(iField) = num_dens(iField) * chargeNr(iField) &
          &                * scheme%mixture%faradayLB
      end do
      charge_dens = sum(chargeTerm)

      ! electric field term: n_k * z_k * Faraday * electricField
      ! Force on each species
      ! d^m_k = weight_m*cxDir_m ( min_a(m_a)*n * F_k + n_k*z_k Faraday E )
      do iField = 1, nFields
        spcForce(:, iField) = EF_elem(:) * diffForce_fac      &
          & * ( chargeTerm(iField) - massFrac(iField)*charge_dens )
      end do
!write(dbgUnit(1),*) 'spcForce ', spcForce

      ! Update souce depends on nInputStates
      ! if nInputStates = 1, it is field source else it is global source
      do iField = 1, nInputStates
        depField = varSys%method%val(fun%srcTerm_varPos)%input_varPos(iField)

        do iDir = 1, QQ
          forceTerm = scheme%layout%fStencil%cxDirRK( 1, iDir ) &
            &         * spcForce(1, depField)                   &
            &       + scheme%layout%fStencil%cxDirRK( 2, iDir ) &
            &         * spcForce(2, depField)                   &
            &       + scheme%layout%fStencil%cxDirRK( 3, iDir ) &
            &         * spcForce(3, depField)


!write(dbgUnit(1),*) 'cxDirRK ', scheme%layout%fStencil%cxDirRK( :, iDir )
!write(dbgUnit(1),*) 'spcForce depField ', spcForce(:, depField)

          outState(                                                         &
            & ?SAVE?(iDir,depField,posInTotal,QQ,nScalars,nPdfSize,neigh) ) &
            & = outState(                                                   &
            & ?SAVE?(iDir,depField,posInTotal,QQ,nScalars,nPdfSize,neigh) ) &
            !& + force(iDir)
            & + scheme%layout%weight( iDir ) * forceTerm
        end do ! iDir
!write(dbgUnit(1),*) depField, 'force: ', force
!write(dbgUnit(1),*) depField, 'forceTerm ', forceTerm
!write(dbgUnit(1),*) depField, 'forceMom rec ', matmul(scheme%layout%moment%toMoments%A, forceTerm)
      end do !iField

    end do !iElem 

!write(dbgUnit(1),*)    

  end subroutine applySrc_electricFieldMSLiquid_diff
  ! ************************************************************************ !
 

?? copy :: applySource_header(applySrc_electricFieldMSLiquid_1order_old)
    ! -------------------------------------------------------------------- !
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    real(kind=rk) :: electricField(fun%elemLvl(iLevel)%nElems*3)
    real(kind=rk) :: EF_elem(3)
    integer :: iElem, nElems, iDir, posInTotal
    integer :: iField, nFields, depField, nScalars, QQ, nInputStates
    !mass density of nSpecies
    real(kind=rk) :: mass_dens( varSys%nStateVars )
    !number density of nSpecies 
    real(kind=rk) :: num_dens( varSys%nStateVars )
    real(kind=rk) :: massFrac( varSys%nStateVars )
    real(kind=rk) :: molWeigRatio( varSys%nStateVars )
    real(kind=rk) :: molWeight( varSys%nStateVars )
    real(kind=rk) :: avgMolWeight, charge_dens, diffForce_fac
    real(kind=rk), dimension(varSys%nStateVars) :: chargeNr, chargeTerm
    real(kind=rk) :: gasConst, temp0, minMolWeight, forceTerm,  mixDiffForce(3)
    real(kind=rk), dimension(3, varSys%nStateVars ) :: mixForce,       &
      & diffForce, electricTerm, spcForce
    ! -------------------------------------------------------------------- !
!write(dbgUnit(1),*) 'source variable: ', trim(varSys%varname%val(fun%srcTerm_varPos))
    ! convert c pointer to solver type fortran pointer
    call c_f_pointer( varSys%method%val( fun%srcTerm_varPos )%method_data, &
      &               fPtr ) 
    scheme => fPtr%solverData%scheme

    ! Number of elements to apply source terms
    nElems = fun%elemLvl(iLevel)%nElems

    ! Get electrical force which is refered in config file either its
    ! spacetime variable or operation variable
    call varSys%method%val(fun%data_varPos)%get_valOfIndex( &
      & varSys  = varSys,                                   &
      & time    = time,                                     &
      & iLevel  = iLevel,                                   &
      & idx     = fun%elemLvl(iLevel)%idx(1:nElems),        &
      & nVals   = nElems,                                   & 
      & res     = electricField                             )

    ! convert physical to lattice
    electricField = electricField * fPtr%solverData%physics%coulomb0 &
      &           / fPtr%solverData%physics%fac(iLevel)%force

    ! number of pdf states this source depends on
    ! last input is spacetime function so it is neglected
    nInputStates = varSys%method%val(fun%srcTerm_varPos)%nInputs - 1
    ! constant parameter
    nFields = scheme%nFields
    QQ = scheme%layout%fStencil%QQ
    nScalars = varSys%nScalars
    ! minimum molecular weight
    minMolWeight = minval(scheme%field(:)%fieldProp%species%molWeight)
    molWeigRatio = scheme%field(:)%fieldProp%species%molWeigRatio
    molWeight = scheme%field(:)%fieldProp%species%molWeight
!write(dbgUnit(1),*) 'avgMolWeight ', avgMolWeight
!write(dbgUnit(1),*) 'ratio ', molWeight/avgMolWeight
    ! species charge
    chargeNr(:) = scheme%field(:)%fieldProp%species%chargeNr
    ! gas constant
    gasConst = scheme%mixture%gasConst_R_LB
    ! temperature
    temp0 = scheme%mixture%temp0LB
    diffForce_fac = minMolWeight / (gasConst*temp0)

    ! update source for each element
    do iElem = 1, nElems
!write(dbgUnit(1),*) 'iElem ', iElem    
      ! to access level wise state array
      posInTotal = fun%elemLvl(iLevel)%posInTotal(iElem)

      mass_dens = 0.0_rk
      do iField = 1, nFields
        ! compute field density and first moments
        do iDir = 1, QQ 
          !field density
          mass_dens( iField ) = mass_dens( iField ) + inState(           &
            & ?FETCH?(iDir,iField,posInTotal,QQ,nScalars,nPdfSize,neigh) )
        end do
      end do
      !x_plus = neigh()

      !mass fraction
      massFrac(:) = mass_dens(:)/sum(mass_dens)

      ! number density of all species
      num_dens(:) = mass_dens(:) * scheme%field(:)%fieldProp%species &
        &                                         %molWeightInv

      ! mean molecular weight of mixture
      avgMolWeight = sum(mass_dens)/sum(num_dens)
!write(dbgUnit(1),*) 'mass_dens ', mass_dens
!write(dbgUnit(1),*) 'num_dens ', num_dens
!write(dbgUnit(1),*) 'avgMolWeight ', avgMolWeight

      ! compute external forcing term
      ! d^m_k = w_m*c_m*( n0*min_a(m_a)*F_k + (1/cs2)*(n_k*z_k*F*E) )
      ! F_k is diffsive forcing term
      EF_elem = electricField((iElem-1)*3+1 : iElem*3)
!write(dbgUnit(1),*) 'electricField ', EF_elem
      ! n_k * z_k 
      do iField = 1, nFields
        chargeTerm(iField) = num_dens(iField) * chargeNr(iField) &
          &                * scheme%mixture%faradayLB
      end do
      charge_dens = sum(chargeTerm)

      ! electric field term: n_k * z_k * Faraday * electricField
      do iField = 1, nFields
!        electricTerm(:, iField) = cs2inv * chargeTerm(iField) * EF_elem(:)

!        electricTerm(:, iField) = cs2inv * EF_elem(:) &
!          & * ( chargeTerm(iField) - massFrac(iField)*charge_dens )

        electricTerm(:, iField) = EF_elem(:) * ( diffForce_fac      &
          & * ( chargeTerm(iField) - massFrac(iField)*charge_dens ) &
          & + cs2inv * massFrac(iField)*charge_dens )

      end do
!write(dbgUnit(1),*) 'electricTerm ', electricTerm

      spcForce = electricTerm
!write(dbgUnit(1),*) 'spcForce ', spcForce

      ! Update souce depends on nInputStates
      ! if nInputStates = 1, it is field source else it is global source
      do iField = 1, nInputStates
        depField = varSys%method%val(fun%srcTerm_varPos)%input_varPos(iField)

        do iDir = 1, QQ
          forceTerm = scheme%layout%fStencil%cxDirRK( 1, iDir ) &
            &         * spcForce(1, depField)                   &
            &       + scheme%layout%fStencil%cxDirRK( 2, iDir ) &
            &         * spcForce(2, depField)                   &
            &       + scheme%layout%fStencil%cxDirRK( 3, iDir ) &
            &         * spcForce(3, depField)

!write(dbgUnit(1),*) 'cxDirRK ', scheme%layout%fStencil%cxDirRK( :, iDir )
!write(dbgUnit(1),*) 'spcForce depField ', spcForce(:, depField)

          outState(                                                         &
            & ?SAVE?(iDir,depField,posInTotal,QQ,nScalars,nPdfSize,neigh) ) &
            & = outState(                                                   &
            & ?SAVE?(iDir,depField,posInTotal,QQ,nScalars,nPdfSize,neigh) ) &
            !& + force(iDir)
            & + scheme%layout%weight( iDir ) * forceTerm
        end do ! iDir
!write(dbgUnit(1),*) depField, 'force: ', force
!write(dbgUnit(1),*) depField, 'forceTerm ', forceTerm
!write(dbgUnit(1),*) depField, 'forceMom rec ', matmul(scheme%layout%moment%toMoments%A, forceTerm)
      end do !iField

    end do !iElem 

!write(dbgUnit(1),*)    

  end subroutine applySrc_electricFieldMSLiquid_1order_old
  ! ************************************************************************ !

  ! ************************************************************************ !
   !> Update state with source variable "electric_field"
   !! Simuilar to derive routine but it updates the state whereas derive 
   !! is used for tracking
   !! @todo species electricField
?? copy :: applySource_header(applySrc_electricFieldMSLiquid_2order)
    ! -------------------------------------------------------------------- !
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    real(kind=rk) :: electricField(fun%elemLvl(iLevel)%nElems*3)
    real(kind=rk) :: EF_elem(3)
    integer :: iElem, nElems, iDir, posInTotal, vPos
    integer :: iField, iField_2, nFields, depField, nScalars, QQ, nInputStates
    !mass density of nSpecies
    real(kind=rk), dimension(varSys%nStateVars) :: mass_dens, num_dens, &
      & massFrac, moleFrac, molWeight, phi, chargeNr, chargeTerm, spcForceTerm
    real(kind=rk) :: avgMolWeight, paramB, omega
    real(kind=rk) :: gasConst, temp0, minMolWeight
    real(kind=rk), dimension(3, varSys%nStateVars ) :: mixForce,       &
      & diffForce, electricTerm, spcForce
    real(kind=rk), dimension(varSys%nStateVars, varSys%nStateVars) :: matA, &
      & invA, resi_coeff
    real(kind=rk), dimension(3, varSys%nStateVars) :: first_moments, velocity, &
      & eqVel, force_moments
    real(kind=rk), allocatable :: forceTerm(:), omegaMoments(:,:), &
      & omegaMomForce(:,:)
    real(kind=rk), allocatable :: fneq(:), fneq_om(:), force_om(:)
    real(kind=rk) :: velAvg(3), ucxQuadTerm, totMassDens, totNum_dens
    real(kind=rk) :: usqr, ucx, feq, charge_dens 
    integer :: stateVarMap(varSys%nStateVars)
    real(kind=rk) :: pdfTmp( varSys%nScalars )  !< temporary local pdf values
    real(kind=rk) :: cxcxTerm(3,3), forceQuadTerm, forceUxTerm, diffForceUxTerm
    ! -------------------------------------------------------------------- !
!write(dbgUnit(1),*) 'source variable: ', trim(varSys%varname%val(fun%srcTerm_varPos))
    ! convert c pointer to solver type fortran pointer
    call c_f_pointer( varSys%method%val( fun%srcTerm_varPos )%method_data, &
      &               fPtr ) 
    scheme => fPtr%solverData%scheme

    ! Number of elements to apply source terms
    nElems = fun%elemLvl(iLevel)%nElems

    ! Get electrical force which is refered in config file either its
    ! spacetime variable or operation variable
    call varSys%method%val(fun%data_varPos)%get_valOfIndex( &
      & varSys  = varSys,                                   &
      & time    = time,                                     &
      & iLevel  = iLevel,                                   &
      & idx     = fun%elemLvl(iLevel)%idx(1:nElems),        &
      & nVals   = nElems,                                   & 
      & res     = electricField                             )

    ! convert physical to lattice
    electricField = electricField * fPtr%solverData%physics%coulomb0 &
      &           / fPtr%solverData%physics%fac(iLevel)%force

    ! number of pdf states this source depends on
    ! last input is spacetime function so it is neglected
    nInputStates = varSys%method%val(fun%srcTerm_varPos)%nInputs - 1
    ! constant parameter
    nFields = scheme%nFields
    QQ = scheme%layout%fStencil%QQ

    nScalars = varSys%nScalars
    stateVarMap = scheme%stateVarMap%varPos%val(:)
    ! minimum molecular weight
    minMolWeight = minval(scheme%field(:)%fieldProp%species%molWeight)
    phi = scheme%field(:)%fieldProp%species%molWeigRatio
    molWeight = scheme%field(:)%fieldProp%species%molWeight
!write(dbgUnit(1),*) 'avgMolWeight ', avgMolWeight
!write(dbgUnit(1),*) 'ratio ', molWeight/avgMolWeight
    ! species charge
    chargeNr(:) = scheme%field(:)%fieldProp%species%chargeNr
    ! gas constant
    gasConst = scheme%mixture%gasConst_R_LB
    ! temperature
    temp0 = scheme%mixture%temp0LB
    omega = scheme%mixture%relaxLvl(iLevel)%omega_diff

    !resistivities
    do iField = 1, nFields
      resi_coeff(iField,:) = scheme%field(iField)%fieldProp%species%resi_coeff
    enddo

    paramB = scheme%mixture%paramB

    allocate(forceTerm(QQ))
    allocate(omegaMoments(QQ,QQ))
    allocate(omegaMomForce(QQ,QQ))
    allocate(fneq(QQ))
    allocate(fneq_om(QQ))
    allocate(force_om(QQ))

    nodeloop: do iElem = 1, nElems
!write(dbgUnit(1),*) 'iElem', iElem

      mass_dens = 0._rk
      first_moments = 0.0_rk
      velocity = 0.0_rk
      eqVel = 0.0_rk
      pdfTmp = 0._rk

      do iField = 1, nFields
        ! compute field density and first moments
        do iDir = 1, QQ
          vPos = varSys%method%val( stateVarMap(iField) )%state_varPos(iDir)
          !store all field pdf in single element to local array pdfTmp
          pdfTmp( vPos ) =                                               &
& instate( ?FETCH?( iDir, iField, iElem, QQ, nScalars, nPdfSize,neigh ) )
          !field density
          mass_dens( iField ) = mass_dens( iField ) + pdfTmp( vPos )

          !field momentum (rho*u)
          first_moments( 1, iField ) = first_moments( 1, iField )              &
            & + pdfTmp( vPos ) * scheme%layout%fStencil%cxDir( 1, iDir )

          first_moments( 2, iField ) = first_moments( 2, iField )              &
            & + pdfTmp( vPos ) * scheme%layout%fStencil%cxDir( 2, iDir )

          first_moments( 3, iField ) = first_moments( 3, iField )              &
            & + pdfTmp( vPos ) * scheme%layout%fStencil%cxDir( 3, iDir )
!write(dbgunit,*) 's',s,'iDir', iDir, layout%fStencil%cxDir(3,iDir), pdfTmp( vPos ), gqzsigma(s)
        enddo
      enddo

      !total mass density
      totmassDens = sum(mass_dens)

      !mass fraction
      massFrac(:) = mass_dens(:)/totmassDens

      ! solve linear system of equation for actual moments
      ! number density of all species
      num_dens(:) = mass_dens(:)/molWeight(:)

      !total number density
      totNum_dens = sum(num_dens(:))

      !mole fraction
      moleFrac(:) =  num_dens(:)/totNum_dens

      ! mean molecular weight of mixture
      avgMolWeight = totmassDens/totNum_dens
      !Electric force per element E * Faraday
      EF_elem = electricField((iElem-1)*3+1 : iElem*3)
      ! n_k * z_k 
      do iField = 1, nFields
        chargeTerm(iField) = num_dens(iField) * chargeNr(iField) &
          &                * scheme%mixture%faradayLB
      end do
      charge_dens = sum(chargeTerm)
!write(dbgUnit(1),*) 'massdens ', mass_dens
!write(dbgUnit(1),*) 'numdens ', num_dens
!write(dbgUnit(1),*) 'chargeTerm', chargeTerm
!write(dbgUnit(1),*) 'EF_elem ', EF_elem
!write(dbgUnit(1),*) 'charge_dens ', charge_dens
!write(dbgUnit(1),*) 'fac ', minMolWeight/(gasConst*temp0)
      do iField = 1, nFields
!         spcForceTerm(iField) = phi(iField)*cs2*mass_dens(iField) &
!           &                  / ( num_dens(iField) * gasConst * temp0 )
!        electricTerm(:, iField) = minMolWeight / (gasConst*temp0)      &
!          &                     * (chargeTerm(iField)                  &
!          &                     - massFrac(iField)*charge_dens ) * EF_elem(:)

!        electricTerm(:, iField) = ( minMolWeight / (gasConst*temp0)       &
!          &                         * ( chargeTerm(iField)                   &
!          &                         - massFrac(iField)*charge_dens )         &
!          &                         + cs2inv * chargeTerm(iField) ) * EF_elem(:)

!        electricTerm(:, iField) = ( minMolWeight / (gasConst*temp0)       &
!          &                         * chargeTerm(iField)                   &
!          &                         + cs2inv * chargeTerm(iField) ) * EF_elem(:)

!        electricTerm(:, iField) = minMolWeight / (gasConst*temp0)       &
!          &                     * chargeTerm(iField) * EF_elem(:)

!        electricTerm(:, iField) = cs2inv * chargeTerm(iField) * EF_elem(:)

        diffForce(:, iField) = minMolWeight / (gasConst*temp0) &
          &                  * ( chargeTerm(iField)            &
          &                  - massFrac(iField)*charge_dens )  &
          &                  * EF_elem(:)
        electricTerm(:, iField) = chargeTerm(iField) * EF_elem(:)

!\todo activate
        first_moments(:, iField) = first_moments(:, iField)     &
          &                      + 0.5_rk*chargeTerm(iField)*EF_elem(:)
      end do  
!write(dbgUnit(1),*) 'spcForceTerm ', spcForceTerm      
!write(dbgUnit(1),*) 'electricTerm ', electricTerm
!write(dbgUnit(1),*) 'firstMoments ', first_moments

      matA = 0.0_rk
      !build up matrix to solver LSE for actual velocity
      do iField = 1, nFields
        !set diagonal part 
        matA(iField, iField) = 1.0_rk
        do iField_2 = 1, nFields
          matA(iField, iField) = matA(iField, iField) + omega * 0.5_rk         &
            &                  * resi_coeff(iField, iField_2) * phi(iField)    &
            &                  * moleFrac(iField_2) / paramB
        end do
        !set non-diagonal part
        do iField_2 = 1, nFields
          matA(iField, iField_2) = matA(iField, iField_2) - omega * 0.5_rk     &
            &                    * resi_coeff(iField, iField_2) * phi(iField_2)&
            &                    * moleFrac(iField) / paramB
        end do
      end do

      ! invert matrix
      invA = invert_matrix( matA )

      ! actual velocity of all species
      velocity(1, :) = matmul( invA, first_moments(1,:) ) / mass_dens(:)
      velocity(2, :) = matmul( invA, first_moments(2,:) ) / mass_dens(:)
      velocity(3, :) = matmul( invA, first_moments(3,:) ) / mass_dens(:)

      ! compute equilibrim velocity
      do iField = 1, nFields
        eqVel( :, iField ) = velocity( :, iField )
        do iField_2 = 1, nFields
          eqVel( :, iField ) = eqVel( :, iField )                              &
            &                + resi_coeff( iField, iField_2 ) * phi(iField)    &
            &                * moleFrac(iField_2)                              &
            &                * (velocity(:, iField_2) - velocity(:,iField))    &
            &                / paramB 
        end do  
      end do

      !compute mass averaged mixture velocity
      velAvg(1) = dot_product( mass_dens, velocity(1,:) )/totmassDens
      velAvg(2) = dot_product( mass_dens, velocity(2,:) )/totmassDens
      velAvg(3) = dot_product( mass_dens, velocity(3,:) )/totmassDens

      !compute equilibrium and do collision
      do iField = 1, nFields
        !omega moments = (M^-1 RelaxMat M)*(I+(M^-1 RelMat M)/2)^-1
        omegaMoments = scheme%field(iField)%fieldProp%species%mrt(iLevel)%omegaMoments
        omegaMomForce = scheme%field(iField)%fieldProp%species%mrt(iLevel)%omegaMomForce

        usqr = dot_product( velAvg, velAvg ) 
        usqr = usqr * t2cs2inv

        do iDir = 1, QQ
          vPos = varSys%method%val( stateVarMap(iField) )%state_varPos(iDir)

          ucx = scheme%layout%fStencil%cxDirRK( 1, iDir )*eqVel(1, iField) &
            & + scheme%layout%fStencil%cxDirRK( 2, iDir )*eqVel(2, iField) &
            & + scheme%layout%fStencil%cxDirRK( 3, iDir )*eqVel(3, iField)

          ucxQuadTerm = dot_product(                                    &
            &           scheme%layout%fStencil%cxDirRK(:, iDir), velAvg )

          feq = scheme%layout%weight(iDir) * mass_dens(iField) * ( phi(iField) &
            & + ucx * cs2inv + ucxQuadTerm * ucxQuadTerm * t2cs4inv - usqr )

          cxcxTerm = 0.0_rk
          if ( iDir == scheme%layout%fStencil%restPosition ) then
          ! equilibrium at rest
            select case( trim(scheme%header%layout) )
            case('d2q9')
              feq = scheme%layout%weight( iDir ) * mass_dens(iField) * ( &
                    & ( 9._rk - 5._rk * phi(iField) )/4._rk - usqr )
              cxcxTerm(1,1) = scheme%layout%fStencil%cxcx(1,iDir) - cs2
              cxcxTerm(2,2) = scheme%layout%fStencil%cxcx(2,iDir) - cs2
              cxcxTerm(1,2) = scheme%layout%fStencil%cxcx(3,iDir)
              cxcxTerm(2,1) = scheme%layout%fStencil%cxcx(3,iDir)
 
            case('d3q19')
              feq = scheme%layout%weight( iDir ) * mass_dens(iField) * ( &
                    & ( 3._rk - 2._rk * phi(iField) ) - usqr )

              cxcxTerm(1,1) = scheme%layout%fStencil%cxcx(1,iDir) - cs2
              cxcxTerm(2,2) = scheme%layout%fStencil%cxcx(2,iDir) - cs2
              cxcxTerm(3,3) = scheme%layout%fStencil%cxcx(3,iDir) - cs2
              cxcxTerm(1,2) = scheme%layout%fStencil%cxcx(4,iDir)
              cxcxTerm(2,1) = scheme%layout%fStencil%cxcx(4,iDir)
              cxcxTerm(2,3) = scheme%layout%fStencil%cxcx(5,iDir)
              cxcxTerm(3,2) = scheme%layout%fStencil%cxcx(5,iDir)
              cxcxTerm(1,3) = scheme%layout%fStencil%cxcx(6,iDir)
              cxcxTerm(3,1) = scheme%layout%fStencil%cxcx(6,iDir)
 
            end select
          end if  
          fneq(iDir) = ( feq - pdfTmp( vPos ) )

          forceQuadTerm = dot_product( electricTerm(:, iField), &
            &             matmul( cxcxTerm, velocity(:,iField) ) )

          forceUxTerm = dot_product( scheme%layout%fStencil%cxDirRK( :, iDir ),&
            &                        electricTerm(:, iField) )
  
          diffForceUxTerm = dot_product( scheme%layout%fStencil%cxDirRK( :, iDir ),&
            &                        diffForce(:, iField) )

          forceTerm(iDir) = scheme%layout%weight(iDir)        &
            & * ( cs2inv * forceUxTerm + cs4inv * forceQuadTerm )!+ diffForceUxTerm) 
!write(dbgUnit(1),*) 'forceQuadTerm ', forceQuadTerm, ' forceUxTerm ',forceUxTerm
!write(dbgUnit(1),*) 'diffForceUxTerm ', diffForceUxTerm
        enddo !iDir

!write(dbgUnit(1),*) 'forceTerm ', forceTerm
        fneq_om = matmul( omegaMoments, fneq )  
        
        force_om = matmul( omegaMomForce, forceTerm )

        do iDir = 1, QQ
          vPos = varSys%method%val( stateVarMap(iField) )%state_varPos(iDir)
          outstate(                                               &
& ?SAVE?( iDir, iField, iElem, QQ, nScalars, nPdfSize,neigh ) ) = &
            !& pdfTmp( vPos ) + fneq_om( iDir ) + forceTerm( iDir )
            & pdfTmp( vPos ) + fneq_om( iDir ) + force_om( iDir )
        enddo !iDir

      enddo !iField
    enddo nodeloop

    deallocate(forceTerm)
    deallocate(omegaMoments)
    deallocate(omegaMomForce)
    deallocate(fneq)
    deallocate(fneq_om)
    deallocate(force_om)

  end subroutine applySrc_electricFieldMSLiquid_2order
  ! ************************************************************************ !

?? copy :: applySource_header(applySrc_electricFieldMSLiquid_2order_new)
    ! -------------------------------------------------------------------- !
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    real(kind=rk) :: electricField(fun%elemLvl(iLevel)%nElems*3)
    real(kind=rk) :: EF_elem(3)
    integer :: iElem, nElems, iDir, posInTotal, vPos
    integer :: iField, iField_2, nFields, depField, nScalars, QQ, nInputStates
    !mass density of nSpecies
    real(kind=rk), dimension(varSys%nStateVars) :: mass_dens, num_dens, &
      & massFrac, moleFrac, molWeight, phi, chargeNr, chargeTerm, spcForceTerm
    real(kind=rk) :: avgMolWeight, paramB, omega
    real(kind=rk) :: gasConst, temp0, minMolWeight
    real(kind=rk), dimension(3, varSys%nStateVars ) :: mixForce,       &
      & diffForce, electricTerm, spcForce
    real(kind=rk), dimension(varSys%nStateVars, varSys%nStateVars) :: matA, &
      & invA, resi_coeff
    real(kind=rk), dimension(3, varSys%nStateVars) :: first_moments, velocity, &
      & eqVel, force_moments
    real(kind=rk), allocatable :: forceTerm(:), omegaMoments(:,:), &
      & omegaMomForce(:,:)
    real(kind=rk), allocatable :: fneq(:), fneq_om(:), force_om(:)
    real(kind=rk) :: velAvg(3), ucxQuadTerm, totMassDens, totNum_dens, diffForce_fac
    real(kind=rk) :: usqr, ucx, feq, charge_dens 
    integer :: stateVarMap(varSys%nStateVars)
    real(kind=rk) :: pdfTmp( varSys%nScalars )  !< temporary local pdf values
    real(kind=rk) :: cxcxTerm(3,3), forceQuadTerm, forceUxTerm, diffForceUxTerm
    ! -------------------------------------------------------------------- !
!write(dbgUnit(1),*) 'source variable: ', trim(varSys%varname%val(fun%srcTerm_varPos))
    ! convert c pointer to solver type fortran pointer
    call c_f_pointer( varSys%method%val( fun%srcTerm_varPos )%method_data, &
      &               fPtr ) 
    scheme => fPtr%solverData%scheme

    ! Number of elements to apply source terms
    nElems = fun%elemLvl(iLevel)%nElems

    ! Get electrical force which is refered in config file either its
    ! spacetime variable or operation variable
    call varSys%method%val(fun%data_varPos)%get_valOfIndex( &
      & varSys  = varSys,                                   &
      & time    = time,                                     &
      & iLevel  = iLevel,                                   &
      & idx     = fun%elemLvl(iLevel)%idx(1:nElems),        &
      & nVals   = nElems,                                   & 
      & res     = electricField                             )

    ! convert physical to lattice
    electricField = electricField * fPtr%solverData%physics%coulomb0 &
      &           / fPtr%solverData%physics%fac(iLevel)%force

    ! number of pdf states this source depends on
    ! last input is spacetime function so it is neglected
    nInputStates = varSys%method%val(fun%srcTerm_varPos)%nInputs - 1
    ! constant parameter
    nFields = scheme%nFields
    QQ = scheme%layout%fStencil%QQ

    nScalars = varSys%nScalars
    stateVarMap = scheme%stateVarMap%varPos%val(:)
    ! minimum molecular weight
    minMolWeight = minval(scheme%field(:)%fieldProp%species%molWeight)
    phi = scheme%field(:)%fieldProp%species%molWeigRatio
    molWeight = scheme%field(:)%fieldProp%species%molWeight
!write(dbgUnit(1),*) 'avgMolWeight ', avgMolWeight
!write(dbgUnit(1),*) 'ratio ', molWeight/avgMolWeight
    ! species charge
    chargeNr(:) = scheme%field(:)%fieldProp%species%chargeNr
    ! gas constant
    gasConst = scheme%mixture%gasConst_R_LB
    ! temperature
    temp0 = scheme%mixture%temp0LB
    omega = scheme%mixture%relaxLvl(iLevel)%omega_diff
    diffForce_fac = minMolWeight / (gasConst*temp0)

    !resistivities
    do iField = 1, nFields
      resi_coeff(iField,:) = scheme%field(iField)%fieldProp%species%resi_coeff
    enddo

    paramB = scheme%mixture%paramB

    allocate(forceTerm(QQ))
    allocate(omegaMoments(QQ,QQ))
    allocate(omegaMomForce(QQ,QQ))
    allocate(fneq(QQ))
    allocate(fneq_om(QQ))
    allocate(force_om(QQ))

    nodeloop: do iElem = 1, nElems
!write(dbgUnit(1),*) 'iElem', iElem

      mass_dens = 0._rk
      first_moments = 0.0_rk
      velocity = 0.0_rk
      eqVel = 0.0_rk
      pdfTmp = 0._rk

      do iField = 1, nFields
        ! compute field density and first moments
        do iDir = 1, QQ
          vPos = varSys%method%val( stateVarMap(iField) )%state_varPos(iDir)
          !store all field pdf in single element to local array pdfTmp
          pdfTmp( vPos ) =                                               &
& instate( ?FETCH?( iDir, iField, iElem, QQ, nScalars, nPdfSize,neigh ) )
          !field density
          mass_dens( iField ) = mass_dens( iField ) + pdfTmp( vPos )

          !field momentum (rho*u)
          first_moments( 1, iField ) = first_moments( 1, iField )              &
            & + pdfTmp( vPos ) * scheme%layout%fStencil%cxDir( 1, iDir )

          first_moments( 2, iField ) = first_moments( 2, iField )              &
            & + pdfTmp( vPos ) * scheme%layout%fStencil%cxDir( 2, iDir )

          first_moments( 3, iField ) = first_moments( 3, iField )              &
            & + pdfTmp( vPos ) * scheme%layout%fStencil%cxDir( 3, iDir )
!write(dbgunit,*) 's',s,'iDir', iDir, layout%fStencil%cxDir(3,iDir), pdfTmp( vPos ), gqzsigma(s)
        enddo
      enddo

      !total mass density
      totmassDens = sum(mass_dens)

      !mass fraction
      massFrac(:) = mass_dens(:)/totmassDens

      ! solve linear system of equation for actual moments
      ! number density of all species
      num_dens(:) = mass_dens(:)/molWeight(:)

      !total number density
      totNum_dens = sum(num_dens(:))

      !mole fraction
      moleFrac(:) =  num_dens(:)/totNum_dens

      ! mean molecular weight of mixture
      avgMolWeight = totmassDens/totNum_dens
      !Electric force per element E * Faraday
      EF_elem = electricField((iElem-1)*3+1 : iElem*3)
      ! n_k * z_k 
      do iField = 1, nFields
        chargeTerm(iField) = num_dens(iField) * chargeNr(iField) &
          &                * scheme%mixture%faradayLB
      end do
      charge_dens = sum(chargeTerm)
!write(dbgUnit(1),*) 'massdens ', mass_dens
!write(dbgUnit(1),*) 'numdens ', num_dens
!write(dbgUnit(1),*) 'chargeTerm', chargeTerm
!write(dbgUnit(1),*) 'EF_elem ', EF_elem
!write(dbgUnit(1),*) 'charge_dens ', charge_dens
!write(dbgUnit(1),*) 'fac ', minMolWeight/(gasConst*temp0)
      do iField = 1, nFields
!         spcForceTerm(iField) = phi(iField)*cs2*mass_dens(iField) &
!           &                  / ( num_dens(iField) * gasConst * temp0 )
!        electricTerm(:, iField) = minMolWeight / (gasConst*temp0)      &
!          &                     * (chargeTerm(iField)                  &
!          &                     - massFrac(iField)*charge_dens ) * EF_elem(:)

!        electricTerm(:, iField) = ( minMolWeight / (gasConst*temp0)       &
!          &                         * ( chargeTerm(iField)                   &
!          &                         - massFrac(iField)*charge_dens )         &
!          &                         + cs2inv * chargeTerm(iField) ) * EF_elem(:)

!        electricTerm(:, iField) = ( minMolWeight / (gasConst*temp0)       &
!          &                         * chargeTerm(iField)                   &
!          &                         + cs2inv * chargeTerm(iField) ) * EF_elem(:)

!        electricTerm(:, iField) = minMolWeight / (gasConst*temp0)       &
!          &                     * chargeTerm(iField) * EF_elem(:)

!        electricTerm(:, iField) = cs2inv * chargeTerm(iField) * EF_elem(:)

!        diffForce(:, iField) = minMolWeight / (gasConst*temp0) &
!          &                  * ( chargeTerm(iField)            &
!          &                  - massFrac(iField)*charge_dens )  &
!          &                  * EF_elem(:)
!        electricTerm(:, iField) = chargeTerm(iField) * EF_elem(:)

        electricTerm(:, iField) = EF_elem(:) * ( diffForce_fac      &
          & * ( chargeTerm(iField) - massFrac(iField)*charge_dens ) &
          & + cs2inv * massFrac(iField)*charge_dens )

!\todo activate
        first_moments(:, iField) = first_moments(:, iField)     &
          &                      + 0.5_rk*cs2*electricTerm(:, iField)
      end do  
!write(dbgUnit(1),*) 'spcForceTerm ', spcForceTerm      
!write(dbgUnit(1),*) 'electricTerm ', electricTerm
!write(dbgUnit(1),*) 'firstMoments ', first_moments

      matA = 0.0_rk
      !build up matrix to solver LSE for actual velocity
      do iField = 1, nFields
        !set diagonal part 
        matA(iField, iField) = 1.0_rk
        do iField_2 = 1, nFields
          matA(iField, iField) = matA(iField, iField) + omega * 0.5_rk         &
            &                  * resi_coeff(iField, iField_2) * phi(iField)    &
            &                  * moleFrac(iField_2) / paramB
        end do
        !set non-diagonal part
        do iField_2 = 1, nFields
          matA(iField, iField_2) = matA(iField, iField_2) - omega * 0.5_rk     &
            &                    * resi_coeff(iField, iField_2) * phi(iField_2)&
            &                    * moleFrac(iField) / paramB
        end do
      end do

      ! invert matrix
      invA = invert_matrix( matA )

      ! actual velocity of all species
      velocity(1, :) = matmul( invA, first_moments(1,:) ) / mass_dens(:)
      velocity(2, :) = matmul( invA, first_moments(2,:) ) / mass_dens(:)
      velocity(3, :) = matmul( invA, first_moments(3,:) ) / mass_dens(:)

      ! compute equilibrim velocity
      do iField = 1, nFields
        eqVel( :, iField ) = velocity( :, iField )
        do iField_2 = 1, nFields
          eqVel( :, iField ) = eqVel( :, iField )                              &
            &                + resi_coeff( iField, iField_2 ) * phi(iField)    &
            &                * moleFrac(iField_2)                              &
            &                * (velocity(:, iField_2) - velocity(:,iField))    &
            &                / paramB 
        end do  
      end do

      !compute mass averaged mixture velocity
      velAvg(1) = dot_product( mass_dens, velocity(1,:) )/totmassDens
      velAvg(2) = dot_product( mass_dens, velocity(2,:) )/totmassDens
      velAvg(3) = dot_product( mass_dens, velocity(3,:) )/totmassDens

      !compute equilibrium and do collision
      do iField = 1, nFields
        !omega moments = (M^-1 RelaxMat M)*(I+(M^-1 RelMat M)/2)^-1
        omegaMoments = scheme%field(iField)%fieldProp%species%mrt(iLevel)%omegaMoments
        omegaMomForce = scheme%field(iField)%fieldProp%species%mrt(iLevel)%omegaMomForce

        usqr = dot_product( velAvg, velAvg ) 
        usqr = usqr * t2cs2inv

        do iDir = 1, QQ
          vPos = varSys%method%val( stateVarMap(iField) )%state_varPos(iDir)

          ucx = scheme%layout%fStencil%cxDirRK( 1, iDir )*eqVel(1, iField) &
            & + scheme%layout%fStencil%cxDirRK( 2, iDir )*eqVel(2, iField) &
            & + scheme%layout%fStencil%cxDirRK( 3, iDir )*eqVel(3, iField)

          ucxQuadTerm = dot_product(                                    &
            &           scheme%layout%fStencil%cxDirRK(:, iDir), velAvg )

          feq = scheme%layout%weight(iDir) * mass_dens(iField) * ( phi(iField) &
            & + ucx * cs2inv + ucxQuadTerm * ucxQuadTerm * t2cs4inv - usqr )

          cxcxTerm = 0.0_rk
          if ( iDir == scheme%layout%fStencil%restPosition ) then
          ! equilibrium at rest
            select case( trim(scheme%header%layout) )
            case('d2q9')
              feq = scheme%layout%weight( iDir ) * mass_dens(iField) * ( &
                    & ( 9._rk - 5._rk * phi(iField) )/4._rk - usqr )
              cxcxTerm(1,1) = scheme%layout%fStencil%cxcx(1,iDir) - cs2
              cxcxTerm(2,2) = scheme%layout%fStencil%cxcx(2,iDir) - cs2
              cxcxTerm(1,2) = scheme%layout%fStencil%cxcx(3,iDir)
              cxcxTerm(2,1) = scheme%layout%fStencil%cxcx(3,iDir)
 
            case('d3q19')
              feq = scheme%layout%weight( iDir ) * mass_dens(iField) * ( &
                    & ( 3._rk - 2._rk * phi(iField) ) - usqr )

              cxcxTerm(1,1) = scheme%layout%fStencil%cxcx(1,iDir) - cs2
              cxcxTerm(2,2) = scheme%layout%fStencil%cxcx(2,iDir) - cs2
              cxcxTerm(3,3) = scheme%layout%fStencil%cxcx(3,iDir) - cs2
              cxcxTerm(1,2) = scheme%layout%fStencil%cxcx(4,iDir)
              cxcxTerm(2,1) = scheme%layout%fStencil%cxcx(4,iDir)
              cxcxTerm(2,3) = scheme%layout%fStencil%cxcx(5,iDir)
              cxcxTerm(3,2) = scheme%layout%fStencil%cxcx(5,iDir)
              cxcxTerm(1,3) = scheme%layout%fStencil%cxcx(6,iDir)
              cxcxTerm(3,1) = scheme%layout%fStencil%cxcx(6,iDir)
 
            end select
          end if  
          fneq(iDir) = ( feq - pdfTmp( vPos ) )

          forceQuadTerm = dot_product( electricTerm(:, iField), &
            &             matmul( cxcxTerm, velocity(:,iField) ) )

          forceUxTerm = dot_product( scheme%layout%fStencil%cxDirRK( :, iDir ),&
            &                        electricTerm(:, iField) )
  
!          diffForceUxTerm = dot_product( scheme%layout%fStencil%cxDirRK( :, iDir ),&
!            &                        diffForce(:, iField) )

          forceTerm(iDir) = scheme%layout%weight(iDir)        &
            & * ( forceUxTerm + cs2inv * forceQuadTerm )!+ diffForceUxTerm) 
!write(dbgUnit(1),*) 'forceQuadTerm ', forceQuadTerm, ' forceUxTerm ',forceUxTerm
!write(dbgUnit(1),*) 'diffForceUxTerm ', diffForceUxTerm
        enddo !iDir

!write(dbgUnit(1),*) 'forceTerm ', forceTerm
        fneq_om = matmul( omegaMoments, fneq )  
        
        force_om = matmul( omegaMomForce, forceTerm )

        do iDir = 1, QQ
          vPos = varSys%method%val( stateVarMap(iField) )%state_varPos(iDir)
          outstate(                                               &
& ?SAVE?( iDir, iField, iElem, QQ, nScalars, nPdfSize,neigh ) ) = &
            !& pdfTmp( vPos ) + fneq_om( iDir ) + forceTerm( iDir )
            & pdfTmp( vPos ) + fneq_om( iDir ) + force_om( iDir )
        enddo !iDir

      enddo !iField
    enddo nodeloop

    deallocate(forceTerm)
    deallocate(omegaMoments)
    deallocate(omegaMomForce)
    deallocate(fneq)
    deallocate(fneq_om)
    deallocate(force_om)

  end subroutine applySrc_electricFieldMSLiquid_2order_new
 

  ! ************************************************************************ !
   !> Update state with source variable "gravity_field"
   !! Simuilar to derive routine but it updates the state whereas derive 
   !! is used for tracking
?? copy :: applySource_header(applySrc_gravityFieldMSLiquid)
    ! -------------------------------------------------------------------- !
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    real(kind=rk) :: gravityField(fun%elemLvl(iLevel)%nElems*3)
    integer :: iElem, nElems, iDir, posInTotal
    integer :: iField, nFields, depField, nScalars, QQ, nInputStates
    !mass density of nSpecies
    real(kind=rk) :: mass_dens( varSys%nStateVars )
    real(kind=rk) :: massFrac( varSys%nStateVars )
    real(kind=rk) :: gasConst, temp0, minMolWeight, forceTerm, mixDiffForce(3)
    real(kind=rk), dimension(3, varSys%nStateVars ) :: mixForce,       &
      & diffForce, gravityTerm, spcForce
    integer :: stateVarMap(varSys%nStateVars)
    ! -------------------------------------------------------------------- !
    !write(*,*) 'source variable: ', trim(varSys%varname%val(fun%srcTerm_varPos))
    ! convert c pointer to solver type fortran pointer
    call c_f_pointer( varSys%method%val( fun%srcTerm_varPos )%method_data, &
      &               fPtr ) 
    scheme => fPtr%solverData%scheme

    ! Number of elements to apply source terms
    nElems = fun%elemLvl(iLevel)%nElems

    ! Get gravitational force which is refered in config file either its
    ! spacetime variable or operation variable
    call varSys%method%val(fun%data_varPos)%get_valOfIndex( &
      & varSys  = varSys,                                   &
      & time    = time,                                     &
      & iLevel  = iLevel,                                   &
      & idx     = fun%elemLvl(iLevel)%idx(1:nElems),        &
      & nVals   = nElems,                                   & 
      & res     = gravityField                              )

    ! convert physical to lattice
    gravityField = gravityField * fPtr%solverData%physics%dtLvl(iLevel)**2 &
      &           / fPtr%solverData%physics%dxLvl(iLevel)

    ! number of pdf states this source depends on
    ! last input is spacetime function so it is neglected
    nInputStates = varSys%method%val(fun%srcTerm_varPos)%nInputs - 1
    ! constant parameter
    nFields = scheme%nFields
    QQ = scheme%layout%fStencil%QQ
    nScalars = varSys%nScalars
    stateVarMap = scheme%stateVarMap%varPos%val(:)

    ! update source for each element
    do iElem = 1, nElems
      ! to access level wise state array
      posInTotal = fun%elemLvl(iLevel)%posInTotal(iElem)

      mass_dens = 0.0_rk
      do iField = 1, nFields
        ! compute field density and first moments
        do iDir = 1, QQ 
          !field density
          mass_dens( iField ) = mass_dens( iField ) + inState(          &
            &?FETCH?(iDir,iField,posInTotal,QQ,nScalars,nPdfSize,neigh) )
        end do
      end do  


      ! compute external forcing term
      ! d^m_k = w_m*c_m*( (1/cs2)*(\rho_k*g) )
      ! F_k is diffsive forcing term
      ! external constant force term: y_k * gravityField
      do iField = 1, nFields
        spcForce(:, iField) = cs2inv * (mass_dens(iField)/sum(mass_dens))  & 
          &                 * gravityField((iElem-1)*3+1 : iElem*3)
      end do

      do iField = 1, nInputStates
        depField = varSys%method%val(fun%srcTerm_varPos)%input_varPos(iField)
        do iDir = 1, QQ
          forceTerm =  dot_product(                               & 
            &          scheme%layout%fStencil%cxDirRK( :, iDir ), &
            &          spcForce(:, depField) )

          outState(                                                         &
            & ?SAVE?(iDir,depField,posInTotal,QQ,nScalars,nPdfSize,neigh) ) &
            & = outState(                                                   &
            & ?SAVE?(iDir,depField,posInTotal,QQ,nScalars,nPdfSize,neigh) ) &
            & + scheme%layout%weight( iDir ) * forceTerm
        end do
      end do
    end do !iElem 
  end subroutine applySrc_gravityFieldMSLiquid
  ! ************************************************************************ !


  ! ************************************************************************ !
   !> Update state with source variable "electric_field"
   !! Simuilar to derive routine but it updates the state whereas derive 
   !! is used for tracking
   !! @todo species electricField
?? copy :: applySource_header(applySrc_electricFieldMSLiquid_WTDF)
    ! -------------------------------------------------------------------- !
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    real(kind=rk) :: electricField(fun%elemLvl(iLevel)%nElems*3)
    real(kind=rk) :: EF_elem(3)
    integer :: iElem, nElems, iDir, posInTotal
    integer :: iField, nFields, depField, nScalars, QQ, nInputStates
    integer :: iField_2
    !mass density of nSpecies
    real(kind=rk) :: mass_dens( varSys%nStateVars )
    !number density of nSpecies 
    real(kind=rk) :: num_dens( varSys%nStateVars )
    real(kind=rk), dimension(varSys%nStateVars) :: massFrac, moleFrac 
    real(kind=rk), dimension(varSys%nStateVars) :: chargeNr
    real(kind=rk) :: gasConst, temp0, minMolWeight, forceTerm, mixDiffForce(3)
    real(kind=rk), dimension(3, varSys%nStateVars ) :: mixForce,       &
      & diffForce, electricTerm, spcForce, diffForce_WTF
    real(kind=rk), dimension(varSys%nStateVars, varSys%nStateVars) :: &
      & resi_coeff, thermodynamic_fac, &
      & inv_thermodyn_fac, diff_coeff
    real(kind=rk) :: press, temp
    real(kind=rk) :: phy_moleDens_fac, diffForce_fac
    integer :: stateVarMap(varSys%nStateVars)
    ! -------------------------------------------------------------------- !
    !write(*,*) 'source variable: ', trim(varSys%varname%val(fun%srcTerm_varPos))
    ! convert c pointer to solver type fortran pointer
    call c_f_pointer( varSys%method%val( fun%srcTerm_varPos )%method_data, &
      &               fPtr ) 
    scheme => fPtr%solverData%scheme

    ! Number of elements to apply source terms
    nElems = fun%elemLvl(iLevel)%nElems

    ! Get electrical force which is refered in config file either its
    ! spacetime variable or operation variable
    call varSys%method%val(fun%data_varPos)%get_valOfIndex( &
      & varSys  = varSys,                                   &
      & time    = time,                                     &
      & iLevel  = iLevel,                                   &
      & idx     = fun%elemLvl(iLevel)%idx(1:nElems),        &
      & nVals   = nElems,                                   & 
      & res     = electricField                             )

    ! convert physical to lattice
    electricField = electricField * fPtr%solverData%physics%coulomb0 &
      &           / fPtr%solverData%physics%fac(iLevel)%force

    ! number of pdf states this source depends on
    ! last input is spacetime function so it is neglected
    nInputStates = varSys%method%val(fun%srcTerm_varPos)%nInputs - 1
    ! constant parameter
    nFields = scheme%nFields
    QQ = scheme%layout%fStencil%QQ
    nScalars = varSys%nScalars
    stateVarMap = scheme%stateVarMap%varPos%val(:)
    do iField = 1, nFields
      ! diffusivity coefficients
      diff_coeff(iField, :) =                                &
        & scheme%field(iField)%fieldProp%species%diff_coeff(:)
    end do
    !KM @todo check moleDens for multilevel
    phy_moleDens_fac = fPtr%solverData%physics%moleDens0
    ! minimum molecular weight
    minMolWeight = minval(scheme%field(:)%fieldProp%species%molWeight)
    ! species charge
    chargeNr(:) = scheme%field(:)%fieldProp%species%chargeNr
    ! gas constant
    gasConst = scheme%mixture%gasConst_R_LB
    ! temperature
    temp0 = scheme%mixture%temp0LB
    ! temperature
    temp = scheme%mixture%temp0
    ! atmospheric pressure
    press = scheme%mixture%atm_press
    diffForce_fac = minMolWeight / (gasConst*temp0)

    ! update source for each element
    do iElem = 1, nElems
      ! to access level wise state array
      posInTotal = fun%elemLvl(iLevel)%posInTotal(iElem)

      mass_dens = 0.0_rk
      do iField = 1, nFields
        ! compute field density and first moments
        do iDir = 1, QQ 
          !field density
          mass_dens( iField ) = mass_dens( iField ) + inState(           &
            & ?FETCH?(iDir,iField,posInTotal,QQ,nScalars,nPdfSize,neigh) )
        end do
      end do  

      !mass fraction
      massFrac(:) = mass_dens(:)/sum(mass_dens)

      ! number density of all species
      num_dens(:) = mass_dens(:) * scheme%field(:)%fieldProp%species &
        &                                         %molWeightInv
!write(dbgUnit(1),*) 'mass_dens ', mass_dens
!write(dbgUnit(1),*) 'num_dens ', num_dens
      ! mole fraction 
      moleFrac = num_dens/sum(num_dens)

      ! MS-Diff coeff matrix from C++ code
      call mus_calc_MS_DiffMatrix( nFields, temp, press,               &
        &                        num_dens*phy_moleDens_fac, diff_coeff )
      ! Convert to lattice unit
      resi_coeff = fPtr%solverData%physics%fac(iLevel)%diffusivity/diff_coeff

      ! Thermodynamic factor from C++ code
      call mus_calc_thermFactor( nFields, temp, press, moleFrac, &
        &                        thermodynamic_fac )

      inv_thermodyn_fac = invert_matrix( thermodynamic_fac )

      ! compute external forcing term
      ! d^m_k = w_m*c_m*( n0*min_a(m_a)*F_k + (1/cs2)*(n_k*z_k*F*E) )
      ! F_k is diffsive forcing term

      EF_elem = electricField((iElem-1)*3+1 : iElem*3)
      !write(*,*) 'electricField ', EF_elem

      ! electric field term: n_k * z_k * Faraday * electricField
      do iField = 1, nFields
        electricTerm(:, iField) = num_dens(iField) * chargeNr(iField) &
          &                     * scheme%mixture%faradayLB            &
          &                     * EF_elem(:)
      end do
!write(dbgUnit(1),*) 'electricTerm ', electricTerm

      ! total mixture force in diffusive forcing term
      ! totMixForce = sum_l(n_l z_l Faraday E)
      mixDiffForce(1) = sum(electricTerm(1, :))
      mixDiffForce(2) = sum(electricTerm(2, :))
      mixDiffForce(3) = sum(electricTerm(3, :))
!write(dbgUnit(1),*) 'totMixForce ', mixDiffForce 

      ! diffusive forcing term in the Maxwell-Stefan equations
      ! F_k = 1/(nRT) rho_k (z_k Faraday/m_k E) 
      !     - y_k/(nRT) sum_l(rho_l (z_l Faraday/m_l E)
      ! diffForce = (electric term - fraction of totMixForce) *
      ! 1/(nRT) * n0 * min_a(m_a)
      ! Here n and n0 will cancel out so they are not included in computation
      do iField = 1, nFields
        diffForce(:, iField) = ( electricTerm(:, iField)            &
          &                  - massFrac(iField) * mixDiffForce(:) ) &
          &                  / ( gasConst * temp0 ) 
      end do
!write(dbgUnit(1),*) 'diffForce ', diffForce      
      ! multiply diffForce with thermFacInv * min_a(m_a)
      ! diffusive driving force term:
      !  min_a(m_a) * c_m* w_m * sum_l(themFacInv_k_l * n * F_l)
      diffForce_WTF = 0.0_rk
      do iField = 1, nFields
        do iField_2 = 1, nFields
          diffForce_WTF(:, iField ) = diffForce_WTF(:, iField)                 &
            &                       + inv_thermodyn_fac(iField, iField_2)      &
            &                       * diffForce(:, iField_2)
        end do
        diffForce_WTF(:, iField ) = minMolWeight * diffForce_WTF(:, iField)
      end do

      ! forcing term in the mixture for momentum equation
      ! mixForce = (1/cs2)*(rho_k*g + n_k*z_k*F*E)
      mixForce = cs2inv * electricTerm

      ! Force on each species
      ! d^m_k = weight_m*cxDir_m ( min_a(m_a)*n * F_k + n_k*z_k Faraday E )
      spcForce = diffForce_WTF + mixForce

      ! Update souce depends on nInputStates
      ! if nInputStates = 1, it is field source else it is global source
      do iField = 1, nInputStates
        depField = varSys%method%val(fun%srcTerm_varPos)%input_varPos(iField)
        do iDir = 1, QQ
          forceTerm =  dot_product(                               & 
            &          scheme%layout%fStencil%cxDirRK( :, iDir ), &
            &          spcForce(:, depField) )

          outState(                                                         &
            & ?SAVE?(iDir,depField,posInTotal,QQ,nScalars,nPdfSize,neigh) ) &
            & = outState(                                                   &
            & ?SAVE?(iDir,depField,posInTotal,QQ,nScalars,nPdfSize,neigh) ) &
            & + scheme%layout%weight( iDir ) * forceTerm
!write(dbgUnit(1),*) iField, 'forceTerm ', scheme%layout%weight( iDir ) * forceTerm
        end do
      end do

    end do !iElem 

  end subroutine applySrc_electricFieldMSLiquid_WTDF
  ! ************************************************************************ !

  ! ************************************************************************ !
?? copy :: applySource_header(applySrc_electricFieldMSLiquid_diff_WTDF)
    ! -------------------------------------------------------------------- !
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    real(kind=rk) :: electricField(fun%elemLvl(iLevel)%nElems*3)
    real(kind=rk) :: EF_elem(3)
    integer :: iElem, nElems, iDir, posInTotal
    integer :: iField, nFields, depField, nScalars, QQ, nInputStates
    integer :: iField_2
    !mass density of nSpecies
    real(kind=rk) :: mass_dens( varSys%nStateVars )
    !number density of nSpecies 
    real(kind=rk) :: num_dens( varSys%nStateVars )
    real(kind=rk) :: massFrac( varSys%nStateVars )
    real(kind=rk) :: moleFrac( varSys%nStateVars )
    real(kind=rk) :: molWeigRatio( varSys%nStateVars )
    real(kind=rk) :: molWeight( varSys%nStateVars )
    real(kind=rk) :: avgMolWeight, charge_dens, diffForce_fac
    real(kind=rk), dimension(varSys%nStateVars) :: chargeNr, chargeTerm
    real(kind=rk) :: gasConst, temp0, minMolWeight, forceTerm,  mixDiffForce(3)
    real(kind=rk), dimension(3, varSys%nStateVars ) :: mixForce,       &
      & diffForce_WTF, electricTerm, spcForce
    real(kind=rk), dimension(varSys%nStateVars, varSys%nStateVars) :: &
      & thermodynamic_fac, inv_thermodyn_fac
    real(kind=rk) :: press, temp
    ! -------------------------------------------------------------------- !
!write(dbgUnit(1),*) 'source variable: ', trim(varSys%varname%val(fun%srcTerm_varPos))
    ! convert c pointer to solver type fortran pointer
    call c_f_pointer( varSys%method%val( fun%srcTerm_varPos )%method_data, &
      &               fPtr ) 
    scheme => fPtr%solverData%scheme

    ! Number of elements to apply source terms
    nElems = fun%elemLvl(iLevel)%nElems

    ! Get electrical force which is refered in config file either its
    ! spacetime variable or operation variable
    call varSys%method%val(fun%data_varPos)%get_valOfIndex( &
      & varSys  = varSys,                                   &
      & time    = time,                                     &
      & iLevel  = iLevel,                                   &
      & idx     = fun%elemLvl(iLevel)%idx(1:nElems),        &
      & nVals   = nElems,                                   & 
      & res     = electricField                             )

    ! convert physical to lattice
    electricField = electricField * fPtr%solverData%physics%coulomb0 &
      &           / fPtr%solverData%physics%fac(iLevel)%force

    ! number of pdf states this source depends on
    ! last input is spacetime function so it is neglected
    nInputStates = varSys%method%val(fun%srcTerm_varPos)%nInputs - 1
    ! constant parameter
    nFields = scheme%nFields
    QQ = scheme%layout%fStencil%QQ
    nScalars = varSys%nScalars

    ! minimum molecular weight
    minMolWeight = minval(scheme%field(:)%fieldProp%species%molWeight)
    molWeigRatio = scheme%field(:)%fieldProp%species%molWeigRatio
    molWeight = scheme%field(:)%fieldProp%species%molWeight
    ! species charge
    chargeNr(:) = scheme%field(:)%fieldProp%species%chargeNr
    ! gas constant
    gasConst = scheme%mixture%gasConst_R_LB
    ! temperature
    temp0 = scheme%mixture%temp0LB
    ! temperature
    temp = scheme%mixture%temp0
    ! atmospheric pressure
    press = scheme%mixture%atm_press
    diffForce_fac = minMolWeight / (gasConst*temp0)

    ! update source for each element
    do iElem = 1, nElems
      ! to access level wise state array
      posInTotal = fun%elemLvl(iLevel)%posInTotal(iElem)

      mass_dens = 0.0_rk
      do iField = 1, nFields
        ! compute field density and first moments
        do iDir = 1, QQ 
          !field density
          mass_dens( iField ) = mass_dens( iField ) + inState(           &
            & ?FETCH?(iDir,iField,posInTotal,QQ,nScalars,nPdfSize,neigh) )
        end do
      end do
      !x_plus = neigh()

      !mass fraction
      massFrac(:) = mass_dens(:)/sum(mass_dens)

      ! number density of all species
      num_dens(:) = mass_dens(:) * scheme%field(:)%fieldProp%species &
        &                                         %molWeightInv

      ! mole fraction 
      moleFrac = num_dens/sum(num_dens)
      ! Thermodynamic factor from C++ code
      call mus_calc_thermFactor( nFields, temp, press, moleFrac, &
        &                        thermodynamic_fac )

      inv_thermodyn_fac = invert_matrix( thermodynamic_fac )

      ! compute external forcing term
      ! d^m_k = w_m*c_m*( n0*min_a(m_a)*F_k + (1/cs2)*(n_k*z_k*F*E) )
      ! F_k is diffsive forcing term
      EF_elem = electricField((iElem-1)*3+1 : iElem*3)
      ! n_k * z_k 
      do iField = 1, nFields
        chargeTerm(iField) = num_dens(iField) * chargeNr(iField) &
          &                * scheme%mixture%faradayLB
      end do
      charge_dens = sum(chargeTerm)

      ! electric field term: n_k * z_k * Faraday * electricField
      do iField = 1, nFields
        electricTerm(:, iField) = EF_elem(:) * diffForce_fac      &
          & * ( chargeTerm(iField) - massFrac(iField)*charge_dens )
      end do
!write(dbgUnit(1),*) 'electricTerm ', electricTerm

      ! multiply diffForce with thermFacInv * min_a(m_a)
      ! diffusive driving force term:
      !  min_a(m_a) * c_m* w_m * sum_l(themFacInv_k_l * n * F_l)
      diffForce_WTF = 0.0_rk
      do iField = 1, nFields
        do iField_2 = 1, nFields
          diffForce_WTF(:, iField ) = diffForce_WTF(:, iField)                 &
            &                       + inv_thermodyn_fac(iField, iField_2)      &
            &                       * electricTerm(:, iField_2)
        end do
        diffForce_WTF(:, iField ) = minMolWeight * diffForce_WTF(:, iField)
      end do

      ! Update souce depends on nInputStates
      ! if nInputStates = 1, it is field source else it is global source
      do iField = 1, nInputStates
        depField = varSys%method%val(fun%srcTerm_varPos)%input_varPos(iField)

        do iDir = 1, QQ
          ! Force on each species
          ! d^m_k = weight_m*cxDir_m ( min_a(m_a)*n * F_k)
          forceTerm = scheme%layout%fStencil%cxDirRK( 1, iDir ) &
            &         * diffForce_WTF(1, depField)                   &
            &       + scheme%layout%fStencil%cxDirRK( 2, iDir ) &
            &         * diffForce_WTF(2, depField)                   &
            &       + scheme%layout%fStencil%cxDirRK( 3, iDir ) &
            &         * diffForce_WTF(3, depField)

          outState(                                                         &
            & ?SAVE?(iDir,depField,posInTotal,QQ,nScalars,nPdfSize,neigh) ) &
            & = outState(                                                   &
            & ?SAVE?(iDir,depField,posInTotal,QQ,nScalars,nPdfSize,neigh) ) &
            !& + force(iDir)
            & + scheme%layout%weight( iDir ) * forceTerm
        end do ! iDir
      end do !iField

    end do !iElem 

  end subroutine applySrc_electricFieldMSLiquid_diff_WTDF
  ! ************************************************************************ !
 
  ! ************************************************************************ !
   !> Update state with source variable "gravity_field"
   !! Simuilar to derive routine but it updates the state whereas derive 
   !! is used for tracking
?? copy :: applySource_header(applySrc_gravityFieldMSLiquid_WTDF)
    ! -------------------------------------------------------------------- !
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    real(kind=rk) :: gravityField(fun%elemLvl(iLevel)%nElems*3)
    integer :: iElem, nElems, iDir, posInTotal
    integer :: iField, nFields, depField, nScalars, QQ, nInputStates
    integer :: iField_2
    !mass density of nSpecies
    real(kind=rk) :: mass_dens( varSys%nStateVars )
    real(kind=rk) :: num_dens( varSys%nStateVars )
    real(kind=rk), dimension(varSys%nStateVars) :: massFrac, moleFrac
    real(kind=rk) :: gasConst, temp0, minMolWeight, forceTerm, mixDiffForce(3)
    real(kind=rk), dimension(3, varSys%nStateVars ) :: mixForce,       &
      & diffForce, gravityTerm, spcForce, diffForce_WTF
    real(kind=rk), dimension(varSys%nStateVars, varSys%nStateVars) :: &
      & resi_coeff, thermodynamic_fac, &
      & inv_thermodyn_fac, diff_coeff
    real(kind=rk) :: press, temp
    real(kind=rk) :: phy_moleDens_fac
    integer :: stateVarMap(varSys%nStateVars)
    ! -------------------------------------------------------------------- !
    !write(*,*) 'source variable: ', trim(varSys%varname%val(fun%srcTerm_varPos))
    ! convert c pointer to solver type fortran pointer
    call c_f_pointer( varSys%method%val( fun%srcTerm_varPos )%method_data, &
      &               fPtr ) 
    scheme => fPtr%solverData%scheme

    ! Number of elements to apply source terms
    nElems = fun%elemLvl(iLevel)%nElems

    ! Get gravitational force which is refered in config file either its
    ! spacetime variable or operation variable
    call varSys%method%val(fun%data_varPos)%get_valOfIndex( &
      & varSys  = varSys,                                   &
      & time    = time,                                     &
      & iLevel  = iLevel,                                   &
      & idx     = fun%elemLvl(iLevel)%idx(1:nElems),        &
      & nVals   = nElems,                                   & 
      & res     = gravityField                              )

    ! convert physical to lattice
    gravityField = gravityField * fPtr%solverData%physics%dtLvl(iLevel)**2 &
      &           / fPtr%solverData%physics%dxLvl(iLevel)

    ! number of pdf states this source depends on
    ! last input is spacetime function so it is neglected
    nInputStates = varSys%method%val(fun%srcTerm_varPos)%nInputs - 1
    ! constant parameter
    nFields = scheme%nFields
    QQ = scheme%layout%fStencil%QQ
    nScalars = varSys%nScalars
    stateVarMap = scheme%stateVarMap%varPos%val(:)
    do iField = 1, nFields
      ! diffusivity coefficients
      diff_coeff(iField, :) =                                     &
        & scheme%field(iField)%fieldProp%species%diff_coeff(:)
    end do    
    !KM @todo check moleDens for multilevel
    phy_moleDens_fac = fPtr%solverData%physics%moleDens0
    ! minimum molecular weight
    minMolWeight = minval(scheme%field(:)%fieldProp%species%molWeight)
    ! gas constant
    gasConst = scheme%mixture%gasConst_R_LB
    ! temperature
    temp0 = scheme%mixture%temp0LB
    ! temperature
    temp = scheme%mixture%temp0
    ! atmospheric pressure
    press = scheme%mixture%atm_press

    ! update source for each element
    do iElem = 1, nElems
      ! to access level wise state array
      posInTotal = fun%elemLvl(iLevel)%posInTotal(iElem)

      mass_dens = 0.0_rk
      do iField = 1, nFields
        ! compute field density and first moments
        do iDir = 1, QQ 
          !field density
          mass_dens( iField ) = mass_dens( iField ) + inState(           &
            & ?FETCH?(iDir,iField,posInTotal,QQ,nScalars,nPdfSize,neigh) )
        end do
      end do

      !mass fraction
      massFrac(:) = mass_dens(:)/sum(mass_dens)

      ! number density of all species
      num_dens(:) = mass_dens(:) * scheme%field(:)%fieldProp%species &
        &                                         %molWeightInv

      ! mole fraction 
      moleFrac = num_dens/sum(num_dens)

      ! MS-Diff coeff matrix from C++ code
      call mus_calc_MS_DiffMatrix( nFields, temp, press,               &
        &                        num_dens*phy_moleDens_fac, diff_coeff )
      ! Convert to lattice unit
      resi_coeff = fPtr%solverData%physics%fac(iLevel)%diffusivity/diff_coeff

      ! Thermodynamic factor from C++ code
      call mus_calc_thermFactor( nFields, temp, press, moleFrac, &
        &                        thermodynamic_fac )

      inv_thermodyn_fac = invert_matrix( thermodynamic_fac )

      ! compute external forcing term
      ! d^m_k = w_m*c_m*( n0*min_a(m_a)*F_k + (1/cs2)*(n_k*z_k*F*E) )
      ! F_k is diffsive forcing term

      ! electric field term: rho_k * gravityField
      do iField = 1, nFields
        gravityTerm(:, iField) = mass_dens(iField) *                 & 
          &                      gravityField((iElem-1)*3+1 : iElem*3)
      end do

      ! total mixture force in diffusive forcing term
      ! totMixForce = sum_l(rho_k * g)
      mixDiffForce(1) = sum(gravityTerm(1, :))
      mixDiffForce(2) = sum(gravityTerm(2, :))
      mixDiffForce(3) = sum(gravityTerm(3, :))

      ! diffusive forcing term in the Maxwell-Stefan equations
      ! F_k = 1/(nRT) rho_k (g) 
      !     - y_k/(nRT) sum_l(rho_l (g)
      ! diffForce = (electric term - fraction of totMixForce) *
      ! 1/(nRT) * n0 * min_a(m_a)
      ! Here n and n0 will cancel out so they are not included in computation
      do iField = 1, nFields
        diffForce(:, iField) = ( gravityTerm(:, iField)             &
          &                  - massFrac(iField) * mixDiffForce(:) ) &
          &                  / ( gasConst * temp0 ) 
      end do

      ! multiply diffForce with thermFacInv * min_a(m_a)
      ! diffusive driving force term:
      !  min_a(m_a) * c_m* w_m * sum_l(themFacInv_k_l * n * F_l)
      diffForce_WTF = 0.0_rk
      do iField = 1, nFields
        do iField_2 = 1, nFields
          diffForce_WTF(:, iField ) = diffForce_WTF(:, iField)                 &
            &                       + inv_thermodyn_fac(iField, iField_2)      &
            &                       * diffForce(:, iField_2)
        end do
        diffForce_WTF(:, iField ) = minMolWeight * diffForce_WTF(:, iField)
      end do

      ! forcing term in the mixture for momentum equation
      ! mixForce = (1/cs2)*(rho_k*g + n_k*z_k*F*E)
      mixForce = cs2inv * gravityTerm

      ! Force on each species
      ! d^m_k = weight_m*cxDir_m ( min_a(m_a)*n * F_k + rho_k * g )
      spcForce = diffForce_WTF + mixForce

      do iField = 1, nInputStates
        depField = varSys%method%val(fun%srcTerm_varPos)%input_varPos(iField)
        do iDir = 1, QQ
          forceTerm =  dot_product(                               & 
            &          scheme%layout%fStencil%cxDirRK( :, iDir ), &
            &          spcForce(:, depField) )

          outState(                                                         &
            & ?SAVE?(iDir,depField,posInTotal,QQ,nScalars,nPdfSize,neigh) ) &
            & = outState(                                                   &
            & ?SAVE?(iDir,depField,posInTotal,QQ,nScalars,nPdfSize,neigh) ) &
            & + scheme%layout%weight( iDir ) * forceTerm
        end do
      end do
    end do !iElem 
  end subroutine applySrc_gravityFieldMSLiquid_WTDF
  ! ************************************************************************ !


  ! ************************************************************************ !
  !> This routine computes equilbrium from density and velocity
  !! This must comply with mus_variable_module%derive_FromMacro
?? copy :: deriveFromMacro_header(deriveEquilMSLiquid_FromMacro)
    ! -------------------------------------------------------------------- !
    real(kind=rk) :: fEq(layout%fStencil%QQ)
    integer :: QQ, iElem, iFld, nFields, offset
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    real(kind=rk) :: phi, resi_coeff(varSys%nStateVars), paramBInv, theta_eq
    !mass density of nSpecies
    real(kind=rk) :: mass_dens( varSys%nStateVars )
    !number density of nSpecies 
    real(kind=rk) :: num_dens( varSys%nStateVars )
    !mole fraction
    real(kind=rk) :: moleFraction( varSys%nStateVars )
    real(kind=rk) :: totNum_densInv
    real(kind=rk) :: vel( 3, varSys%nStateVars )
    real(kind=rk) :: weight0Inv
    ! -------------------------------------------------------------------- !
    call C_F_POINTER( varSys%method%val(iField)%method_Data, fPtr )
    scheme => fPtr%solverData%scheme

    QQ = layout%fStencil%QQ 
    nFields = scheme%nFields
    ! molecular weight ratio
    phi = scheme%field(iField)%fieldProp%species%molWeigRatio
    ! resistivity coefficients
    resi_coeff(:) =                                     &
      & scheme%field(iField)%fieldProp%species%resi_coeff(:)

    paramBInv = 1.0_rk / scheme%mixture%paramB

    theta_eq = scheme%mixture%theta_eq

    weight0Inv = 1.0_rk / layout%weight(layout%fStencil%restPosition)

    do iElem = 1, nElems
      offset = (iElem-1)*nFields
      ! get species density and velocity of iElem
      do ifld = 1, nFields
        mass_dens(ifld) = density( offset + ifld )
        vel(:, ifld) = velocity(:,offset+ifld)
      end do
      num_dens(:) = mass_dens(:) * scheme%field(:)%fieldProp%species &
        &                                         %molWeightInv
      ! number density
      totNum_densInv = 1.0_rk/sum(num_dens)
      ! molefraction
      moleFraction = num_dens * totNum_densInv

      ! compute equilibrium from macroscopic quantities
      fEq = equilFromMacro( iField       = iField,       &
        &                   mass_dens    = mass_dens,    &
        &                   moleFraction = moleFraction, &
        &                   velocity     = vel,          &
        &                   layout       = layout,       &
        &                   nFields      = nFields,      &
        &                   paramBInv    = paramBInv,    &
        &                   phi          = phi,          &
        &                   resi_coeff   = resi_coeff,   &
        &                   theta_eq     = theta_eq,     &
        &                   weight0Inv   = weight0Inv    )


      res( (iElem-1)*QQ+1: iElem*QQ ) = fEq
    end do
  end subroutine deriveEquilMSLiquid_FromMacro
  ! ************************************************************************ !


  ! ************************************************************************ !
  !> This routine computes velocity from state array
  !! This must comply with interface in [[mus_variable_module:derive_FromState]]
?? copy :: deriveFromState_header( deriveVelMSLiquid_FromState )
    ! -------------------------------------------------------------------- !
    integer :: iElem, iComp, iFld
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    integer :: varPos, QQ, nFields
    real(kind=rk) :: tmpPDF(layout%fStencil%QQ)
    real(kind=rk) :: vel( 3 )
    !mass density of nSpecies
    real(kind=rk) :: mass_dens( varSys%nStateVars )
    !number density of nSpecies 
    real(kind=rk) :: num_dens( varSys%nStateVars )
    !mole fraction
    real(kind=rk) :: moleFraction( varSys%nStateVars )
    real(kind=rk) :: totNum_densInv
    !first moments of nSpecies
    real(kind=rk) :: first_moments( 3, varSys%nStateVars )
    !momentum from linear system of equation
    real(kind=rk) :: momentum( 3, varSys%nStateVars )
    !parameters from solver specific conf
    !field specific info from field table
    real(kind=rk), dimension(varSys%nStateVars) :: phi
    real(kind=rk) :: resi_coeff( varSys%nStateVars, varSys%nStateVars )
    !mixture info
    real(kind=rk) :: omega_diff, omega_fac, paramBInv
    integer :: stateVarMap(varSys%nStateVars)
    ! -------------------------------------------------------------------- !
    call C_F_POINTER( varSys%method%val(iField)%method_Data, fPtr )
    scheme => fPtr%solverData%scheme

    nFields = scheme%nFields
    stateVarMap = scheme%stateVarMap%varPos%val(:)

    do iFld = 1, nFields
      ! species properties
      ! molecular weight ratio
      phi(iFld) = scheme%field(iFld)%fieldProp%species%molWeigRatio
      ! resistivity coefficients
      resi_coeff(iFld, :) =                                &
        & scheme%field(iFld)%fieldProp%species%resi_coeff(:)
    end do 

    QQ = layout%fStencil%QQ
    omega_diff = scheme%mixture%omega_diff
    omega_fac = omega_diff * 0.5_rk

    paramBInv = 1.0_rk / scheme%mixture%paramB

    do iElem = 1, nElems
      do iFld = 1, nFields

        do iComp = 1, QQ
          varPos = varSys%method%val(stateVarMap(iFld))%state_varPos(iComp)

          tmpPDF(iComp) = state(                       &
            ! position of this state variable in the state array
            & ?IDX?(varPos, iElem, varSys%nScalars, nElems) )
        end do

        ! mass density of species
        mass_dens(iFld ) = sum( tmpPDF )

        ! number density of species
        num_dens(iFld) = mass_dens(iFld)                      &
          & * scheme%field(iFld)%fieldProp%species%molWeightInv

        ! velocity, first moments
        do iComp = 1, 3 
          first_moments(iComp, iFld) = sum( tmpPDF *        &
            & scheme%layout%fStencil%cxDirRK(iComp, :) )
        end do  

      end do !iFld

      ! total number density Inv
      totNum_densInv = 1.0_rk/sum(num_dens(:))

      ! molefraction
      moleFraction(:) = num_dens(:)*totNum_densInv

      ! momentum of all species
      momentum = momentumFromMacroLSE( moleFraction  = moleFraction,   &
        &                              first_moments = first_moments,  &
        &                              nFields       = nFields,        &
        &                              phi           = phi,            &
        &                              resi_coeff    = resi_coeff,     &
        &                              omega_fac     = omega_fac,      &
        &                              paramBInv     = paramBInv       )

      ! find required species velocity   
      vel = momentum(:, iField) / mass_dens(iField)

      ! copy the results to the res 
      res( (iElem-1)*3 + 1 : iElem*3) = vel
    end do

  end subroutine deriveVelMSLiquid_FromState
  ! ************************************************************************ !


  ! ************************************************************************ !
  !> This routine computes momentum from state array
  !! This must comply with interface in [[mus_variable_module:derive_FromState]]
?? copy :: deriveFromState_header( deriveMomMSLiquid_FromState )
    ! -------------------------------------------------------------------- !
    integer :: iElem, iComp, iFld
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    integer :: varPos, QQ, nFields
    real(kind=rk) :: tmpPDF(layout%fStencil%QQ)
    !mass density of nSpecies
    real(kind=rk) :: mass_dens( varSys%nStateVars )
    !number density of nSpecies 
    real(kind=rk) :: num_dens( varSys%nStateVars )
    !mole fraction
    real(kind=rk) :: moleFraction( varSys%nStateVars )
    real(kind=rk) :: totNum_densInv
    !first moments of nSpecies
    real(kind=rk) :: first_moments( 3, varSys%nStateVars )
    !momentum from linear system of equation
    real(kind=rk) :: momentum( 3, varSys%nStateVars )
    !parameters from solver specific conf
    !field specific info from field table
    real(kind=rk), dimension(varSys%nStateVars) :: phi
    real(kind=rk) :: resi_coeff( varSys%nStateVars, varSys%nStateVars )
    !mixture info
    real(kind=rk) :: omega_diff, omega_fac, paramBInv
    integer :: stateVarMap(varSys%nStateVars)
    ! -------------------------------------------------------------------- !
    call C_F_POINTER( varSys%method%val(iField)%method_Data, fPtr )
    scheme => fPtr%solverData%scheme

    nFields = scheme%nFields
    stateVarMap = scheme%stateVarMap%varPos%val(:)

    do iFld = 1, nFields
      ! species properties
      ! molecular weight ratio
      phi(iFld) = scheme%field(iFld)%fieldProp%species%molWeigRatio
      ! resistivity coefficients
      resi_coeff(iFld, :) =                                &
        & scheme%field(iFld)%fieldProp%species%resi_coeff(:)
    end do 

    QQ = layout%fStencil%QQ
    omega_diff = scheme%mixture%omega_diff
    omega_fac = omega_diff * 0.5_rk

    paramBInv = 1.0_rk / scheme%mixture%paramB

    do iElem = 1, nElems
      do iFld = 1, nFields

        do iComp = 1, QQ
          varPos = varSys%method%val(stateVarMap(iFld))%state_varPos(iComp)

          tmpPDF(iComp) = state(                       &
            ! position of this state variable in the state array
            & ?IDX?(varPos, iElem, varSys%nScalars, nElems) )
        end do

        ! mass density of species
        mass_dens(iFld ) = sum( tmpPDF )

        ! number density of species
        num_dens(iFld) = mass_dens(iFld)                      &
          & * scheme%field(iFld)%fieldProp%species%molWeightInv

        ! velocity, first moments
        do iComp = 1, 3 
          first_moments(iComp, iFld) = sum( tmpPDF *   &
            & scheme%layout%fStencil%cxDirRK(iComp, :) )
        end do  

      end do !iFld

      ! total number density Inv
      totNum_densInv = 1.0_rk/sum(num_dens(:))

      ! molefraction
      moleFraction(:) = num_dens(:)*totNum_densInv

      ! momentum of all species
      momentum = momentumFromMacroLSE( moleFraction  = moleFraction,   &
        &                              first_moments = first_moments,  &
        &                              nFields       = nFields,        &
        &                              phi           = phi,            &
        &                              resi_coeff    = resi_coeff,     &
        &                              omega_fac     = omega_fac,      &
        &                              paramBInv     = paramBInv       )

      ! copy the results to the res 
      res( (iElem-1)*3 + 1 : iElem*3) = momentum(:, iField)
    end do

  end subroutine deriveMomMSLiquid_FromState
  ! ************************************************************************ !


  ! ************************************************************************ !
  !> This routine computes velocities of all species from state array
  !! This must comply with interface in [[mus_variable_module:derive_FromState]]
?? copy :: deriveFromState_header( deriveVelocitiesMSLiquid_FromState )
    ! -------------------------------------------------------------------- !
    integer :: iElem, iComp, iFld
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    integer :: varPos, QQ, nFields
    real(kind=rk) :: tmpPDF(layout%fStencil%QQ)
    real(kind=rk) :: vel( 3 )
    !mass density of nSpecies
    real(kind=rk) :: mass_dens( varSys%nStateVars )
    !number density of nSpecies 
    real(kind=rk) :: num_dens( varSys%nStateVars )
    !mole fraction
    real(kind=rk) :: moleFraction( varSys%nStateVars )
    real(kind=rk) :: totNum_densInv
    !first moments of nSpecies
    real(kind=rk) :: first_moments( 3, varSys%nStateVars )
    !momentum from linear system of equation
    real(kind=rk) :: momentum( 3, varSys%nStateVars )
    !parameters from solver specific conf
    !field specific info from field table
    real(kind=rk), dimension(varSys%nStateVars) :: phi
    real(kind=rk) :: resi_coeff( varSys%nStateVars, varSys%nStateVars )
    !mixture info
    real(kind=rk) :: omega_diff, omega_fac, paramBInv
    integer :: stateVarMap(varSys%nStateVars)
    ! -------------------------------------------------------------------- !
    call C_F_POINTER( varSys%method%val(iField)%method_Data, fPtr )
    scheme => fPtr%solverData%scheme

    nFields = scheme%nFields
    stateVarMap = scheme%stateVarMap%varPos%val(:)

    do iFld = 1, nFields
      ! species properties
      ! molecular weight ratio
      phi(iFld) = scheme%field(iFld)%fieldProp%species%molWeigRatio
      ! resistivity coefficients
      resi_coeff(iFld, :) =                                &
        & scheme%field(iFld)%fieldProp%species%resi_coeff(:)
    end do

    QQ = layout%fStencil%QQ
    omega_diff = scheme%mixture%omega_diff
    omega_fac = omega_diff * 0.5_rk

    paramBInv = 1.0_rk / scheme%mixture%paramB
    res = 0.0_rk

    do iElem = 1, nElems
      do iFld = 1, nFields

        do iComp = 1, QQ
          varPos = varSys%method%val(stateVarMap(iFld))%state_varPos(iComp)

          tmpPDF(iComp) = state(                       &
            ! position of this state variable in the state array
            & ?IDX?(varPos, iElem, varSys%nScalars, nElems) )
        end do

        ! mass density of species
        mass_dens(iFld ) = sum( tmpPDF )

        ! number density of species
        num_dens(iFld) = mass_dens(iFld)                      &
          & * scheme%field(iFld)%fieldProp%species%molWeightInv

        ! velocity, first moments
        do iComp = 1, 3
          first_moments(iComp, iFld) = sum( tmpPDF *   &
            & scheme%layout%fStencil%cxDirRK(iComp, :) )
        end do

      end do !iFld

      ! total number density Inv
      totNum_densInv = 1.0_rk/sum(num_dens(:))

      ! molefraction
      moleFraction(:) = num_dens(:)*totNum_densInv

      ! momentum of all species
      momentum = momentumFromMacroLSE( moleFraction  = moleFraction,   &
        &                              first_moments = first_moments,  &
        &                              nFields       = nFields,        &
        &                              phi           = phi,            &
        &                              resi_coeff    = resi_coeff,     &
        &                              omega_fac     = omega_fac,      &
        &                              paramBInv     = paramBInv       )

     ! copy the results to the res
      do iFld = 1, nFields
        ! find required species velocity
        vel = momentum(:, iFld)/mass_dens(iFld)
        res( (iElem-1)*nFields*3 + (iFld-1)*3 + 1) = vel(1)
        res( (iElem-1)*nFields*3 + (iFld-1)*3 + 2) = vel(2)
        res( (iElem-1)*nFields*3 + (iFld-1)*3 + 3) = vel(3)
      end do
    end do

  end subroutine deriveVelocitiesMSLiquid_FromState
  ! ************************************************************************ !


  ! ************************************************************************ !
  !> This routine computes momentum of all species from state array
  !! This must comply with interface in [[mus_variable_module:derive_FromState]]
?? copy :: deriveFromState_header( deriveMomentaMSLiquid_FromState )
    ! -------------------------------------------------------------------- !
    integer :: iElem, iComp, iFld
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    integer :: varPos, QQ, nFields
    real(kind=rk) :: tmpPDF(layout%fStencil%QQ)
    !mass density of nSpecies
    real(kind=rk) :: mass_dens( varSys%nStateVars )
    !number density of nSpecies 
    real(kind=rk) :: num_dens( varSys%nStateVars )
    !mole fraction
    real(kind=rk) :: moleFraction( varSys%nStateVars )
    real(kind=rk) :: totNum_densInv
    !first moments of nSpecies
    real(kind=rk) :: first_moments( 3, varSys%nStateVars )
    !momentum from linear system of equation
    real(kind=rk) :: momentum( 3, varSys%nStateVars )
    !parameters from solver specific conf
    !field specific info from field table
    real(kind=rk), dimension(varSys%nStateVars) :: phi
    real(kind=rk) :: resi_coeff( varSys%nStateVars, varSys%nStateVars )
    !mixture info
    real(kind=rk) :: omega_diff, omega_fac, paramBInv
    integer :: stateVarMap(varSys%nStateVars)
    ! -------------------------------------------------------------------- !
    call C_F_POINTER( varSys%method%val(iField)%method_Data, fPtr )
    scheme => fPtr%solverData%scheme

    nFields = scheme%nFields
    stateVarMap = scheme%stateVarMap%varPos%val(:)

    do iFld = 1, nFields
      ! species properties
      ! molecular weight ratio
      phi(iFld) = scheme%field(iFld)%fieldProp%species%molWeigRatio
      ! resistivity coefficients
      resi_coeff(iFld, :) =                                &
        & scheme%field(iFld)%fieldProp%species%resi_coeff(:)
    end do 

    QQ = layout%fStencil%QQ
    omega_diff = scheme%mixture%omega_diff
    omega_fac = omega_diff * 0.5_rk

    paramBInv = 1.0_rk / scheme%mixture%paramB

    do iElem = 1, nElems
      do iFld = 1, nFields

        do iComp = 1, QQ
          varPos = varSys%method%val(iFld)%state_varPos(iComp)

          tmpPDF(iComp) = state(                       &
            ! position of this state variable in the state array
            & ?IDX?( varPos, iElem, varSys%nScalars, nElems ) )
        end do

        ! mass density of species
        mass_dens(iFld ) = sum( tmpPDF )

        ! number density of species
        num_dens(iFld) = mass_dens(iFld)                      &
          & * scheme%field(iFld)%fieldProp%species%molWeightInv

        ! velocity, first moments
        do iComp = 1, 3 
          first_moments(iComp, iFld) = sum( tmpPDF *   &
            & scheme%layout%fStencil%cxDirRK(iComp, :) )
        end do  

      end do !iFld

      ! total number density Inv
      totNum_densInv = 1.0_rk/sum(num_dens(:))

      ! molefraction
      moleFraction(:) = num_dens(:)*totNum_densInv

      ! momentum of all species
      momentum = momentumFromMacroLSE( moleFraction  = moleFraction,   &
        &                              first_moments = first_moments,  &
        &                              nFields       = nFields,        &
        &                              phi           = phi,            &
        &                              resi_coeff    = resi_coeff,     &
        &                              omega_fac     = omega_fac,      &
        &                              paramBInv     = paramBInv       )

     ! copy the results to the res 
      do iFld = 1, nFields
        res( (iElem-1)*nFields*3 + (iFld-1)*3 + 1) = momentum(1, iFld)
        res( (iElem-1)*nFields*3 + (iFld-1)*3 + 2) = momentum(2, iFld)
        res( (iElem-1)*nFields*3 + (iFld-1)*3 + 3) = momentum(3, iFld)
      end do
 
    end do

  end subroutine deriveMomentaMSLiquid_FromState
  ! ************************************************************************ !


  ! ************************************************************************ !
  !> This routine computes equilibrium from state array
  !! This must comply with interface in [[mus_variable_module:derive_FromState]]
?? copy :: deriveFromState_header( deriveEqMSLiquid_FromState )
    ! -------------------------------------------------------------------- !
    integer :: iElem, iComp, QQ, iFld, nFields
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    integer :: varPos
    real(kind=rk) :: tmpPDF(layout%fStencil%QQ)
    !mass density of nSpecies
    real(kind=rk) :: mass_dens( varSys%nStateVars )
    !number density of nSpecies 
    real(kind=rk) :: num_dens( varSys%nStateVars )
    !mole fraction
    real(kind=rk) :: moleFraction( varSys%nStateVars )
    real(kind=rk) :: totNum_densInv
    !first moments of nSpecies
    real(kind=rk) :: first_moments( 3, varSys%nStateVars )
    !momentum from linear system of equation
    real(kind=rk) :: momentum( 3, varSys%nStateVars )
    real(kind=rk) :: vel( 3, varSys%nStateVars )
    !parameters from solver specific conf
    !field specific info from field table
    real(kind=rk), dimension(varSys%nStateVars) :: phi
    real(kind=rk) :: resi_coeff( varSys%nStateVars, varSys%nStateVars )
    !mixture info
    real(kind=rk) :: omega_diff, paramBInv, theta_eq, weight0Inv, omega_fac
    real(kind=rk) :: fEq(layout%fStencil%QQ)
    integer :: stateVarMap(varSys%nStateVars)
    ! -------------------------------------------------------------------- !
    call C_F_POINTER( varSys%method%val(iField)%method_Data, fPtr )
    scheme => fPtr%solverData%scheme

    nFields = scheme%nFields
    QQ = layout%fStencil%QQ
    stateVarMap = scheme%stateVarMap%varPos%val(:)

    do iFld = 1, nFields
      ! species properties
      ! molecular weight ratio
      phi(iFld) = scheme%field(iFld)%fieldProp%species%molWeigRatio
      ! resistivity coefficients
      resi_coeff(iFld, :) =                                &
        & scheme%field(iFld)%fieldProp%species%resi_coeff(:)
    end do 

    omega_diff = scheme%mixture%omega_diff
    omega_fac = omega_diff * 0.5_rk

    paramBInv = 1.0_rk / scheme%mixture%paramB
    theta_eq = scheme%mixture%theta_eq

    weight0Inv = 1.0_rk / layout%weight( layout%fStencil%restPosition )
    do iElem = 1, nElems
      do iFld = 1, nFields

        do iComp = 1, QQ 
          varPos = varSys%method%val(stateVarMap(iFld))%state_varPos(iComp)

          tmpPDF(iComp) = state(                       &
            ! position of this state variable in the state array
            & ?IDX?(varPos, iElem, varSys%nScalars, nElems ) )
        end do

        ! mass density of species
        mass_dens(iFld ) = sum( tmpPDF )

        ! number density of species
        num_dens(iFld) = mass_dens(iFld)                      &
          & * scheme%field(iFld)%fieldProp%species%molWeightInv

        ! velocity, first moments
        do iComp = 1, 3 
          first_moments(iComp, iFld) = sum( tmpPDF *   &
            & scheme%layout%fStencil%cxDirRK(iComp, :) )
        end do  

      end do !iFld

      ! total number density Inv
      totNum_densInv = 1.0_rk/sum(num_dens(:))

      ! molefraction
      moleFraction(:) = num_dens(:)*totNum_densInv

      ! momentum of all species
      momentum = momentumFromMacroLSE( moleFraction  = moleFraction,   &
        &                              first_moments = first_moments,  &
        &                              nFields       = nFields,        &
        &                              phi           = phi,            &
        &                              resi_coeff    = resi_coeff,     &
        &                              omega_fac     = omega_fac,      &
        &                              paramBInv     = paramBInv       )

      !velocity of all species
      do iFld = 1, nFields
        vel( :, iFld) = momentum( :, iFld) / mass_dens(iFld)
      end do

      ! compute equilibrium from macroscopic quantities
      fEq = equilFromMacro( iField       = iField,                &
        &                   mass_dens    = mass_dens,             &
        &                   moleFraction = moleFraction,          &
        &                   velocity     = vel,                   &
        &                   layout       = scheme%layout,         &
        &                   nFields      = nFields,               &
        &                   paramBInv    = paramBInv,             &
        &                   phi          = phi(iField),           &
        &                   resi_coeff   = resi_coeff(iField, :), &
        &                   theta_eq     = theta_eq,              &
        &                   weight0Inv   = weight0Inv             )

      ! copy the results to the res 
      res( (iElem-1)*QQ + 1 : iElem*QQ) = fEq 
 
    end do !iElem
  end subroutine deriveEqMSLiquid_FromState
  ! ************************************************************************ !

! **************************************************************************** !
  !> This routine computes auxField 'density and velocity' of given field
  !! from state array.
  !! velocity of original PDF is computed in this routine by solving LSE.
  !! This must comply with interface in 
  !! [[mus_derVarPos_type_module:derive_AuxFromState]]
?? copy :: deriveAuxFromState_header( deriveAuxMSLiquid_fromState )
    ! ------------------------------------------------------------------------ !
    integer :: iElem, iDir, pdfPos, iFld, nFields, elemOff, dens_pos, vel_pos(3)
    !mass density of nSpecies
    real(kind=rk) :: mass_dens( varSys%nStateVars ), inv_rho
    !number density of nSpecies 
    real(kind=rk) :: num_dens( varSys%nStateVars )
    !mole fraction
    real(kind=rk) :: moleFraction( varSys%nStateVars )
    real(kind=rk) :: totNum_densInv
    !first moments of nSpecies
    real(kind=rk) :: first_moments( 3, varSys%nStateVars )
    !momentum from linear system of equation
    real(kind=rk) :: momentum( 3, varSys%nStateVars )
    !parameters from solver specific conf
    !field specific info from field table
    real(kind=rk), dimension(varSys%nStateVars) :: phi
    real(kind=rk) :: resi_coeff( varSys%nStateVars, varSys%nStateVars )
    !mixture info
    real(kind=rk) :: omega_fac, paramBInv
    real(kind=rk) :: pdf( stencil%QQ )
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    ! ------------------------------------------------------------------------ !
    call C_F_POINTER( varSys%method%val(iField)%method_Data, fPtr )
    scheme => fPtr%solverData%scheme

    nFields = scheme%nFields

    do iFld = 1, nFields
      ! species properties
      ! molecular weight ratio
      phi(iFld) = scheme%field(iFld)%fieldProp%species%molWeigRatio
      ! resistivity coefficients
      resi_coeff(iFld, :) =                                &
        & scheme%field(iFld)%fieldProp%species%resi_coeff(:)
    end do

    omega_fac = scheme%mixture%omega_diff * 0.5_rk

    paramBInv = 1.0_rk / scheme%mixture%paramB

    !NEC$ ivdep
    do iElem = 1, nElems
      !NEC$ shortloop
      do iFld = 1, nFields
        do iDir = 1, stencil%QQ
          pdfPos = varSys%method%val(scheme%derVarPos(iFld)%pdf) &
            &                   %state_varPos(iDir)
          pdf(iDir) = state( ?IDX?(pdfPos, iElem, varSys%nScalars, nElems) )
        end do

        ! mass density of species
        mass_dens(iFld ) = sum(pdf)

        ! number density of species
        num_dens(iFld) = mass_dens(iFld)                      &
          & * scheme%field(iFld)%fieldProp%species%molWeightInv

        ! momentum of all species
        first_moments(1, iFld) = sum( pdf * stencil%cxDirRK(1, :) )
        first_moments(2, iFld) = sum( pdf * stencil%cxDirRK(2, :) )
        first_moments(3, iFld) = sum( pdf * stencil%cxDirRK(3, :) )
      end do !iFld

      ! total number density Inv
      totNum_densInv = 1.0_rk/sum(num_dens(:))

      ! molefraction
      moleFraction(:) = num_dens(:)*totNum_densInv

      ! momentum of all species
      momentum = momentumFromMacroLSE( moleFraction  = moleFraction,   &
        &                              first_moments = first_moments,  &
        &                              nFields       = nFields,        &
        &                              phi           = phi,            &
        &                              resi_coeff    = resi_coeff,     &
        &                              omega_fac     = omega_fac,      &
        &                              paramBInv     = paramBInv       )

      ! position of density and velocity of current field in auxField array
      dens_pos = varSys%method%val(derVarPos%density)%auxField_varPos(1)
      vel_pos = varSys%method%val(derVarPos%velocity)%auxField_varPos(:)

      ! element offset for auxField
      elemOff = (iElem-1)*varSys%nAuxScalars
      ! store field density
      auxField(elemOff + dens_pos) = mass_dens(iField)
      ! store field velocity
      inv_rho = 1.0_rk/mass_dens(iField)
      auxField(elemOff + vel_pos(1)) = momentum(1, iField) * inv_rho
      auxField(elemOff + vel_pos(2)) = momentum(2, iField) * inv_rho
      auxField(elemOff + vel_pos(3)) = momentum(3, iField) * inv_rho

    end do !iElem

  end subroutine deriveAuxMSLiquid_fromState
! **************************************************************************** !

! **************************************************************************** !
  !> This routine computes auxField 'density and velocity' of given field
  !! from state array with thermodynamic factot.
  !! velocity of original PDF is computed in this routine by solving LSE.
  !! This must comply with interface in 
  !! [[mus_derVarPos_type_module:derive_AuxFromState]]
?? copy :: deriveAuxFromState_header( deriveAuxMSLiquid_fromState_WTDF )
    ! ------------------------------------------------------------------------ !
    integer :: iElem, iDir, pdfPos, iFld, nFields, elemOff, dens_pos, vel_pos(3)
    !mass density of nSpecies
    real(kind=rk) :: mass_dens( varSys%nStateVars ), inv_rho
    !number density of nSpecies 
    real(kind=rk) :: num_dens( varSys%nStateVars )
    !mole fraction
    real(kind=rk) :: moleFraction( varSys%nStateVars )
    real(kind=rk) :: totNum_densInv, press, temp, phy_moleDens_fac
    !first moments of nSpecies
    real(kind=rk) :: first_moments( 3, varSys%nStateVars )
    !momentum from linear system of equation
    real(kind=rk) :: momentum( 3, varSys%nStateVars )
    !parameters from solver specific conf
    !field specific info from field table
    real(kind=rk), dimension(varSys%nStateVars) :: phi
    real(kind=rk), dimension(varSys%nStateVars, varSys%nStateVars) :: &
      & resi_coeff, thermodynamic_fac, &
      & inv_thermodyn_fac, diff_coeff
    !mixture info
    real(kind=rk) :: omega_fac, paramBInv
    real(kind=rk) :: pdf( stencil%QQ )
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    ! ------------------------------------------------------------------------ !
    call C_F_POINTER( varSys%method%val(iField)%method_Data, fPtr )
    scheme => fPtr%solverData%scheme

    nFields = scheme%nFields

    do iFld = 1, nFields
      ! species properties
      ! molecular weight ratio
      phi(iFld) = scheme%field(iFld)%fieldProp%species%molWeigRatio
      ! resistivity coefficients
      resi_coeff(iFld, :) =                                &
        & scheme%field(iFld)%fieldProp%species%resi_coeff(:)
    end do

    phy_moleDens_fac = fPtr%solverData%physics%moleDens0
    omega_fac = scheme%mixture%omega_diff * 0.5_rk

    paramBInv = 1.0_rk / scheme%mixture%paramB

    ! temperature
    temp = scheme%mixture%temp0
    ! atmospheric pressure
    press = scheme%mixture%atm_press

    !NEC$ ivdep
    do iElem = 1, nElems
      !NEC$ shortloop
      do iFld = 1, nFields
        do iDir = 1, stencil%QQ
          pdfPos = varSys%method%val(scheme%derVarPos(iFld)%pdf) &
            &                   %state_varPos(iDir)
          pdf(iDir) = state( ?IDX?(pdfPos, iElem, varSys%nScalars, nSize) )
        end do

        ! mass density of species
        mass_dens(iFld ) = sum(pdf)

        ! number density of species
        num_dens(iFld) = mass_dens(iFld)                      &
          & * scheme%field(iFld)%fieldProp%species%molWeightInv

        ! momentum of all species
        first_moments(1, iFld) = sum( pdf * stencil%cxDirRK(1, :) )
        first_moments(2, iFld) = sum( pdf * stencil%cxDirRK(2, :) )
        first_moments(3, iFld) = sum( pdf * stencil%cxDirRK(3, :) )
      end do !iFld

      ! total number density Inv
      totNum_densInv = 1.0_rk/sum(num_dens(:))

      ! molefraction
      moleFraction(:) = num_dens(:)*totNum_densInv

      ! MS-Diff coeff matrix from C++ code
      call mus_calc_MS_DiffMatrix( nFields, temp, press,                 &
        &                          num_dens*phy_moleDens_fac, diff_coeff )

      ! Convert to lattice unit
      resi_coeff = fPtr%solverData%physics%fac(iLevel)%diffusivity/diff_coeff

      ! Thermodynamic factor from C++ code
      call mus_calc_thermFactor( nFields, temp, press, moleFraction, &
        &                        thermodynamic_fac )

      inv_thermodyn_fac = invert_matrix( thermodynamic_fac )

      ! momentum of all species
      momentum = momentumFromMacroLSE_WTDF(                                 &
        &                            moleFraction      = moleFraction,      &
        &                            first_moments     = first_moments,     &
        &                            nFields           = nFields,           &
        &                            inv_thermodyn_fac = inv_thermodyn_fac, &
        &                            phi               = phi,               &
        &                            resi_coeff        = resi_coeff,        &
        &                            omega_fac         = omega_fac,         &
        &                            paramBInv         = paramBInv          )

      ! position of density and velocity of current field in auxField array
      dens_pos = varSys%method%val(derVarPos%density)%auxField_varPos(1)
      vel_pos = varSys%method%val(derVarPos%velocity)%auxField_varPos(:)

      ! element offset for auxField
      elemOff = (iElem-1)*varSys%nAuxScalars
      ! store field density
      auxField(elemOff + dens_pos) = mass_dens(iField)
      ! store field velocity
      inv_rho = 1.0_rk/mass_dens(iField)
      auxField(elemOff + vel_pos(1)) = momentum(1, iField) * inv_rho
      auxField(elemOff + vel_pos(2)) = momentum(2, iField) * inv_rho
      auxField(elemOff + vel_pos(3)) = momentum(3, iField) * inv_rho

    end do !iElem

  end subroutine deriveAuxMSLiquid_fromState_WTDF
! **************************************************************************** !

  ! ************************************************************************** !
  !> This routine computes equilbrium from auxField
?? copy :: deriveEquilFromAux_header(deriveEquilMSLiquid_fromAux)
    ! ------------------------------------------------------------------------ !
    integer :: iElem, iDir, QQ, iFld, nFields, elemOff, dens_pos, vel_pos(3)
    real(kind=rk) :: fEq_loc(layout%fStencil%QQ)
    real(kind=rk) :: phi, resi_coeff(varSys%nStateVars), paramBInv, theta_eq
    !mass density of nSpecies
    real(kind=rk) :: mass_dens( varSys%nStateVars )
    real(kind=rk) :: num_dens( varSys%nStateVars )
    !mass fraction
    real(kind=rk) :: moleFraction( varSys%nStateVars )
    real(kind=rk) :: totNum_densInv
    real(kind=rk) :: vel( 3, varSys%nStateVars )
    real(kind=rk) :: weight0Inv
    type(mus_scheme_type), pointer :: scheme
    type(mus_varSys_data_type), pointer :: fPtr
    ! ------------------------------------------------------------------------ !
    call C_F_POINTER( varSys%method%val(iField)%method_Data, fPtr )
    scheme => fPtr%solverData%scheme

    nFields = scheme%nFields
    QQ = layout%fStencil%QQ

    ! molecular weight ratio
    phi = scheme%field(iField)%fieldProp%species%molWeigRatio
    ! resistivity coefficients
    resi_coeff(:) =                                     &
      & scheme%field(iField)%fieldProp%species%resi_coeff(:)

    paramBInv = 1.0_rk / scheme%mixture%paramB

    theta_eq = scheme%mixture%theta_eq

    weight0Inv = 1.0_rk / layout%weight(layout%fStencil%restPosition)
    !NEC$ ivdep
    do iElem = 1, nElems
      ! element offset
      elemoff = (iElem-1)*varSys%nAuxScalars

      ! get all field density and velocity to compute equilibrium velocity
      !NEC$ shortloop
      do iFld = 1, nFields
        ! field density
        dens_pos = varSys%method%val(scheme%derVarPos(iFld)%density) &
          &                     %auxField_varPos(1)
        mass_dens(iFld) = auxField(elemOff + dens_pos)

        ! field velocity
        vel_pos = varSys%method%val(scheme%derVarPos(iFld)%velocity) &
          &                    %auxField_varPos(:)
        vel(1, iFld) = auxField(elemOff+vel_pos(1))
        vel(2, iFld) = auxField(elemOff+vel_pos(2))
        vel(3, iFld) = auxField(elemOff+vel_pos(3))
      end do !iFld

      ! number density
      num_dens(:) = mass_dens(:) * scheme%field(:)%fieldProp%species &
        &                                         %molWeightInv
      ! number density
      totNum_densInv = 1.0_rk/sum(num_dens)
      ! molefraction
      moleFraction = num_dens * totNum_densInv

      ! compute equilibrium from macroscopic quantities
      fEq_loc = equilFromMacro( iField       = iField,       &
        &                       mass_dens    = mass_dens,    &
        &                       moleFraction = moleFraction, &
        &                       velocity     = vel,          &
        &                       layout       = layout,       &
        &                       nFields      = nFields,      &
        &                       paramBInv    = paramBInv,    &
        &                       phi          = phi,          &
        &                       resi_coeff   = resi_coeff,   &
        &                       theta_eq     = theta_eq,     &
        &                       weight0Inv   = weight0Inv    )

      fEq( (iElem-1)*QQ+1: iElem*QQ ) = fEq_loc
    end do
  end subroutine deriveEquilMSLiquid_fromAux
  ! ************************************************************************** !


  ! ************************************************************************ !
  !> Equlibrium velocity from macro
  pure function equilVelFromMacro( iField, moleFraction, velocity, nFields,    &
    &                              paramBInv, phi, resi_coeff ) result( eqVel )
    ! ---------------------------------------------------------------------------
    !> current field
    integer, intent(in) :: iField
    !> number of species
    integer, intent(in) :: nFields
    !> mole fraction of all species
    real(kind=rk), intent(in) :: moleFraction(nFields)
    !> velocity of all species
    real(kind=rk), intent(in) :: velocity(3,nFields)
    !> free parameter B
    real(kind=rk), intent(in) :: paramBInv
    !> molecular weight ratio of iField
    real(kind=rk), intent(in) :: phi
    !> resistivity coefficients of iField
    real(kind=rk), intent(in) :: resi_coeff(nFields)
    !> return equilibrium velocity
    real(kind=rk) :: eqVel(3)
    ! ---------------------------------------------------------------------------
    integer :: ifld
    ! ---------------------------------------------------------------------------
    eqVel(:) = velocity(:,iField)
    do ifld = 1, nFields
      eqVel(:) = eqVel(:) + resi_coeff(ifld)*phi*molefraction(ifld)  &
        &      * ( velocity(:,ifld) - velocity(:,iField) ) * paramBInv
    end do

  end function equilVelFromMacro
  ! ************************************************************************ !


  ! ************************************************************************ !
  !> Equlibrium velocity from macro with thermodynamic factor
  pure function equilVelFromMacroWTDF( iField, mass_dens, moleFraction,        &
    &                                  velocity, nFields, inv_thermodyn_fac,   &
    &                                  paramBInv, phi, resi_coeff )            &
    &                                  result( eqVel )
    ! -------------------------------------------------------------------- !
    !> current field
    integer, intent(in) :: iField
    !> number of species
    integer, intent(in) :: nFields
    !> mass density of all species
    real(kind=rk), intent(in) :: mass_dens(nFields)
    !> mole fraction of all species
    real(kind=rk), intent(in) :: moleFraction(nFields)
    !> velocity of all species
    real(kind=rk), intent(in) :: velocity(3,nFields)
    !> inverse of thermodynamic factor
    real(kind=rk), intent(in) :: inv_thermodyn_fac(nFields, nFields)
    !> free parameter B
    real(kind=rk), intent(in) :: paramBInv
    !> molecular weight ratio
    real(kind=rk), intent(in) :: phi(nFields)
    !> resistivity coefficients
    real(kind=rk), intent(in) :: resi_coeff(nFields, nFields)
    !> return equilibrium velocity
    real(kind=rk) :: eqVel(3)
    ! -------------------------------------------------------------------- !
    integer :: iField_2, iField_3
    ! -------------------------------------------------------------------- !
    ! computation equilibrium vel with thermodynamic factor is done on momentum
    ! space
    eqVel(:) = mass_dens(iField)*velocity( :, iField )
    do iField_2 = 1, nFields
      do iField_3 = 1, nFields
        eqVel(:) = eqVel(:) + inv_thermodyn_fac(iField, iField_2)             &
          &                 * mass_dens(iField_2)                             &
          &                 * resi_coeff( iField_2, iField_3 ) * phi(iField_2)&
          &                 * moleFraction(iField_3)                          &
          &                 * (velocity(:, iField_3) - velocity(:,iField_2))  &
          &                 * paramBInv 
      end do    
    end do  
    eqVel = eqVel / mass_dens(iField)

  end function equilVelFromMacroWTDF
  ! ************************************************************************ !


  ! ************************************************************************ !
  !> derive untransformed pdf velocity of species by solving system of 
  !! equations of nSpecies
  function momentumFromMacroLSE( moleFraction, first_Moments, nFields, phi,    &
    &                            resi_coeff, omega_fac, paramBInv )            &
    &                            result(momentum)
    ! -------------------------------------------------------------------- !
    !> number of species
    integer, intent(in) :: nFields
    !> molefraction  of all species
    real(kind=rk), intent(in) :: moleFraction(nFields)
    !> momentum from transformed pdf of all species
    real(kind=rk), intent(in) :: first_Moments(3, nFields)
    !> free parameter B
    real(kind=rk), intent(in) :: paramBInv
    !> molecular weight ratio of all species
    real(kind=rk), intent(in) :: phi(nFields)
    !> resistivity coefficients
    real(kind=rk), intent(in) :: resi_coeff(nFields, nFields)
    !> relaxation parameter, omega_diff*0.5_rk
    real(kind=rk), intent(in) :: omega_fac
    !> return actual momentum
    real(kind=rk) :: momentum(3, nFields)
    ! -------------------------------------------------------------------- !
    integer :: ifield, ifieldDia, ifieldNonDia, icomp
    real(kind=rk) :: matrixA( nFields, nFields ), invA( nFields, nFields )
    ! -------------------------------------------------------------------- !
    ! build up the equation system for momentum
    matrixA = 0.0_rk
    do iField = 1, nFields
      ! set diagonal part
      matrixA( iField, iField ) = 1.0_rk
      do iFieldDia = 1, nFields
        matrixA( iField, iField ) = matrixA( iField, iField )     &
          & + omega_fac * resi_coeff( iField, iFieldDia )         &
          & * phi( iField ) * moleFraction( iFieldDia ) * paramBInv
      end do
      ! set nonDiagonal
      do iFieldNonDia = 1, nFields
        matrixA( iField, iFieldNonDia ) = matrixA( iField, iFieldNonDia ) &
          & - omega_fac * resi_coeff( iField, iFieldNonDia )              &
          & * phi( iFieldNonDia ) * moleFraction( iField ) * paramBInv
      end do
    end do
      
    invA = invert_matrix( matrixA )

    ! momentum of all species
    momentum = 0.0_rk
    do iComp = 1, 3
      momentum( iComp, : ) = matmul( invA, first_Moments( iComp, : ) )
    end do

  end function momentumFromMacroLSE
  ! ************************************************************************ !


  ! ************************************************************************ !
  !> derive untransformed pdf velocity of species by solving system of 
  !! equations of nSpecies
  function momentumFromMacroLSE_WTDF( moleFraction, first_Moments, nFields,    &
    &                                 inv_thermodyn_fac, phi, resi_coeff,      &
    &                                 omega_fac, paramBInv ) result(momentum)
    ! -------------------------------------------------------------------- !
    !> number of species
    integer, intent(in) :: nFields
    !> molefraction  of all species
    real(kind=rk), intent(in) :: moleFraction(nFields)
    !> momentum from transformed pdf of all species
    real(kind=rk), intent(in) :: first_Moments(3, nFields)
    !> inverse of thermodynamic factor
    real(kind=rk), intent(in) :: inv_thermodyn_fac(nFields, nFields)
    !> free parameter B
    real(kind=rk), intent(in) :: paramBInv
    !> molecular weight ratio of all species
    real(kind=rk), intent(in) :: phi(nFields)
    !> resistivity coefficients
    real(kind=rk), intent(in) :: resi_coeff(nFields, nFields)
    !> relaxation parameter, omega_diff*0.5_rk
    real(kind=rk), intent(in) :: omega_fac
    !> return actual momentum
    real(kind=rk) :: momentum(3, nFields)
    ! -------------------------------------------------------------------- !
    integer :: ifield, ifield_2, ifield_3, icomp
    real(kind=rk) :: matrixA( nFields, nFields ), invA( nFields, nFields )
    ! -------------------------------------------------------------------- !
    matrixA = 0.0_rk
    !build up matrix to solver LSE for actual velocity
    do iField = 1, nFields
      !set diagonal part 
      matrixA(iField, iField) = 1.0_rk
      do iField_2 = 1, nFields
        do iField_3 = 1, nFields
          matrixA(iField, iField_2) = matrixA(iField, iField_2) + omega_fac  &
            &                    * inv_thermodyn_fac(iField, iField_2)       &
            &                    * resi_coeff(iField_2, iField_3)            &
            &                    * phi(iField_2) * moleFraction(iField_3)    &
            &                    * paramBInv
        end do
      end do
      !set non-diagonal part
      do iField_2 = 1, nFields
        do iField_3 = 1, nFields
          matrixA(iField, iField_3) = matrixA(iField, iField_3) - omega_fac  &
            &                    * inv_thermodyn_fac(iField, iField_2)       &
            &                    * resi_coeff(iField_2, iField_3)            &
            &                    * phi(iField_3) * moleFraction(iField_2)    &
            &                    * paramBInv
        end do
      end do
    end do
      
    invA = invert_matrix( matrixA )

    ! momentum of all species
    momentum = 0.0_rk
    do iComp = 1, 3
      momentum( iComp, : ) = matmul( invA, first_Moments( iComp, : ) )
    end do

  end function momentumFromMacroLSE_WTDF
  ! ************************************************************************ !


  ! ************************************************************************ !
  !> derive equilibrium from macro
  pure function equilFromMacro( iField, mass_dens, moleFraction, velocity,     &
    &                           layout, nFields, phi, paramBInv, resi_coeff,   &
    &                           theta_eq, weight0Inv )  result(fEq)
    ! -------------------------------------------------------------------- !
    !> current field
    integer, intent(in) :: iField
    !> number of species
    integer, intent(in) :: nFields
    !> mass density  of all species
    real(kind=rk), intent(in) :: mass_dens(nFields)
    !> molefraction  of all species
    real(kind=rk), intent(in) :: moleFraction(nFields)
    !> velocity of all species
    real(kind=rk), intent(in) :: velocity(3, nFields)
    !> scheme layout contains stencil definition and lattice weight
    type(mus_scheme_layout_type), intent(in) :: layout
    !> molecular weight ratio of iField
    real(kind=rk), intent(in) :: phi
    !> free parameter B
    real(kind=rk), intent(in) :: paramBInv
    !> resistivity coefficients
    real(kind=rk), intent(in) :: resi_coeff(nFields)
    !> parameter to tune mixture velocity in equilibrium quadratic term
    real(kind=rk), intent(in) :: theta_eq
    !> Inverse of lattice weight ar restPosition
    real(kind=rk), intent(in) :: weight0Inv
    !> return equilibrium
    real(kind=rk) :: fEq(layout%fStencil%QQ)
    ! -------------------------------------------------------------------- !
    integer :: iDir, QQ
    real(kind=rk) :: totMass_densInv
    real(kind=rk) :: ucx, usq, ucxQuadTerm, velAvg(3), velQuadTerm(3), eqVel(3)
    ! -------------------------------------------------------------------- !
    QQ = layout%fStencil%QQ 

    ! compute equilibrium velocity
    eqVel = equilVelFromMacro( iField       = iField,       &
      &                        moleFraction = moleFraction, &
      &                        velocity     = velocity,     &
      &                        nFields      = nFields,      &
      &                        paramBInv    = paramBInv,    &
      &                        phi          = phi,          &
      &                        resi_coeff   = resi_coeff    ) 

    ! total mass density inverse
    totMass_densInv = 1.0_rk/sum(mass_dens)

    ! mass averaged mixture velocity
    velAvg(1) = dot_product( mass_dens(:), velocity(1, :) )*totMass_densInv
    velAvg(2) = dot_product( mass_dens(:), velocity(2, :) )*totMass_densInv
    velAvg(3) = dot_product( mass_dens(:), velocity(3, :) )*totMass_densInv

    ! velocity in quadratic term of equilibrium
    velQuadTerm(:) = theta_eq*velAvg(:) + (1.0_rk-theta_eq)*eqVel(:)
    
    ! Calculate the square of velocity
    usq = dot_product( velQuadTerm,velQuadTerm ) * t2cs2inv

    do iDir = 1, QQ
      ! Velocity times lattice unit velocity
      ucx = dot_product( layout%fStencil%cxDirRK(:, iDir), eqVel )

      ucxQuadTerm = dot_product( layout%fStencil%cxDirRK(:, iDir), &
        &                        velQuadTerm )

      ! calculate equilibrium
      fEq(iDir) = layout%weight( iDir ) * mass_dens(iField)  &
        &         * ( phi + ucx * cs2inv                     &
        &         + ucxQuadTerm * ucxQuadTerm * t2cs4inv     &
        &         - usq                                      )
    end do ! iDir

    fEq(layout%fStencil%restPosition) =                 &
      & layout%weight( layout%fStencil%restPosition )   &
      & * mass_dens(iField)                               &
      & * ( weight0Inv + (1.0_rk - weight0Inv)*phi - usq  )

  end function equilFromMacro
  ! ************************************************************************ !


  ! ************************************************************************ !
  !> derive equilibrium from macro
  pure function equilFromMacroWTDF( iField, mass_dens, moleFraction, velocity, &
    &                               inv_thermodyn_fac, layout, nFields, phi,   &
    &                               paramBInv, resi_coeff, theta_eq,           &
    &                               weight0Inv )  result(fEq)
    ! -------------------------------------------------------------------- !
    !> current field
    integer, intent(in) :: iField
    !> number of species
    integer, intent(in) :: nFields
    !> mass density  of all species
    real(kind=rk), intent(in) :: mass_dens(nFields)
    !> molefraction  of all species
    real(kind=rk), intent(in) :: moleFraction(nFields)
    !> velocity of all species
    real(kind=rk), intent(in) :: velocity(3, nFields)
    !> inverse of thermodynamic factor
    real(kind=rk), intent(in) :: inv_thermodyn_fac(nFields, nFields)
    !> scheme layout contains stencil definition and lattice weight
    type(mus_scheme_layout_type), intent(in) :: layout
    !> molecular weight ratio of iField
    real(kind=rk), intent(in) :: phi(nFields)
    !> free parameter B
    real(kind=rk), intent(in) :: paramBInv
    !> resistivity coefficients
    real(kind=rk), intent(in) :: resi_coeff(nFields, nFields)
    !> parameter to tune mixture velocity in equilibrium quadratic term
    real(kind=rk), intent(in) :: theta_eq
    !> Inverse of lattice weight ar restPosition
    real(kind=rk), intent(in) :: weight0Inv
    !> return equilibrium
    real(kind=rk) :: fEq(layout%fStencil%QQ)
    ! -------------------------------------------------------------------- !
    integer :: iDir, QQ
    real(kind=rk) :: totMass_densInv
    real(kind=rk) :: ucx, usq, ucxQuadTerm, velAvg(3), velQuadTerm(3), eqVel(3)
    ! -------------------------------------------------------------------- !
    QQ = layout%fStencil%QQ 

    ! compute equilibrium velocity
    eqVel = equilVelFromMacroWTDF( iField            = iField,            &
      &                            mass_dens         = mass_dens,         &
      &                            moleFraction      = moleFraction,      &
      &                            velocity          = velocity,          &
      &                            nFields           = nFields,           &
      &                            inv_thermodyn_fac = inv_thermodyn_fac, &
      &                            paramBInv         = paramBInv,         &
      &                            phi               = phi,               &
      &                            resi_coeff        = resi_coeff         )

    ! total mass density inverse
    totMass_densInv = 1.0_rk/sum(mass_dens)

    ! mass averaged mixture velocity
    velAvg(1) = dot_product( mass_dens(:), velocity(1, :) )*totMass_densInv
    velAvg(2) = dot_product( mass_dens(:), velocity(2, :) )*totMass_densInv
    velAvg(3) = dot_product( mass_dens(:), velocity(3, :) )*totMass_densInv

    ! velocity in quadratic term of equilibrium
    velQuadTerm(:) = theta_eq*velAvg(:) + (1.0_rk-theta_eq)*eqVel(:)
    
    ! Calculate the square of velocity
    usq = dot_product( velQuadTerm,velQuadTerm ) * t2cs2inv

    do iDir = 1, QQ
      ! Velocity times lattice unit velocity
      ucx = dot_product( layout%fStencil%cxDirRK(:, iDir), eqVel )

      ucxQuadTerm = dot_product( layout%fStencil%cxDirRK(:, iDir), &
        &                        velQuadTerm )

      ! calculate equilibrium
      fEq(iDir) = layout%weight( iDir ) * mass_dens(iField) &
        &         * ( phi(iField) + ucx * cs2inv            &
        &         + ucxQuadTerm * ucxQuadTerm * t2cs4inv    &
        &         - usq                                     )
    end do ! iDir
    
    fEq(layout%fStencil%restPosition) =                           &
      & layout%weight( layout%fStencil%restPosition )             &
      & * mass_dens(iField)                                       &
      & * ( weight0Inv + (1.0_rk - weight0Inv)*phi(iField) - usq  )

  end function equilFromMacroWTDF
  ! ************************************************************************ !


! ****************************************************************************** !
  !> Calculate the density of a given set of elements (sum up all links).
  !! This routine is used to compute density for all scheme kinds
  !! For multi-species, it can compute both species density and mixture density
  !!
  !! The interface has to comply to the abstract interface
  !! [[tem_varSys_module:tem_varSys_proc_element]].
  !!
  recursive subroutine derivedensityMS(fun, varsys, elempos, time, tree, &
      &                                                nelems, ndofs, res)
    !--------------------------------------------------------------------------!
    !> description of the method to obtain the variables, here some preset
    !! values might be stored, like the space time function to use or the
    !! required variables.
    class(tem_varsys_op_type), intent(in) :: fun

    !> the variable system to obtain the variable from.
    type(tem_varsys_type), intent(in)     :: varsys

    !> position of the treeid of the element to get the variable for in the
    !! global treeid list.
    integer, intent(in)                   :: elempos(:)

    !> point in time at which to evaluate the variable.
    type(tem_time_type), intent(in)       :: time

    !> global treelm mesh info
    type(treelmesh_type), intent(in)      :: tree

    !> number of values to obtain for this variable (vectorized access).
    integer, intent(in)                   :: nelems

    !> number of degrees of freedom within an element.
    integer, intent(in)                   :: ndofs

    !> resulting values for the requested variable.
    !!
    !! linearized array dimension:
    !! (n requested entries) x (ncomponents of this variable)
    !! x (ndegrees of freedom)
    !! access: (ielem-1)*fun%ncomponents*ndofs +
    !!         (idof-1)*fun%ncomponents + icomp
    real(kind=rk), intent(out)            :: res(:)
    !--------------------------------------------------------------------------!
    type(mus_varSys_data_type), pointer   :: fPtr
    real(kind=rk)                         :: dens
    integer                               :: iElem, iComp, iLevel, iDep
    integer                               :: pdfPos, statePos, varPos
    integer                               :: nSize, nCompPDF
    ! ---------------------------------------------------------------------------
    call C_F_POINTER( fun%method_Data, fPtr )

    ! res is always AOS layout
    res = 0.0_rk

    do iElem = 1, nElems
      ! if state array is defined level wise then use levelPointer(pos)
      ! to access state array
      statePos = fPtr%solverData%geometry%levelPointer( elemPos(iElem) )
      iLevel = tem_levelOf( tree%treeID( elemPos(iElem) ) )
      nSize = fPtr%solverData%scheme%pdf( iLevel )%nSize

      dens = 0.0_rk
      ! use iDep here to use this routine to compute also mixture density
      ! in multi-species
      do iDep = 1, fun%nInputs
        pdfPos = fun%input_varPos(iDep)
        nCompPDF = varSys%method%val(pdfPos)%nComponents
        do iComp = 1, nCompPDF
          varPos = varSys%method%val(pdfPos)%state_varPos(iComp)
          dens = dens + &
            & fPtr%solverData%scheme%state( iLevel )%val(        &
            ! position of this state variable in the state array
            & ?IDX?( varPos, statePos, varSys%nScalars, nSize ), &
            & fPtr%solverData%scheme%pdf( iLevel )%nNext )
        end do !iComp
      end do !iDep
      res( iElem ) = dens

    end do ! iElem

  end subroutine deriveDensityMS
! ****************************************************************************** !

end module mus_derQuanMSLiquid_module
! **************************************************************************** !

