! See copyright notice in the COPYRIGHT file.
! ****************************************************************************** !
!> author: seyfettin bilgi
!! This module provides the MUSUBI specific functions for calculating
!! macroscopic quantities from the state variables.
!! The depending common interface between MUSUBI and ATELES is defined in the
!! tem_derived_module. The functionality for accessing a variable from the state
!! and evaluating a lua function are also provided in the tem_derived module.
!! A Novel lattice boltzmann model for nernstPlanck equation 
!! author> Zhenhua Chai , Baochang Shi
!! A Coupled Lattice Boltzmann method to solve  Nernst -Planck Model for 
!! simulating Electro-Osmotic Flows
?? include 'header/lbm_macros.inc'
?? include 'header/lbm_deriveMacros.inc'
?? include 'header/lbm_interfaceMacros.inc'
?? include 'treelm/source/deriveMacros.inc'
module mus_derQuanNernstPlanck_module
  use iso_c_binding, only: c_loc, c_ptr, c_f_pointer
  ! include treelm modules
  use env_module,               only: rk, long_k, globalMaxLevels, labelLen
  use tem_param_module,         only: div1_2, div1_3, div1_54, div1_9, div3_4, &
    &                                 sqrt3, cs2inv, cs2, t2cs2inv, t2cs4inv,  &
    &                                 cs4inv
  use tem_aux_module,           only: tem_abort
  use tem_varSys_module,        only: tem_varSys_type, tem_varSys_op_type,     &
    &                                 tem_varSys_append_derVar,                &
    &                                 tem_varSys_proc_point,                   &
    &                                 tem_varSys_proc_element,                 &
    &                                 tem_varSys_proc_setParams,               &
    &                                 tem_varSys_proc_getParams,               &
    &                                 tem_varSys_proc_setupIndices,            &
    &                                 tem_varSys_proc_getValOfIndex
  use tem_variable_module,      only: tem_variable_type
  use tem_stencil_module,       only: tem_stencilHeader_type
  use tem_logging_module,       only: logUnit
  use tem_topology_module,      only: tem_levelOf
  use tem_time_module,          only: tem_time_type
  use treelmesh_module,         only: treelmesh_type
  use tem_debug_module,         only: dbgUnit
  use tem_operation_var_module, only: tem_opVar_setupIndices,      &
    &                                 tem_get_new_varSys_data_ptr, &
    &                                 tem_evalAdd_forElement,      &
    &                                 tem_evalAdd_fromIndex,       &
    &                                 tem_opVar_setParams,         &
    &                                 tem_opVar_getParams
  use tem_grow_array_module,    only: grw_labelarray_type, append

  ! include musubi modules
  use mus_source_var_module,         only: mus_source_op_type
  use mus_scheme_header_module,      only: mus_scheme_header_type
  use mus_scheme_layout_module,      only: mus_scheme_layout_type
  use mus_varSys_module,             only: mus_varSys_data_type,             &
    &                                      mus_varSys_solverData_type,       &
    &                                      mus_get_new_solver_ptr,           &
    &                                      mus_deriveVar_ForPoint,           &
    &                                      mus_generic_varFromPDF_fromIndex, &
    &                                      mus_generic_fromPDF_forElement,   &
    &                                      mus_derive_fromPDF
  use mus_scheme_type_module,        only: mus_scheme_type
  use mus_field_prop_module,         only: mus_field_prop_type
  use mus_operation_var_module,      only: mus_opVar_setupIndices 
  use mus_derVarPos_type_module,     only: mus_derVarPos_type
  use mus_physics_module,            only: mus_convertFac_type, &
    &                                      gasConst_R, faraday

  ! include aotus modules
  use aotus_module, only: flu_State


  implicit none

  private

!KM!  public :: mus_append_derVar_nernstPlanck
  public :: deriveAuxNP_fromState
  public :: deriveEquilNP_fromAux

  public :: mus_deriveMoleDensity
  public :: deriveMoleDensity_forElement
  public :: deriveMoleDensity_fromIndex

!!  public :: deriveSrc_electricField
  public :: applySrc_electricFieldNP
 
contains

  ! **************************************************************************** !
  !> subroutine to add derive variables for weakly compressible PB
  !! (schemekind = 'nernstPlanck') to the varsys.
  !! A Coupled Lattice Boltzmann Method to Solve Nernst-Planck Model
  !! for Simulating Electro-Osmotic flows
  !! author> Xuguang yang
  !KM! \todo currently this is replaced by mus_append_derMixVar_MS since
  !KM! only mixture variable is appended in this routine
!KM!  subroutine mus_append_derVar_nernstPlanck( varSys, solverData, fldLabel, &
!KM!    &                                        nFields, derVarName ) 
!KM!    ! ---------------------------------------------------------------------------
!KM!    !> global variable system
!KM!    type(tem_varSys_type), intent(inout)  :: varSys
!KM!
!KM!    !> Contains pointer to solver data types
!KM!    type(mus_varSys_solverData_type), target, intent(in) :: solverData
!KM!
!KM!    !> number of fields
!KM!    integer, intent(in)                      :: nFields
!KM!
!KM!    !> array of field label prefix. Size=nFields
!KM!    character(len=*), intent(in)              :: fldLabel(:)
!KM!
!KM!    !> array of derive physical variables
!KM!    type(grw_labelarray_type), intent(inout) :: derVarName
!KM!    ! ---------------------------------------------------------------------------
!KM!    ! number of derive variables
!KM!    integer :: iVar, nComponents, addedPos, iField
!KM!    logical :: wasAdded
!KM!    character(len=labelLen), allocatable ::  input_varname(:)
!KM!    character(len=labelLen)  ::  varName
!KM!    procedure(tem_varSys_proc_point), pointer :: get_point => NULL()
!KM!    procedure(tem_varSys_proc_element), pointer :: get_element => NULL()
!KM!    procedure(tem_varSys_proc_setParams), pointer :: set_params => null()
!KM!    procedure(tem_varSys_proc_getParams), pointer :: get_params => null()
!KM!    procedure(tem_varSys_proc_setupIndices), pointer :: &
!KM!      &                                      setup_indices => null()
!KM!    procedure(tem_varSys_proc_getValOfIndex), pointer :: &
!KM!      &                                       get_valOfIndex => null()
!KM!    type(c_ptr) :: method_data  
!KM!    character(len=labelLen) :: derVarName_loc
!KM!    ! ---------------------------------------------------------------------------
!KM!    nullify(get_point, get_element, set_params, get_params, setup_indices, &
!KM!      &     get_valOfIndex)
!KM!
!KM!    derVarName_loc = 'mole_density'
!KM!    ! mole_density of each field is already added as auxiliary variable
!KM!    ! so only add mole_density for mixture
!KM!
!KM!    ! set pointers for mixture. mixture quantity is obtained by summing up
!KM!    ! species quantities
!KM!    get_element => tem_evalAdd_forElement
!KM!    get_point => mus_deriveVar_ForPoint
!KM!    setup_indices => tem_opVar_setupIndices
!KM!    get_valOfIndex => tem_evalAdd_fromIndex
!KM!    method_data = tem_get_new_varSys_data_ptr(method_data)
!KM!    set_params => tem_opVar_setParams 
!KM!    get_params => tem_opVar_getParams
!KM!
!KM!    nComponents = 1
!KM!    varname = trim(adjustl(derVarName_loc))
!KM!    allocate(input_varname(nFields))
!KM!    do iField = 1, nFields
!KM!      input_varname(iField) = trim(fldLabel(iField))//trim(varname)
!KM!    end do  
!KM!
!KM!    ! append variable to varSys
!KM!    call tem_varSys_append_derVar(  me             = varSys,         &
!KM!      &                             varName        = trim(varname),  &
!KM!      &                             nComponents    = nComponents,    &
!KM!      &                             input_varname  = input_varname,  &
!KM!      &                             method_data    = method_data,    &
!KM!      &                             get_point      = get_point,      &
!KM!      &                             get_element    = get_element,    &
!KM!      &                             set_params     = set_params,     &
!KM!      &                             get_params     = get_params,     &
!KM!      &                             setup_indices  = setup_indices,  &
!KM!      &                             get_valOfIndex = get_valOfIndex, &
!KM!      &                             pos            = addedPos,       &
!KM!      &                             wasAdded       = wasAdded        )
!KM!
!KM!    if (wasAdded) then
!KM!      write(logUnit(10),*) ' Appended variable:'//trim(varname)
!KM!    else if (addedpos < 1) then
!KM!      write(logUnit(1),*) 'Error: variable '//trim(varname)// &
!KM!        &                 ' is not added to variable system'
!KM!    end if
!KM!
!KM!    deallocate(input_varname)
!KM!
!KM!  end subroutine mus_append_derVar_nernstPlanck
  ! ************************************************************************** !


! **************************************************************************** !
  !> This routine computes auxField 'mole_density' from state array
  !! This must comply with interface in 
  !! [[mus_derVarPos_type_module:derive_AuxFromState]]
?? copy :: deriveAuxFromState_header( deriveAuxNP_fromState )
    ! ------------------------------------------------------------------------ !
    integer :: iElem, iDir, pdfPos
    real(kind=rk) :: pdf( stencil%QQ )
    ! ------------------------------------------------------------------------ !
    !NEC$ ivdep
    do iElem = 1, nElems
      !NEC$ shortloop
      do iDir = 1, stencil%QQ
        pdfPos = varSys%method%val(derVarPos%pdf)%state_varPos(iDir) 
        pdf(iDir) = state( ?IDX?(pdfPos, iElem, varSys%nScalars, nSize) )
      end do

      ! element offset is not required because passive scalar has only
      ! one aux scalar
      ! density
      auxField(iElem) = sum(pdf)
    end do

  end subroutine deriveAuxNP_fromState
! **************************************************************************** !

  ! ************************************************************************** !
  !> This routine computes equilbrium from auxField
?? copy :: deriveEquilFromAux_header(deriveEquilNP_fromAux)
    ! ------------------------------------------------------------------------ !
    ! ------------------------------------------------------------------------ !
    !KM: \todo add transport velocity as auxField
    write(logUnit(1),*) 'ERROR: Equilibrium calculation requires transport '
    write(logUnit(1),*) 'velocity and it is not provided to this routine'
    write(logUnit(1),*) 'Solution: use deriveEquilPS_fromMacro'
    call tem_abort()
  end subroutine deriveEquilNP_fromAux
  ! ************************************************************************** !

  ! ************************************************************************** !
  !> Calculate the potential of a given set of pdfs of elements
  !!
  !! The interface has to comply to the abstract interface
  !! [[mus_varSys_module:mus_derive_fromPDF]].
  !!
  recursive subroutine mus_deriveMoleDensity(fun, varsys, stencil, iLevel, &
    &                                      pdf, res, nVals )
    !> description of the method to obtain the variables, here some preset
    !! values might be stored, like the space time function to use or the
    !! required variables.
    class(tem_varsys_op_type), intent(in)     :: fun
    !> the variable system to obtain the variable from.
    type(tem_varsys_type), intent(in)         :: varsys
    !> fluid stencil defintion
    type(tem_stencilHeader_type), intent(in)  :: stencil
    !> current Level
    integer, intent(in)                       :: iLevel
    !> pdf array
    real(kind=rk), intent(in)                 :: pdf(:)
    !> results
    real(kind=rk), intent(out)                :: res(:)
    !> nVals to get
    integer, intent(in)                       :: nVals
    ! ---------------------------------------------------------------------------
    integer                                   :: iVal
    integer                                   :: nCompPDF
    ! ---------------------------------------------------------------------------
    nCompPDF = varSys%method%val(fun%input_varPos(1))%nComponents

    res = 0.0_rk
    do iVal = 1, nVals
      res(iVal) = sum( pdf( (iVal-1)*nCompPDF + 1: iVal*nCompPDF) )
    end do !iVal
  end subroutine mus_deriveMoleDensity
! ****************************************************************************** !


  ! ************************************************************************** !
  !> Calculate the potential of a given set of elements (sum up all links).
  !! This routine is used to compute potential for all scheme kinds
  !!
  !! The interface has to comply to the abstract interface
  !! [[tem_varSys_module:tem_varSys_proc_element]].
  !!
?? copy :: get_element_headtxt(deriveMoleDensity_forElement)
    ! ---------------------------------------------------------------------------
    !> Function pointer to perform specific operation.
    procedure(mus_derive_fromPDF), pointer :: fnCalcPtr
    ! ---------------------------------------------------------------------------

    fnCalcPtr => mus_deriveMoleDensity

    call mus_generic_fromPDF_forElement( &
      &  fun       = fun,                &
      &  varSys    = varSys,             &
      &  elempos   = elempos,            &
      &  tree      = tree,               &
      &  time      = time,               &
      &  nVals     = nElems,             &
      &  fnCalcPtr = fnCalcPtr,          &
      &  nDofs     = nDofs,              &
      &  res       = res                 )

  end subroutine deriveMoleDensity_forElement
  ! ************************************************************************** !

  ! ************************************************************************** !
  !> Calculate the potential of a given set of elements (sum up all links).
  !! This routine is used to compute potential for all scheme kinds
  !!
  !! The interface has to comply to the abstract interface
  !! [[tem_varSys_module:tem_varSys_proc_getValofIndex]].
  !!
?? copy :: get_valofindex_headtxt(deriveMoleDensity_fromIndex)
    ! ---------------------------------------------------------------------------
    !> Function pointer to perform specific operation.
    procedure(mus_derive_fromPDF), pointer  :: fnCalcPtr
    ! ---------------------------------------------------------------------------

    fnCalcPtr => mus_deriveMoleDensity

    call mus_generic_varFromPDF_fromIndex( &
      &  fun       = fun,                  &
      &  varSys    = varSys,               &
      &  time      = time,                 &
      &  iLevel    = iLevel,               &
      &  idx       = idx,                  &
      &  nVals     = nVals,                &
      &  fnCalcPtr = fnCalcPtr,            &
      &  res       = res                   )

  end subroutine deriveMoleDensity_fromIndex
  ! ************************************************************************** !

!!  ! ************************************************************************** !
!!  !> Calculate charge density source variable referred in config file 
!!  !!
!!  !! The interface has to comply to the abstract interface
!!  !! [[tem_varSys_module:tem_varSys_proc_element]].
!!  !! 
!!?? copy :: get_element_headtxt(deriveSrc_electricField)
!!    ! -------------------------------------------------------------------- !
!!    type(mus_varSys_data_type), pointer :: fPtr
!!    type(mus_scheme_type), pointer :: scheme
!!    real(kind=rk) :: electricField(nElems*3)
!!    real(kind=rk) :: EF_elem(3)
!!    integer :: stFun_pos
!!    integer :: iElem, nElems, iDir, iLevel, posInTotal
!!    integer :: iField, nFields, depField, nScalars, QQ, nInputStates
!!    real(kind=rk) :: num_dens( varSys%nStateVars )
!!    real(kind=rk) :: gasConst, temp0
!!    real(kind=rk), dimension(varSys%nStateVars) :: chargeNr
!!    real(kind=rk), dimension(3, varSys%nStateVars ) ::  electricTerm
!!    real(kind=rk), dimension(varSys%nStateVars, varSys%nStateVars) :: diff_coeff
!!    real(kind=rk), dimension(varSys%nStateVars) :: omega
!!    real(kind=rk) :: temp, force_fac
!!    integer :: nSize
!!    integer :: stateVarMap(varSys%nStateVars)
!!    ! -------------------------------------------------------------------- !
!!    ! convert c pointer to solver type fortran pointer
!!    call c_f_pointer( fun%method_data, fPtr )
!!    scheme => fPtr%solverData%scheme
!!
!!    ! last input var is spacetime function
!!    stFun_pos = fun%input_varPos(fun%nInputs)
!!
!!    ! Get Electric field which might be defined to vary in space and time
!!    call varSys%method%val(stfun_pos)%get_element( varSys  = varSys,       &
!!      &                                            elemPos = elemPos,      &
!!      &                                            time    = time,         &
!!      &                                            tree    = tree,         &
!!      &                                            nElems  = nElems,       &
!!      &                                            ndofs   = 1,            &
!!      &                                            res     = electricField )
!!
!!    ! convert physical to lattice
!!    electricField = electricField * fPtr%solverData%physics%coulomb0 &
!!      &           / fPtr%solverData%physics%fac(iLevel)%force
!!
!!    ! number of pdf states this source depends on
!!    ! last input is spacetime function so it is neglected
!!    nInputStates = fun%nInputs - 1
!!
!!    ! constant parameter
!!    nFields = scheme%nFields
!!    QQ = scheme%layout%fStencil%QQ
!!    nScalars = varSys%nScalars
!!    stateVarMap = scheme%stateVarMap%varPos%val(:)
!!    ! diffusivity coefficients
!!    diff_coeff(:) = scheme%field(:)%fieldProp%species%diff_coeff(1)
!!    ! gas constant
!!    gasConst = scheme%mixture%gasConst_R_LB
!!    ! temperature
!!    temp0 = scheme%mixture%temp0LB
!!    ! temperature
!!    temp = scheme%mixture%temp0
!!    ! omega
!!    omega(:) =   scheme%field(:)%fieldProp%species%omega
!!
!!    force_fac = cs2inv * scheme%mixture%faradayLB / (gasConst*temp0)
!!    ! update source for each element
!!    do iElem = 1, nElems
!!      posInTotal = fPtr%solverData%geometry%levelPointer( elemPos(iElem) )
!!      iLevel = tem_levelOf( tree%treeID( elemPos(iElem) ) )
!!      nSize = scheme%pdf(iLevel)%nSize
!!
!!      ! S = (D*w/cs^2)*(z_i*F/(R*T))*n_i.E
!!      num_dens = 0.0_rk
!!      do iField = 1, nFields
!!        ! compute field density and first moments
!!        do iDir = 1, QQ 
!!          !field density
!!          num_dens( iField ) = num_dens( iField ) + inState(           &
!!            & ?FETCH?(iDir,iField,posInTotal,QQ,nScalars,nPdfSize,neigh) )
!!        end do
!!      end do
!!
!!      ! convert physical to lattice
!!      EF_elem = electricField((iElem-1)*3+1 : iElem*3)
!!
!!    !write(*,*) 'electricField ', EF_elem
!!      ! electric field term: n_k * z_k * Faraday * electricField
!!      do iField = 1, nFields
!!        electricTerm(:, iField) = force_fac * diff_coeff(iField)      &
!!          &                     * omega(iField) * chargeNr(iField)  &
!!          &                     * EF_elem(:) * num_dens(iField)
!!      end do
!!    !write(dbgUnit(1),*) 'electricTerm ', electricTerm
!!
!!      ! copy the results to the res
!!      res( (iElem-1)*fun%nComponents + 1 : iElem*fun%nComponents) =  electricTerm
!!!write(dbgUnit(1),*) 'derive forceTerm ', forceTerm
!!    end do !iElem
!!
!!  end subroutine deriveSrc_electricField
!!! ****************************************************************************** !

! ****************************************************************************** !
   !> Update state with source variable "electric field"
   !!
   !! $$ \S_j = \w_j*\c_j*\S $$
   !! 
   !! Where \S is the source term 
   !! S = (D*w/cs^2)*(z_i*F/(R*T))*n_i.E
   !! Simuilar to derive routine but it updates the state whereas derive 
   !! is used for tracking.
?? copy :: applySource_header(applySrc_electricFieldNP)
     ! -------------------------------------------------------------------- !
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    integer :: stFun_pos
    real(kind=rk) :: electricField(fun%elemLvl(iLevel)%nElems*3)
    integer :: nSize
    real(kind=rk) :: EF_elem(3)
    integer :: iElem, nElems, iDir, posInTotal
    integer :: iField, nFields, depField, nScalars, QQ, nInputStates
    !number density of nSpecies 
    real(kind=rk) :: num_dens( varSys%nStateVars )
    real(kind=rk), dimension(varSys%nStateVars) :: chargeNr
    real(kind=rk) :: gasConstLB, temp, minMolWeight,  mixDiffForce(3)
    real(kind=rk), dimension(3, varSys%nStateVars ) :: electricTerm
    real(kind=rk), dimension(varSys%nStateVars) :: diff_coeff
    real(kind=rk), dimension(varSys%nStateVars) :: omega
    integer :: stateVarMap(varSys%nStateVars)
    real(kind=rk) :: force_fac, faradayLB
    real(kind=rk) :: forceTerm
    ! -------------------------------------------------------------------- !
!write(dbgUnit(1),*) 'source variable: ', trim(varSys%varname%val(fun%srcTerm_varPos))
    ! convert c pointer to solver type fortran pointer
    call c_f_pointer( varSys%method%val( fun%srcTerm_varPos )%method_data, &
      &               fPtr ) 
    scheme => fPtr%solverData%scheme

    ! Number of elements to apply source terms
    nElems = fun%elemLvl(iLevel)%nElems

    ! Get electrical force which is refered in config file either its
    ! spacetime variable or operation variable
    call varSys%method%val(fun%data_varPos)%get_valOfIndex( &
      & varSys  = varSys,                                   &
      & time    = time,                                     &
      & iLevel  = iLevel,                                   &
      & idx     = fun%elemLvl(iLevel)%idx(1:nElems),        &
      & nVals   = nElems,                                   & 
      & res     = electricField                             )

      ! convert physical to lattice
    electricField = electricField * fPtr%solverData%physics%coulomb0 &
      &           / fPtr%solverData%physics%fac(iLevel)%force

    ! number of pdf states this source depends on
    ! last input is spacetime function so it is neglected
    nInputStates =  varSys%method%val(fun%srcTerm_varPos)%nInputs - 1

    ! constant parameter
    nFields = scheme%nFields
    QQ = scheme%layout%fStencil%QQ
    nScalars = varSys%nScalars
    stateVarMap = scheme%stateVarMap%varPos%val(:)
    ! diffusivity coefficients
    do iField = 1 , nFields
      diff_coeff(iField) = scheme%field(iField)%fieldProp%species%diff_coeff(1)
      ! omega
      omega(iField) =   scheme%field(iField)%fieldProp%species%omega
      chargeNr(iField) =   scheme%field(iField)%fieldProp%species%chargeNr
    end do   
    ! gas constant
    gasConstLB = gasConst_R / fPtr%solverData%physics%fac(iLevel)%gasConst
    faradayLB = faraday / fPtr%solverData%physics%fac(iLevel)%faraday
    ! temperature
    temp = scheme%nernstPlanck%temp

    force_fac = cs2inv * faradayLB / (gasConstLB*temp)
    !write(*,*) 'force_fac ', force_fac
    ! update source for each element
    do iElem = 1, nElems
      posInTotal = fun%elemLvl(iLevel)%posInTotal(iElem)
      nSize = scheme%pdf(iLevel)%nSize

      ! S = (D*w/cs^2)*(z_i*F/(R*T))*n_i.E
      num_dens = 0.0_rk
      do iField = 1, nFields
        ! compute field density and first moments
        do iDir = 1, QQ 
          !field density
          num_dens( iField ) = num_dens( iField ) + inState(           &
            & ?FETCH?(iDir,iField,posInTotal,QQ,nScalars,nPdfSize,neigh) )
        end do
      end do

      ! convert physical to lattice
      EF_elem = electricField((iElem-1)*3+1 : iElem*3)

      !write(*,*) 'electricField ', EF_elem
      ! electric field term: n_k * z_k * Faraday * electricField
      do iField = 1, nFields
        electricTerm(:, iField) = force_fac * diff_coeff(iField)      &
          &                     * omega(iField) * chargeNr(iField)  &
          &                     * EF_elem(:) * num_dens(iField)
      end do
      !write(*,*) 'electricTerm ', electricTerm

      do iField = 1, nInputStates
        depField = varSys%method%val(fun%srcTerm_varPos)%input_varPos(iField)

        do iDir = 1, QQ
          ! Force on each species
          ! d^m_k = weight_m*cxDir_m ( y_k*\rho_e Faraday E )
          forceTerm = scheme%layout%fStencil%cxDirRK( 1, iDir ) &
            &         * electricTerm(1, depField)               &
            &       + scheme%layout%fStencil%cxDirRK( 2, iDir ) &
            &         * electricTerm(2, depField)               &
            &       + scheme%layout%fStencil%cxDirRK( 3, iDir ) &
            &         * electricTerm(3, depField)


          outState(                                                         &
            & ?SAVE?(iDir,depField,posInTotal,QQ,nScalars,nPdfSize,neigh) ) &
            & = outState(                                                   &
            & ?SAVE?(iDir,depField,posInTotal,QQ,nScalars,nPdfSize,neigh) ) &
            !& + force(iDir)
            & + scheme%layout%weight( iDir ) * forceTerm
        end do ! iDir
      end do !iField
    end do !iElem

  end subroutine applySrc_electricFieldNP
! ****************************************************************************** !



end module mus_derQuanNernstPlanck_module
