! See copyright notice in the COPYRIGHT file.
! ****************************************************************************** !
!> This module contains routines which initiliaze advection relaxation and
!! flow field for multispecies lbm gas model and liquid model.
module mus_initMultispecies_module

  ! include treelm modules
  use env_module,         only: labelLen
  use tem_aux_module,     only: tem_abort
  use tem_logging_module, only: logUnit

  ! include musubi modules
  use mus_MSGas_module, only: bgk_advRel_MSGas,                                &
    &                         bgk_advRel_d3q19f3_MSGas
  use mus_MSLiquid_module, only: bgk_advRel_MSLiquid,                          &
    &                            bgk_advRel_MSLiquid_WTDF,                     &
    &                            bgk_advRel_d3q19f3_MSLiquid,                  &
    &                            bgk_forcing_advRel_MSLiquid,                  &
    &                            mrt_advRel_MSLiquid,                          &
    &                            mrt_advRel_MSLiquid_WTDF,                     &
    &                            mrt_advRel_d3q19f3_MSLiquid
  use mus_scheme_type_module, only: kernel

  implicit none

  private

  public :: mus_init_advRel_multispecies_gas
  public :: mus_init_advRel_multispecies_liquid

contains

! ****************************************************************************** !
  !> Initialize the relaxation model for multispecies gas model
  subroutine mus_init_advRel_multispecies_gas( relaxation, layout, nFields, &
    &                                          compute )
    ! ---------------------------------------------------------------------------
    character(len=labelLen), intent(in) :: relaxation
    character(len=labelLen), intent(in) :: layout
    integer,                 intent(in) :: nFields
    procedure( kernel ), pointer, intent(out) :: compute
    ! ---------------------------------------------------------------------------

    select case( trim( relaxation ))
    case( 'bgk' )
      select case( trim(layout) )
      case( 'd3q19' )
        if ( nFields == 3 ) then
          write(logUnit(1),*) 'Choosen optimized d3q19f3 multispecies '//      &
            &                 'gas kernel'
          compute => bgk_advRel_d3q19f3_MSGas
        else
          write(logUnit(1),*) 'Choosen testing multispecies gas kernel'
          compute => bgk_advRel_MSGas
        end if
      case( 'd2q9' )
        write(logUnit(1),*) 'Choosen testing multispecies gas kernel'
        compute => bgk_advRel_MSGas
      case default
        write(logUnit(1),*) 'Stencil '//trim(layout)//' is not supported yet!'
        call tem_abort()
      end select ! layout
    case default
      write(logUnit(1),*) 'Relaxation '//trim(relaxation)//' is not supported!'
      call tem_abort()
    end select ! relaxation

  end subroutine mus_init_advRel_multispecies_gas
! ****************************************************************************** !

! ****************************************************************************** !
  !> Initialize the relaxation model for multispecies liquid model
  subroutine mus_init_advRel_multispecies_liquid( relaxation, layout, nFields,&
    &                                             compute )
    ! ---------------------------------------------------------------------------
    character(len=labelLen), intent(in) :: relaxation
    character(len=labelLen), intent(in) :: layout
    integer,                 intent(in) :: nFields
    procedure( kernel ), pointer, intent(out) :: compute
    ! ---------------------------------------------------------------------------

    select case( trim(relaxation) )
    case( 'bgk' )
      select case( trim(layout) )
      case( 'd3q19' )
        if( nFields==3 ) then
          write(logUnit(1),*) 'Choosen optimized d3q19f3 multispecies '//      &
            &                 'liquid kernel'
          compute => bgk_advRel_d3q19f3_MSLiquid
        else
          compute => bgk_advRel_MSLiquid
        end if
      case( 'd2q9' )
        compute => bgk_advRel_MSLiquid
      case default
        write(logUnit(1),*) 'Stencil '//trim(layout)//' is not supported yet!'
        call tem_abort()
      end select
    case( 'bgk_forcing' )
      select case( trim(layout) )
      case( 'd3q19' , 'd2q9')
        write(logUnit(1),*) 'Choosen unoptimized bgk forcing multispecies '//  &
          &                 'liquid kernel'
        compute => bgk_forcing_advRel_MSLiquid
      case default
        write(logUnit(1),*) 'Stencil '//trim(layout)//' is not supported yet!'
        call tem_abort()
      end select
    case( 'bgk_withthermodynfac' )
      select case( trim(layout) )
      case( 'd3q19' , 'd2q9' )
        write(logUnit(1),*) 'Choosen unoptimized bgk with thermodynamic '//  &
          &             'factor multispecies liquid kernel'
        compute => bgk_advRel_MSLiquid_WTDF
      case default
        write(logUnit(1),*) 'Stencil '//trim(layout)//' is not supported yet!'
        call tem_abort()
      end select
    case( 'mrt_withthermodynfac' )
      select case( trim(layout) )
      case( 'd3q19' , 'd2q9' )
        write(logUnit(1),*) 'Choosen unoptimized mrt multispecies liquid '// &
          &                 'kernel with thermodynamic factor'
        compute => mrt_advRel_MSLiquid_WTDF
      case default
        write(logUnit(1),*) 'Stencil '//trim(layout)//' is not supported yet!'
        call tem_abort()
      end select

    case( 'mrt' )
      select case( trim(layout) )
      case( 'd3q19' )
        if( nFields==3 ) then
          write(logUnit(1),*) 'Choosen optimized mrt d3q19f3 multispecies '//  &
            &                 'liquid kernel'
          compute => mrt_advRel_d3q19f3_MSLiquid
        else
          write(logUnit(1),*) 'Choosen unoptimized mrt multispecies liquid '// &
            &                 'kernel'
          compute => mrt_advRel_MSLiquid
        end if
      case( 'd2q9' )
        write(logUnit(1),*) 'Choosen unoptimized mrt multispecies liquid kernel'
        compute => mrt_advRel_MSLiquid
      case default
        write(logUnit(1),*) 'Stencil '//trim(layout)//' is not supported yet!'
        call tem_abort()
      end select
    case default
      write(logUnit(1),*) 'Relaxation '//trim(relaxation)//' is not supported!'
      call tem_abort()
    end select ! relaxation

  end subroutine mus_init_advRel_multispecies_liquid
! ****************************************************************************** !

end module mus_initMultispecies_module
