! See copyright notice in the COPYRIGHT file.
!> This module contains an interface for external C++ code to compute
!! liquid mixture property like thermodynamic factor and 
!! Maxwell-Stefan Diffusivity coefficients
module mus_eNRTL_module
  use env_module,                  only: rk  
  use tem_logging_module,          only: logUnit

  use, intrinsic :: iso_c_binding, only: c_char, c_int
  
  implicit none

  private 

  public :: mus_init_eNRTL
  public :: mus_calc_thermFactor
  public :: mus_calc_MS_DiffMatrix

  interface mus_calc_thermFactor
    module procedure mus_calc_thermFactor_single
  end interface mus_calc_thermFactor

  interface mus_calc_MS_DiffMatrix
    module procedure mus_calc_MS_DiffMatrix_single
  end interface mus_calc_MS_DiffMatrix

contains

! ****************************************************************************** !
  !> Dummy routine which sets thermodynamic factor matrix to diagonal matrix
  subroutine mus_calc_thermFactor_single( nFields, temp, press, mole_frac,     &
    &                                     therm_factors )
    ! --------------------------------------------------------------------------!
    !> number of fields in mixture
    integer, intent(in) :: nFields
    !> mixture temperature
    real(kind=rk), intent(in) :: temp
    !> mixture pressure
    real(kind=rk), intent(in) :: press
    !> mole fraction of all species of single element
    real(kind=rk), intent(in) :: mole_frac(nFields)
    !> thermodynamic factor matrix
    real(kind=rk), intent(out) :: therm_factors(nFields,nFields)
    ! --------------------------------------------------------------------------!
    integer :: iField
    ! --------------------------------------------------------------------------!

    therm_factors = 0.0_rk
    do iField = 1,nFields
      therm_factors(iField, iField) = 1.0_rk
    end do  

    end subroutine mus_calc_thermFactor_single  
! ****************************************************************************** !
  
! ****************************************************************************** !
  !> Dummy routine which sets diffusivity coeff matrix to diagonal matrix
  subroutine mus_calc_MS_DiffMatrix_single( nFields, temp, press, mole_frac,   &
    &                                     D_ij_out )
    ! --------------------------------------------------------------------------!
    !> number of fields in mixture
    integer, intent(in) :: nFields
    !> mixture temperature
    real(kind=rk), intent(in) :: temp
    !> mixture pressure
    real(kind=rk), intent(in) :: press
    !> mole fraction of all species of single element
    real(kind=rk), intent(in) :: mole_frac(nFields)
    !> thermodynamic factor matrix
    real(kind=rk), intent(out) :: D_ij_out(nFields,nFields)
    ! --------------------------------------------------------------------------!
    D_ij_out = 1.0_rk
  end subroutine mus_calc_MS_DiffMatrix_single
! ****************************************************************************** !

! ****************************************************************************** !
  !> Dummy function to init_enrtl
  function mus_init_eNRTL( filename, nFields ) result(success)
    ! --------------------------------------------------------------------------!
    character(kind=c_char), dimension(*) :: filename
    !> Number of fields defined in the property file
    integer, intent(out) :: nFields
    !> Result, indicating the status of encode
    logical :: success
    ! --------------------------------------------------------------------------!
    write(logUnit(1),*) 'Using eNRTL dummy module'
    success = .true.
    nFields = 0
  end function mus_init_eNRTL
! ****************************************************************************** !

end module mus_eNRTL_module
