! See copyright notice in the COPYRIGHT file.
! ****************************************************************************** !
!> author: Kannan Masilamani
!! This module provides variable to extract when turbulence model is active
?? include 'treelm/source/deriveMacros.inc'
module mus_turbulence_var_module
  use iso_c_binding, only: c_loc, c_ptr, c_f_pointer

  ! include treelm modules
  use env_module,               only: rk, labelLen
  use tem_logging_module,       only: logUnit
  use tem_variable_module,      only: tem_variable_type
  use tem_topology_module,      only: tem_levelOf
  use tem_time_module,          only: tem_time_type
  use treelmesh_module,         only: treelmesh_type
  use tem_varSys_module,        only: tem_varSys_type, tem_varSys_op_type,     &
    &                                 tem_varSys_append_derVar,                &
    &                                 tem_varSys_proc_point,                   &
    &                                 tem_varSys_proc_element,                 &
    &                                 tem_varSys_proc_setParams,               &
    &                                 tem_varSys_proc_getParams,               &
    &                                 tem_varSys_proc_setupIndices,            &
    &                                 tem_varSys_proc_getValOfIndex,           &
    &                                 tem_varSys_setupIndices_dummy,           &
    &                                 tem_varSys_getValOfIndex_dummy,          &
    &                                 tem_varSys_setParams_dummy,              &
    &                                 tem_varSys_getParams_dummy
  use tem_aux_module,           only: tem_abort
  use tem_grow_array_module,    only: grw_labelarray_type, append

  ! include musubi modules
  use mus_scheme_type_module,        only: mus_scheme_type
  use mus_varSys_module,             only: mus_varSys_data_type,             &
    &                                      mus_varSys_solverData_type,       &
    &                                      mus_get_new_solver_ptr,           &
    &                                      mus_deriveVar_forPoint
  use mus_turbulence_module,         only: mus_turbulence_config_type, &
    &                                      mus_turbulence_data_type

  implicit none
  private

  public :: mus_append_turbVar

contains

  ! ************************************************************************** !
  !> subroutine to add variables for turbulence model
  subroutine mus_append_turbVar( varSys, solverData, derVarName, turbConfig )
    ! --------------------------------------------------------------------------
    !> global variable system
    type(tem_varSys_type), intent(inout)  :: varSys

    !> Contains pointer to solver data types
    type(mus_varSys_solverData_type), target, intent(in) :: solverData

    !> array of derive physical variables
    type(grw_labelarray_type), intent(inout) :: derVarName
    
    !> turbulence definition
    type(mus_turbulence_config_type), intent(in) :: turbConfig 
    ! --------------------------------------------------------------------------
    ! number of derive variables
    integer :: addedPos
    logical :: wasAdded
    procedure(tem_varSys_proc_point), pointer :: get_point => NULL()
    procedure(tem_varSys_proc_element), pointer :: get_element => NULL()
    procedure(tem_varSys_proc_setParams), pointer :: set_params => null()
    procedure(tem_varSys_proc_getParams), pointer :: get_params => null()
    procedure(tem_varSys_proc_setupIndices), pointer :: &
      &                                      setup_indices => null()
    procedure(tem_varSys_proc_getValOfIndex), pointer :: &
      &                                       get_valOfIndex => null()
    character(len=labelLen) :: derVarName_loc
    ! ---------------------------------------------------------------------------
    nullify(get_point, get_element, set_params, get_params, setup_indices, &
      &     get_valOfIndex)

    write(logUnit(1),*) 'Appending turbulence variables '  

    ! append turbulent vicosity
    derVarName_loc = 'turb_viscosity'
    call append(derVarName, derVarName_loc)

    ! assign function pointers only for get_element and get_point because
    ! this variable will be only used for tracking
    get_element => access_turbVisc_forElement 
    get_point => mus_deriveVar_forPoint
    ! for other function pointers assign dummy routines
    setup_indices => tem_varSys_setupIndices_dummy 
    get_valOfIndex => tem_varSys_getValOfIndex_dummy
    set_params => tem_varSys_setParams_dummy
    get_params => tem_varSys_getParams_dummy

    ! append variable to varSys
    call tem_varSys_append_derVar(                            &
      &  me             = varSys,                             &
      &  varName        = 'turb_viscosity',                   &
      &  nComponents    = 1,                                  &
      &  method_data    = mus_get_new_solver_ptr(solverData), &
      &  get_point      = get_point,                          &
      &  get_element    = get_element,                        &
      &  set_params     = set_params,                         &
      &  get_params     = get_params,                         &
      &  setup_indices  = setup_indices,                      &
      &  get_valOfIndex = get_valOfIndex,                     &
      &  pos            = addedPos,                           &
      &  wasAdded       = wasAdded                            )

    if (wasAdded) then
      write(logUnit(10),*) ' Appended variable: turb_viscosity'
    else if (addedpos < 1) then
      call tem_abort('Error: variable turb_viscosity' &
        &         // ' is not added to variable system')
    end if

    if (.not. turbConfig%compSR_fromPDF) then
      derVarName_loc = 'grad_velocity'
      call append(derVarName, derVarName_loc)
      ! assign function pointers only for get_element and get_point because
      ! this variable will be only used for tracking
      get_element => derive_gradU_forElement 
      get_point => mus_deriveVar_forPoint
      ! for other function pointers assign dummy routines
      setup_indices => tem_varSys_setupIndices_dummy 
      get_valOfIndex => tem_varSys_getValOfIndex_dummy
      set_params => tem_varSys_setParams_dummy
      get_params => tem_varSys_getParams_dummy
  
      ! append variable to varSys
      call tem_varSys_append_derVar(                            &
        &  me             = varSys,                             &
        &  varName        = 'grad_velocity',                    &
        &  nComponents    = 9,                                  &
        &  method_data    = mus_get_new_solver_ptr(solverData), &
        &  get_point      = get_point,                          &
        &  get_element    = get_element,                        &
        &  set_params     = set_params,                         &
        &  get_params     = get_params,                         &
        &  setup_indices  = setup_indices,                      &
        &  get_valOfIndex = get_valOfIndex,                     &
        &  pos            = addedPos,                           &
        &  wasAdded       = wasAdded                            )

      if (wasAdded) then
        write(logUnit(10),*) ' Appended variable: grad_velocity'
      else if (addedpos < 1) then
        call tem_abort('Error: variable grad_velocity' &
          &         // ' is not added to variable system')
      end if
    end if  

  end subroutine mus_append_turbVar 
  ! ************************************************************************** !
 

! **************************************************************************** !
  !> This routine returns the turbulent viscosity 
  !!
  !! The interface has to comply to the abstract interface
  !! [[tem_varSys_module:tem_varSys_proc_element]].
?? copy :: get_element_headtxt(access_turbVisc_forElement)
    ! --------------------------------------------------------------------------
    integer :: statePos, iElem, iComp, iLevel
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    ! --------------------------------------------------------------------------
    call C_F_POINTER( fun%method_Data, fPtr )
    scheme => fPtr%solverData%scheme

    ! res is always AOS layout
    res = 0.0_rk
    do iElem = 1, nElems
      ! if state array is defined level wise then use levelPointer(pos)
      ! to access state array
      statePos = fPtr%solverData%geometry%levelPointer( elemPos(iElem) )
      iLevel = tem_levelOf( tree%treeID( elemPos(iElem) ) )
      res( iElem ) = scheme%turbulence%dataOnLvl( iLevel )%visc( statePos )
    end do !iElem

  end subroutine access_turbVisc_forElement
! ****************************************************************************** !

! **************************************************************************** !
  !> This routine returns the velocity gradient 
  !!
  !! The interface has to comply to the abstract interface
  !! [[tem_varSys_module:tem_varSys_proc_element]].
?? copy :: get_element_headtxt(derive_gradU_forElement)
    ! --------------------------------------------------------------------------
    integer :: statePos, iElem, iComp, iLevel
    type(mus_varSys_data_type), pointer :: fPtr
    type(mus_scheme_type), pointer :: scheme
    real(kind=rk) :: gradU(3,3)
    integer :: velPos(3), nAuxScalars, elemOff
    ! --------------------------------------------------------------------------
    call C_F_POINTER( fun%method_Data, fPtr )
    scheme => fPtr%solverData%scheme
    velPos = varSys%method%val(scheme%derVarPos(1)%velocity)%auxField_varPos(1:3)
    nAuxScalars = varSys%nAuxScalars

    ! res is always AOS layout
    res = 0.0_rk
    do iElem = 1, nElems
      ! if state array is defined level wise then use levelPointer(pos)
      ! to access state array
      statePos = fPtr%solverData%geometry%levelPointer( elemPos(iElem) )
      iLevel = tem_levelOf( tree%treeID( elemPos(iElem) ) )

      ! compute velocity gradient from velocity field stored in turbData
      gradU = getGradU( iElem    = statePos,                            &
        &               turbData = scheme%turbulence%dataOnLvl(iLevel), &
        &               auxField = scheme%auxField(iLevel)%val(:)       ) 

      elemOff= (iElem-1)*9
      res( elemOff + 1 ) = gradU(1,1) !dudx
      res( elemOff + 2 ) = gradU(1,2) !dudy
      res( elemOff + 3 ) = gradU(1,3) !dudz
      res( elemOff + 4 ) = gradU(2,1) !dvdx
      res( elemOff + 5 ) = gradU(2,2) !dvdy
      res( elemOff + 6 ) = gradU(2,3) !dvdz
      res( elemOff + 7 ) = gradU(3,1) !dwdx
      res( elemOff + 8 ) = gradU(3,2) !dwdy
      res( elemOff + 9 ) = gradU(3,3) !dwdz
    end do !iElem

    contains

    ! ************************************************************************ !
    !> This function computes gradient of velocity from gradient and veleocity 
    !! data.
    !! Gradient is computed using central difference.
    !! if an element has an boundary then neighbor refers to current element
    !! then forward difference is used
    pure function getGradU( iElem, turbData, auxField) result(gradU)
      ! ------------------------------------------------------------------------
      !> element index to compute velocity gradient
      integer, intent(in) :: iElem
      !> turbulence data
      type(mus_turbulence_data_type), intent(in) :: turbData
      !> auxField
      real(kind=rk), intent(in) :: auxField(:)
      !> output: gradient of velocity
      real(kind=rk) :: gradU(3, 3)
      ! ------------------------------------------------------------------------
      integer :: iRow, iCol
      integer :: leftNgh, rightNgh
      real(kind=rk) :: FDcoeff, leftVel(3), rightVel(3)
      ! ------------------------------------------------------------------------
      gradU = 0.0_rk
      do iCol = 1, 3 
        leftNgh = turbData%gradData%neighPos(iElem, iCol, 1)
        rightNgh = turbData%gradData%neighPos(iElem, iCol ,2)
        ! left and right velocity for each direction
        leftvel(1) = auxField( (leftNgh-1)*nAuxScalars+velPos(1) )
        leftvel(2) = auxField( (leftNgh-1)*nAuxScalars+velPos(2) )
        leftvel(3) = auxField( (leftNgh-1)*nAuxScalars+velPos(3) )

        rightvel(1) = auxField( (rightNgh-1)*nAuxScalars+velPos(1) )
        rightvel(2) = auxField( (rightNgh-1)*nAuxScalars+velPos(2) )
        rightvel(3) = auxField( (rightNgh-1)*nAuxScalars+velPos(3) )

        ! finite difference coefficients
        FDcoeff  = turbData%gradData%FDcoeff(iElem, iCol)
        do iRow = 1, 3
          gradU(iRow, iCol) = (rightVel(iRow)-leftVel(iRow))*FDcoeff
        end do
      end do  
    end function getGradU
    ! ************************************************************************ !

  end subroutine derive_gradU_forElement
! ****************************************************************************** !

end module mus_turbulence_var_module

