-- Musubi configuration file. 
require 'seeder'

--flow parameters
nu_phy = 1e-3 --m^2/s
rho0_p = 1.0
rho0_L = 1.0
cs2 = 1.0/3.0

Re = 20
u_mean_phy = Re*nu_phy/height
u_max_phy = 3.0*u_mean_phy/2.0

ref_press = 1.0
--press_drop = 1e-4
press_drop = 8*u_max_phy*rho0_p*nu_phy*length/height^2
press_grad = press_drop/length

function u_inflow(x,y,z) 
  return (0.5/(rho0_p*nu_phy))*press_grad*y*(height-y)
  --return (0.5/(rho0_p*nu_phy))*press_grad*(height*0.5-y)*(height*0.5+y)
end

u_max_phy = u_inflow(0.0,height/2.0,0.0)
u_mean_phy = 2.0*u_max_phy/3.0

Re = u_mean_phy*height/nu_phy

scaling = 'acoustic'

--scaling = 'diffusive'
if scaling == 'acoustic' then
--acoustic scaling
  u_max_L = 0.05
  dt = u_max_L*dx/u_max_phy
  u_mean_L = 2.0*u_max_L/3.0
  nu_L = nu_phy*dt/dx^2.
  omega = 1.0/(3.0*nu_L+0.5)
else
--diffusive scaling
  omega = 1.8
  nu_L = (1.0/omega-0.5)/3.0
  dt = nu_L*dx^2/nu_phy
  u_mean_L = u_mean_phy*dt/dx
  u_max_L = 3.0*u_mean_L/2.0
end

-- Analytical solutions
function wssA(x, y, z)
  return nu_phy * 2 * math.abs(y-height*0.5) * u_max_phy / (height*0.5)^2.0
end
function strainRateA(x, y, z)
  return -1*(y) * u_max_phy / R / R
end
function pressA(x,y,z)
  return ref_press+press_drop*(length-x)/length
end

inletBC = 'inlet_ubb'
outletBC = 'outlet_expol'
wallBC = 'wall'


-- Simulation name
simulation_name = 'channel'
mesh = 'mesh/' -- Mesh information
printRuntimeInfo = false
control_routine = 'fast'
-- Time step settigs
tmax_p = 100000 --s
tmax =  tmax_p/dt    -- total iteration number
interval_p = 1 -- s
interval = interval_p/dt
sim_control = {
  time_control = { 
    max = {sim=tmax_p,iter=tmax}, 
    interval = {sim=interval_p,iter=interval}
   },
  abort_criteria = {
    stop_file = 'stop',
    steady_state = true
  }
}

estart = {
      ead = 'restart/channel2D_lastHeader.lua',
      write = 'restart/',
}

fluid = { 
           omega = omega,
           rho0 = rho0_p }

interpolation_method = 'linear'                     

-- Initial condition 
initial_condition = { 
                      pressure = ref_press, 
                      velocityX = u_inflow,
                      velocityY = 0.0,
                      velocityZ = 0.0 }

identify = {label='2D',layout='d3q19', relaxation='bgk',
            kind = 'lbm_incomp'
            }

-- needed to dump variable in physical unit
physics = { dt = dt, rho0 = rho0_p }            

-- Boundary conditions
boundary_condition = {  
  {
    label = 'west', 
    kind = inletBC,
    velocityX = u_inflow,
    velocityY = 0.0, velocityZ = 0.0
  },  
  {
    label = 'east',
    kind = outletBC,
    pressure = ref_press
  }, 
  {   
    label = 'north', 
    kind = wallBC,
   fac = 1.0 
  },
  {
    label = 'south', 
    kind = wallBC,
   fac = 1.0 
  },
 }

variable = {
  { name='vel_an', ncomponents=1, vartype = 'st_fun', st_fun=u_inflow },
  { name='wss_an', ncomponents=1, vartype = 'st_fun', st_fun=wssA },
  { name='press_an', ncomponents=1, vartype = 'st_fun', st_fun=pressA },
  { name='diff_press', ncomponents=1, vartype = 'operation', 
    operation = {kind='difference', input_varname={'pressure_phy','press_an'} }
  },
  { name='diff_vel', ncomponents=1, vartype = 'operation', 
    operation = {kind='difference', input_varname={'vel_mag_phy','vel_an'} }
  },
  { name='diff_wss', ncomponents=1, vartype = 'operation', 
    operation = {kind='difference', input_varname={'wss_phy','wss_an'} }
  },
}

-- Tracking              
tracking = {
{
  label = 'probe', 
  folder = 'tracking/',
 variable = {'pressure_phy','velocity_phy'}, 
 shape = {kind = 'canoND', object = {origin ={length*0.5,height*0.5,zpos} } },
 time_control = {min= {iter= 1}, max = {iter= tmax}, interval = {iter= 10}},
  format = 'ascii'      
},
{
  label = 'hvs', 
  folder = 'tracking/',
  variable = {'pressure_phy','velocity_phy'}, 
  shape = {kind = 'all'},
  time_control = {min= tmax_p, max = tmax_p, interval = tmax_p},
  format = 'harvester'      
},
{
  label = 'pressInlet', 
  folder = 'tracking/',
 variable = {'pressure_phy'}, 
 shape = {kind = 'canoND', object = {origin ={0.0,0.0,zpos},
                                     vec = {0.0,height,0.0},
                                     segments = nHeight+2} },
 time_control = {min = {iter= tmax}, max = {iter= tmax}, interval = {iter= tmax+1}},
  format = 'asciiSpatial'      
},
{
  label = 'pressOutlet', 
  folder = 'tracking/',
 variable = {'pressure_phy'}, 
 shape = {kind = 'canoND', object = {origin ={length,0.0,zpos},
                                     vec = {0.0,height,0.0},
                                     segments = nHeight+2} },
 time_control = {min = {iter= tmax}, max = {iter= tmax}, interval = {iter= tmax+1}},
  format = 'asciiSpatial'      
},
{
  label = 'press_line', 
  folder = 'tracking/',
 variable = {'pressure_phy','press_an','diff_press'}, 
 shape = {kind = 'canoND', object = {origin ={0.0,height*0.5,zpos},
                                     vec = {length,0.0,0.0},
                                     segments = nLength+2} },
 time_control = {min = {iter= tmax}, max = {iter= tmax}, interval = {iter= tmax+1}},
  format = 'asciiSpatial'      
},
{
  label = 'vel_centerLine', 
  folder = 'tracking/',
 variable = {'velocity_phy'}, 
 shape = {kind = 'canoND', object = {origin ={0.0,height*0.5,zpos},
                                     vec = {length,0.0,0.0},
                                     segments = nLength+2} },
 time_control = {min = {iter= tmax}, max = {iter= tmax}, interval = {iter= tmax+1}},
  format = 'asciiSpatial'      
},
{ label = 'press_l2norm',
  variable = {
    'diff_press', 'press_an',
              },
 shape = {kind = 'canoND', object = {origin ={0.0,height*0.5,zpos},
                                     vec = {length,0.0,0.0},
                                     segments = nLength+2} },
  time_control = { min = {iter=tmax}, max = {iter=tmax}, interval = {iter=tmax+1} },
  reduction = {'l2norm', 'l2norm'},
  folder = './tracking/',
  format = 'asciiSpatial',
},
{
  label = 'velProfile', 
  folder = 'tracking/',
 variable = {'velocity_phy'}, 
 shape = {kind = 'canoND', object = {origin ={length*0.5,0.0,zpos},
                                     vec = {0.0,height,0.0},
                                     segments = nHeight+2} },
 time_control = {min = {iter= tmax}, max = {iter= tmax}, interval = {iter= tmax+1}},
  format = 'asciiSpatial'      
},
{ label = 'convergence',
  variable = {'pressure_phy','vel_mag_phy'}, 
  shape = {kind = 'all'},
  time_control = {min = 1, max = tmax_p, interval = 10*dt},
  format='convergence',
  reduction = {'average','average'},
  convergence = {norm='average', nvals = 50, absolute = true,
  condition = { threshold = 1.e-12, operator = '<=' }}
},
{ label = 'wss_spatial',
  variable = { 'wss_phy', 'wss_an', 'diff_wss'},
 shape = {kind = 'canoND', object = {origin ={length*0.5,0.0,zpos},
                                       vec = { 0.0, height, 0.0},
                                       segments = {nHeight+2} 
                              }
  },
  time_control = { min = {iter=tmax}, max = {iter=tmax}, interval = {iter=tmax+1} },
  folder = './tracking/',
  format = 'asciiSpatial',},

{ label = 'wss_l2norm',
  variable = {
    'diff_wss','wss_an',
              },
 shape = {kind = 'canoND', object = {origin ={length*0.5,0.0,zpos},
                                       vec = { 0.0, height, 0.0},
                                       segments = {nHeight+2} 
                              }
  },
  time_control = { min = {iter=tmax}, max = {iter=tmax}, interval = {iter=tmax+1} },
  reduction = {'l2norm', 'l2norm'},
  folder = './tracking/',
  format = 'asciiSpatial',},

{ label = 'vel_spatial',
  variable = {
    'vel_mag_phy',
    'vel_an', 'diff_vel'
              },
 shape = {kind = 'canoND', object = {origin ={length*0.5,0.0,zpos},
                                       vec = { 0.0, height, 0.0},
                                       segments = {nHeight+2} 
                              }
  },
  time_control = { min = {iter=tmax}, max = {iter=tmax}, interval = {iter=tmax+1} },
  folder = './tracking/',
  format = 'asciiSpatial',
},
{ label = 'vel_l2norm',
  variable = {
    'diff_vel','vel_an',
              },
 shape = {kind = 'canoND', object = {origin ={length*0.5,0.0,zpos},
                                       vec = { 0.0, height, 0.0},
                                       segments = {nHeight+2} 
                              }
  },
  time_control = { min = {iter=tmax}, max = {iter=tmax}, interval = {iter=tmax+1} },
  reduction = {'l2norm', 'l2norm'},
  folder = './tracking/',
  format = 'asciiSpatial',
}
}


