-- This is the configuration file for musubi. As it is a simple case, the mesh
-- is defined internally such that seeder is not needed.

--! [Local variables]
-- Flow parameters
-- Kinematic viscosity of the fluid [m^2/s]
--nu_phy = 1e-3
nu_phy = 0.1 -- otherwise it crashes for the low resolution we will use
-- Density of the fluid [kg/m^3]
rho0_phy = 1.0
-- Physical time step (default value if not specified)
--dt = 1.0
dt = 0.5

-- Parameters for gaussian pulse in pressure
-- Background pressure
background = 1.0
center = { 5.0, 5.0, 5.0 }
halfwidth = 1.0
amplitude = 1.20
background = 1.000

-- Mesh information
length = 10.0
refinementLevel = 4
-- Here it is a simple cube, which can be defined internally
mesh = { predefined = 'cube',
         origin = {0.0, 0.0, 0.0},
         length = length,
         refinementLevel = refinementLevel
       }
-- Calculate dx based on length and refinementLevel
dx = length / 2.0^refinementLevel

-------------- Compute physical time step from lattice Mach number ---------------
---- Lattice Mach number
--Ma_lat = 0.1 -- 0.05
---- Lattice speed of sound
--cs_lat = math.sqrt(1.0/3.0)
---- Lattice maximum velocity
--vel_lat = Ma_lat * cs_lat
---- Physical time step computed from physical and lattice velocity
--dt = dx * vel_lat / vel_max_phy
----------------------------------------------------------------------------------

----------------------------- Time settings ------------------------------------
-- Number of iterations used to run the simulation.
-- tmax_iter is also number of lattice iterations
tmax_iter =  50
-- Starting time for tracking output
trac_start = 0
-- Starting time for restart output
rest_start = 0
------------------------- End of time settings ---------------------------------
--! [Global variables]

--------------------------- Musubi configuration -------------------------------
-- Simulation name used in tracking and restart outputs
simulation_name = 'gaussianPulse'
-- Print runtime information like memory usage at the end of the simulation
printRuntimeInfo = false
-- File to write measurement time of simulation for each solver step
timing_file = 'timing.res'
-- Scaling for multilevel simulation
scaling = 'acoustic'
-- Debug outputs to write additional information
NOdebug = {
  logging = {
    level = 1,
    filename = 'dbg',
    root_only = false -- all involved MPI processes writes output
  }
}

--! [Simulation control]
sim_control = {
  time_control = {
    max = { sim = tmax_iter },
    interval = { iter = 5 }
   },
}
--! [Simulation control]

--! [Physics parameters]
-- Required to convert physical unit to lattice unit
physics = {
  dt = dt,
  rho0 = rho0_phy
}
--! [Physics parameters]

--! [Scheme identifier]
identify = {
  layout = 'd3q19',   -- Stencil
  relaxation = 'bgk', -- Collision
  kind = 'fluid_incompressible' -- Physics
}
--! [Scheme identifier]

--! [Fluid]
fluid = {
  kinematic_viscosity = nu_phy
}
--! [Fluid]

--! [Initial condition]
initial_condition = {
  pressure = {
  -- Make use of predefined gausspulse function, which is defined in
  -- treelm/source/tem_spatial_module.f90
  predefined = 'gausspulse',
  center= center,
  halfwidth = halfwidth,
  amplitude = amplitude,
  background = background
  },
  velocityX = 0.0,
  velocityY = 0.0,
  velocityZ = 0.0
}
--! [Initial condition]

--! [Tracking]
tracking = {
  -- Track density, pressure and velocity over line at center.
  {
    label = 'pressAlongLength',
    folder = 'tracking/',
    variable = { 'density_phy', 'pressure_phy', 'velocity_phy'},
    shape = {
      kind = 'canoND',
      object = {
        origin = { 0.0, 0.5*length, 0.5*length },
        vec = { length, 0.0, 0.0 },
      }
     },
    time_control = {
      min = 0,
      max = tmax_iter,
      interval = tmax_iter
    },
    -- ascii format writes variable values and simulation time into a single
    -- file at every tracking interval. It is usally used for point tracking.
    output = {format = 'asciiSpatial'}
  },
  -- Track density, pressure and velocity at the center of the channel over time
  {
    label = 'probeAt1',
    folder = 'tracking/',
    variable = { 'density_phy', 'pressure_phy', 'velocity_phy' },
    shape = {
      kind = 'canoND',
      object = {
        origin = { 1.0, 5.0, 5.0 }
      }
     },
    time_control = {
      min = 0,
      max = tmax_iter,
      interval = {iter=1}
    },
    -- ascii format writes variable values and simulation time into a single
    -- file at every tracking interval. It is usally used for point tracking.
    output = {format = 'ascii'}
  },
}
--! [Tracking]
--
--! [Restart]
restart = {
  NOread = 'restart/gaussianPulse_lastHeader.lua',
  write = 'restart/',
}
--! [Restart]

--------------------------- Musubi configuration -------------------------------
