﻿/*
[momiji music component library]
---------------------------------------------------------------------
Momiji.Core.Winmm.cpp
	
---------------------------------------------------------------------
Copyright (C) 2011 tyiki badwell {miria@users.sourceforge.jp}.

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/gpl-3.0.html>.
---------------------------------------------------------------------
*/
#include "StdAfx.h"

#include "Momiji.Core.Buffer.h"

namespace Momiji {
namespace Core {
namespace Buffer {

	generic<typename DATA_TYPE> 
	BufferPool<DATA_TYPE>::Buffer::Buffer(Allocator^ allocator)
		:_allocator(allocator)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif

		this->Alloc();
	}

	generic<typename DATA_TYPE> 
	BufferPool<DATA_TYPE>::Buffer::~Buffer()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif
		this->!Buffer();
	}

	generic<typename DATA_TYPE> 
	BufferPool<DATA_TYPE>::Buffer::!Buffer()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif
		this->Free();
	}

	generic<typename DATA_TYPE>
	void BufferPool<DATA_TYPE>::Buffer::Alloc()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif

		this->_buffer = this->_allocator(); 
		this->_bufferHandle = InteropServices::GCHandle::Alloc(this->_buffer, InteropServices::GCHandleType::Pinned);
	}

	generic<typename DATA_TYPE>
	void BufferPool<DATA_TYPE>::Buffer::Free()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif
		if (this->_bufferHandle.IsAllocated)
		{
			this->_bufferHandle.Free();
			delete this->_buffer;
		}
	}

	generic<typename DATA_TYPE>
	System::IntPtr BufferPool<DATA_TYPE>::Buffer::GetBufferIntPtr()
	{
		return this->_bufferHandle.AddrOfPinnedObject();
	}

	generic<typename DATA_TYPE>
	DATA_TYPE% BufferPool<DATA_TYPE>::Buffer::GetBuffer()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif
		
		return this->_buffer;
	}






	generic<typename DATA_TYPE>
	BufferPool<DATA_TYPE>::BufferPool(
		System::UInt32 defaultPoolSize,
		Allocator^ allocator
	):	_allocator(allocator),
		_buffers(gcnew System::Collections::Generic::List<Buffer^>()),
		_idleBuffers(gcnew System::Collections::Generic::Dictionary<System::IntPtr, Buffer^>()),
		_busyBuffers(gcnew System::Collections::Generic::Dictionary<System::IntPtr, Buffer^>()),
		_defaultPoolSize(defaultPoolSize)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif

		for (System::UInt32 i = 0; i < this->_defaultPoolSize; i++)
		{
			this->MakeBuffer();
		}
	}

	generic<typename DATA_TYPE> 
	BufferPool<DATA_TYPE>::~BufferPool()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif
		this->!BufferPool();
	}

	generic<typename DATA_TYPE> 
	BufferPool<DATA_TYPE>::!BufferPool()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif
		
		for each (Buffer^ buffer in this->_buffers)
		{
			delete buffer;
		}
		this->_buffers->Clear();
		this->_idleBuffers->Clear();
		this->_busyBuffers->Clear();
	}

	generic<typename DATA_TYPE> 
	BufferPool<DATA_TYPE>::Buffer^ BufferPool<DATA_TYPE>::Get()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif

		System::Threading::Monitor::Enter(this);
		try
		{
			if (this->_idleBuffers->Count == 0)
			{
				//暇なバッファが無ければ、１回分のバッファを新規作成
				this->MakeBuffer();
			}

			System::Collections::Generic::KeyValuePair<System::IntPtr, Buffer^> pair = 
				System::Linq::Enumerable::First(this->_idleBuffers);
			#ifdef _DEBUG
				System::Console::WriteLine("[{0}] buffer ptr[{1}]", __FUNCTION__, pair.Key);
			#endif

			Buffer^ buffer = pair.Value;
			auto intPtr = buffer->GetBufferIntPtr();
			if (!this->_idleBuffers->Remove(intPtr))
			{
				throw gcnew System::Exception("バッファの確保に失敗しました。");
			}
			this->_busyBuffers->Add(intPtr, buffer);

			//#ifdef _DEBUG
				System::Console::WriteLine("[{0}] alloc[{1}] idle[{2}] busy[{3}]", __FUNCTION__, this->_buffers->Count, this->_idleBuffers->Count, this->_busyBuffers->Count);
			//#endif

			return buffer;
		}
		finally
		{
			System::Threading::Monitor::Exit(this);
		}
	}

	generic<typename DATA_TYPE> 
	BufferPool<DATA_TYPE>::Buffer^ BufferPool<DATA_TYPE>::GetBusy(System::IntPtr bufferPtr)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}][{1}]",__FUNCTION__, bufferPtr);
		#endif

		BufferPool<DATA_TYPE>::Buffer^ buffer = nullptr;
		if (!this->_busyBuffers->TryGetValue(bufferPtr, buffer))
		{
			throw gcnew System::Exception("使用済みバッファの取得に失敗しました。");
		}

		return buffer;
	}

	generic<typename DATA_TYPE> 
	void BufferPool<DATA_TYPE>::Release(Buffer^ buffer)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif

		System::Threading::Monitor::Enter(this);
		try
		{
			auto intPtr = buffer->GetBufferIntPtr();
			if (!this->_busyBuffers->Remove(intPtr))
			{
				throw gcnew System::Exception("使用済みバッファの開放に失敗しました。");
			}
			#ifdef _DEBUG
				System::Console::WriteLine("[{0}] buffer ptr[{1}]", __FUNCTION__, intPtr);
			#endif
			this->_idleBuffers->Add(intPtr, buffer);

			//#ifdef _DEBUG
				System::Console::WriteLine("[{0}] alloc[{1}] idle[{2}] busy[{3}]", __FUNCTION__, this->_buffers->Count, this->_idleBuffers->Count, this->_busyBuffers->Count);
			//#endif
		}
		finally
		{
			System::Threading::Monitor::Exit(this);
		}
	}

	generic<typename DATA_TYPE> 
	BufferPool<DATA_TYPE>::Buffer^ BufferPool<DATA_TYPE>::MakeBuffer()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif

		BufferPool<DATA_TYPE>::Buffer^ buffer = gcnew BufferPool<DATA_TYPE>::Buffer(this->_allocator);
		this->_buffers->Add(buffer);
		this->_idleBuffers->Add(buffer->GetBufferIntPtr(), buffer);

		//#ifdef _DEBUG
			System::Console::WriteLine("[{0}] alloc[{1}] idle[{2}] busy[{3}]", __FUNCTION__, this->_buffers->Count, this->_idleBuffers->Count, this->_busyBuffers->Count);
		//#endif

		return buffer;
	}

	generic<typename DATA_TYPE>
	System::Boolean BufferPool<DATA_TYPE>::IsBusy()
	{
		return (this->_busyBuffers->Count > 0);
	}

}
}
}
