﻿/*
[momiji music component library]
---------------------------------------------------------------------
Momiji.Core.DeviceInfo.cpp
	device information.
---------------------------------------------------------------------
Copyright (C) 2011 tyiki badwell {miria@users.sourceforge.jp}.

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/gpl-3.0.html>.
---------------------------------------------------------------------
*/
#include "StdAfx.h"

#include "Momiji.Interop.Setupapi.h"
#include "Momiji.Interop.Kernel32.h"

#include "Momiji.Core.DeviceInfo.h"

namespace Momiji {
namespace Core {
namespace DeviceInfo {


	Devices::Devices(
		Interop::Guiddef::Guid category,
		Interop::Setupapi::DIGCF digcf,
		System::String^ deviceName
	)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}][category:{1}][digcf:{2:F}][{3}]",__FUNCTION__, category, digcf, deviceName);
		#endif

		this->_category = category;

		this->_handle =
			Interop::Setupapi::Function::SetupDiGetClassDevs(
				this->_category,
				deviceName,
				InteropServices::HandleRef(nullptr, System::IntPtr::Zero),
				digcf
			);

		if (_handle->IsInvalid)
		{
			auto error = InteropServices::Marshal::GetHRForLastWin32Error();
			#ifdef _DEBUG
				System::Console::WriteLine("[{0}] error {1:X} {2}",__FUNCTION__, error, InteropServices::Marshal::GetExceptionForHR(error)->ToString());
			#endif
			InteropServices::Marshal::ThrowExceptionForHR(error);
		}
	}

	Devices::~Devices()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif
		this->!Devices();
	}

	Devices::!Devices()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif
		this->_handle->Close();
	}

	Devices::Detail::Detail(
		Devices^ devices,
		System::UInt32 index
	):
		_devices(devices)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif

		this->GetDetail(index);
	}

	Devices::Detail::~Detail()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif
		this->!Detail();
	}

	Devices::Detail::!Detail()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif
	}

	void Devices::Detail::GetDetail(
		System::UInt32 index
	)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}][index: {1}]",__FUNCTION__, index);
		#endif

		auto data = Interop::Setupapi::SpDeviceInterfaceData();
		data.cbSize = safe_cast<System::UInt32>(InteropServices::Marshal::SizeOf(data));
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] data->cbSize {1}",__FUNCTION__, data.cbSize);
		#endif
		if (
			!Interop::Setupapi::Function::SetupDiEnumDeviceInterfaces(
				this->_devices->_handle,
				nullptr,
				this->_devices->_category,
				index,
				data
			)
		)
		{
			auto error = InteropServices::Marshal::GetHRForLastWin32Error();
			#ifdef _DEBUG
				System::Console::WriteLine("[{0}] error [{1}]",__FUNCTION__, InteropServices::Marshal::GetExceptionForHR(error)->ToString());
			#endif
			if (error == 0x80070103) //ERROR_NO_MORE_ITEMS
			{
				this->_noMoreItems = true;
				return;
			}
			else
			{
				InteropServices::Marshal::ThrowExceptionForHR(error);
			}
		}
			
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] data [{1}]", __FUNCTION__, data);
		#endif

		this->_data = data;

		//データサイズ確認
		System::UInt32 reqired = 0;
		
		if (
			!Interop::Setupapi::Function::SetupDiGetDeviceInterfaceDetail(
				this->_devices->_handle,
				this->_data,
				System::IntPtr::Zero,
				0,
				reqired,
				System::IntPtr::Zero
			)
		)
		{
			auto error = InteropServices::Marshal::GetHRForLastWin32Error();
			#ifdef _DEBUG
				System::Console::WriteLine("[{0}] error [{1}]",__FUNCTION__, InteropServices::Marshal::GetExceptionForHR(error)->ToString());
			#endif
			if (error != 0x8007007A)//ERROR_INSUFFICIENT_BUFFER
			{
				InteropServices::Marshal::ThrowExceptionForHR(error);
			}
		}

		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] reqired {1}",__FUNCTION__, reqired);
		#endif
		
		//・・・はしてるものの、結局今は、1024byte固定で読み込んでいる
		auto detail = Interop::Setupapi::SpDeviceInterfaceDetailData();
		detail.cbSize = 8;//safe_cast<System::UInt32>(InteropServices::Marshal::SizeOf(detail->cbSize) + InteropServices::Marshal::SystemDefaultCharSize);

		auto info = Interop::Setupapi::SpDevinfoData();
		info.cbSize = safe_cast<System::UInt32>(InteropServices::Marshal::SizeOf(info));
		
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] detail->cbSize {1}",__FUNCTION__, detail.cbSize);
		#endif

		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] info->cbSize {1}",__FUNCTION__, info.cbSize);
		#endif

		if (
			!Interop::Setupapi::Function::SetupDiGetDeviceInterfaceDetail(
				this->_devices->_handle,
				this->_data,
				detail,
				reqired,
				reqired,
				info
			)
		)
		{
			auto error = InteropServices::Marshal::GetHRForLastWin32Error();
			InteropServices::Marshal::ThrowExceptionForHR(error);
		}

		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] detail [{1}]",__FUNCTION__, detail);
			System::Console::WriteLine("[{0}] info [{1}]", __FUNCTION__, info);
		#endif

		this->_devicePath = detail.DevicePath;
		this->_info = info;
	}

	Interop::Setupapi::SpDeviceInterfaceData^ Devices::Detail::GetAlias(
		Interop::Guiddef::Guid subCategory
	)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}][{1}]",__FUNCTION__, subCategory);
		#endif
		auto data = Interop::Setupapi::SpDeviceInterfaceData();
		data.cbSize = safe_cast<System::UInt32>(InteropServices::Marshal::SizeOf(data));

		if (
			!Interop::Setupapi::Function::SetupDiGetDeviceInterfaceAlias(
				this->_devices->_handle,
				this->_data,
				subCategory,
				data
			)
		)
		{
			#ifdef _DEBUG
				auto error = InteropServices::Marshal::GetHRForLastWin32Error();
				System::Console::WriteLine("[{0}] error [{1}]",__FUNCTION__, InteropServices::Marshal::GetExceptionForHR(error)->ToString());
			#endif
			return nullptr;
		}

		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] alias [{1}]", __FUNCTION__, data);
		#endif
		return data;
	}

	System::String^ Devices::Detail::GetDeviceRegistryProperty(
		Interop::Setupapi::SPDRP spdrp
	)
	{
		#ifdef _DEBUG
		//	System::Console::WriteLine("[{0}][{1}]",__FUNCTION__, spdrp);
		#endif
		System::UInt32 regDataType = 0;
		System::UInt32 reqired = 0;
		if (
			!Interop::Setupapi::Function::SetupDiGetDeviceRegistryProperty(
				this->_devices->_handle,
				this->_info,
				spdrp,
				regDataType,
				nullptr,
				0,
				reqired
			)
		)
		{
			auto error = InteropServices::Marshal::GetHRForLastWin32Error();
			#ifdef _DEBUG
			//	System::Console::WriteLine("[{0}] error {1:X}",__FUNCTION__, error);
			#endif
			if (error != 0x8007007A)//ERROR_INSUFFICIENT_BUFFER
			{
				#ifdef _DEBUG
					System::Console::WriteLine("[{0}] {1} [{2}]",__FUNCTION__, spdrp, InteropServices::Marshal::GetExceptionForHR(error)->ToString());
				#endif
				return nullptr;
			}
		}
		#ifdef _DEBUG
		//	System::Console::WriteLine("[{0}] reqired {1}",__FUNCTION__, reqired);
		#endif
		auto buf = gcnew System::Text::StringBuilder(safe_cast<System::Int32>(reqired));

		if (
			!Interop::Setupapi::Function::SetupDiGetDeviceRegistryProperty(
				this->_devices->_handle,
				this->_info,
				spdrp,
				regDataType,
				buf,
				reqired,
				reqired
			)
		)
		{
			auto error = InteropServices::Marshal::GetHRForLastWin32Error();
			#ifdef _DEBUG
			//	System::Console::WriteLine("[{0}] error {1:X}",__FUNCTION__, error);
			#endif
			InteropServices::Marshal::ThrowExceptionForHR(error);
		}

		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] {1} [{2}]",__FUNCTION__, spdrp, buf->ToString());
		#endif
		return buf->ToString();
	}

	System::String^ Devices::Detail::GetDeviceRegistryProperty(
		System::String^ name
	)
	{
		#ifdef _DEBUG
		//	System::Console::WriteLine("[{0}][{1}]",__FUNCTION__, name);
		#endif
		Interop::Setupapi::Function::RegKey^ regKey;
		
		{
			regKey =
				Interop::Setupapi::Function::SetupDiOpenDeviceInterfaceRegKey(
					this->_devices->_handle,
					this->_data,
					0,
					Interop::Kernel32::ACCESS_TYPES::KEY_READ
				);
			if (regKey->IsInvalid)
			{
				#ifdef _DEBUG
					auto error = InteropServices::Marshal::GetHRForLastWin32Error();
					System::Console::WriteLine("[{0}] error {1:X}",__FUNCTION__, InteropServices::Marshal::GetExceptionForHR(error)->ToString());
				#endif
				return nullptr;
			}
		}

		try
		{
			System::UInt32 type = 0;
			System::UInt32 size = 1024;
			auto data = InteropServices::Marshal::AllocHGlobal(size);
			try
			{
				System::UInt32 error =
					Interop::Setupapi::Function::RegQueryValueEx(
						regKey,
						name,
						System::IntPtr::Zero,
						type,
						data,
						size
					);
				#ifdef _DEBUG
				//	System::Console::WriteLine("[{0}] error {1:X}",__FUNCTION__, error);
				#endif
				if (error != 0)
				{
					#ifdef _DEBUG
						System::Console::WriteLine("[{0}] msg {1}",__FUNCTION__, InteropServices::Marshal::GetExceptionForHR(error)->ToString());
					#endif
					return nullptr;
				}

				auto result = InteropServices::Marshal::PtrToStringAuto(data);
				#ifdef _DEBUG
					System::Console::WriteLine("[{0}] {1} [{2}]",__FUNCTION__, name, result);
				#endif
				return result;
			}
			finally
			{
				InteropServices::Marshal::FreeHGlobal(data);
			}
		}
		finally
		{
			delete regKey;
		}
	}



	Devices::DetailEnum::DetailEnum(
		Devices^ devices
	):
		_devices(devices),
		_index(0) //一般的な用法からは外して、MoveNext後に+1している
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif
	}

	System::Boolean Devices::DetailEnum::MoveNext()
	{
		auto detail =
			gcnew Detail(
				this->_devices,
				this->_index++
			);

		this->_detail =
			(detail->IsNoMoreItems)
				? nullptr
				: detail
				;
		return !detail->IsNoMoreItems;
	}

	void Devices::DetailEnum::Reset()
	{
		this->_detail = nullptr;
		this->_index = 0;
	}


}
}
}
