﻿/*
[momiji music component library]
---------------------------------------------------------------------
Momiji.Core.Midi.Out.cpp
	midi output component.
---------------------------------------------------------------------
Copyright (C) 2011 tyiki badwell {miria@users.sourceforge.jp}.

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/gpl-3.0.html>.
---------------------------------------------------------------------
*/
#include "StdAfx.h"

#include "Momiji.Interop.Winmm.h"
#include "Momiji.Core.Midi.Out.h"
#include "Momiji.Core.Midi.Data.h"

namespace Momiji {
namespace Core {
namespace Midi {
namespace Out {

	void Device::OnEventHandler(
		System::Object^ sender, 
		Interop::Winmm::DriverCallBack::DriverEventArgs^ args
	)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] [{1}] start", __FUNCTION__, System::Threading::Thread::CurrentThread->GetHashCode());
		#endif

		switch (args->uMsg)
		{
		case Interop::Winmm::DriverCallBack::MM_EXT_WINDOW_MESSAGE::MOM_OPEN	:
			{
				this->DoOpen(args->dw1, args->dw2);
				break;
			}
		case Interop::Winmm::DriverCallBack::MM_EXT_WINDOW_MESSAGE::MOM_CLOSE:
			{
				this->DoClose(args->dw1, args->dw2);
				break;
			}
		case Interop::Winmm::DriverCallBack::MM_EXT_WINDOW_MESSAGE::MOM_DONE	:
			{//Param1 = MIDIHDR
				this->DoDone(args->dw1, args->dw2);
				break;
			}
		}
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] [{1}] end", __FUNCTION__, System::Threading::Thread::CurrentThread->GetHashCode());
		#endif
	}

	void Device::DoOpen(System::IntPtr dwParam1, System::IntPtr dwParam2)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] [{1,8:X}][{2,8:X}]", __FUNCTION__, dwParam1, dwParam2);
		#endif
		this->OnOpen(this, System::EventArgs::Empty);
	}

	void Device::DoClose(System::IntPtr dwParam1, System::IntPtr dwParam2)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] [{1,8:X}][{2,8:X}]", __FUNCTION__, dwParam1, dwParam2);
		#endif
		this->OnClose(this, System::EventArgs::Empty);
	}

	void Device::DoDone(System::IntPtr dwParam1, System::IntPtr dwParam2)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] [{1,8:X}][{2,8:X}]", __FUNCTION__, dwParam1, dwParam2);

			auto midiHeader = 
				safe_cast<Interop::Winmm::MidiHeader^>(InteropServices::Marshal::PtrToStructure(dwParam1, Interop::Winmm::MidiHeader::typeid));

			System::Console::WriteLine("[{0}] {1}", __FUNCTION__, midiHeader);
			System::Console::Write("[{0}] ", __FUNCTION__);
			for (System::UInt32 i = 0; i < midiHeader->bufferLength; i++)
			{
				System::Console::Write("[{0,2:X}]", InteropServices::Marshal::ReadByte(midiHeader->data,i));
			}
			System::Console::WriteLine();
		#endif

		this->Unprepare(dwParam1);
	}

	System::UInt32 Device::GetNumDevices()
	{
		return Interop::Winmm::Function::midiOutGetNumDevs();
	}

	Interop::Winmm::MidiOutCapabilities^ Device::GetCapabilities(
		const System::UInt32 deviceID
	)
	{
		auto caps = gcnew Interop::Winmm::MidiOutCapabilities();
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] caps size {1}",__FUNCTION__, InteropServices::Marshal::SizeOf(caps));
		#endif
		auto mmResult = 
			Interop::Winmm::Function::midiOutGetDevCaps(
				deviceID, 
				caps, 
				InteropServices::Marshal::SizeOf(caps)
			);
		if (mmResult != Interop::Winmm::MMRESULT::NOERROR)
		{
			throw gcnew MidiOutException(mmResult);
		}
		return caps;
	}

	Device::Device(
		const System::UInt32 deviceID
	):	_deviceID(deviceID)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] deviceID {1}",__FUNCTION__, this->_deviceID);
		#endif

		this->Open();
	}

	Device::~Device()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif
		this->!Device();
	}

	Device::!Device()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif
		this->Close();
	}

	Interop::Winmm::MidiOutCapabilities^ Device::GetCapabilities()
	{
		return Device::GetCapabilities(this->_deviceID);
	}

	void Device::Open()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif

		this->_callBack = gcnew Core::Winmm::DriverCallBack(true); //イベントは非同期で動かす
		this->_callBack->OnEvent += gcnew System::EventHandler<Interop::Winmm::DriverCallBack::DriverEventArgs^>(this, &Device::OnEventHandler);

		this->_headerPool =
			gcnew Core::Buffer::BufferPool<Interop::Winmm::MidiHeader^>(
				2,		//２回分のバッファを用意
				gcnew Core::Buffer::BufferPool<Interop::Winmm::MidiHeader^>::Allocator(this, &Device::AllocateHeader) 
			);

		this->_bufferPool =
			gcnew Core::Buffer::BufferPool<array<System::Byte>^>(
				2,		//２回分のバッファを用意
				gcnew Core::Buffer::BufferPool<array<System::Byte>^>::Allocator(this, &Device::AllocateBuffer) 
			);

		auto mmResult = 
			Interop::Winmm::Function::midiOutOpen(
				this->_handle,
				this->_deviceID,
				this->_callBack->GetDriverCallBackProc(),
				System::IntPtr::Zero,
				Interop::Winmm::DriverCallBack::TYPE::FUNCTION
			);
		if (mmResult != Interop::Winmm::MMRESULT::NOERROR) {
			throw gcnew MidiOutException(mmResult);
		}

		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] Handle invalid:{1} closed:{2}",__FUNCTION__, this->_handle->IsInvalid, this->_handle->IsClosed);
		#endif
	}

	void Device::Close()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] Handle invalid:{1} closed:{2}",__FUNCTION__, this->_handle->IsInvalid, this->_handle->IsClosed);
		#endif
		if (
			!this->_handle->IsInvalid
		&&	!this->_handle->IsClosed
		)
		{
			{
				auto mmResult = Interop::Winmm::Function::midiOutReset(this->_handle);
				if (mmResult != Interop::Winmm::MMRESULT::NOERROR)
				{
					throw gcnew MidiOutException(mmResult);
				}
				#ifdef _DEBUG
					System::Console::WriteLine("[{0}] midiOutReset OK",__FUNCTION__);
				#endif
			}

			//バッファの開放待ち
			while(this->_headerPool->IsBusy())
			{
				#ifdef _DEBUG
					System::Console::WriteLine("[{0}] wait for unprepare headers ...",__FUNCTION__);
				#endif
				System::Threading::Thread::Sleep(5);
			}

			this->_handle->Close();
		}
		else
		{
			#ifdef _DEBUG
				System::Console::WriteLine("[{0}] openしていない状態なので、無視します。", __FUNCTION__);
			#endif
		}

		if (this->_callBack != nullptr)
		{
			this->_callBack->OnEvent -= gcnew System::EventHandler<Interop::Winmm::DriverCallBack::DriverEventArgs^>(this, &Device::OnEventHandler);
			delete this->_callBack;
		}

		if (this->_headerPool != nullptr)
		{
			delete this->_headerPool;
		}
	}

	void Device::SendShort(
		const System::UInt32 dwMsg
	)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] dwMsg {1,4:X}", __FUNCTION__, dwMsg);
		#endif
		if (
			this->_handle->IsInvalid
		||	this->_handle->IsClosed
		)
		{
			#ifdef _DEBUG
				System::Console::WriteLine("[{0}] openしていない状態なので、無視します。", __FUNCTION__);
			#endif
			return;
		}

		auto mmResult = 
			Interop::Winmm::Function::midiOutShortMsg(
				this->_handle,
				dwMsg
			);

		if (mmResult != Interop::Winmm::MMRESULT::NOERROR)
		{
			throw gcnew MidiOutException(mmResult);
		}
	}

	void Device::SendLong(
		array<System::Byte>^ data, System::UInt32 useSize
	)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif
		#ifdef _DEBUG
			System::Console::Write("[{0}] ", __FUNCTION__);
			for (auto i = 0; i < data->Length; i++)
			{
				System::Console::Write("[{0,2:X}]", data[i]);
			}
			System::Console::WriteLine();
		#endif

		if (
			this->_handle->IsInvalid
		||	this->_handle->IsClosed
		)
		{
			#ifdef _DEBUG
				System::Console::WriteLine("[{0}] openしていない状態なので、無視します。", __FUNCTION__);
			#endif
			return;
		}

		System::IntPtr headerPtr = this->Prepare(data, useSize);

		auto mmResult = 
			Interop::Winmm::Function::midiOutLongMsg(
				this->_handle,
				headerPtr,
				InteropServices::Marshal::SizeOf(Interop::Winmm::MidiHeader::typeid)
			);
		if (mmResult != Interop::Winmm::MMRESULT::NOERROR)
		{
			this->Unprepare(headerPtr);
			throw gcnew MidiOutException(mmResult);
		}
	}

	System::IntPtr Device::Prepare(array<System::Byte>^ data, System::UInt32 useSize)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif

		auto header = this->_headerPool->Get();
		auto buffer = this->_bufferPool->Get();

		{
			System::Array::Copy(data, buffer->GetBuffer(), static_cast<System::Int32>(useSize));

			auto midiHeader = header->GetBuffer();
			midiHeader->bufferLength = useSize;
			midiHeader->data = buffer->GetBufferIntPtr();

			#ifdef _DEBUG
				{
					System::Console::WriteLine("[{0}] midiInPrepareHeader before", __FUNCTION__);
					System::Console::WriteLine("[{0}] {1}", __FUNCTION__, midiHeader);
				}
			#endif
		}

		auto mmResult = 
			Interop::Winmm::Function::midiOutPrepareHeader(
				this->_handle,
				header->GetBufferIntPtr(),
				InteropServices::Marshal::SizeOf(Interop::Winmm::MidiHeader::typeid)
			);
		if (mmResult != Interop::Winmm::MMRESULT::NOERROR)
		{
			this->_headerPool->Release(header);
			this->_bufferPool->Release(buffer);
			throw gcnew MidiOutException(mmResult);
		}
		#ifdef _DEBUG
			{
				auto midiHeader = header->GetBuffer();
				System::Console::WriteLine("[{0}] midiOutPrepareHeader OK", __FUNCTION__);
				System::Console::WriteLine("[{0}] {1}", __FUNCTION__, midiHeader);
			}
		#endif

		return header->GetBufferIntPtr();
	}

	void Device::Unprepare(System::IntPtr headerPtr)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif

		auto header = this->_headerPool->GetBusy(headerPtr);
		
		#ifdef _DEBUG
			{
				auto midiHeader = header->GetBuffer();
				System::Console::WriteLine("[{0}] midiOutUnprepareHeader before", __FUNCTION__);
				System::Console::WriteLine("[{0}] {1}", __FUNCTION__, midiHeader);
			}
		#endif

		auto mmResult = 
			Interop::Winmm::Function::midiOutUnprepareHeader(
				this->_handle,
				headerPtr,
				InteropServices::Marshal::SizeOf(Interop::Winmm::MidiHeader::typeid)
			);
		if (mmResult != Interop::Winmm::MMRESULT::NOERROR)
		{
			throw gcnew MidiOutException(mmResult);
		}

		auto bufferPtr = header->GetBuffer()->data;
		this->_headerPool->Release(header);

		auto buffer = this->_bufferPool->GetBusy(bufferPtr);
		this->_bufferPool->Release(buffer);

		#ifdef _DEBUG
			{
				auto midiHeader = header->GetBuffer();
				System::Console::WriteLine("[{0}] midiOutUnprepareHeader after", __FUNCTION__);
				System::Console::WriteLine("[{0}] {1}", __FUNCTION__, midiHeader);
			}
		#endif
	}

	Interop::Winmm::MidiHeader^ Device::AllocateHeader()
	{
		return gcnew Interop::Winmm::MidiHeader();
	}

	array<System::Byte>^ Device::AllocateBuffer()
	{
		return gcnew array<System::Byte>(256); //SMFとして最大長 + 1
	}

	System::String^ MidiOutException::Initialize()
	{
		auto errorMessage = System::String::Empty;
		auto buf = gcnew System::Text::StringBuilder(256);

		auto r = 
			Interop::Winmm::Function::midiOutGetErrorText(
				this->_mmResult, 
				buf, 
				buf->Capacity
			);
		if (r == Interop::Winmm::MMRESULT::NOERROR)
		{
			errorMessage = buf->ToString();
		}
		else
		{
			errorMessage = "不明なエラー";
		}

		return errorMessage + "[" + this->_mmResult.ToString("D") + "]";
	}








	Devices::Devices()
		: _outs(gcnew System::Collections::Generic::List<Device^>)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif
	}

	Devices::~Devices()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif
		this->!Devices();
	}

	Devices::!Devices()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif
		for each (Device^ d in this->_outs)
		{
			delete d;
		}
		this->_outs->Clear();
	}

	void Devices::AddPort(System::UInt32 deviceID)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif
		this->_outs->Add(gcnew Device(deviceID));
	}

	void Devices::Send(Core::IData^ data)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif

		Core::Midi::MidiData^ midiData = dynamic_cast<Core::Midi::MidiData^>(data);
		if (midiData == nullptr)
		{
			return;
		}

		System::UInt16 p = midiData->port;
		if (p >= this->_outs->Count)
		{
			p = 0;
		}

		Device^ d = this->_outs[p];

		Core::Midi::ShortData^ shortData = dynamic_cast<Core::Midi::ShortData^>(midiData);
		if (shortData != nullptr)
		{
			d->SendShort(shortData->shortData);
			return;
		}

		Core::Midi::LongData^ longData = dynamic_cast<Core::Midi::LongData^>(midiData);
		if (longData != nullptr)
		{
			d->SendLong(longData->longData, longData->longData->Length);
			return;
		}

	}
}
}
}
}
