﻿/*
[momiji music component library]
---------------------------------------------------------------------
Momiji.Core.Vst.Host.h
	vst host component.
---------------------------------------------------------------------
Copyright (C) 2011 tyiki badwell {miria@users.sourceforge.jp}.

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/gpl-3.0.html>.
---------------------------------------------------------------------
*/
#include "StdAfx.h"

#include "Momiji.Core.Vst.Host.Master.h"

namespace Momiji{
namespace Core {
namespace Vst {
namespace Host {

	Master::AudioMasterCallBack::AudioMasterCallBack()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif

		this->_callBack = InteropServices::GCHandle::Alloc(gcnew Interop::Vst::AudioMasterCallBack::Delegate(this, &AudioMasterCallBack::AudioMasterCallBackProc));
	}

	Master::AudioMasterCallBack::~AudioMasterCallBack()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif
		this->!AudioMasterCallBack();
	}

	Master::AudioMasterCallBack::!AudioMasterCallBack()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif
		if (this->_callBack.IsAllocated)
		{
			this->_callBack.Free();
		}
	}

	Interop::Vst::AudioMasterCallBack::Delegate^ Master::AudioMasterCallBack::GetAudioMasterCallBackProc()
	{
		return safe_cast<Interop::Vst::AudioMasterCallBack::Delegate^>(this->_callBack.Target);
	}

	System::IntPtr Master::AudioMasterCallBack::AudioMasterCallBackProc(
		System::IntPtr/*AEffect^*/		effect,
		Interop::Vst::AudioMasterOpcodes	opcode,
		System::Int32		index,
		System::IntPtr		value,
		System::IntPtr		ptr,
		System::Single		opt
	)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] [{1,8:X}][{2:f}][{3,8:X}][{4,8:X}][{5,8:X}][{6}]", __FUNCTION__, effect, opcode, index, value, ptr, opt);
		#endif
		auto params = gcnew Interop::Vst::AudioMasterCallBack::AudioMasterEventArgs();
		try
		{
			params->effect = effect;
			params->opcode = opcode;
			params->index = index;
			params->value = value;
			params->ptr = ptr;
			params->opt = opt;

			this->DoEvent(params);
		}
		catch(System::Exception^ e)
		{
			#ifdef _DEBUG
				System::Console::WriteLine("[{0}] コールバック中のエラー[{1}]", __FUNCTION__, e->ToString());
			#endif
		}
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] OUT", __FUNCTION__);
		#endif

		return params->returnValue;
	}

	void Master::AudioMasterCallBack::DoEvent(
		System::Object^	stateInfo
	)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] [{1}] start", __FUNCTION__, System::Threading::Thread::CurrentThread->GetHashCode());
		#endif

		auto params = safe_cast<Interop::Vst::AudioMasterCallBack::AudioMasterEventArgs^>(stateInfo);
		try
		{
			this->OnEvent(this, params);
		}
		catch(System::Exception^ e)
		{
			#ifdef _DEBUG
				System::Console::WriteLine("[{0}] コールバック中のエラー[{1}]", __FUNCTION__, e->ToString());
			#endif
		}

		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] [{1}] end", __FUNCTION__, System::Threading::Thread::CurrentThread->GetHashCode());
		#endif
	}

	void Master::OnEventHandler(
		System::Object^ sender, 
		Interop::Vst::AudioMasterCallBack::AudioMasterEventArgs^ args
	)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] [{1}] start", __FUNCTION__, System::Threading::Thread::CurrentThread->GetHashCode());
		#endif

		switch(args->opcode)
		{
		case Interop::Vst::AudioMasterOpcodes::audioMasterVersion:
			{
				args->returnValue = System::IntPtr(2400);
				break;
			}
		case Interop::Vst::AudioMasterOpcodes::audioMasterGetTime:
			{
				args->returnValue = this->GetTimeInfo(0);
				break;
			}
		}
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] [{1}] end", __FUNCTION__, System::Threading::Thread::CurrentThread->GetHashCode());
		#endif
	}


	Master::Master(
		System::String^ library
	): _library(library)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] library {1}",__FUNCTION__, this->_library);
		#endif

		this->_timeInfo = gcnew Interop::Vst::VstTimeInfo();
		this->_timeInfoHandle = InteropServices::GCHandle::Alloc(this->_timeInfo, InteropServices::GCHandleType::Pinned);

		this->_timeInfo->samplePos = 0.0;
		this->_timeInfo->sampleRate = 0;
		this->_timeInfo->nanoSeconds = 0.0;
		this->_timeInfo->ppqPos = 0.0;
		this->_timeInfo->tempo = 240.0;
		this->_timeInfo->barStartPos = 0.0;
		this->_timeInfo->cycleStartPos = 0.0;
		this->_timeInfo->cycleEndPos = 0.0;
		this->_timeInfo->timeSigNumerator = 4;
		this->_timeInfo->timeSigDenominator = 4;
		this->_timeInfo->smpteOffset = 0;
		this->_timeInfo->smpteFrameRate = 1;
		this->_timeInfo->samplesToNextClock = 0;
		this->_timeInfo->flags = Interop::Vst::VstTimeInfo::VstTimeInfoFlags::kVstTempoValid;

		this->Open();
	}

	Master::~Master()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif
		this->!Master();
	}

	Master::!Master()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif

		this->Suspend();
		this->Close();

		if (this->_timeInfoHandle.IsAllocated)
		{
			this->_timeInfoHandle.Free();
			delete this->_timeInfo;
		}
	}

	void Master::Open()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif

		this->_callBack = gcnew AudioMasterCallBack();
		this->_callBack->OnEvent += gcnew System::EventHandler<Interop::Vst::AudioMasterCallBack::AudioMasterEventArgs^>(this, &Master::OnEventHandler);

		this->_dll = Interop::Kernel32::Function::LoadLibrary(this->_library);
		if (this->_dll->IsInvalid)
		{
			auto error = InteropServices::Marshal::GetHRForLastWin32Error();
			System::Console::WriteLine("[{0}] error {1}",__FUNCTION__, error);
			InteropServices::Marshal::ThrowExceptionForHR(error);
		}
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] Handle invalid:{1} closed:{2}",__FUNCTION__, this->_dll->IsInvalid, this->_dll->IsClosed);
		#endif

		auto proc = Interop::Kernel32::Function::GetProcAddress(this->_dll, "VSTPluginMain");
		if (proc == System::IntPtr::Zero)
		{
			proc = Interop::Kernel32::Function::GetProcAddress(this->_dll, "main");
		}

		if (proc == System::IntPtr::Zero)
		{
			auto error = InteropServices::Marshal::GetHRForLastWin32Error();
			System::Console::WriteLine("[{0}] error {1}",__FUNCTION__, error);
			InteropServices::Marshal::ThrowExceptionForHR(error);
		}

		auto vstPluginMain = 
			safe_cast<Interop::Vst::VSTPluginMain^>(InteropServices::Marshal::GetDelegateForFunctionPointer(proc, Interop::Vst::VSTPluginMain::typeid));

		this->_aeffectPtr = vstPluginMain(this->_callBack->GetAudioMasterCallBackProc());
		this->_aeffect = 
			safe_cast<Interop::Vst::AEffect^>(InteropServices::Marshal::PtrToStructure(this->_aeffectPtr, Interop::Vst::AEffect::typeid));

		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] magic:{1} ",__FUNCTION__, this->_aeffect->magic);
			System::Console::WriteLine("[{0}] dispatcher:{1} ",__FUNCTION__, this->_aeffect->dispatcher);
			System::Console::WriteLine("[{0}] processDeprecated:{1} ",__FUNCTION__, this->_aeffect->processDeprecated);
			System::Console::WriteLine("[{0}] setParameter:{1} ",__FUNCTION__, this->_aeffect->setParameter);
			System::Console::WriteLine("[{0}] getParameter:{1} ",__FUNCTION__, this->_aeffect->getParameter);

			System::Console::WriteLine("[{0}] numPrograms:{1} ",__FUNCTION__, this->_aeffect->numPrograms);
			System::Console::WriteLine("[{0}] numParams:{1} ",__FUNCTION__, this->_aeffect->numParams);
			System::Console::WriteLine("[{0}] numInputs:{1} ",__FUNCTION__, this->_aeffect->numInputs);
			System::Console::WriteLine("[{0}] numOutputs:{1} ",__FUNCTION__, this->_aeffect->numOutputs);
			System::Console::WriteLine("[{0}] flags:{1:f} ",__FUNCTION__, this->_aeffect->flags);

			System::Console::WriteLine("[{0}] resvd1:{1} ",__FUNCTION__, this->_aeffect->resvd1);
			System::Console::WriteLine("[{0}] resvd2:{1} ",__FUNCTION__, this->_aeffect->resvd2);

			System::Console::WriteLine("[{0}] initialDelay:{1} ",__FUNCTION__, this->_aeffect->initialDelay);

			System::Console::WriteLine("[{0}] realQualitiesDeprecated:{1} ",__FUNCTION__, this->_aeffect->realQualitiesDeprecated);
			System::Console::WriteLine("[{0}] offQualitiesDeprecated:{1} ",__FUNCTION__, this->_aeffect->offQualitiesDeprecated);
			System::Console::WriteLine("[{0}] ioRatioDeprecated:{1} ",__FUNCTION__, this->_aeffect->ioRatioDeprecated);
			System::Console::WriteLine("[{0}] object:{1} ",__FUNCTION__, this->_aeffect->object);
			System::Console::WriteLine("[{0}] user:{1} ",__FUNCTION__, this->_aeffect->user);

			System::Console::WriteLine("[{0}] uniqueID:{1} ",__FUNCTION__, this->_aeffect->uniqueID);
			System::Console::WriteLine("[{0}] version:{1} ",__FUNCTION__, this->_aeffect->version);

			System::Console::WriteLine("[{0}] processReplacing:{1} ",__FUNCTION__, this->_aeffect->processReplacing);
			System::Console::WriteLine("[{0}] processDoubleReplacing:{1} ",__FUNCTION__, this->_aeffect->processDoubleReplacing);
			System::Console::WriteLine("[{0}] future:{1} ",__FUNCTION__, this->_aeffect->future);
		#endif

		if (this->_aeffect->dispatcher != System::IntPtr::Zero)
		{
			this->_dispatcher = 
				safe_cast<Interop::Vst::AEffectDispatcherProc^>(InteropServices::Marshal::GetDelegateForFunctionPointer(this->_aeffect->dispatcher, Interop::Vst::AEffectDispatcherProc::typeid));
		}

		if (this->_aeffect->setParameter != System::IntPtr::Zero)
		{
			this->_setParameter =
				safe_cast<Interop::Vst::AEffectSetParameterProc^>(InteropServices::Marshal::GetDelegateForFunctionPointer(this->_aeffect->setParameter, Interop::Vst::AEffectSetParameterProc::typeid));
		}

		if (this->_aeffect->getParameter != System::IntPtr::Zero)
		{
			this->_getParameter =
				safe_cast<Interop::Vst::AEffectGetParameterProc^>(InteropServices::Marshal::GetDelegateForFunctionPointer(this->_aeffect->getParameter, Interop::Vst::AEffectGetParameterProc::typeid));
		}

		if (this->_aeffect->processReplacing != System::IntPtr::Zero)
		{
			this->_processReplacing =
				safe_cast<Interop::Vst::AEffectProcessProc^>(InteropServices::Marshal::GetDelegateForFunctionPointer(this->_aeffect->processReplacing, Interop::Vst::AEffectProcessProc::typeid));
		}

		if (this->_aeffect->processDoubleReplacing != System::IntPtr::Zero)
		{
			this->_processDoubleReplacing =
				safe_cast<Interop::Vst::AEffectProcessDoubleProc^>(InteropServices::Marshal::GetDelegateForFunctionPointer(this->_aeffect->processDoubleReplacing, Interop::Vst::AEffectProcessDoubleProc::typeid));
		}

		{//effOpenに戻り値の定義は無い
			auto result =
				this->Dispatcher(
					Interop::Vst::AEffectOpcodes::effOpen, 
					0, 
					System::IntPtr::Zero, 
					System::IntPtr::Zero, 
					0
				);
		}

	}

	void Master::Close()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] Handle invalid:{1} closed:{2}",__FUNCTION__, this->_dll->IsInvalid, this->_dll->IsClosed);
		#endif
		if (
			!this->_dll->IsInvalid
		&&	!this->_dll->IsClosed
		)
		{
			if (this->_aeffect != nullptr)
			{//effCloseに戻り値の定義は無い
				auto result =
					this->Dispatcher(
						Interop::Vst::AEffectOpcodes::effClose, 
						0, 
						System::IntPtr::Zero, 
						System::IntPtr::Zero, 
						0
					);
			}

			this->_dll->Close();
		}
		else
		{
			#ifdef _DEBUG
				System::Console::WriteLine("[{0}] openしていない状態なので、無視します。", __FUNCTION__);
			#endif
		}

		if (this->_callBack != nullptr)
		{
			this->_callBack->OnEvent -= gcnew System::EventHandler<Interop::Vst::AudioMasterCallBack::AudioMasterEventArgs^>(this, &Master::OnEventHandler);
			delete this->_callBack;
		}

	}

	System::IntPtr Master::Dispatcher(
		Interop::Vst::AEffectOpcodes	opcode,
		System::Int32	index,
		System::IntPtr	value,
		System::IntPtr	ptr,
		System::Single	opt
	)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] Handle invalid:{1} closed:{2}",__FUNCTION__, this->_dll->IsInvalid, this->_dll->IsClosed);
		#endif

		if (this->_dispatcher == nullptr)
		{
			throw gcnew VstException("dispatcherが無い");
		}

		auto result =
			this->_dispatcher(
				this->_aeffectPtr,
				opcode, 
				index, 
				value, 
				ptr, 
				opt
			);
		#ifdef _DEBUG
		//	System::Console::WriteLine("[{0}] {1} result:{2}",__FUNCTION__, opcode, result);
		#endif

		if (opcode == Interop::Vst::AEffectOpcodes::effClose)
		{
			this->_aeffectPtr = System::IntPtr::Zero;

			this->_aeffect = nullptr;
			this->_dispatcher = nullptr;
			this->_processReplacing = nullptr;
			this->_processDoubleReplacing = nullptr;
			this->_setParameter = nullptr;
			this->_getParameter = nullptr;
		}

		return result;
	}

	void Master::Process(
		Buffer::VstBuffer<System::Single>^ inputs,
		Buffer::VstBuffer<System::Single>^ outputs,
		System::Int32	sampleFrames
	)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] Handle invalid:{1} closed:{2} sampleFrames:{3}",__FUNCTION__, this->_dll->IsInvalid, this->_dll->IsClosed, sampleFrames);
		#endif

		if (this->_processReplacing == nullptr)
		{
			throw gcnew VstException("processReplacingが無い");
		}

		this->_processReplacing(
			this->_aeffectPtr,
			inputs->AddrOfPinnedObject(), 
			outputs->AddrOfPinnedObject(), 
			sampleFrames
		);
	}

	void Master::Process(
		Buffer::VstBuffer<System::Double>^ inputs,
		Buffer::VstBuffer<System::Double>^ outputs,
		System::Int32	sampleFrames
	)
	{
		#ifdef _DEBUG
	//		System::Console::WriteLine("[{0}] Handle invalid:{1} closed:{2}",__FUNCTION__, this->_dll->IsInvalid, this->_dll->IsClosed);
		#endif

		if (this->_processDoubleReplacing == nullptr)
		{
			throw gcnew VstException("processDoubleReplacingが無い");
		}

		this->_processDoubleReplacing(
			this->_aeffectPtr,
			inputs->AddrOfPinnedObject(), 
			outputs->AddrOfPinnedObject(), 
			sampleFrames
		);
	}

	void Master::SetParameter(
		System::Int32	index,
		System::Single	parameter
	)
	{
		#ifdef _DEBUG
		//	System::Console::WriteLine("[{0}] Handle invalid:{1} closed:{2}",__FUNCTION__, this->_dll->IsInvalid, this->_dll->IsClosed);
		#endif

		if (this->_setParameter == nullptr)
		{
			throw gcnew VstException("setParameterが無い");
		}

		this->_setParameter(
			this->_aeffectPtr,
			index, 
			parameter
		);
	}

	System::Single Master::GetParameter(
		System::Int32	index
	)
	{
		#ifdef _DEBUG
		//	System::Console::WriteLine("[{0}] Handle invalid:{1} closed:{2}",__FUNCTION__, this->_dll->IsInvalid, this->_dll->IsClosed);
		#endif

		if (this->_getParameter == nullptr)
		{
			throw gcnew VstException("getParameterが無い");
		}

		return 
			this->_getParameter(
				this->_aeffectPtr,
				index
			);
	}

	bool Master::String2Parameter(System::Int32 index, System::String^ value)
	{
		auto ptr = InteropServices::Marshal::StringToHGlobalAnsi(value);
		try
		{
			return 
				(this->Dispatcher(
					Interop::Vst::AEffectOpcodes::effString2Parameter, 
					index, 
					System::IntPtr::Zero, 
					ptr, 
					0
				) != System::IntPtr::Zero);
		}
		finally
		{
			InteropServices::Marshal::FreeHGlobal(ptr);
		}
	}








	System::String^ Master::GetString(Interop::Vst::AEffectOpcodes opcode, System::Int32 index, System::Int32 length)
	{
		//文字長制限に従っていないプラグインがあるので、バッファを多めに取っておく
		auto ptr = InteropServices::Marshal::AllocHGlobal(length+1);
		InteropServices::Marshal::WriteByte(ptr, 0, 0); //先頭をnullクリア
		try
		{
			if (this->Dispatcher(
					opcode, 
					index, 
					System::IntPtr::Zero,
					ptr, 
					0
				) == System::IntPtr::Zero)
			{//失敗しているかどうかの判断は、１バイト目がゼロのままだったら、ということにする。
				if (InteropServices::Marshal::ReadByte(ptr, 0) == 0)
				{
					return "(failed)";
				}
			}
			InteropServices::Marshal::WriteByte(ptr, length, 0);	//念のため終端を設定
			return InteropServices::Marshal::PtrToStringAnsi(ptr);
		}
		finally
		{
			InteropServices::Marshal::FreeHGlobal(ptr);
		}
	}

	//State Transitions
	void Master::Suspend()
	{
		this->Dispatcher(
			Interop::Vst::AEffectOpcodes::effMainsChanged, 
			0, 
			System::IntPtr(0),
			System::IntPtr::Zero, 
			0
		);
	}

	void Master::Resume()
	{
		this->Dispatcher(
			Interop::Vst::AEffectOpcodes::effMainsChanged, 
			0, 
			System::IntPtr(1),
			System::IntPtr::Zero, 
			0
		);
	}

	//Processing
	void Master::SetSampleRate(System::Single sampleRate)
	{
		if (this->_timeInfo->sampleRate == sampleRate)
		{
			//変わっていないので何もしない
			return;
		}

		this->_timeInfo->sampleRate = sampleRate;

		this->Dispatcher(
			Interop::Vst::AEffectOpcodes::effSetSampleRate, 
			0, 
			System::IntPtr::Zero, 
			System::IntPtr::Zero, 
			sampleRate
		);
	}

	void Master::SetBlockSize(System::Int32 blockSize)
	{
		this->Dispatcher(
			Interop::Vst::AEffectOpcodes::effSetBlockSize, 
			0, 
			System::IntPtr(blockSize),
			System::IntPtr::Zero, 
			0
		);
	}

	bool Master::CanParameterBeAutomated (System::Int32 index)
	{
		return 
			(this->Dispatcher(
					Interop::Vst::AEffectOpcodes::effCanBeAutomated, 
					index, 
					System::IntPtr::Zero, 
					System::IntPtr::Zero, 
					0
			) != System::IntPtr::Zero);
	}

	Interop::Vst::VstParameterProperties^ Master::GetParameterProperties(System::Int32 index)
	{
		auto ptr = InteropServices::Marshal::AllocHGlobal(InteropServices::Marshal::SizeOf(Interop::Vst::VstParameterProperties::typeid));
		try
		{
			if (this->Dispatcher(
					Interop::Vst::AEffectOpcodes::effGetParameterProperties, 
					index, 
					System::IntPtr::Zero,
					ptr, 
					0
				) == System::IntPtr::Zero)
			{
				throw gcnew VstException("effGetParameterProperties失敗");
			}

			return safe_cast<Interop::Vst::VstParameterProperties^>(InteropServices::Marshal::PtrToStructure(ptr, Interop::Vst::VstParameterProperties::typeid));
		}
		finally
		{
			InteropServices::Marshal::FreeHGlobal(ptr);
		}
	}

	System::Int32 Master::GetProgram()
	{
		auto result =
			this->Dispatcher(
				Interop::Vst::AEffectOpcodes::effGetProgram, 
				0, 
				System::IntPtr::Zero, 
				System::IntPtr::Zero, 
				0
			);

		return result.ToInt32();
	}

	void Master::SetProgram(System::Int32 program)
	{
		if (this->Dispatcher(
				Interop::Vst::AEffectOpcodes::effSetProgram, 
				0, 
				System::IntPtr(program),
				System::IntPtr::Zero, 
				0
			) == System::IntPtr::Zero)
		{
			throw gcnew VstException("effSetProgram失敗");
		}
	}

	void Master::SetProgramName(System::String^ name)
	{
		auto ptr = InteropServices::Marshal::StringToHGlobalAnsi(name);
		try
		{
			if (this->Dispatcher(
					Interop::Vst::AEffectOpcodes::effSetProgramName, 
					0, 
					System::IntPtr::Zero,
					ptr, 
					0
				) == System::IntPtr::Zero)
			{
				throw gcnew VstException("effSetProgramName失敗");
			}
		}
		finally
		{
			InteropServices::Marshal::FreeHGlobal(ptr);
		}
	}

	System::String^ Master::GetProgramName()
	{
		return 
			this->GetString(
				Interop::Vst::AEffectOpcodes::effGetProgramName, 
				0,
				255//Interop::Vst::VstStringConstants::kVstMaxProgNameLen
			);
	}

	System::String^ Master::GetParameterLabel(System::Int32 index)
	{
		return
			this->GetString(
				Interop::Vst::AEffectOpcodes::effGetParamLabel, 
				index,
				255//Interop::Vst::VstStringConstants::kVstMaxParamStrLen 
			);
	}

	System::String^ Master::GetParameterDisplay(System::Int32 index)
	{
		return
			this->GetString(
				Interop::Vst::AEffectOpcodes::effGetParamDisplay, 
				index,
				255//Interop::Vst::VstStringConstants::kVstMaxParamStrLen 
			);
	}

	System::String^ Master::GetParameterName(System::Int32 index)
	{
		return
			this->GetString(
				Interop::Vst::AEffectOpcodes::effGetParamName, 
				index,
				255//Interop::Vst::VstStringConstants::kVstMaxParamStrLen 
			);
	}

	System::String^ Master::GetProgramNameIndexed(System::Int32 index)
	{
		return
			this->GetString(
				Interop::Vst::AEffectOpcodes::effGetProgramNameIndexed, 
				index,
				255//Interop::Vst::VstStringConstants::kVstMaxProgNameLen 
			);
	}

	bool Master::BeginSetProgram ()
	{
		return 
			(this->Dispatcher(
				Interop::Vst::AEffectOpcodes::effBeginSetProgram, 
				0, 
				System::IntPtr::Zero, 
				System::IntPtr::Zero, 
				0
			) != System::IntPtr::Zero);
	}

	bool Master::EndSetProgram ()
	{
		return 
			(this->Dispatcher(
				Interop::Vst::AEffectOpcodes::effEndSetProgram, 
				0, 
				System::IntPtr::Zero, 
				System::IntPtr::Zero, 
				0
			) != System::IntPtr::Zero);
	}


	Interop::Vst::VstPinProperties^ Master::GetInputProperties(System::Int32 index)
	{
		auto ptr = InteropServices::Marshal::AllocHGlobal(InteropServices::Marshal::SizeOf(Interop::Vst::VstPinProperties::typeid));
		try
		{
			if (this->Dispatcher(
					Interop::Vst::AEffectOpcodes::effGetInputProperties, 
					index, 
					System::IntPtr::Zero,
					ptr,
					0
				) == System::IntPtr::Zero)
			{
				throw gcnew VstException("effGetInputProperties失敗");
			}

			return safe_cast<Interop::Vst::VstPinProperties^>(InteropServices::Marshal::PtrToStructure(ptr, Interop::Vst::VstPinProperties::typeid));
		}
		finally
		{
			InteropServices::Marshal::FreeHGlobal(ptr);
		}
	}

	Interop::Vst::VstPinProperties^ Master::GetOutputProperties(System::Int32 index)
	{
		auto ptr = InteropServices::Marshal::AllocHGlobal(InteropServices::Marshal::SizeOf(Interop::Vst::VstPinProperties::typeid));
		try
		{
			if (this->Dispatcher(
					Interop::Vst::AEffectOpcodes::effGetOutputProperties, 
					index, 
					System::IntPtr::Zero,
					ptr,
					0
				) == System::IntPtr::Zero)
			{
				throw gcnew VstException("effGetOutputProperties失敗");
			}
			
			return safe_cast<Interop::Vst::VstPinProperties^>(InteropServices::Marshal::PtrToStructure(ptr, Interop::Vst::VstPinProperties::typeid));
		}
		finally
		{
			InteropServices::Marshal::FreeHGlobal(ptr);
		}
	}


	System::String^ Master::GetEffectName()
	{
		return
			this->GetString(
				Interop::Vst::AEffectOpcodes::effGetEffectName, 
				0,
				255//Interop::Vst::VstStringConstants::kVstMaxEffectNameLen 
			);
	}

	System::String^ Master::GetVendorString()
	{
		return
			this->GetString(
				Interop::Vst::AEffectOpcodes::effGetVendorString, 
				0,
				255//Interop::Vst::VstStringConstants::kVstMaxVendorStrLen 
			);
	}

	System::String^ Master::GetProductString()
	{
		return
			this->GetString(
				Interop::Vst::AEffectOpcodes::effGetProductString, 
				0,
				255//Interop::Vst::VstStringConstants::kVstMaxProductStrLen 
			);
	}

	System::Int32 Master::ProcessEvents(array<Interop::Vst::VstEvent>^ events)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] start",__FUNCTION__);
		#endif

		auto e = gcnew Interop::Vst::VstEvents();
		e->numEvents = events->Length;

		//VstEventの転機先ポインタを保持する
		auto eventPtrs = gcnew array<System::IntPtr>(e->numEvents);

		auto ptr = 
			InteropServices::Marshal::AllocHGlobal(
				InteropServices::Marshal::SizeOf(Interop::Vst::VstEvents::typeid) +
				(InteropServices::Marshal::SizeOf(System::IntPtr::typeid) * e->numEvents)
			);

		try
		{
			auto cursor = ptr;

			InteropServices::Marshal::StructureToPtr(e, cursor, false);
			cursor += InteropServices::Marshal::SizeOf(Interop::Vst::VstEvents::typeid);

			for (int idx = 0; idx < e->numEvents; idx++)
			{
				auto ev = events[idx];
				ev.byteSize = 
					InteropServices::Marshal::SizeOf(Interop::Vst::VstEvents::typeid)
					-(InteropServices::Marshal::SizeOf(System::Int32::typeid) + InteropServices::Marshal::SizeOf(System::Int32::typeid)) //VSTの仕様だとこの分はサイズから除くことになっている
					;

				auto eventPtr = InteropServices::Marshal::AllocHGlobal(InteropServices::Marshal::SizeOf(Interop::Vst::VstEvent::typeid));
				eventPtrs[idx] = eventPtr;

				InteropServices::Marshal::StructureToPtr(ev, eventPtr, false);

				#ifdef _DEBUG
					System::Console::WriteLine("[{0}] {1}",__FUNCTION__, ev.ToString());
				#endif
			}
			InteropServices::Marshal::Copy(eventPtrs, 0, cursor, e->numEvents);

			auto result =
				this->Dispatcher(
					Interop::Vst::AEffectOpcodes::effProcessEvents, 
					0, 
					System::IntPtr::Zero,
					ptr, 
					0
				);

			return result.ToInt32();
		}
		finally
		{
			for (int idx = 0; idx < e->numEvents; idx++)
			{
				auto eventPtr = eventPtrs[idx];
				if (eventPtr == System::IntPtr::Zero)
				{
					continue;
				}
				InteropServices::Marshal::FreeHGlobal(eventPtr);
			}
			InteropServices::Marshal::FreeHGlobal(ptr);
		}
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] end",__FUNCTION__);
		#endif
	}

	System::Int32 Master::StartProcess()
	{
		auto result =
			this->Dispatcher(
				Interop::Vst::AEffectOpcodes::effStartProcess, 
				0, 
				System::IntPtr::Zero,
				System::IntPtr::Zero, 
				0
			);
		return result.ToInt32();
	}

	System::Int32 Master::StopProcess()
	{
		auto result =
			this->Dispatcher(
				Interop::Vst::AEffectOpcodes::effStopProcess, 
				0, 
				System::IntPtr::Zero,
				System::IntPtr::Zero, 
				0
			);
		return result.ToInt32();
	}


	System::IntPtr Master::GetTimeInfo(System::Int32 filter)
	{
		return this->_timeInfoHandle.AddrOfPinnedObject();
	}
}
}
}
}
