﻿/*
[momiji music component library]
---------------------------------------------------------------------
Momiji.Core.Wave.Out.cpp
	wave output component.
---------------------------------------------------------------------
Copyright (C) 2011 tyiki badwell {miria@users.sourceforge.jp}.

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/gpl-3.0.html>.
---------------------------------------------------------------------
*/
#include "StdAfx.h"

#include "Momiji.Interop.Winmm.h"
#include "Momiji.Core.Wave.Out.h"
#include "Momiji.Core.Wave.Data.h"

namespace Momiji {
namespace Core {
namespace Wave {
namespace Out {

	void Device::OnEventHandler(
		System::Object^ sender, 
		Interop::Winmm::DriverCallBack::DriverEventArgs^ args
	)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] [{1}] start", __FUNCTION__, System::Threading::Thread::CurrentThread->GetHashCode());
		#endif

		switch (args->uMsg)
		{
		case Interop::Winmm::DriverCallBack::MM_EXT_WINDOW_MESSAGE::WOM_OPEN	:
			{
				this->DoOpen(args->dw1, args->dw2);
				break;
			}
		case Interop::Winmm::DriverCallBack::MM_EXT_WINDOW_MESSAGE::WOM_CLOSE:
			{
				this->DoClose(args->dw1, args->dw2);
				break;
			}
		case Interop::Winmm::DriverCallBack::MM_EXT_WINDOW_MESSAGE::WOM_DONE	:
			{
				this->DoDone(args->dw1, args->dw2);
				break;
			}
		}
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] [{1}] end", __FUNCTION__, System::Threading::Thread::CurrentThread->GetHashCode());
		#endif
	}

	void Device::DoOpen(System::IntPtr dwParam1, System::IntPtr dwParam2)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] [{1,8:X}][{2,8:X}]", __FUNCTION__, dwParam1, dwParam2);
		#endif
		this->OnOpen(this, System::EventArgs::Empty);
	}

	void Device::DoClose(System::IntPtr dwParam1, System::IntPtr dwParam2)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] [{1,8:X}][{2,8:X}]", __FUNCTION__, dwParam1, dwParam2);
		#endif
		this->OnClose(this, System::EventArgs::Empty);
	}

	void Device::DoDone(System::IntPtr dwParam1, System::IntPtr dwParam2)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] [{1,8:X}][{2,8:X}]", __FUNCTION__, dwParam1, dwParam2);

			auto waveHeader = safe_cast<Interop::Winmm::WaveHeader^>(InteropServices::Marshal::PtrToStructure(dwParam1, Interop::Winmm::WaveHeader::typeid));
			System::Console::WriteLine("[{0}] {1}", __FUNCTION__, waveHeader);
		#endif

		UnpreparedEventArgs^ eventArgs = gcnew UnpreparedEventArgs();
		eventArgs->bufferPtr = this->Unprepare(dwParam1);

		this->OnDone(this, eventArgs);
	}

	System::UInt32 Device::GetNumDevices()
	{
		return Interop::Winmm::Function::waveOutGetNumDevs();
	}

	Interop::Winmm::WaveOutCapabilities^ Device::GetCapabilities(
		const System::UInt32 deviceID
	)
	{
		auto caps = gcnew Interop::Winmm::WaveOutCapabilities();
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] caps size {1}",__FUNCTION__, InteropServices::Marshal::SizeOf(caps));
		#endif
		auto mmResult = 
			Interop::Winmm::Function::waveOutGetDevCaps(
				deviceID, 
				caps, 
				InteropServices::Marshal::SizeOf(caps)
			);
		if (mmResult != Interop::Winmm::MMRESULT::NOERROR)
		{
			throw gcnew WaveOutException(mmResult);
		}
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] caps productName {1}",__FUNCTION__, caps->productName);
		#endif

		return caps;
	}

	Device::Device(
		System::UInt32 deviceID,
		System::UInt16 channels,
		System::UInt32 samplesPerSecond,
		System::UInt16 bitsPerSample,
		Interop::Winmm::WaveFormatExtensiblePart::SPEAKER	channelMask,
		Interop::Guiddef::Guid formatSubType,
		System::UInt32 samplesPerBuffer
	): 
		_deviceID(deviceID), 
		_channels(channels), 
		_samplesPerSecond(samplesPerSecond), 
		_bitsPerSample(bitsPerSample),
		_channelMask(channelMask),
		_formatSubType(formatSubType),
		_samplesPerBuffer(samplesPerBuffer)
	{
		#ifdef _DEBUG
			System::Diagnostics::Trace::WriteLine(System::String::Format("[{0}] deviceID {1} start",__FUNCTION__, this->_deviceID));
			System::Console::WriteLine("[{0}] deviceID {1} start",__FUNCTION__, this->_deviceID);
		#endif

		this->Open();
		
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] end",__FUNCTION__);
		#endif
	}

	Device::~Device()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] start",__FUNCTION__);
		#endif
		this->!Device();
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] end",__FUNCTION__);
		#endif
	}

	Device::!Device()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] start",__FUNCTION__);
		#endif
		this->Close();
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] end",__FUNCTION__);
		#endif
	}

	Interop::Winmm::WaveOutCapabilities^ Device::GetCapabilities()
	{
		return Device::GetCapabilities(this->_deviceID);
	}

	void Device::Open()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif

		this->_callBack = gcnew Core::Winmm::DriverCallBack(true); //イベントは非同期で動かす
		this->_callBack->OnEvent += gcnew System::EventHandler<Interop::Winmm::DriverCallBack::DriverEventArgs^>(this, &Device::OnEventHandler);

		auto format = Interop::Winmm::WaveFormatExtensible();
		format.wfe.formatType				= Interop::Winmm::WaveFormatEx::FORMAT::EXTENSIBLE;
		format.wfe.channels					= this->_channels;
		format.wfe.samplesPerSecond			= this->_samplesPerSecond;
		format.wfe.bitsPerSample			= this->_bitsPerSample;
		format.wfe.blockAlign				= format.wfe.channels * format.wfe.bitsPerSample / 8; 
		format.wfe.averageBytesPerSecond	= format.wfe.samplesPerSecond * format.wfe.blockAlign;
		format.wfe.size						= safe_cast<System::UInt16>(InteropServices::Marshal::SizeOf(Interop::Winmm::WaveFormatExtensiblePart::typeid));

		format.exp.validBitsPerSample		= format.wfe.bitsPerSample;	//TODO: 実際にハードウェアでサポートしている限界に揃える
		format.exp.channelMask				= this->_channelMask;
		format.exp.subFormat				= this->_formatSubType;


		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] format [{1}]",__FUNCTION__, format);
		#endif

		this->_headerPool =
			gcnew Core::Buffer::BufferPool<Interop::Winmm::WaveHeader^>(
				((this->_samplesPerSecond / (this->_samplesPerBuffer * format.wfe.blockAlign)) + 2), //１秒分＋２回
				gcnew Core::Buffer::BufferPool<Interop::Winmm::WaveHeader^>::Allocator(this, &Device::AllocateHeader) 
			);
			
		auto mmResult = 
			Interop::Winmm::Function::waveOutOpen(
				this->_handle,
				this->_deviceID,
				format,
				this->_callBack->GetDriverCallBackProc(),
				System::IntPtr::Zero,
				(
						Interop::Winmm::DriverCallBack::TYPE::FUNCTION
					|	Interop::Winmm::DriverCallBack::TYPE::WAVE_FORMAT_DIRECT
				//	|	Interop::Winmm::DriverCallBack::TYPE::WAVE_ALLOWSYNC
				)
			);
		if (mmResult != Interop::Winmm::MMRESULT::NOERROR)
		{
			throw gcnew WaveOutException(mmResult);
		}

		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] Handle invalid:{1} closed:{2}",__FUNCTION__, this->_handle->IsInvalid, this->_handle->IsClosed);
		#endif
	}

	void Device::Close()
	{
		if (this->_handle != nullptr)
		{
			#ifdef _DEBUG
				System::Console::WriteLine("[{0}] Handle invalid:{1} closed:{2}",__FUNCTION__, this->_handle->IsInvalid, this->_handle->IsClosed);
			#endif
			if (
				!this->_handle->IsInvalid
			&&	!this->_handle->IsClosed
			)
			{
				this->Reset();

				//バッファの開放待ち
				while(this->_headerPool->IsBusy())
				{
					#ifdef _DEBUG
						System::Console::WriteLine("[{0}] wait for unprepare headers ...",__FUNCTION__);
					#endif
					System::Threading::Thread::Sleep(5);
				}

				this->_handle->Close();
			}
			else
			{
				#ifdef _DEBUG
					System::Console::WriteLine("[{0}] openしていない状態なので、無視します。", __FUNCTION__);
				#endif
			}
		}

		if (this->_callBack != nullptr)
		{
			this->_callBack->OnEvent -= gcnew System::EventHandler<Interop::Winmm::DriverCallBack::DriverEventArgs^>(this, &Device::OnEventHandler);
			delete this->_callBack;
		}

		if (this->_headerPool != nullptr)
		{
			delete this->_headerPool;
		}
	}

	void Device::Reset()
	{
		if (
			this->_handle->IsInvalid
		||	this->_handle->IsClosed
		)
		{
			#ifdef _DEBUG
				System::Console::WriteLine("[{0}] openしていない状態なので、無視します。", __FUNCTION__);
			#endif
			return;
		}

		auto mmResult = Interop::Winmm::Function::waveOutReset(this->_handle);
		if (mmResult != Interop::Winmm::MMRESULT::NOERROR)
		{
			throw gcnew WaveOutException(mmResult);
		}
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] waveOutReset OK",__FUNCTION__);
		#endif
	}
	
	//TODO 要同期
	System::IntPtr Device::Prepare(System::IntPtr data, System::UInt32 useSize)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif

		auto header = this->_headerPool->Get();

		{
			auto waveHeader = header->GetBuffer();
			waveHeader->data = data;
			waveHeader->bufferLength = useSize;
			waveHeader->flags = (Interop::Winmm::WaveHeader::FLAG::BEGINLOOP | Interop::Winmm::WaveHeader::FLAG::ENDLOOP);
			waveHeader->loops = 1;

			waveHeader->bytesRecorded = 0;
			waveHeader->user = System::IntPtr::Zero;
			waveHeader->next = System::IntPtr::Zero;
			waveHeader->reserved = System::IntPtr::Zero;

			#ifdef _DEBUG
				{
					System::Console::WriteLine("[{0}] waveOutPrepareHeader before", __FUNCTION__);
					System::Console::WriteLine("[{0}] {1}", __FUNCTION__, waveHeader);
				}
			#endif
		}

		auto mmResult = 
			Interop::Winmm::Function::waveOutPrepareHeader(
				this->_handle,
				header->GetBufferIntPtr(),
				InteropServices::Marshal::SizeOf(Interop::Winmm::WaveHeader::typeid)
			);
		if (mmResult != Interop::Winmm::MMRESULT::NOERROR)
		{
			this->_headerPool->Release(header);

			throw gcnew WaveOutException(mmResult);
		}
		#ifdef _DEBUG
			{
				auto waveHeader = header->GetBuffer();
				System::Console::WriteLine("[{0}] waveOutPrepareHeader after", __FUNCTION__);
				System::Console::WriteLine("[{0}] {1}", __FUNCTION__, waveHeader);
			}
		#endif

		return header->GetBufferIntPtr();
	}

	//TODO 要同期
	System::IntPtr Device::Unprepare(System::IntPtr headerPtr)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif

		Core::Buffer::BufferPool<Interop::Winmm::WaveHeader^>::Buffer^ header = this->_headerPool->GetBusy(headerPtr);
			
		#ifdef _DEBUG
			{
				auto waveHeader = header->GetBuffer();
				System::Console::WriteLine("[{0}] waveOutUnprepareHeader before", __FUNCTION__);
				System::Console::WriteLine("[{0}] {1}", __FUNCTION__, waveHeader);
			}
		#endif

		auto mmResult = 
			Interop::Winmm::Function::waveOutUnprepareHeader(
				this->_handle,
				headerPtr,
				InteropServices::Marshal::SizeOf(Interop::Winmm::WaveHeader::typeid)
			);
		if (mmResult != Interop::Winmm::MMRESULT::NOERROR)
		{
			throw gcnew WaveOutException(mmResult);
		}

		auto result = header->GetBuffer()->data;

		this->_headerPool->Release(header);

		#ifdef _DEBUG
			{
				auto waveHeader = header->GetBuffer();
				System::Console::WriteLine("[{0}] waveOutUnprepareHeader after", __FUNCTION__);
				System::Console::WriteLine("[{0}] {1}", __FUNCTION__, waveHeader);
			}
		#endif

		return result;
	}

	void Device::Send(
		System::IntPtr data, System::UInt32 useSize
	)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif

		if (
			this->_handle->IsInvalid
		||	this->_handle->IsClosed
		)
		{
			#ifdef _DEBUG
				System::Console::WriteLine("[{0}] openしていない状態なので、無視します。", __FUNCTION__);
			#endif
			return;
		}

		System::IntPtr headerPtr = this->Prepare(data, useSize);

		auto mmResult = 
			Interop::Winmm::Function::waveOutWrite(
				this->_handle,
				headerPtr,
				InteropServices::Marshal::SizeOf(Interop::Winmm::WaveHeader::typeid)
			);
		if (mmResult != Interop::Winmm::MMRESULT::NOERROR)
		{
			this->Unprepare(headerPtr);
			throw gcnew WaveOutException(mmResult);
		}
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] waveOutWrite OK",__FUNCTION__);
		#endif
	}

	Interop::Winmm::WaveHeader^ Device::AllocateHeader()
	{
		return gcnew Interop::Winmm::WaveHeader();
	}

	System::String^ WaveOutException::Initialize()
	{
		auto errorMessage = System::String::Empty;
		auto buf = gcnew System::Text::StringBuilder(256);

		auto r = 
			Interop::Winmm::Function::waveOutGetErrorText(
				this->_mmResult, 
				buf, 
				buf->Capacity
			);
		if (r == Interop::Winmm::MMRESULT::NOERROR)
		{
			errorMessage = buf->ToString();
		}
		else
		{
			errorMessage = "不明なエラー";
		}

		return errorMessage + "[" + this->_mmResult.ToString("D") + "]";
	}

}
}
}
}
