﻿/*
[momiji music component library]
---------------------------------------------------------------------
Momiji.Core.Winmm.h
	windows multi media dll importer.
---------------------------------------------------------------------
Copyright (C) 2011 tyiki badwell {miria@users.sourceforge.jp}.

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/gpl-3.0.html>.
---------------------------------------------------------------------
*/
#pragma once

#using <mscorlib.dll>
#include "Momiji.Interop.Guiddef.h"

using namespace System::Runtime;

namespace Momiji{
namespace Interop {
namespace Winmm {

#define WIN_MM_DLLIMPORT		WIN32_DLL_IMPORT("winmm")

	/**
	<remarks>
	MMRESULT
	</remarks>
	*/
	public enum class MMRESULT: System::UInt32 {
		NOERROR = 0,

		WAVERR_BASE = 32,

		MIDIERR_BASE = 64,

		TIMERR_BASE = 96,
		TIMERR_NOCANDO = (TIMERR_BASE+1),      // request not completed
	};

	public value class DriverCallBack
	{
	public:
		enum class TYPE: System::UInt32 {
			TYPEMASK   = 0x00070000l,	// callback type mask
			NULL       = 0x00000000l,	// no callback 
			WINDOW     = 0x00010000l,	// dwCallback is a HWND
			TASK       = 0x00020000l,	// dwCallback is a HTASK
			FUNCTION   = 0x00030000l,	// dwCallback is a FARPROC
			THREAD     = TASK,			// thread ID replaces 16 bit task
			EVENT      = 0x00050000l,	// dwCallback is an EVENT Handle

			WAVE_FORMAT_QUERY							= 0x0001,
			WAVE_ALLOWSYNC								= 0x0002,
			WAVE_MAPPED									= 0x0004,
			WAVE_FORMAT_DIRECT							= 0x0008,
			WAVE_FORMAT_DIRECT_QUERY					= (WAVE_FORMAT_QUERY | WAVE_FORMAT_DIRECT),
			WAVE_MAPPED_DEFAULT_COMMUNICATION_DEVICE	= 0x0010,
		};

		enum class MM_EXT_WINDOW_MESSAGE: System::UInt32 {
			WOM_OPEN		= 0x3BB,	// waveform output
			WOM_CLOSE		= 0x3BC,
			WOM_DONE		= 0x3BD,

			WIM_OPEN		= 0x3BE,	// waveform input
			WIM_CLOSE		= 0x3BF,
			WIM_DATA		= 0x3C0,

			MIM_OPEN		= 0x3C1,	// MIDI input 
			MIM_CLOSE		= 0x3C2,
			MIM_DATA		= 0x3C3,
			MIM_LONGDATA	= 0x3C4,
			MIM_ERROR		= 0x3C5,
			MIM_LONGERROR	= 0x3C6,
			MIM_MOREDATA	= 0x3CC,	// MIM_DONE w/ pending events 

			MOM_OPEN		= 0x3C7,	// MIDI output 
			MOM_CLOSE		= 0x3C8,
			MOM_DONE		= 0x3C9,
			MOM_POSITIONCB	= 0x3CA,	// Callback for MEVT_POSITIONCB 

		};

		ref class DriverEventArgs: System::EventArgs
		{
		public:
			System::IntPtr	hdrvr;
			MM_EXT_WINDOW_MESSAGE	uMsg;
			System::IntPtr	dwUser;
			System::IntPtr	dw1;
			System::IntPtr	dw2;
		};

		delegate void Delegate(
			System::IntPtr	hdrvr,
			MM_EXT_WINDOW_MESSAGE	uMsg,
			System::IntPtr	dwUser,
			System::IntPtr	dw1,
			System::IntPtr	dw2
		);

	};

	// MIDI data block header
	WIN32_DLL_STRUCTLAYOUT public ref struct MidiHeader {
		// flags for dwFlags field of MIDIHDR structure 
		[System::FlagsAttribute]
		enum class FLAG: System::UInt32 {
			DONE		= 0x00000001,	// done bit
			PREPARED	= 0x00000002,	// set if header prepared
			INQUEUE		= 0x00000004,	// reserved for driver
			ISSTRM		= 0x00000008,	// Buffer is stream buffer
		};

		System::IntPtr		data;			// pointer to locked data block
		System::UInt32		bufferLength;	// length of data in data block
		System::UInt32		bytesRecorded;	// used for input only
		System::IntPtr		user;			// for client's use
		FLAG				flags;			// assorted flags (see defines)
		System::UInt32		next;			// reserved for driver
		System::IntPtr		reserved;		// reserved for driver
		System::UInt32		offset;			// Callback offset into buffer
		//[InteropServices::MarshalAs(InteropServices::UnmanagedType::ByValArray, SizeConst=8)]
		//	array<System::IntPtr>^ reserve;	// Reserved for MMSYSTEM

		System::IntPtr		reserve1;
		System::IntPtr		reserve2;
		System::IntPtr		reserve3;
		System::IntPtr		reserve4;
		System::IntPtr		reserve5;
		System::IntPtr		reserve6;
		System::IntPtr		reserve7;
		System::IntPtr		reserve8;
			
		virtual System::String^ ToString() override
		{
			return 
				"data ["+data.ToString("X")+"] bufferLength ["+bufferLength+"] bytesRecorded ["+bytesRecorded+"] flags ["+flags.ToString("F")+"] next ["+next+"] reserved ["+reserved+"] offset ["+offset+"]";
		}
	};

	WIN32_DLL_STRUCTLAYOUT public ref struct MidiOutCapabilities {
		// flags for wTechnology field of MIDIOUTCAPS structure
		enum class MOD: System::UInt16 {
			MIDIPORT    = 1,  // output port 
			SYNTH       = 2,  // generic internal synth 
			SQSYNTH     = 3,  // square wave internal synth 
			FMSYNTH     = 4,  // FM internal synth
			MAPPER      = 5,  // MIDI mapper
			WAVETABLE   = 6,  // hardware wavetable synth
			SWSYNTH     = 7,  // software synth
		};
	
		//flags for dwSupport field of MIDIOUTCAPS structure
		[System::FlagsAttribute]
		enum class MIDICAPS: System::UInt32 {
			VOLUME        =  0x0001,  // supports volume control
			LRVOLUME      =  0x0002,  // separate left-right volume control
			CACHE         =  0x0004,
			STREAM        =  0x0008,  // driver supports midiStreamOut directly
		};

		System::UInt16			manufacturerID;
		System::UInt16			productID;
		System::UInt32			driverVersion;
		[InteropServices::MarshalAs(InteropServices::UnmanagedType::ByValTStr, SizeConst=32)]
			System::String^		productName;
		MOD						technology;
		System::UInt16			voices;
		System::UInt16			notes;
		System::UInt16			channelMask;
		MIDICAPS				support;

		Guiddef::Guid			manufacturerGuid;      // for extensible MID mapping 
		Guiddef::Guid			productGuid;           // for extensible PID mapping
		Guiddef::Guid			nameGuid;              // for name lookup in registry
	};

	WIN32_DLL_STRUCTLAYOUT public ref struct MidiInCapabilities {
		//flags for dwSupport field of MIDIOUTCAPS structure
		[System::FlagsAttribute]
		enum class MIDICAPS: System::UInt32 {
			VOLUME        =  0x0001,  // supports volume control
			LRVOLUME      =  0x0002,  // separate left-right volume control
			CACHE         =  0x0004,
			STREAM        =  0x0008,  // driver supports midiStreamOut directly
		};

		System::UInt16			manufacturerID;
		System::UInt16			productID;
		System::UInt32			driverVersion;
		[InteropServices::MarshalAs(InteropServices::UnmanagedType::ByValTStr, SizeConst=32)]
			System::String^		productName;
		MIDICAPS				support;

		Guiddef::Guid			manufacturerGuid;      // for extensible MID mapping 
		Guiddef::Guid			productGuid;           // for extensible PID mapping
		Guiddef::Guid			nameGuid;              // for name lookup in registry
	};

	public value class TimerCallBack
	{
	public:
		enum class TYPE: System::UInt32 {
			// flags for fuEvent parameter of timeSetEvent() function
			ONESHOT					= 0x0000,	// program timer for single event
			PERIODIC				= 0x0001,	// program for continuous periodic event
			
			CALLBACK_FUNCTION		= 0x0000,	// callback is function
			CALLBACK_EVENT_SET		= 0x0010,	// callback is event - use SetEvent
			CALLBACK_EVENT_PULSE	= 0x0020,	// callback is event - use PulseEvent
			
			KILL_SYNCHRONOUS		= 0x0100,	// This flag prevents the event from occurring
												// after the user calls timeKillEvent() to
												// destroy it.
		};

		ref class TimerEventArgs: System::EventArgs
		{
		public:
			System::UInt32	uTimerID;
			System::UInt32	uMsg;
			System::IntPtr	dwUser;
			System::IntPtr	dw1;
			System::IntPtr	dw2;
		};

		delegate void Delegate(
			System::UInt32	uTimerID,
			System::UInt32	uMsg,
			System::IntPtr	dwUser,
			System::IntPtr	dw1,
			System::IntPtr	dw2
		);
	};

	WIN32_DLL_STRUCTLAYOUT public ref struct TimeCapabilities {
		System::UInt32	periodMin;	// minimum period supported
		System::UInt32	periodMax;	// maximum period supported
	};


	// flags for dwFlags parameter in waveOutOpen() and waveInOpen()
	[System::FlagsAttribute]
	public enum class WAVE: System::UInt32 {
		FORMAT_QUERY						= 0x0001,
		ALLOWSYNC							= 0x0002,
		MAPPED								= 0x0004,
		FORMAT_DIRECT						= 0x0008,
		FORMAT_DIRECT_QUERY					= (FORMAT_QUERY | FORMAT_DIRECT),
		MAPPED_DEFAULT_COMMUNICATION_DEVICE	= 0x0010,
	};

	// wave data block header
	WIN32_DLL_STRUCTLAYOUT public ref struct WaveHeader {
		// flags for dwFlags field of WAVEHDR
		[System::FlagsAttribute]
		enum class FLAG: System::UInt32 {
			DONE		= 0x00000001,	// done bit
			PREPARED	= 0x00000002,	// set if this header has been prepared
			BEGINLOOP	= 0x00000004,	// loop start block
			ENDLOOP		= 0x00000008,	// loop end block
			INQUEUE		= 0x00000010,	// reserved for driver
		};

		System::IntPtr		data;			// pointer to locked data buffer
		System::UInt32		bufferLength;	// length of data buffer
		System::UInt32		bytesRecorded;	// used for input only
		System::IntPtr		user;			// for client's use
		FLAG				flags;			// assorted flags (see defines)
		System::UInt32		loops;			// loop control counter
		System::IntPtr		next;			// reserved for driver
		System::IntPtr		reserved;		// reserved for driver

		virtual System::String^ ToString() override
		{
			return 
				"data["+data+"] bufferLength["+bufferLength+"] bytesRecorded["+bytesRecorded+"] user["+user+"] flags["+flags.ToString("F")+"] loops["+loops+"] next["+next+"] reserved["+reserved+"]";
		}
	};

	//defines for dwFormat field of WAVEINCAPS and WAVEOUTCAPS
	[System::FlagsAttribute]
	public enum class WAVE_FORMAT: System::UInt32 {
		FORMAT_INVALID	= 0x00000000,	// invalid format
		FORMAT_1M08		= 0x00000001,	// 11.025 kHz, Mono,   8-bit 
		FORMAT_1S08		= 0x00000002,	// 11.025 kHz, Stereo, 8-bit 
		FORMAT_1M16		= 0x00000004,	// 11.025 kHz, Mono,   16-bit
		FORMAT_1S16		= 0x00000008,	// 11.025 kHz, Stereo, 16-bit
		FORMAT_2M08		= 0x00000010,	// 22.05  kHz, Mono,   8-bit 
		FORMAT_2S08		= 0x00000020,	// 22.05  kHz, Stereo, 8-bit 
		FORMAT_2M16		= 0x00000040,	// 22.05  kHz, Mono,   16-bit
		FORMAT_2S16		= 0x00000080,	// 22.05  kHz, Stereo, 16-bit
		FORMAT_4M08		= 0x00000100,	// 44.1   kHz, Mono,   8-bit 
		FORMAT_4S08		= 0x00000200,	// 44.1   kHz, Stereo, 8-bit 
		FORMAT_4M16		= 0x00000400,	// 44.1   kHz, Mono,   16-bit
		FORMAT_4S16		= 0x00000800,	// 44.1   kHz, Stereo, 16-bit
		FORMAT_44M08	= 0x00000100,	// 44.1   kHz, Mono,   8-bit 
		FORMAT_44S08	= 0x00000200,	// 44.1   kHz, Stereo, 8-bit 
		FORMAT_44M16	= 0x00000400,	// 44.1   kHz, Mono,   16-bit
		FORMAT_44S16	= 0x00000800,	// 44.1   kHz, Stereo, 16-bit
		FORMAT_48M08	= 0x00001000,	// 48     kHz, Mono,   8-bit 
		FORMAT_48S08	= 0x00002000,	// 48     kHz, Stereo, 8-bit 
		FORMAT_48M16	= 0x00004000,	// 48     kHz, Mono,   16-bit
		FORMAT_48S16	= 0x00008000,	// 48     kHz, Stereo, 16-bit
		FORMAT_96M08	= 0x00010000,	// 96     kHz, Mono,   8-bit 
		FORMAT_96S08	= 0x00020000,	// 96     kHz, Stereo, 8-bit 
		FORMAT_96M16	= 0x00040000,	// 96     kHz, Mono,   16-bit
		FORMAT_96S16	= 0x00080000,	// 96     kHz, Stereo, 16-bit
	};

	WIN32_DLL_STRUCTLAYOUT public ref struct WaveOutCapabilities {
		//flags for dwSupport field of WAVEOUTCAPS
		[System::FlagsAttribute]
		enum class WAVECAPS: System::UInt32 {
			PITCH			= 0x0001,	// supports pitch control
			PLAYBACKRATE	= 0x0002,	// supports playback rate control
			VOLUME			= 0x0004,	// supports volume control
			LRVOLUME		= 0x0008,	// separate left-right volume control
			SYNC			= 0x0010,
			SAMPLEACCURATE	= 0x0020,
		};

		System::UInt16			manufacturerID;
		System::UInt16			productID;
		System::UInt32			driverVersion;
		[InteropServices::MarshalAs(InteropServices::UnmanagedType::ByValTStr, SizeConst=32)]
				System::String^		productName;
		WAVE_FORMAT				formats;
		System::UInt16			channels;
		System::UInt16			reserved1;
		WAVECAPS				support;
		Guiddef::Guid			manufacturerGuid;      // for extensible MID mapping 
		Guiddef::Guid			productGuid;           // for extensible PID mapping
		Guiddef::Guid			nameGuid;              // for name lookup in registry
	};

	WIN32_DLL_STRUCTLAYOUT public ref struct WaveInCapabilities {
		System::UInt16			manufacturerID;
		System::UInt16			productID;
		System::UInt32			driverVersion;
		[InteropServices::MarshalAs(InteropServices::UnmanagedType::ByValTStr, SizeConst=32)]
			System::String^		productName;
		WAVE_FORMAT				formats;
		System::UInt16			channels;
		System::UInt16			reserved1;
		Guiddef::Guid			manufacturerGuid;      // for extensible MID mapping 
		Guiddef::Guid			productGuid;           // for extensible PID mapping
		Guiddef::Guid			nameGuid;              // for name lookup in registry
	};


	// general extended waveform format structure
	// Use this for all NON PCM formats
	// (information common to all formats)
	WIN32_DLL_STRUCTLAYOUT public value struct WaveFormatEx
	{
		// flags for wFormatTag field of WAVEFORMAT
		enum class FORMAT: System::UInt16 {
			PCM			= 0x0001,
			ADPCM		= 0x0002,
			IEEE_FLOAT	= 0x0003,

			EXTENSIBLE	= 0xFFFE,
		};

		FORMAT			formatType;					// format type
		System::UInt16	channels;					// number of channels (i.e. mono, stereo...)
		System::UInt32	samplesPerSecond;			// sample rate
		System::UInt32	averageBytesPerSecond;		// for buffer estimation
		System::UInt16	blockAlign;					// block size of data
		System::UInt16	bitsPerSample;				// Number of bits per sample of mono data
		System::UInt16	size;						// The count in bytes of the size of extra information (after cbSize)

		virtual System::String^ ToString() override
		{
			return 
				"formatType["+formatType.ToString("F")+"] channels["+channels+"] samplesPerSecond["+samplesPerSecond+"] averageBytesPerSecond["+averageBytesPerSecond+"] blockAlign["+blockAlign+"] bitsPerSample["+bitsPerSample+"] size["+size+"]";
		}
	};

	WIN32_DLL_STRUCTLAYOUT public value struct WaveFormat
	{
		WaveFormatEx::FORMAT	formatType;				// format type
		System::UInt16			channels;				// number of channels (i.e. mono, stereo...)
		System::UInt32			samplesPerSecond;		// sample rate
		System::UInt32			averageBytesPerSecond;	// for buffer estimation
		System::UInt16			blockAlign;				// block size of data

		virtual System::String^ ToString() override
		{
			return 
				"formatType["+formatType.ToString("F")+"] channels["+channels+"] samplesPerSecond["+samplesPerSecond+"] averageBytesPerSecond["+averageBytesPerSecond+"] blockAlign["+blockAlign+"]";
		}
	};

	WIN32_DLL_STRUCTLAYOUT public value struct PcmWaveFormat
	{
		WaveFormat		wf;
		System::UInt16	bitsPerSample;				// Number of bits per sample of mono data

		virtual System::String^ ToString() override
		{
			return 
				"wf["+wf.ToString()+"] bitsPerSample["+bitsPerSample+"]";
		}
	};

	//
	//  New wave format development should be based on the
	//  WAVEFORMATEXTENSIBLE structure. WAVEFORMATEXTENSIBLE allows you to
	//  avoid having to register a new format tag with Microsoft. Simply
	//  define a new GUID value for the WAVEFORMATEXTENSIBLE.SubFormat field
	//  and use WAVE_FORMAT_EXTENSIBLE in the
	//  WAVEFORMATEXTENSIBLE.Format.wFormatTag field.
	//
	WIN32_DLL_STRUCTLAYOUT public value struct WaveFormatExtensiblePart
	{
		[System::FlagsAttribute]
		enum class SPEAKER: System::UInt32 {
			FRONT_LEFT				= 0x00000001,
			FRONT_RIGHT				= 0x00000002,
			FRONT_CENTER			= 0x00000004,
			LOW_FREQUENCY			= 0x00000008,
			BACK_LEFT				= 0x00000010,
			BACK_RIGHT				= 0x00000020,
			FRONT_LEFT_OF_CENTER	= 0x00000040,
			FRONT_RIGHT_OF_CENTER	= 0x00000080,
			BACK_CENTER				= 0x00000100,
			SIDE_LEFT				= 0x00000200,
			SIDE_RIGHT				= 0x00000400,
			TOP_CENTER				= 0x00000800,
			TOP_FRONT_LEFT			= 0x00001000,
			TOP_FRONT_CENTER		= 0x00002000,
			TOP_FRONT_RIGHT			= 0x00004000,
			TOP_BACK_LEFT			= 0x00008000,
			TOP_BACK_CENTER			= 0x00010000,
			TOP_BACK_RIGHT			= 0x00020000,
			RESERVED				= 0x7FFC0000,	// Bit mask locations reserved for future use
			ALL						= 0x80000000,	// Used to specify that any possible permutation of speaker configurations
		};

		System::UInt16	validBitsPerSample;
		SPEAKER			channelMask; 
		Guiddef::Guid	subFormat;

		virtual System::String^ ToString() override
		{
			return 
				"validBitsPerSample["+validBitsPerSample+"] channelMask["+channelMask.ToString("F")+"] subFormat["+subFormat+"]";
		}
	};
	

	//
	//  extended waveform format structure used for all non-PCM formats. this
	//  structure is common to all non-PCM formats.
	//
	WIN32_DLL_STRUCTLAYOUT public value struct WaveFormatExtensible
	{
		WaveFormatEx				wfe;
		WaveFormatExtensiblePart	exp;

		virtual System::String^ ToString() override
		{
			return 
				"wfe["+wfe.ToString()+"] exp["+exp.ToString()+"]";
		}

	};

	public value class Function
	{
	public:
		[System::Security::Permissions::SecurityPermissionAttribute(System::Security::Permissions::SecurityAction::InheritanceDemand, UnmanagedCode = true)]
		ref class MidiOut
			: public Microsoft::Win32::SafeHandles::SafeHandleZeroOrMinusOneIsInvalid
		{
		public:
			MidiOut()
				: Microsoft::Win32::SafeHandles::SafeHandleZeroOrMinusOneIsInvalid(true)
			{
				#ifdef _DEBUG
					System::Console::WriteLine("[{0}]",__FUNCTION__);
				#endif
			};

		protected:
			[ConstrainedExecution::ReliabilityContract(ConstrainedExecution::Consistency::WillNotCorruptState, ConstrainedExecution::Cer::MayFail)]
			virtual System::Boolean ReleaseHandle() override
			{
				#ifdef _DEBUG
					System::Console::WriteLine("[{0}]",__FUNCTION__);
				#endif

				MMRESULT mmResult = midiOutClose(this->handle);

				#ifdef _DEBUG
					System::Console::WriteLine("[{0}] midiOutClose result {1}",__FUNCTION__, mmResult);
				#endif
				return (mmResult == Interop::Winmm::MMRESULT::NOERROR);
			};
		};

		WIN_MM_DLLIMPORT static System::UInt32 midiOutGetNumDevs();

		WIN_MM_DLLIMPORT static MMRESULT midiOutGetDevCaps( 
			[InteropServices::In]	System::UInt32			uDeviceID, 
			[InteropServices::Out]	MidiOutCapabilities^	pmoc, 
			[InteropServices::In]	System::UInt32			cbmoc
		);

		WIN_MM_DLLIMPORT static MMRESULT midiOutGetErrorText(
			[InteropServices::In]	MMRESULT						mmrError, 
			[InteropServices::Out]	System::Text::StringBuilder^	pszText, 
			[InteropServices::In]	System::UInt32					cchText
		);

		WIN_MM_DLLIMPORT static MMRESULT midiOutOpen(
			[InteropServices::Out]	MidiOut^%					phmo, 
			[InteropServices::In]	System::UInt32				uDeviceID, 
			[InteropServices::In][InteropServices::MarshalAs(InteropServices::UnmanagedType::FunctionPtr)]
									DriverCallBack::Delegate^	dwCallback, 
			[InteropServices::In]	System::IntPtr				dwInstance, 
			[InteropServices::In]	DriverCallBack::TYPE		fdwOpen
		);
	private:
		WIN_MM_DLLIMPORT static MMRESULT midiOutClose(
			[InteropServices::In]	System::IntPtr hmo
		);
	public:
		WIN_MM_DLLIMPORT static MMRESULT midiOutShortMsg(
			[InteropServices::In]	MidiOut^		hmo,
			[InteropServices::In]	System::UInt32	dwMsg
		);

		WIN_MM_DLLIMPORT static MMRESULT midiOutPrepareHeader(
			[InteropServices::In]	MidiOut^		hmo,
			[InteropServices::In]	System::IntPtr	pmh, 
			[InteropServices::In]	System::UInt32	cbmh
		);

		WIN_MM_DLLIMPORT static MMRESULT midiOutUnprepareHeader(
			[InteropServices::In]	MidiOut^		hmo,
			[InteropServices::In]	System::IntPtr	pmh, 
			[InteropServices::In]	System::UInt32	cbmh
		);

		WIN_MM_DLLIMPORT static MMRESULT midiOutLongMsg(
			[InteropServices::In]	MidiOut^		hmo,
			[InteropServices::In]	System::IntPtr	pmh, 
			[InteropServices::In]	System::UInt32	cbmh
		);

		WIN_MM_DLLIMPORT static MMRESULT midiOutReset(
			[InteropServices::In]	MidiOut^	hmo
		);

		//WINMMAPI MMRESULT WINAPI midiOutCachePatches( __in HMIDIOUT hmo, __in UINT uBank, __in_ecount(MIDIPATCHSIZE) LPWORD pwpa, __in UINT fuCache);
		//WINMMAPI MMRESULT WINAPI midiOutCacheDrumPatches( __in HMIDIOUT hmo, __in UINT uPatch, __in_ecount(MIDIPATCHSIZE) LPWORD pwkya, __in UINT fuCache);
		
		WIN_MM_DLLIMPORT static MMRESULT midiOutGetID(
			[InteropServices::In]	MidiOut^		hmo,
			[InteropServices::Out]	System::UInt32%	puDeviceID
		);

		WIN_MM_DLLIMPORT static MMRESULT midiOutMessage(
			[InteropServices::In]	MidiOut^		hmo,
			[InteropServices::In]	System::UInt32	uMsg, 
			[InteropServices::In]	System::IntPtr	dw1, 
			[InteropServices::In]	System::IntPtr	dw2
		);

		[System::Security::Permissions::SecurityPermissionAttribute(System::Security::Permissions::SecurityAction::InheritanceDemand, UnmanagedCode = true)]
		ref class MidiIn
			: public Microsoft::Win32::SafeHandles::SafeHandleZeroOrMinusOneIsInvalid
		{
		public:
			MidiIn()
				: Microsoft::Win32::SafeHandles::SafeHandleZeroOrMinusOneIsInvalid(true)
			{
				#ifdef _DEBUG
					System::Console::WriteLine("[{0}]",__FUNCTION__);
				#endif
			};

		protected:
			[ConstrainedExecution::ReliabilityContract(ConstrainedExecution::Consistency::WillNotCorruptState, ConstrainedExecution::Cer::MayFail)]
			virtual System::Boolean ReleaseHandle() override
			{
				#ifdef _DEBUG
					System::Console::WriteLine("[{0}]",__FUNCTION__);
				#endif

				MMRESULT mmResult = midiInClose(this->handle);

				#ifdef _DEBUG
					System::Console::WriteLine("[{0}] midiInClose result {1}",__FUNCTION__, mmResult);
				#endif
				return (mmResult == Interop::Winmm::MMRESULT::NOERROR);
			};
		};

		WIN_MM_DLLIMPORT static System::UInt32 midiInGetNumDevs();

		WIN_MM_DLLIMPORT static MMRESULT midiInGetDevCaps( 
			[InteropServices::In]	System::UInt32		uDeviceID, 
			[InteropServices::Out]	MidiInCapabilities^	pmic, 
			[InteropServices::In]	System::UInt32		cbmoc
		);

		WIN_MM_DLLIMPORT static MMRESULT midiInGetErrorText(
			[InteropServices::In]	MMRESULT						mmrError, 
			[InteropServices::Out]	System::Text::StringBuilder^	pszText, 
			[InteropServices::In]	System::UInt32					cchText
		);

		WIN_MM_DLLIMPORT static MMRESULT midiInOpen(
			[InteropServices::Out]	MidiIn^%					phmi, 
			[InteropServices::In]	System::UInt32				uDeviceID, 
			[InteropServices::In][InteropServices::MarshalAs(InteropServices::UnmanagedType::FunctionPtr)]
									DriverCallBack::Delegate^	dwCallback, 
			[InteropServices::In]	System::IntPtr				dwInstance, 
			[InteropServices::In]	DriverCallBack::TYPE		fdwOpen
		);
	private:
		WIN_MM_DLLIMPORT static MMRESULT midiInClose(
			[InteropServices::In]	System::IntPtr hmi
		);
	public:
		WIN_MM_DLLIMPORT static MMRESULT midiInPrepareHeader(
			[InteropServices::In]	MidiIn^			hmi,
			[InteropServices::In]	System::IntPtr	pmh, 
			[InteropServices::In]	System::UInt32	cbmh
		);

		WIN_MM_DLLIMPORT static MMRESULT midiInUnprepareHeader(
			[InteropServices::In]	MidiIn^			hmi,
			[InteropServices::In]	System::IntPtr	pmh, 
			[InteropServices::In]	System::UInt32	cbmh
		);

		WIN_MM_DLLIMPORT static MMRESULT midiInAddBuffer(
			[InteropServices::In]	MidiIn^			hmi,
			[InteropServices::In]	System::IntPtr	pmh, 
			[InteropServices::In]	System::UInt32	cbmh
		);

		WIN_MM_DLLIMPORT static MMRESULT midiInStart(
			[InteropServices::In]	MidiIn^	hmi
		);

		WIN_MM_DLLIMPORT static MMRESULT midiInStop(
			[InteropServices::In]	MidiIn^	hmi
		);

		WIN_MM_DLLIMPORT static MMRESULT midiInReset(
			[InteropServices::In]	MidiIn^	hmi
		);

		WIN_MM_DLLIMPORT static MMRESULT midiInGetID(
			[InteropServices::In]	MidiIn^			hmi,
			[InteropServices::Out]	System::UInt32%	puDeviceID
		);
		
		WIN_MM_DLLIMPORT static MMRESULT midiInMessage(
			[InteropServices::In]	MidiIn^			hmi,
			[InteropServices::In]	System::UInt32	uMsg, 
			[InteropServices::In]	System::IntPtr	dw1, 
			[InteropServices::In]	System::IntPtr	dw2
		);


		//WINMMAPI MMRESULT WINAPI timeGetSystemTime( __out_bcount(cbmmt) LPMMTIME pmmt, __in UINT cbmmt);


		WIN_MM_DLLIMPORT static System::UInt32 timeGetTime();
				
		WIN_MM_DLLIMPORT static System::UInt32 timeSetEvent(
			[InteropServices::In]	System::UInt32				uDelay, 
			[InteropServices::In]	System::UInt32				uResolution, 
			[InteropServices::In][InteropServices::MarshalAs(InteropServices::UnmanagedType::FunctionPtr)]
									TimerCallBack::Delegate^	fptc, 
			[InteropServices::In]	System::IntPtr				dwUser, 
			[InteropServices::In]	TimerCallBack::TYPE			fuEvent
		);

		WIN_MM_DLLIMPORT static MMRESULT timeKillEvent(
			[InteropServices::In]	System::UInt32	uTimerID
		);

		WIN_MM_DLLIMPORT static MMRESULT timeGetDevCaps( 
			[InteropServices::Out]	TimeCapabilities^	ptc, 
			[InteropServices::In]	System::UInt32		cbtc
		);
		/*
		WIN_MM_DLLIMPORT static MMRESULT timeBeginPeriod(
			[InteropServices::In]	System::UInt32	uPeriod
		);

		WIN_MM_DLLIMPORT static MMRESULT timeEndPeriod(
			[InteropServices::In]	System::UInt32	uPeriod
		);
		*/

		[System::Security::Permissions::SecurityPermissionAttribute(System::Security::Permissions::SecurityAction::InheritanceDemand, UnmanagedCode = true)]
		ref class WaveOut
			: public Microsoft::Win32::SafeHandles::SafeHandleZeroOrMinusOneIsInvalid
		{
		public:
			WaveOut()
				: Microsoft::Win32::SafeHandles::SafeHandleZeroOrMinusOneIsInvalid(true)
			{
				#ifdef _DEBUG
					System::Console::WriteLine("[{0}]",__FUNCTION__);
				#endif
			};

		protected:
			[ConstrainedExecution::ReliabilityContract(ConstrainedExecution::Consistency::WillNotCorruptState, ConstrainedExecution::Cer::MayFail)]
			virtual System::Boolean ReleaseHandle() override
			{
				#ifdef _DEBUG
					System::Console::WriteLine("[{0}]",__FUNCTION__);
				#endif

				MMRESULT mmResult = waveOutClose(this->handle);

				#ifdef _DEBUG
					System::Console::WriteLine("[{0}] waveOutClose result {1}",__FUNCTION__, mmResult);
				#endif
				return (mmResult == Interop::Winmm::MMRESULT::NOERROR);
			};
		};

		WIN_MM_DLLIMPORT static System::UInt32 waveOutGetNumDevs();

		WIN_MM_DLLIMPORT static MMRESULT waveOutGetDevCaps( 
			[InteropServices::In]	System::UInt32			uDeviceID, 
			[InteropServices::Out]	WaveOutCapabilities^	pwoc, 
			[InteropServices::In]	System::UInt32			cbwoc
		);

		WIN_MM_DLLIMPORT static MMRESULT waveOutGetVolume(
			[InteropServices::In]	WaveOut^		hwo,
			[InteropServices::Out]	System::UInt32%	pdwVolume
		);

		WIN_MM_DLLIMPORT static MMRESULT waveOutSetVolume(
			[InteropServices::In]	WaveOut^		hwo,
			[InteropServices::In]	System::UInt32	pdwVolume
		);

		WIN_MM_DLLIMPORT static MMRESULT waveOutGetErrorText(
			[InteropServices::In]	MMRESULT						mmrError, 
			[InteropServices::Out]	System::Text::StringBuilder^	pszText, 
			[InteropServices::In]	System::UInt32					cchText
		);

		WIN_MM_DLLIMPORT static MMRESULT waveOutOpen(
			[InteropServices::Out]	WaveOut^%					phwo, 
			[InteropServices::In]	System::UInt32				uDeviceID, 
			[InteropServices::In]	WaveFormatExtensible%		pwfx, 
			[InteropServices::In][InteropServices::MarshalAs(InteropServices::UnmanagedType::FunctionPtr)]
									DriverCallBack::Delegate^	dwCallback, 
			[InteropServices::In]	System::IntPtr				dwInstance, 
			[InteropServices::In]	DriverCallBack::TYPE		fdwOpen
		);
	private:
		WIN_MM_DLLIMPORT static MMRESULT waveOutClose(
			[InteropServices::In]	System::IntPtr	hwo
		);
	public:
		WIN_MM_DLLIMPORT static MMRESULT waveOutPrepareHeader(
			[InteropServices::In]	WaveOut^		hwo,
			[InteropServices::In]	System::IntPtr	pwh, 
			[InteropServices::In]	System::UInt32	cbwh
		);

		WIN_MM_DLLIMPORT static MMRESULT waveOutUnprepareHeader(
			[InteropServices::In]	WaveOut^		hwo,
			[InteropServices::In]	System::IntPtr	pwh, 
			[InteropServices::In]	System::UInt32	cbwh
		);

		WIN_MM_DLLIMPORT static MMRESULT waveOutWrite(
			[InteropServices::In]	WaveOut^		hwo,
			[InteropServices::In]	System::IntPtr	pwh, 
			[InteropServices::In]	System::UInt32	cbwh
		);

		WIN_MM_DLLIMPORT static MMRESULT waveOutPause(
			[InteropServices::In]	WaveOut^		hwo
		);

		WIN_MM_DLLIMPORT static MMRESULT waveOutRestart(
			[InteropServices::In]	WaveOut^		hwo
		);

		WIN_MM_DLLIMPORT static MMRESULT waveOutReset(
			[InteropServices::In]	WaveOut^		hwo
		);

		WIN_MM_DLLIMPORT static MMRESULT waveOutBreakLoop(
			[InteropServices::In]	WaveOut^		hwo
		);

		/*
		WINMMAPI MMRESULT WINAPI waveOutGetPosition( __in HWAVEOUT hwo, __inout_bcount(cbmmt) LPMMTIME pmmt, __in UINT cbmmt);
		WINMMAPI MMRESULT WINAPI waveOutGetPitch( __in HWAVEOUT hwo, __out LPDWORD pdwPitch);
		WINMMAPI MMRESULT WINAPI waveOutSetPitch( __in HWAVEOUT hwo, __in DWORD dwPitch);
		WINMMAPI MMRESULT WINAPI waveOutGetPlaybackRate( __in HWAVEOUT hwo, __out LPDWORD pdwRate);
		WINMMAPI MMRESULT WINAPI waveOutSetPlaybackRate( __in HWAVEOUT hwo, __in DWORD dwRate);
		WINMMAPI MMRESULT WINAPI waveOutGetID( __in HWAVEOUT hwo, __out LPUINT puDeviceID);
		WINMMAPI MMRESULT WINAPI waveOutMessage( __in_opt HWAVEOUT hwo, __in UINT uMsg, __in DWORD_PTR dw1, __in DWORD_PTR dw2);
		*/

		[System::Security::Permissions::SecurityPermissionAttribute(System::Security::Permissions::SecurityAction::InheritanceDemand, UnmanagedCode = true)]
		ref class WaveIn
			: public Microsoft::Win32::SafeHandles::SafeHandleZeroOrMinusOneIsInvalid
		{
		public:
			WaveIn()
				: Microsoft::Win32::SafeHandles::SafeHandleZeroOrMinusOneIsInvalid(true)
			{
				#ifdef _DEBUG
					System::Console::WriteLine("[{0}]",__FUNCTION__);
				#endif
			};

		protected:
			[ConstrainedExecution::ReliabilityContract(ConstrainedExecution::Consistency::WillNotCorruptState, ConstrainedExecution::Cer::MayFail)]
			virtual System::Boolean ReleaseHandle() override
			{
				#ifdef _DEBUG
					System::Console::WriteLine("[{0}]",__FUNCTION__);
				#endif

				MMRESULT mmResult = waveInClose(this->handle);

				#ifdef _DEBUG
					System::Console::WriteLine("[{0}] waveInClose result {1}",__FUNCTION__, mmResult);
				#endif
				return (mmResult == Interop::Winmm::MMRESULT::NOERROR);
			};
		};

		WIN_MM_DLLIMPORT static System::UInt32 waveInGetNumDevs();

		WIN_MM_DLLIMPORT static MMRESULT waveInGetDevCaps( 
			[InteropServices::In]	System::UInt32		uDeviceID, 
			[InteropServices::Out]	WaveInCapabilities%	pwic, 
			[InteropServices::In]	System::UInt32		cbwic
		);

		WIN_MM_DLLIMPORT static MMRESULT waveInGetErrorText(
			[InteropServices::In]	MMRESULT						mmrError, 
			[InteropServices::Out]	System::Text::StringBuilder^	pszText, 
			[InteropServices::In]	System::UInt32					cchText
		);

		WIN_MM_DLLIMPORT static MMRESULT waveInOpen(
			[InteropServices::Out]	WaveIn^%					phwi, 
			[InteropServices::In]	System::UInt32				uDeviceID, 
			[InteropServices::In]	WaveFormatExtensible%		pwfx, 
			[InteropServices::In][InteropServices::MarshalAs(InteropServices::UnmanagedType::FunctionPtr)]
									DriverCallBack::Delegate^	dwCallback, 
			[InteropServices::In]	System::IntPtr				dwInstance, 
			[InteropServices::In]	DriverCallBack::TYPE		fdwOpen
		);
	private:
		WIN_MM_DLLIMPORT static MMRESULT waveInClose(
			[InteropServices::In]	System::IntPtr	hwi
		);
	public:
		WIN_MM_DLLIMPORT static MMRESULT waveInPrepareHeader(
			[InteropServices::In]	WaveIn^			hwi,
			[InteropServices::In]	System::IntPtr	pwh, 
			[InteropServices::In]	System::UInt32	cbwh
		);

		WIN_MM_DLLIMPORT static MMRESULT waveInUnprepareHeader(
			[InteropServices::In]	WaveIn^			hwi,
			[InteropServices::In]	System::IntPtr	pwh, 
			[InteropServices::In]	System::UInt32	cbwh
		);

		WIN_MM_DLLIMPORT static MMRESULT waveInAddBuffer(
			[InteropServices::In]	WaveIn^			hwi,
			[InteropServices::In]	System::IntPtr	pwh, 
			[InteropServices::In]	System::UInt32	cbwh
		);

		WIN_MM_DLLIMPORT static MMRESULT waveInStart(
			[InteropServices::In]	WaveIn^	hwi
		);

		WIN_MM_DLLIMPORT static MMRESULT waveInStop(
			[InteropServices::In]	WaveIn^	hwi
		);

		WIN_MM_DLLIMPORT static MMRESULT waveInReset(
			[InteropServices::In]	WaveIn^	hwi
		);

		//WINMMAPI MMRESULT WINAPI waveInGetPosition( __in HWAVEIN hwi, __inout_bcount(cbmmt) LPMMTIME pmmt, __in UINT cbmmt);
		//WINMMAPI MMRESULT WINAPI waveInGetID( __in HWAVEIN hwi, __in LPUINT puDeviceID);
		//WINMMAPI MMRESULT WINAPI waveInMessage( __in_opt HWAVEIN hwi, __in UINT uMsg, __in_opt DWORD_PTR dw1, __in_opt DWORD_PTR dw2);


	};

}
}
}
