﻿/*
[momiji music component library]
---------------------------------------------------------------------
Momiji.Sequencer.Midi.Smf.h
	stream component of standard midi file.
---------------------------------------------------------------------
Copyright (C) 2011 tyiki badwell {miria@users.sourceforge.jp}.

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/gpl-3.0.html>.
---------------------------------------------------------------------
*/
#pragma once

#using <mscorlib.dll>
#include "Momiji.Core.Interface.h"

using namespace System::Runtime;

namespace Momiji{
namespace Sequencer {
namespace Midi {
namespace Smf {


	public ref class MetaData abstract
		: public Core::IData
	{
	public:
		enum class META_TYPE: System::Byte
		{
			SEQUENCE_NUMBER		= 0x00,	//  {FF: '00' 02     ss ss}
			TEXT_EVENT			= 0x01,	//  {FF: '01' length text}
			COPYRIGHT_NOTICE	= 0x02,	//  {FF: '02' length text}
			TRACK_NAME			= 0x03,	//  {FF: '03' length text}
			INSTRUMENT_NAME		= 0x04,	//  {FF: '04' length text}
			LYRIC				= 0x05,	//  {FF: '05' length text}
			MARKER				= 0x06,	//  {FF: '06' length text}
			CUE_POINT			= 0x07,	//  {FF: '07' length text}
			PROGRAM_NAME		= 0x08,	//  {FF: '08' length text}
			DEVICE_NAME			= 0x09,	//  {FF: '09' length text}

			PORT				= 0x21,	//  {FF: '21' length PORT ??}	< 2バイト目の役割が不明；
			END_OF_TRACK		= 0x2F,	//  {FF: '2F'  00}
			SET_TEMPO			= 0x51,	//  {FF: '51'  03    tt tt tt   = microceconds/quarter note( Tempo = (60×10^6)÷MetaTempo )}
			SMPTE_OFFSET		= 0x54,	//  {FF: '54'  05    hr mn se fr ff}
			TIME_SIGNATURE		= 0x58,	//  {FF: '58'  04    nn dd cc bb}
			/*		{
						 nn / 2^dd
						 cc :the number of MIDI clocks in a metronome click          (MIDI Clocks / Click)
						 dd :the number of notated 32nd-notes in a MIDI quarter-note (32nd Notes / MIDI Clocks)
								 4分音符中の32分音符の数：8に決まってます。

					}*/
			KEY_SIGNATURE		= 0x59,	//  {FF: '59'  02    sf mi}
			/*		{
					sf = -7 : 7 flats
					sf = -1 : 1 flat
					sf = 0  : key of C
					sf = 1  : 1 sharp
					sf = 7  : 7 sharps

					mi = 0  : major key
					mi = 1  : minor key
				}*/
			
			SEQ_SP_METAEVENT	= 0x7F,	//  {FF: '7F'  length data  ユーザ定義メタ}
		};
	private:
		META_TYPE		_type;
		System::Boolean	_conductor;
	public:
		MetaData(
			META_TYPE		type,
			System::Boolean	conductor
		) : _type(type), _conductor(conductor) {}

		property META_TYPE type				{ META_TYPE get() {return this->_type;} }
		property System::Boolean conductor	{ System::Boolean get() {return this->_conductor;} }
	};

	public ref class MetaSequenceNumber sealed
		: public MetaData
	{
	private:
		System::UInt16 _number;

	public:
		MetaSequenceNumber(
			System::UInt16	number
		): MetaData(META_TYPE::SEQUENCE_NUMBER, false), _number(number) {}

		property System::UInt16	number { System::UInt16 get() {return this->_number;} }
	};

	public ref class MetaText sealed
		: public MetaData
	{
	private:
		System::String^ _text;
	public:
		MetaText(
			System::String^ text
		): MetaData(META_TYPE::TEXT_EVENT, false), _text(text) {}

		property System::String^ text { System::String^ get() {return this->_text;} }
	};

	public ref class MetaCopyrightNotice sealed
		: public MetaData
	{
	private:
		System::String^ _text;
	public:
		MetaCopyrightNotice(
			System::String^ text
		): MetaData(META_TYPE::COPYRIGHT_NOTICE, false), _text(text) {}

		property System::String^ text { System::String^ get() {return this->_text;} }
	};

	public ref class MetaTrackName sealed
		: public MetaData
	{
	private:
		System::String^ _text;
	public:
		MetaTrackName(
			System::String^ text
		): MetaData(META_TYPE::TRACK_NAME, false), _text(text) {}

		property System::String^ text { System::String^ get() {return this->_text;} }
	};
	public ref class MetaInstrumentName sealed
		: public MetaData
	{
	private:
		System::String^ _text;
	public:
		MetaInstrumentName(
			System::String^ text
		): MetaData(META_TYPE::INSTRUMENT_NAME, false), _text(text) {}

		property System::String^ text { System::String^ get() {return this->_text;} }
	};
	public ref class MetaLyric sealed
		: public MetaData
	{
	private:
		System::String^ _text;
	public:
		MetaLyric(
			System::String^ text
		): MetaData(META_TYPE::LYRIC, false), _text(text) {}

		property System::String^ text { System::String^ get() {return this->_text;} }
	};
	public ref class MetaMarker sealed
		: public MetaData
	{
	private:
		System::String^ _text;
	public:
		MetaMarker(
			System::String^ text
		): MetaData(META_TYPE::MARKER, false), _text(text) {}

		property System::String^ text { System::String^ get() {return this->_text;} }
	};
	public ref class MetaCuePoint sealed
		: public MetaData
	{
	private:
		System::String^ _text;
	public:
		MetaCuePoint(
			System::String^ text
		): MetaData(META_TYPE::CUE_POINT, false), _text(text) {}

		property System::String^ text { System::String^ get() {return this->_text;} }
	};
	public ref class MetaProgramName sealed
		: public MetaData
	{
	private:
		System::String^ _text;
	public:
		MetaProgramName(
			System::String^ text
		): MetaData(META_TYPE::PROGRAM_NAME, false), _text(text) {}

		property System::String^ text { System::String^ get() {return this->_text;} }
	};
	public ref class MetaDeviceName sealed
		: public MetaData
	{
	private:
		System::String^ _text;
	public:
		MetaDeviceName(
			System::String^ text
		): MetaData(META_TYPE::DEVICE_NAME, false), _text(text) {}

		property System::String^ text { System::String^ get() {return this->_text;} }
	};

	public ref class MetaPort sealed
		: public MetaData
	{
	private:
		System::UInt16 _number;

	public:
		MetaPort(
			System::UInt16	number
		): MetaData(META_TYPE::PORT, false), _number(number) {}

		property System::UInt16	number { System::UInt16 get() {return this->_number;} }
	};

	public ref class MetaEndOfTrack sealed
		: public MetaData
	{
	public: 
		MetaEndOfTrack(
		): MetaData(META_TYPE::END_OF_TRACK, false) {}
	};

	public ref class MetaSetTempo sealed
		: public MetaData
	{
	private:
		System::UInt32 _usecPerQNote;
	public:
		MetaSetTempo(
			System::UInt32	usecPerQNote
		): MetaData(META_TYPE::SET_TEMPO, true), _usecPerQNote(usecPerQNote) {}

		property System::UInt32	usecPerQNote { System::UInt32 get() {return this->_usecPerQNote;} }
	};

	public ref class MetaSMPTEOffset sealed
		: public MetaData
	{
	private:
		System::Byte _hour;
		System::Byte _minute;
		System::Byte _second;
		System::Byte _frame;
		System::Byte _subFrame;
	public:
		MetaSMPTEOffset(
			System::Byte	hour,
			System::Byte	minute,
			System::Byte	second,
			System::Byte	frame,
			System::Byte	subFrame
		): MetaData(META_TYPE::SMPTE_OFFSET, true), _hour(hour), _minute(minute), _second(second), _frame(frame), _subFrame(subFrame) {}

		property System::Byte	hour { System::Byte get() {return this->_hour;} }
		property System::Byte	minute { System::Byte get() {return this->_minute;} }
		property System::Byte	second { System::Byte get() {return this->_second;} }
		property System::Byte	frame { System::Byte get() {return this->_frame;} }
		property System::Byte	subFrame { System::Byte get() {return this->_subFrame;} }
	};

	public ref class MetaTimeSignature sealed
		: public MetaData
	{
	private:
		System::Byte _bar;
		System::Byte _beat;
		System::Byte _metronome;
		System::Byte _notes;
	public:
		MetaTimeSignature(
			System::Byte	bar,
			System::Byte	beat,
			System::Byte	metronome,
			System::Byte	notes
		): MetaData(META_TYPE::TIME_SIGNATURE, true), _bar(bar), _beat(beat), _metronome(metronome), _notes(notes) {}

		property System::Byte	bar { System::Byte get() {return this->_bar;} }
		property System::Byte	beat { System::Byte get() {return this->_beat;} }
		property System::Byte	metronome { System::Byte get() {return this->_metronome;} }
		property System::Byte	notes { System::Byte get() {return this->_notes;} }
	};

	public ref class MetaKeySignature sealed
		: public MetaData
	{
	private:
		System::SByte _key;
		System::Boolean _minor;
	public:
		MetaKeySignature(
			System::SByte	key,
			System::Boolean	minor
		): MetaData(META_TYPE::TIME_SIGNATURE, true), _key(key), _minor(minor) {}

		property System::SByte	key { System::SByte get() {return this->_key;} }
		property System::Boolean	minor { System::Boolean get() {return this->_minor;} }
	};

	public ref class MetaDefault sealed
		: public MetaData
	{
	private:
		System::Byte			_defaultType;
		array<System::Byte>^	_data;
	public:
		MetaDefault(
			System::Byte			defaultType,
			array<System::Byte>^	data
		): MetaData(META_TYPE::SEQ_SP_METAEVENT, false), _defaultType(defaultType), _data(data) {}

		property System::Byte	defaultType { System::Byte get() {return this->_defaultType;} }
		property array<System::Byte>^	data { array<System::Byte>^ get() {return this->_data;} }
	};

	///<summary></summary>
	///
	///
	///
	public ref class SmfStream
		: public Core::IStream
	{
	public:
		ref class SmfTrack:
			public Core::ITrack
		{
		public:
			ref class SmfPacket:
				public Core::IStreamPacket
			{
			private:
				initonly SmfTrack^			_track;
				initonly System::UInt64		_offset;
				initonly System::UInt64		_nextOffset;
				initonly System::UInt64		_tick;
				initonly System::Byte		_status;
				initonly Core::IData^	_data;
				initonly System::Boolean	_isTerminate;

			private:
				System::UInt32 GetDeltaTime(System::UInt64% tempOffset);
				System::Byte NextByte(System::UInt64% tempOffset);
				void NextByte(array<System::Byte>^ dest, System::UInt64% tempOffset, System::Byte start, System::Byte size);

			public:
				SmfPacket(
					SmfTrack^			track,
					System::UInt64		offset,
					System::UInt64		previousTick,
					System::Byte		previousStatus
				);
				~SmfPacket();

			protected:
				!SmfPacket();

			public:
				property Core::IStreamPacket^	next	{ virtual Core::IStreamPacket^ get(); }
				property System::Boolean		hasNext	{ virtual System::Boolean get() {return !this->_isTerminate;} }
				property Core::IData^			data	{ virtual Core::IData^ get() {return this->_data;} }
				property System::UInt64			tick	{ virtual System::UInt64 get() {return this->_tick;} }
			};

		private:
			SmfStream^	_stream;

			System::UInt64	_headOffset;
			System::UInt64	_tailOffset;

			Core::IStreamPacket^	_headPacket;

			//再生中の状態
			Core::IStreamPacket^	_packet;
			System::Double	_nowTick;

			System::Collections::Generic::Dictionary<MetaData::META_TYPE, MetaData^>^ _metaData;

		public:
			SmfTrack(
				SmfStream^ stream,
				System::UInt64 headOffset, 
				System::UInt64 tailOffset
			);
			virtual ~SmfTrack();
		protected:
			!SmfTrack();

		public:
			property Core::IStreamPacket^ head { virtual Core::IStreamPacket^ get(){ return this->_headPacket; } }

			virtual void Rewind();
			virtual array<Core::IStreamPacket^>^ GetStreamPacket(System::Double deltaTime);
		};

		enum class TYPE: System::UInt32 
		{
			HEADER		= 0x4D546864L,	//	MThd: length format ntrks division
			HEADER_R	= 0x6468544DL,	//		読み込み用
			TRACK		= 0x4D54726BL,	//	MTrk: length
			TRACK_R		= 0x6B72544DL,	//	MTrk: length
		};

		enum class FORMAT: System::UInt16 
		{
			SINGLE			= 0,	//	single-track
			MULTIPLE_SYNC	= 1,	//	multiple tracks, synchronous
			MULTIPLE_ASYNC	= 2,	//	multiple tracks, asynchronous
		};

		//SMFHeader Chunk
		WIN32_DLL_STRUCTLAYOUT value struct SMFHEADER_CHUNK 
		{
			System::UInt32	dwType;
			System::UInt32	dwLength;
			System::UInt16	wFormat;
			System::UInt16	wNtrks;
			System::UInt16	wDivition;
		};

		//SMFTrack chunk
		WIN32_DLL_STRUCTLAYOUT value struct SMFTRACK_CHUNK 
		{
			System::UInt32	dwType;
			System::UInt32	dwLength;
		};

	private:
		System::IO::MemoryMappedFiles::MemoryMappedFile^			_mmap;
		System::IO::MemoryMappedFiles::MemoryMappedViewAccessor^	_view;

		SMFHEADER_CHUNK	_smfHeader;
		array<Core::ITrack^>^	_tracks;

	public:
		SmfStream(System::String^ path);
		virtual ~SmfStream();

	protected:
		!SmfStream();

	private:
		void Open(System::String^ path);
		void CheckSMF();

	private:
		System::Double	_speed;

	public:
		virtual array<Core::IStreamPacket^>^ GetStreamPacket(System::Double deltaTime);
		virtual void Rewind();
	};

	public ref class SmfException
		: System::Exception
	{
	public:
		SmfException(System::String^ v): System::Exception(v){};
	};

}
}
}
}
