﻿/*
[momiji music component library]
---------------------------------------------------------------------
Momiji.Sequencer.Midi.Smf.cpp
	stream component of standard midi file.
---------------------------------------------------------------------
Copyright (C) 2011 tyiki badwell {miria@users.sourceforge.jp}.

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/gpl-3.0.html>.
---------------------------------------------------------------------
*/
#include "StdAfx.h"

#include "Momiji.Interop.Winmm.h"
#include "Momiji.Sequencer.Wave.h"

namespace Momiji{
namespace Sequencer {
namespace Wave {

	WaveStream::WaveStream(System::String^ path)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif
		this->Open(path);
		this->CheckWave();
	}

	WaveStream::~WaveStream()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif
		this->!WaveStream();
	}

	WaveStream::!WaveStream()
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}]",__FUNCTION__);
		#endif
		
		if (this->_view != nullptr)
		{
			delete this->_view;
			this->_view = nullptr;
		}
		if (this->_mmap != nullptr)
		{
			delete this->_mmap;
			this->_mmap = nullptr;
		}
	}

	void WaveStream::Open(System::String^ path)
	{
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] {1}",__FUNCTION__, path);
		#endif

		auto info = gcnew System::IO::FileInfo(path);

		this->_mmap = 
			System::IO::MemoryMappedFiles::MemoryMappedFile::CreateFromFile(
				info->FullName,
				System::IO::FileMode::Open,
				"momiji.wav",
				info->Length,
				System::IO::MemoryMappedFiles::MemoryMappedFileAccess::Read
			);

		this->_view =
			this->_mmap->CreateViewAccessor(
				0,
				0,
				System::IO::MemoryMappedFiles::MemoryMappedFileAccess::Read
			);
	}


	void WaveStream::CheckWave()
	{
		System::Int64 offset = 0;
		RIFFHEADER_CHUNK riffHeader;

		this->_view->Read<RIFFHEADER_CHUNK>(offset, riffHeader);

		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] [dwType   {1,8:X}",__FUNCTION__, riffHeader.dwType);
			System::Console::WriteLine("[{0}] [dwLength {1}]",__FUNCTION__, riffHeader.dwLength);
			System::Console::WriteLine("[{0}] [dwFormat {1,8:X}",__FUNCTION__, riffHeader.dwFormat);
		#endif

		if (!System::Enum::IsDefined(TYPE::typeid,	 riffHeader.dwType)) {throw gcnew WaveException("RIFFではない");}
		if (!System::Enum::IsDefined(FORMAT::typeid, riffHeader.dwFormat)) {throw gcnew WaveException("WAVEではない");}

		offset += 12;

		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] [offset   {1}]",__FUNCTION__, offset);
		#endif

		while(offset < riffHeader.dwLength)
		{
			RIFFSUB_CHUNK subTrack;
			this->_view->Read<RIFFSUB_CHUNK>(offset, subTrack);
			offset += 8;

			#ifdef _DEBUG
				System::Console::WriteLine("[{0}] [dwType   {1,8:X}]",__FUNCTION__, subTrack.dwType);
				System::Console::WriteLine("[{0}] [dwLength {1}]",__FUNCTION__, subTrack.dwLength);
			#endif

			//対応していないサブチャンクがあっても、気にしないことにする
			//if (!System::Enum::IsDefined(SUB::typeid, subTrack.dwType)) {throw gcnew WaveException("不明なチャンク");}

			switch (subTrack.dwType) {
			case SUB::fmt__R:
				{
					static const auto pcmSize = safe_cast<System::UInt32>(InteropServices::Marshal::SizeOf(Momiji::Interop::Winmm::PcmWaveFormat::typeid));

					if (subTrack.dwLength == pcmSize)
					{
						Momiji::Interop::Winmm::PcmWaveFormat pcm;
						this->_view->Read<Momiji::Interop::Winmm::PcmWaveFormat>(offset, pcm);

						this->_wfx.formatType				= pcm.wf.formatType;
						this->_wfx.channels					= pcm.wf.channels;
						this->_wfx.samplesPerSecond			= pcm.wf.samplesPerSecond;
						this->_wfx.averageBytesPerSecond	= pcm.wf.averageBytesPerSecond;
						this->_wfx.blockAlign				= pcm.wf.blockAlign;
						this->_wfx.bitsPerSample			= pcm.bitsPerSample;
					}
					else if (subTrack.dwLength > pcmSize)
					{
						this->_view->Read<Momiji::Interop::Winmm::WaveFormatEx>(offset, this->_wfx);
					}

					#ifdef _DEBUG
						System::Console::WriteLine("[{0}] [wfx {1}]",__FUNCTION__, this->_wfx.ToString());
					#endif

					break;
				}

			case SUB::data_R:
				{
					this->_dataStartPosition = offset;
					this->_dataSize = subTrack.dwLength;
					this->_dataSeekPosition = offset;
					break;
				}
			}

			offset += subTrack.dwLength;
			#ifdef _DEBUG
				System::Console::WriteLine("[{0}] [offset {1}]",__FUNCTION__, offset);
			#endif
		}
	}

	//System::UInt32 WaveStream::Read(array<System::Byte>^ buffer, System::UInt32 samplesPerBuffer)
	System::UInt32 WaveStream::Read(Momiji::Core::Buffer::BufferPool<array<System::Byte>^>::Buffer^ buffer, System::UInt32 samplesPerBuffer)
	{
		auto size = (samplesPerBuffer * this->_wfx.channels * this->_wfx.bitsPerSample / 8);
		auto b = buffer->GetBuffer();
		if (b->Length < size) {
			size = b->Length;
			#ifdef _DEBUG
				System::Console::WriteLine("[{0}] 指定したサンプル分を格納するだけのバッファが足りないので、バッファまでの読み込みにします[samplesPerBuffer {1}][buffer length {1}]",__FUNCTION__, samplesPerBuffer, b->Length);
			#endif
		}

		auto leftSize = this->_dataSize - safe_cast<System::UInt32>(this->_dataSeekPosition - this->_dataStartPosition);
		if (leftSize < size) {
			size = leftSize;
		}
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] [size {1}]",__FUNCTION__, size);
		#endif

		//auto result = gcnew array<System::Byte>(size);
		auto readBytes = this->_view->ReadArray<System::Byte>(this->_dataSeekPosition, b, 0, size);
		#ifdef _DEBUG
			System::Console::WriteLine("[{0}] [readBytes {1}]",__FUNCTION__, readBytes);
		#endif

		this->_dataSeekPosition += readBytes;

		return readBytes;
	}

}
}
}
