#include "stdafx.h"
#include <BaseTsd.h>

#include "BMUtil.h"

namespace NTextFinder {
namespace NativeLib {
namespace BMUtil {

    class CStringReader {
    public:
        CStringReader( const wchar_t* str, int length ) : m_begin( str ), m_end( str + length ), m_pos( str ) {}

        bool ReadLine( const wchar_t** begin, const wchar_t** end )
        {
            const wchar_t* ptr = m_pos;
            while (ptr < m_end) {
                const wchar_t ch = *ptr;
                if (ch == L'\r' || ch == L'\n') {
                    *begin = m_pos;
                    *end = ptr;
                    m_pos = ptr + 1;
                    if (ch == L'\r' && m_pos < m_end && m_pos[0] == L'\n') {
                        m_pos++;
                    }
                    return true;
                }
                ptr++;
            }
            if (ptr > m_pos) {
                *begin = m_pos;
                *end = ptr;
                m_pos = ptr;
                return true;
            }
            return false;
        }

    private:
        const wchar_t* m_begin;
        const wchar_t* m_end;
        const wchar_t* m_pos;
    };

    typedef unsigned char byte;

    __inline bool IsWord( const byte* is_word, unsigned int ch )
    {
        return ((is_word[ch / 8] & (1 << (ch & 7))) != 0);
    }

    bool IsOnWordBoundary( const byte* is_word, const wchar_t* begin, const wchar_t* end, const wchar_t* ptr, int length )
    {
        // check the beginning
        if (ptr > begin) {
            const bool w0 = IsWord( is_word, ptr[-1] );
            const bool w1 = IsWord( is_word, ptr[0] );
            if (w0 == w1) { return false; }
        }
        // cehck the ending
        ptr += length;
        if (ptr < end) {
            const bool w0 = IsWord( is_word, ptr[-1] );
            const bool w1 = IsWord( is_word, ptr[0] );
            if (w0 == w1) { return false; }
        }
        return true;
    }

    const wchar_t* IndexIn( const SFindParams& params, const wchar_t* begin, const wchar_t* end )
    {
        const INT_PTR text_length = params.text_length;
        const INT_PTR length = end - begin;
        if (length < text_length) { return NULL; }

        const wchar_t* text = params.text;
        const int* skip = params.skip;
        const int* last_idx_of = params.last_idx_of;

        const wchar_t* ptr = begin;
        const wchar_t* ptr_end = end - text_length;
        while (ptr <= ptr_end) {
            INT_PTR rpos = text_length;
            wchar_t ch;
            while (true) {
                ch = ptr[--rpos];
                if (ch != text[rpos]) { break; }
                if (rpos == 0) { return ptr; }
            }
            const int pos1 = skip[rpos];
            const int pos2 = rpos - last_idx_of[ch];
            const int pos0 = (pos1 > pos2) ? pos1 : pos2;
            ptr += pos0;
        }
        return NULL;
    }

    const wchar_t* Find( const SFindParams& params, const wchar_t* begin, const wchar_t* end )
    {
        const int text_length = params.text_length;
        if (text_length == 0) { return NULL; }

        const byte* is_word = params.is_word;
        const wchar_t* ptr = begin;
        while (ptr < end) {
            const wchar_t* idx = IndexIn( params, ptr, end );
            if (idx == NULL) { return NULL; }
            if (is_word == NULL || IsOnWordBoundary( is_word, begin, end, idx, text_length )) {
                return idx;
            }
            ptr = idx + 1;
        }
        return NULL;
    }

    int FindInLinesNative( const SFindParams& params, std::vector<SFoundLine>& FoundLines )
    {
        CStringReader reader( params.content, params.content_length );
        int line = 0;
        const wchar_t* begin;
        const wchar_t* end;
        while (reader.ReadLine( &begin, &end )) {
            if (const wchar_t* ptr = Find( params, begin, end )) {
                FoundLines.push_back( SFoundLine( line, begin - params.content, end - begin, ptr - begin ) );
            }
            line++;
        }
        return line;
    }

}
}
}
