/**
 * \file rotenc.c
 * \author Shinichiro Nakamura
 * \brief AD799x I2C ADCからデータを読み込み、API経由でアプリケーションへ渡す
 */

#include <t_services.h>
#include <s_services.h>
#include <i2c_subsystem.h>
#include "kernel_id.h"
#include "uzume.h"
#include "rotenc.h"

/**
 * @brief rotencのチャンネル数
 */
#define NUM_CH 4

/**
 * @brief ADC変換値保存用配列
 * @details
 * 変換値を保存する。アクセスは排他的でなければならない。値は符号付き左詰めである。
 * 整数部はないので、1ビットの符号の右に小数部が連なる。
 */
static fract32 values[NUM_CH];

/**
 * \brief rotenc制御タスク
 * \param exinf タスク引数。
 */
void rotenc_task(VP_INT exinf)
{
    while (1) {
        /* 排他区間開始 */
        syscall(wai_sem(ROTENC_SEM));
        values[0] = (((0x0F) << 8) | 0xFF) << 19;
        values[1] = (((0x0F) << 8) | 0xFF) << 19;
        values[2] = (((0x0F) << 8) | 0xFF) << 19;
        values[3] = (((0x0F) << 8) | 0xFF) << 19;
        syscall(sig_sem(ROTENC_SEM));
        /* 排他区間終了 */

        tslp_tsk(1);
    }
}

/**
 * \brief ボリューム値の取得
 * \param ch 0から始まるチャンネル指定番号。(3まで)
 * \return 32bit符号付き固定小数点数のボリューム値
 */

fract32 get_volume(unsigned int ch)
{
    fract32 retval;

    // パラメータチェック
    if (ch >= NUM_CH) {
        ch = NUM_CH - 1;
    }

    /* 排他区間開始 */
    syscall(wai_sem(ROTENC_SEM));
    retval = values[ch];
    syscall(sig_sem(ROTENC_SEM));
    /* 排他区間終了 */

    return retval;
}

/**
 * \brief ボリューム値の設定
 * \param ch 0から始まるチャンネル指定番号。rotencは3まで
 * \param value 32bit符号付き固定小数点数のボリューム設定値
 * \details
 * ボリュームに初期値を与える。
 */
void set_volume(unsigned int ch, fract32 value)
{
}

