/**
 * @file sgtl5000.c
 * @author: Shinichiro Nakamura
 * @brief SGTL5000初期化プログラム
 * @details
 * SGTL5000の初期化を行う。I2Cを使って初期化を行う。
 */

#include "i2s_subsystem.h"
#include "i2c_subsystem.h"

/**
 * \brief コーデック初期化データ列の長さ
 * \details
 * SGTL5000に与える初期化データの長さである。長さはレジスタアドレスとデータの組が
 * 何組であるかを示す。
 */
#define     CODECINITDATALEN        18      /* SGTL5000初期化データ長 */

#ifndef     UZUME_SGTL5000_I2C_PORT
    /**
     * @brief CODECのI2Cポート。
     * SGTL5000が接続されているI2Cペリフェラルのポート番号。デフォルトは0である。他のポートを使いたい時には
     * sys_config.hで宣言すると良い。なお、I2Cコントローラ・ペリフェラルがひとつしかない場合には0を指定する。
     */
#   define  UZUME_SGTL5000_I2C_PORT   0
#endif

#ifndef     UZUME_SGTL5000_I2C_ADRS
    /**
     * @brief SGTL5000のI2Cデバイスアドレス(7bit右づめ)
     */
#   define  UZUME_SGTL5000_I2C_ADRS   0x0A
#endif

/* SGTL5000 TWI初期化データ列 */
static const unsigned char sgtl5000_initdata[CODECINITDATALEN][4] =
{
    {0x00, 0x02, 0x00, 0x73}, // CHIP_DIG Power     0000-0000 0111-0011
    {0x00, 0x30, 0x77, 0xFF}, // CHIP_ANA_POWER     0111-0111 1111-1111
    {0x00, 0x32, 0x83, 0x12}, // CHIP_PLL_CTRL      1000-0011 0001-0010(16/786)
    {0x00, 0x04, 0x00, 0x0B}, // CHIP_CLK_CTRL      0000-0000 0000-1011
    {0x00, 0x06, 0x00, 0x80}, // CHIP_I2S_CTRL      0000-0000 1000-0000
    {0x00, 0x0A, 0x00, 0x10}, // CHIP_SSS_CTRL      0000-0000 0001-0000
    {0x00, 0x0E, 0x02, 0x00}, // CHIP_ADCDAC_CTRL   0000-0010 0000-0000
    {0x00, 0x10, 0x3C, 0x3C}, // CHIP_DAC_VOL
    {0x00, 0x14, 0x01, 0x5F}, // CHIP_PAD_STRENGTH  0000-0001 0101-1111
    {0x00, 0x20, 0x01, 0x00}, // CHIP_ANA_ADC_CTRL  0000-0001 0000-0000
    {0x00, 0x22, 0x18, 0x18}, // CHIP_ANA_HP_CTRL
    {0x00, 0x24, 0x00, 0x04}, // CHIP_ANA_CTRL      0000-0000 0000-0100
    {0x00, 0x26, 0x00, 0x00}, // CHIP_LINREG_CTRL   0000-0000 0000-0000
    {0x00, 0x28, 0x00, 0x00}, // CHIP_REF_CTRL      0000-0000 0000-0000
    {0x00, 0x2A, 0x00, 0x00}, // CHIP_MIC_CTRL      0000-0000 0000-0000
    {0x00, 0x2C, 0x00, 0x00}, // CHIP_LINE_OUT_CTRL 0000-0000 0000-0000
    {0x00, 0x2E, 0x06, 0x06}, // CHIP_LINE_OUT_VOL  0000-0110 0000-0110
    {0x00, 0x34, 0x00, 0x00}, // CHIP_CLK_TOP_CTRL  0000-1000 0000-0000
};

/**
 * @brief コーデックの初期化
 * @details
 * TLV32AIC23Bに初期化数値列をI2C経由で送り込む。タスクコンテキストで呼び出さなければならない。
 * 初期化後、TLV32AIC23Bはクロック信号を生成して動作を開始する。
 */
void init_codec()
{
    int i;

    /* SGTL5000に初期化データを送信する */
    for ( i=0; i< CODECINITDATALEN; i++){
        i2c_master_write(UZUME_SGTL5000_I2C_PORT, UZUME_SGTL5000_I2C_ADRS, sgtl5000_initdata[i], 4);
    }
}

