package jp.gr.java_conf.ykhr.csvutil.core;

import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import jp.gr.java_conf.ykhr.common.converter.Converter;
import jp.gr.java_conf.ykhr.common.converter.HeaderBaseConverter;
import jp.gr.java_conf.ykhr.common.ex.ReadException;
import jp.gr.java_conf.ykhr.common.utils.Utils;
import jp.gr.java_conf.ykhr.csvutil.config.ReaderConfig;
import jp.gr.java_conf.ykhr.csvutil.escaper.DefaultEscaper;
import jp.gr.java_conf.ykhr.csvutil.escaper.Unescaper;

public class CSVReader {
    
    private static final Unescaper DEFAULT_UNESCAPER = new DefaultEscaper();
    private static final Converter DEFAULT_CONVERTER = new HeaderBaseConverter();
    
    private Unescaper unescaper;
    private Converter converter;
    
    public CSVReader() {
        this(DEFAULT_UNESCAPER, DEFAULT_CONVERTER);
    }
    
    public CSVReader(Unescaper unescaper) {
        this(unescaper, DEFAULT_CONVERTER);
    }

    public CSVReader(Converter unescaper) {
        this(DEFAULT_UNESCAPER, unescaper);
    }
    
    public CSVReader(Unescaper unescaper, Converter converter) {
        this.unescaper = unescaper;
        this.converter = converter;
    }
    
    public <T>Collection<T> read(File file, Class<T> clazz) throws ReadException {
        return read(file, clazz, new ReaderConfig());
    }
    
    public <T>Collection<T> read(File file, Class<T> clazz, ReaderConfig config) throws ReadException {
        InputStream is = null;
        
        try {
            is = new FileInputStream(file);
            return read(is, clazz, config);
        } catch (IOException e) {
            throw new ReadException(e);
        } finally {
            Utils.close(is);
        }
    }
    
    public <T>Collection<T> read(InputStream is, Class<T> clazz) throws ReadException {
        return read(is, clazz, new ReaderConfig());
    }
    
    public <T>Collection<T> read(InputStream is, Class<T> clazz, ReaderConfig config) throws ReadException {
        String contents;
        try {
            contents = readFile(is, config.getEncoding());
        } catch (IOException e) {
            throw new ReadException(e);
        }
        Collection<String[]> elements = divideItem(contents, config);
        Collection<T> result = convert(elements, clazz, config);
        return result;
    }

    private <T>Collection<T> convert(Collection<String[]> elements, Class<T> clazz, ReaderConfig config)
    throws ReadException {
        
        Collection<T> result = converter.toBeanCollection(clazz, elements);
        return result;
    }
    
    private Collection<String[]> divideItem(String contents, ReaderConfig config) {
        List<String[]> result = new ArrayList<String[]>();
        
        boolean inItem = false;
        List<String> items = new ArrayList<String>();
        StringBuilder item = new StringBuilder();
        
        for (int i = 0; i < contents.length(); i++) {
            char c = contents.charAt(i);
            
            if (inItem && unescaper.isEscappedValue(contents, i)) {
                item.append(unescaper.unescape(contents, i));
                i += unescaper.skipLengthAfterUnescape();
            } else if (!inItem && equalsString(contents, i, config.getSeparator())) {
                items.add(item.toString());
                item.setLength(0);
                i += config.getSeparator().length() - 1;
            } else if (!inItem && equalsString(contents, i, config.getLineEnding())) {
                items.add(item.toString());
                item.setLength(0);
                result.add(items.toArray(new String[items.size()]));
                items.clear();
                i += config.getLineEnding().length() - 1;
            } else if (!inItem && c == ' ') {
                // do nothing
            } else if (equalsString(contents, i, config.getQuoteString())) {
                inItem = !inItem;
            } else {
                item.append(contents.charAt(i));
            }
        }

        items.add(item.toString());
        if (items.size() == result.get(0).length) {
            result.add(items.toArray(new String[items.size()]));
        }
        
        return result;
    }
    
    private boolean equalsString(String contents, int index, String check) {
        if (contents.length() < index + check.length()) {
            return false;
        }
        String value = contents.substring(index, index + check.length());
        return value.equals(check);
    }
    
    private String readFile(InputStream is, String encoding) throws IOException {
        BufferedInputStream input = new BufferedInputStream(is);
        ByteArrayOutputStream out = new ByteArrayOutputStream(1024);
        byte[] b = new byte[1024];
        int read = 0;
        
        while ((read = input.read(b)) != -1) {
            out.write(b, 0, read);
        }
        
        return out.toString(encoding);
    }
    
    
}
