// Copyright (c) 2019-2020 Alaskan Emily, Transnat Games
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at http://mozilla.org/MPL/2.0/.

#include "rejoy_bsd.hpp"

#include "rejoy_bsd_core.h"
#include "../rejoy_private.hpp"
#include "../unix/rejoy_unix_core.h"

#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <sys/time.h>

#ifdef __OpenBSD__
#include <dev/hid/hid.h>
#else
#pragma warning FIX THIS!
#include <dev/hid/hid.h>
#endif

///////////////////////////////////////////////////////////////////////////////

// malloc, but safe with input of zero.
static void *rejoy_safe_alloc(unsigned n){
#if !((defined __OpenBSD__) || (defined __NetBSD__))    
    if(n == 0)
        return NULL;
#endif
    return malloc(n);
}

///////////////////////////////////////////////////////////////////////////////

#define REJOY_AXIS_SIZE(X) ((X)<<1)

///////////////////////////////////////////////////////////////////////////////

#ifndef __ppc__
#define REJOY_SHIFT_ROUND_UP(X, SHIFT) (((X)+(1<<(SHIFT))-1)>>(SHIFT))
#define REJOY_BUTTON(BUTTONS, I) ((BUTTONS)[(I)>>3])
#define REJOY_BUTTON_MASK(I) (1 << ((I) & 7))
#endif

///////////////////////////////////////////////////////////////////////////////

#ifdef __ppc__
#define REJOY_BUTTON_SIZE(X) X
#else
#define REJOY_BUTTON_SIZE(X) REJOY_SHIFT_ROUND_UP(X, 3)
#endif

///////////////////////////////////////////////////////////////////////////////

#ifdef __ppc__
#define REJOY_HAT_SIZE(X) (X)
#else
#define REJOY_HAT_SIZE(X) REJOY_SHIFT_ROUND_UP(X, 1)
#endif

///////////////////////////////////////////////////////////////////////////////

namespace Rejoy {

///////////////////////////////////////////////////////////////////////////////

void BSDGamepad::setButton(unsigned i, bool b){
    assert(i < m_num_buttons);
#ifdef __ppc__
    m_buttons[i] = b;
#else
    unsigned char &button = REJOY_BUTTON(m_buttons, i);
    const unsigned char mask = REJOY_BUTTON_MASK(i);
    if(b)
        button |= mask;
    else
        button &= ~mask;
#endif
}
    
///////////////////////////////////////////////////////////////////////////////

void BSDGamepad::setHat(unsigned i, unsigned char h){
    assert(i < m_num_hats);
    assert((h & 0xF0) == 0);
#ifdef __ppc__
    m_hats[i] = b;
#else
    unsigned char &hat = m_hats[i>>1];
    if(i & 1){
        hat &= 0x0F;
        hat |= h << 4;
    }
    else{
        hat &= 0xF0;
        hat |= h;
    }
#endif
}

///////////////////////////////////////////////////////////////////////////////

BSDGamepad::BSDGamepad(const char *name,
    int a_fd,
    unsigned num_axes,
    unsigned num_buttons,
    unsigned num_hats,
    int report_id,
    report_desc_t report)
  : UnixGamepad(a_fd, num_axes, num_buttons, num_hats)
  , m_report(report)
  , m_report_id(report_id)
  , m_report_size(hid_report_size(m_report, hid_input, m_report_id))
  , m_report_buffer(malloc(m_report_size))
  , m_kqueue(kqueue())
  // TODO: These could all live together in one buffer!
  , m_axes((short*)rejoy_safe_alloc(REJOY_AXIS_SIZE(num_axes)))
  , m_buttons((unsigned char*)rejoy_safe_alloc(REJOY_BUTTON_SIZE(num_buttons)))
  , m_hats((unsigned char*)rejoy_safe_alloc(REJOY_HAT_SIZE(num_hats))){

#if (defined __OpenBSD__) || (defined REJOY_USE_STRL_FUNCS)
    strlcpy(m_name, name, REJOY_GAMEPAD_NAME_LEN);
#else
    strncpy(m_name, name, REJOY_GAMEPAD_NAME_LEN-1);
    m_name[REJOY_GAMEPAD_NAME_LEN-1] = 0;
#endif
    
    EV_SET(&m_event, a_fd, EVFILT_READ, EV_ADD|EV_ENABLE, 0, 0, 0);
        
    printf("Report size %i\n"
        "If this is regularly above 2048, we should switch to mmap.\n",
        m_report_size);
}
    
///////////////////////////////////////////////////////////////////////////////

BSDGamepad::~BSDGamepad(){
    hid_dispose_report_desc(m_report);
    close(m_kqueue);
    free(m_report_buffer);
    free(m_axes);
    free(m_buttons);
    free(m_hats);
}

///////////////////////////////////////////////////////////////////////////////

void BSDGamepad::update() {
    
    if(!valid())
        return;

    /* Drain all input. */
    bool any_read = false;
    {
        struct kevent out[2];
        struct timespec timer;
        memset(&timer, 0, sizeof(struct timespec));
        
        int n;
        while((n = kevent(m_kqueue, &m_event, 1, out, 2, &timer)) != 0){
            for(int i = 0; i < n; i++){
                if(out[i].flags == EV_ERROR){
                    fprintf(stderr, "Error on %i\n", fd());
                    invalidate();
                    return;
                }
                else if(out[i].filter == EVFILT_READ){
                    if(read(fd(), m_report_buffer, m_report_size) == m_report_size){
                        any_read = true;
                    }
                }
                else{
                    printf("Unknown event on %i\n", fd());
                }
            }
        }
    }
    
    // If there was no new data (testing shows this basically can't happen,
    // but still test for it) then just return immediately.
    if(!any_read)
        return;

    hid_data_t data = hid_start_parse(m_report, 1<<hid_input, m_report_id);
    unsigned num_buttons = 0, num_axes = 0, num_hats = 0;
    assert(data);
    hid_item_t item;
    while(hid_get_item(data, &item) > 0){
        if(item.kind != hid_input)
            continue; // We don't really want this right now... */
        switch(HID_PAGE(item.usage)){
            case HUP_BUTTON:
                assert(num_buttons < m_num_buttons);
                setButton(num_buttons++, hid_get_data(m_report_buffer, &item));
                break;
            case HUP_GENERIC_DESKTOP:
                switch(HID_USAGE(item.usage)){
                    case HUG_X: /* FALLTHROUGH */
                    case HUG_Y: /* FALLTHROUGH */
                    case HUG_Z: /* FALLTHROUGH */
                    case HUG_RX: /* FALLTHROUGH */
                    case HUG_RY: /* FALLTHROUGH */
                    case HUG_RZ: /* FALLTHROUGH */
                    case HUG_SLIDER:
                        assert(num_axes < m_num_axes);
                        setAxis(num_axes++,
                            Rejoy_BSD_ParseAxis(m_report_buffer, &item));
                        break;
                    case HUG_HAT_SWITCH:
                        // Uhhh...some docs would be nice...
                        // We just have to FULLY debug this I guess.
                        assert(num_hats < m_num_hats);
                        num_hats++;
                        break;
                }
                break;
        }
    }
    
    assert(num_axes == m_num_axes);
    assert(num_buttons == m_num_buttons);
    assert(num_hats == m_num_hats);

    hid_end_parse(data);
}

///////////////////////////////////////////////////////////////////////////////

short BSDGamepad::getAxis(unsigned i) const {
    if(REJOY_LIKELY(i < m_num_axes))
        return m_axes[i];
    else
        return 0;
}

///////////////////////////////////////////////////////////////////////////////

bool BSDGamepad::getButton(unsigned i) const {
    if(REJOY_LIKELY(i < m_num_buttons)){
#ifdef __ppc__
        return m_buttons[i];
#else
        const unsigned mask = REJOY_BUTTON_MASK(i);
        return (REJOY_BUTTON(m_buttons, i) & mask) != 0;
#endif
    }
    else{
        return 0;
    }
}
    
///////////////////////////////////////////////////////////////////////////////

unsigned BSDGamepad::getHat(unsigned i) const {
    if(REJOY_LIKELY(i < m_num_hats)){
#ifdef __ppc__
        return m_hats[i];
#else
        const unsigned char raw = m_hats[i>>1];
        if(i & 1)
            return raw >> 4;
        else
            return raw & 0x0F;
#endif
    }
    else{
        return 0;
    }
}

///////////////////////////////////////////////////////////////////////////////

void BSDDriver::enumerateGamepad(const char *path, int fd){
    // TODO: Quite a bit of this is different in old FreeBSD (we don't really
    // care about < 8.0) and current NetBSD (we should fix this!)
    
    // ALSO TODO: move some of this into the _core.c file.
    report_desc_t report = hid_get_report_desc(fd);
    if(!report)
        return;
    // TODO: There is stuff to do with the report ID!
    const int report_id = Rejoy_BSD_ReportID(fd);
    hid_data_t data = hid_start_parse(report, (1<<hid_input)|(1<<hid_collection), report_id);
    hid_item_t item;
    
    if(!data)
        return;
    
    unsigned num_buttons = 0, num_axes = 0, num_hats = 0;
    bool is_gamepad = false;
    while(hid_get_item(data, &item) > 0){
        // if(is_gamepad)
        //     printf("ID: %i \tkind: %i \tusage: %i\n", item.report_ID, item.kind, item.usage);
        switch(HID_PAGE(item.usage)){
            case HUP_BUTTON:
                if(item.kind == hid_input)
                    num_buttons++;
                break;
            case HUP_GENERIC_DESKTOP:
                switch(HID_USAGE(item.usage)){
                    case HUG_JOYSTICK: /* FALLTHOUGH */
                    case HUG_GAME_PAD:
                        if(item.kind == hid_collection)
                            is_gamepad = true;
                        break;
                    case HUG_X: /* FALLTHROUGH */
                    case HUG_Y: /* FALLTHROUGH */
                    case HUG_Z: /* FALLTHROUGH */
                    case HUG_RX: /* FALLTHROUGH */
                    case HUG_RY: /* FALLTHROUGH */
                    case HUG_RZ: /* FALLTHROUGH */
                    case HUG_SLIDER:
                        if(item.kind == hid_input)
                            num_axes++;
                        break;
                    case HUG_HAT_SWITCH:
                        if(item.kind == hid_input)
                            num_hats++;
                        break;
                }
                break;
        }
    }
    
    printf("%s gamepad, %i axes, %i buttons, %i hats\n",
        is_gamepad ? "is" : "is not",
        num_axes,
        num_buttons,
        num_hats);

    hid_end_parse(data);
    if(is_gamepad){
        /* Set non-blocking */
#ifndef __OpenBSD__
        ioctl(fd, F_SETFL, O_NONBLOCK);
#endif

        BSDGamepad *const gamepad = new BSDGamepad(path,
            fd,
            num_axes,
            num_buttons,
            num_hats,
            report_id,
            report);
        SLIST_INSERT_HEAD(&m_gamepads, gamepad, entries);
        m_num_gamepads++;
    }
    else{
        hid_dispose_report_desc(report);
        close(fd);
    }
}


///////////////////////////////////////////////////////////////////////////////

void BSDDriver::update() {
    // Clear the gamepads.
    struct BSDGamepad *gamepad = SLIST_FIRST(&m_gamepads);
    while(gamepad != NULL){
        struct BSDGamepad *next = SLIST_NEXT(gamepad, entries);
        delete gamepad;
        gamepad = next;
    }
    m_num_gamepads = 0;
    
    // Re-initialize the gamepad list.
    SLIST_INIT(&m_gamepads);
    
    Rejoy_Unix_IterateGlob("/dev/uhid*",
        O_RDONLY,
        this,
        BSDDriver::EnumerateGamepad);
}

///////////////////////////////////////////////////////////////////////////////

REJOY_STATIC_INIT(BSD);

///////////////////////////////////////////////////////////////////////////////

} // namespace Rejoy

///////////////////////////////////////////////////////////////////////////////

