// Copyright (c) 2019-2020 Alaskan Emily, Transnat Games
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at http://mozilla.org/MPL/2.0/.

#ifndef REJOY_BSD_HPP
#define REJOY_BSD_HPP
#pragma once

///////////////////////////////////////////////////////////////////////////////

#include "rejoy_bsd_core.h"

#include "../rejoy.hpp"
#include "../rejoy_config.h"
#include "../unix/rejoy_unix.hpp"

#include <stdint.h> // Needed for usbhid.h
#include <stdlib.h> // for malloc+free

#include <sys/queue.h>
#include <sys/event.h>
#include <sys/types.h>

#include <assert.h>
#include <string.h>
#include <stdio.h> // TODO: This is just for debugging the report size.

///////////////////////////////////////////////////////////////////////////////

namespace Rejoy {

///////////////////////////////////////////////////////////////////////////////
// libusbhid gamepad for BSD systems.

// Tagged as struct so that it can be used in SLIST's
struct BSDGamepad : public Rejoy::UnixGamepad {
    SLIST_ENTRY(BSDGamepad) entries;

private:
    
    char m_name[REJOY_GAMEPAD_NAME_LEN];
    report_desc_t m_report;
    
    const int m_report_id;
    const int m_report_size;

    void *const m_report_buffer;
    
    const int m_kqueue;
    struct kevent m_event;
    
    short *const m_axes;
    
    // On PowerPC, each button is an entire byte.
    // Otherwise, each button is a single bit.
    unsigned char *const m_buttons;

    // On PowerPC, each hat is an entire byte.
    // Otherwise, each hat is a nibble (half-byte).
    unsigned char *const m_hats;
    
    inline void setAxis(unsigned i, short a){
        assert(i < m_num_axes);
        m_axes[i] = a;
    } 
    void setButton(unsigned i, bool b);
    void setHat(unsigned i, unsigned char h);

public:
    
    BSDGamepad(const char *name,
        int a_fd,
        unsigned num_axes,
        unsigned num_buttons,
        unsigned num_hats,
        int report_id,
        report_desc_t report);
    
    ~BSDGamepad();
    
    virtual void update();
    virtual const char *name() const { return m_name; }
    
    virtual short getAxis(unsigned i) const;
    virtual bool getButton(unsigned i) const;
    virtual unsigned getHat(unsigned i) const;
};

///////////////////////////////////////////////////////////////////////////////

SLIST_HEAD(BSDGamepadList, BSDGamepad);

///////////////////////////////////////////////////////////////////////////////

class BSDDriver : public Rejoy::Driver {
    
    struct BSDGamepadList m_gamepads;
    
    void enumerateGamepad(const char *path, int fd);
    
    static void EnumerateGamepad(void *arg, const char *path, int fd){
        static_cast<BSDDriver*>(arg)->enumerateGamepad(path, fd);
    }
    
    
public:
    BSDDriver()
      : Driver("bsd") {
        SLIST_INIT(&m_gamepads);
        hid_start(NULL);
    }

    virtual void update();

    virtual Gamepad *getGamepad(unsigned i){
        struct BSDGamepad *gamepad = SLIST_FIRST(&m_gamepads);
        for(unsigned e = 0; e < i; e++){
            assert(gamepad != NULL);
            gamepad = SLIST_NEXT(gamepad, entries);
        }
        return gamepad;
    }
};

///////////////////////////////////////////////////////////////////////////////

} // namespace Rejoy

///////////////////////////////////////////////////////////////////////////////

#endif // REJOY_BSD_HPP

///////////////////////////////////////////////////////////////////////////////

