// Copyright (c) 2019-2020 Alaskan Emily, Transnat Games
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at http://mozilla.org/MPL/2.0/.

#include "rejoy_dinput.hpp"
#include "rejoy_util.h"

///////////////////////////////////////////////////////////////////////////////

#include <string.h>
#include <stddef.h>
#include <limits.h>

///////////////////////////////////////////////////////////////////////////////

namespace Rejoy {

///////////////////////////////////////////////////////////////////////////////

#define REJOY_AXIS      (DIDFT_OPTIONAL | DIDFT_AXIS | DIDFT_ANYINSTANCE)
#define REJOY_BUTTON    (DIDFT_OPTIONAL | DIDFT_BUTTON | DIDFT_ANYINSTANCE)
#define REJOY_POV       (DIDFT_OPTIONAL | DIDFT_POV | DIDFT_ANYINSTANCE)

///////////////////////////////////////////////////////////////////////////////
// Gets the offset of an element's indices inside the DInputGamepadState
// X must be axes, povs, or buttons
#define REJOY_STATE_OFFSET(N, X) \
    (offsetof(struct DInputGamepadState, m_ ## X) + (sizeof(Rejoy_ ## X ## _t) * (N)))

///////////////////////////////////////////////////////////////////////////////
// Gets the offset of an axis.
#define REJOY_DATA_AXIS(N) \
    {NULL, REJOY_STATE_OFFSET(N, axes), REJOY_AXIS, 0}

///////////////////////////////////////////////////////////////////////////////
// Gets the offset of a button.
#define REJOY_DATA_BUTTON(N) \
    {NULL, REJOY_STATE_OFFSET(N, buttons), REJOY_BUTTON, 0}

///////////////////////////////////////////////////////////////////////////////
// Gets the offset of a hat.
#define REJOY_DATA_POV(N) \
    {NULL, REJOY_STATE_OFFSET(N, povs), REJOY_POV, 0}

///////////////////////////////////////////////////////////////////////////////
// TODO: It feels like this could be done smarter. But meh.
static const DIOBJECTDATAFORMAT DInputDataFormatElements[] = {
    REJOY_DATA_AXIS(0),
    REJOY_DATA_AXIS(1),
    REJOY_DATA_AXIS(2),
    REJOY_DATA_AXIS(3),
    REJOY_DATA_AXIS(4),
    REJOY_DATA_AXIS(5),
    REJOY_DATA_AXIS(6),
    REJOY_DATA_AXIS(7),
    REJOY_DATA_POV(0),
    REJOY_DATA_POV(1),
    REJOY_DATA_POV(2),
    REJOY_DATA_POV(3),
    REJOY_DATA_BUTTON(0),
    REJOY_DATA_BUTTON(1),
    REJOY_DATA_BUTTON(2),
    REJOY_DATA_BUTTON(3),
    REJOY_DATA_BUTTON(4),
    REJOY_DATA_BUTTON(5),
    REJOY_DATA_BUTTON(6),
    REJOY_DATA_BUTTON(7),
    REJOY_DATA_BUTTON(8),
    REJOY_DATA_BUTTON(9),
    REJOY_DATA_BUTTON(10),
    REJOY_DATA_BUTTON(11),
    REJOY_DATA_BUTTON(12),
    REJOY_DATA_BUTTON(13),
    REJOY_DATA_BUTTON(14),
    REJOY_DATA_BUTTON(15),
    REJOY_DATA_BUTTON(16),
    REJOY_DATA_BUTTON(17),
    REJOY_DATA_BUTTON(18),
    REJOY_DATA_BUTTON(19),
    REJOY_DATA_BUTTON(20),
    REJOY_DATA_BUTTON(21),
    REJOY_DATA_BUTTON(22),
    REJOY_DATA_BUTTON(23),
    REJOY_DATA_BUTTON(24),
    REJOY_DATA_BUTTON(25),
    REJOY_DATA_BUTTON(26),
    REJOY_DATA_BUTTON(27),
    REJOY_DATA_BUTTON(28),
    REJOY_DATA_BUTTON(29),
    REJOY_DATA_BUTTON(30),
    REJOY_DATA_BUTTON(31)
};

#define REJOY_NUM_FORMAT_DATA \
    (sizeof(DInputDataFormatElements) / sizeof(DIOBJECTDATAFORMAT))

// Clean up after ourselves.
#undef REJOY_DATA_AXIS
#undef REJOY_DATA_BUTTON
#undef REJOY_DATA_POV

///////////////////////////////////////////////////////////////////////////////
// Data format descriptor.
static const DIDATAFORMAT DInputDataFormat = {
    sizeof(DIDATAFORMAT),
    sizeof(DIOBJECTDATAFORMAT),
    DIDF_ABSAXIS,
    sizeof(DInputGamepadState),
    REJOY_NUM_FORMAT_DATA,
    (LPDIOBJECTDATAFORMAT)DInputDataFormatElements
};

///////////////////////////////////////////////////////////////////////////////

void DInputGamepad::init(IDirectInput8 *dinput, HWND helper_window, const GUID &guid){
    
    if(REJOY_LIKELY(dinput->CreateDevice(m_guid, &m_device, NULL) == DI_OK)){
        assert(m_device);
        
        DIDEVCAPS caps;
        caps.dwSize = sizeof(DIDEVCAPS);
        
        if(REJOY_UNLIKELY(FAILED(m_device->SetCooperativeLevel(helper_window,
                DISCL_NONEXCLUSIVE|DISCL_BACKGROUND)) ||
            FAILED(m_device->SetDataFormat(&DInputDataFormat)) ||
            FAILED(m_device->GetCapabilities(&caps)))){
            
            m_device->Release();
            m_device = NULL;
            
            // TODO: stuff?
            // This gets hit later when doing vector<DInputGamepad>::push_back
            // if we don't zero these.
            m_num_axes = 0;
            m_num_buttons = 0;
            m_num_hats = 0;
        }
        else{
            if(caps.dwFlags & DIDC_POLLEDDATAFORMAT)
                m_polled = true;
            
            m_num_axes = caps.dwAxes;
            m_num_buttons = caps.dwButtons;
            m_num_hats = static_cast<short>(caps.dwPOVs);
            
            // TODO: Find out of the SetFormat did this or not?
            assert(m_num_axes <= REJOY_MAX_AXES);
            assert(m_num_buttons <= REJOY_MAX_BUTTONS);
            assert(m_num_hats <= REJOY_MAX_POVS);
            
            if(REJOY_UNLIKELY(m_num_axes > REJOY_MAX_AXES))
                m_num_axes = REJOY_MAX_AXES;
            
            if(REJOY_UNLIKELY(m_num_buttons > REJOY_MAX_BUTTONS))
                m_num_buttons = REJOY_MAX_BUTTONS;
            
            if(REJOY_UNLIKELY(m_num_hats > REJOY_MAX_POVS))
                m_num_hats = REJOY_MAX_POVS;
            
            // Set or get the axis ranges.
            DIPROPRANGE range;
            range.diph.dwSize = sizeof(DIPROPRANGE);
            range.diph.dwHeaderSize = sizeof(DIPROPHEADER);
            range.diph.dwHow = DIPH_BYOFFSET;
            range.lMin = SHRT_MIN;
            range.lMax = SHRT_MAX;
            
            for(unsigned i = 0; i < m_num_axes; i++){
                range.diph.dwObj = REJOY_STATE_OFFSET(i, axes);
                
                // Try to set the axis ranges to be SHRT_MIN to SHRT_MAX
                const HRESULT set_result =
                    m_device->SetProperty(DIPROP_RANGE, &(range.diph));
                if(REJOY_LIKELY(set_result == DI_OK ||
                    set_result == DI_PROPNOEFFECT)){
                    
                    m_axis_ranges[i].axis_min = 0;
                    m_axis_ranges[i].axis_max = 0;
                }
                else{
                    // Just read the inputs if we can't set them.
                    if(REJOY_LIKELY(m_device->GetProperty(DIPROP_RANGE,
                        &(range.diph)) == DI_OK)){
                        
                        m_axis_ranges[i].axis_min = range.lMin;
                        m_axis_ranges[i].axis_max = range.lMax;
                    }
                    else{
                        // TODO: This sucks!
                        m_axis_ranges[i].axis_min = 0;
                        m_axis_ranges[i].axis_max = 0xFFFF;
                    }
                    range.lMin = SHRT_MIN;
                    range.lMax = SHRT_MAX;
                }
            }
            
            // m_device->EnumObjects(DInputGamepad::EnumObject,
            //     this,
            //     DIDFT_AXIS);
                
                //DIDFT_BUTTON|DIDFT_AXIS|DIDFT_POV);
        }
    }
    else{
        assert(m_device == NULL);
        m_device = NULL;
    }
}

///////////////////////////////////////////////////////////////////////////////

void DInputGamepad::update(){
    
    if(REJOY_UNLIKELY(m_device == NULL))
        return;
    
    m_device->Acquire();
    
    if(m_polled)
        m_device->Poll();
    
    m_device->GetDeviceState(sizeof(DInputGamepadState), &m_state);
    m_device->Unacquire();
    
    // Prepare all the axes.
    assert(m_num_axes <= REJOY_MAX_AXES);
    for(unsigned i = 0; i < m_num_axes; i++){
        
        if(m_axis_ranges[i].axis_min == 0 &&
            m_axis_ranges[i].axis_max == 0){
            
            // Already OK
        }
        else{
            const LONG from = m_axis_ranges[i].axis_min;
            const LONG to = m_axis_ranges[i].axis_max;
            m_state.m_axes[i] = Rejoy_ScaleValue(from, to, m_state.m_axes[i]);
        }
    }
}

///////////////////////////////////////////////////////////////////////////////

short DInputGamepad::getAxis(unsigned i) const {
    assert(i < m_num_axes);
    assert(m_num_axes <= REJOY_MAX_AXES);
    
    if(REJOY_LIKELY(i < m_num_axes)){
        return (short)m_state.m_axes[i];
    }
    else{
        return 0;
    }
}

///////////////////////////////////////////////////////////////////////////////

bool DInputGamepad::getButton(unsigned i) const {
    assert(i < m_num_buttons);
    assert(m_num_buttons <= REJOY_MAX_BUTTONS);
    if(REJOY_LIKELY(i < m_num_buttons)){
        return !!m_state.m_buttons[i];
    }
    else{
        return false;
    }
}

///////////////////////////////////////////////////////////////////////////////

bool DInputDriver::enumGamepad(const DIDEVICEINSTANCE &dev){
    m_gamepads.push_back(DInputGamepad(
        m_dinput,
        m_helper_window,
        dev.tszInstanceName,
        dev.guidInstance));
    m_num_gamepads++;
    return true;
}

///////////////////////////////////////////////////////////////////////////////

BOOL DInputDriver::EnumGamepad(const DIDEVICEINSTANCE *dev, void *arg){
    assert(dev != NULL);
    assert(dev->dwSize >= sizeof(DIDEVICEINSTANCE));
    assert(arg != NULL);
    return static_cast<DInputDriver*>(arg)->enumGamepad(*dev);
}

///////////////////////////////////////////////////////////////////////////////

void DInputDriver::init(void){
    
    const HINSTANCE module = GetModuleHandle(NULL);
    
    HRESULT result = DirectInput8Create(module,
        DIRECTINPUT_VERSION,
        IID_IDirectInput8W,
        (void**)&m_dinput,
        NULL);
    
    assert(m_dinput != NULL);
    if(m_dinput == NULL)
        return; // This actual works out OK. m_gamepads remains empty.
    
    // Create the helper window.
    {
        WNDCLASSW clazz;
        const wchar_t *const clazzName = L"CLASS_Rejoy_HelperWindow";
        const wchar_t *const windowName = clazzName+6;
        memset(&clazz, 0, sizeof(WNDCLASS));
        clazz.lpfnWndProc = DefWindowProc;
        clazz.lpszClassName = clazzName;
        clazz.hInstance = module;
        RegisterClassW(&clazz);
        m_helper_window = CreateWindowExW(0,
            clazzName,
            windowName,
            WS_OVERLAPPED,
            CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT,
            HWND_MESSAGE,
            NULL,
            module,
            NULL);
    }
    
    if(!FAILED(result)){
        m_dinput->EnumDevices(DI8DEVCLASS_GAMECTRL,
            DInputDriver::EnumGamepad,
            this,
            DIEDFL_ATTACHEDONLY);
    }
}

///////////////////////////////////////////////////////////////////////////////

Gamepad *DInputDriver::getGamepad(unsigned i){
    if(i < m_gamepads.size())
        return &(m_gamepads[i]);
    else
        return NULL;
}

///////////////////////////////////////////////////////////////////////////////

REJOY_STATIC_INIT(DInput);

///////////////////////////////////////////////////////////////////////////////

} // namespace Rejoy

///////////////////////////////////////////////////////////////////////////////
