// Copyright (c) 2019-2020 Alaskan Emily, Transnat Games
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at http://mozilla.org/MPL/2.0/.

#ifndef REJOY_DINPUT_HPP
#define REJOY_DINPUT_HPP
#pragma once

///////////////////////////////////////////////////////////////////////////////

// Setup for including Windows headers
#define WIN32_LEAN_AND_MEAN

#ifndef DIRECTINPUT_VERSION
#define DIRECTINPUT_VERSION 0x800
#endif

///////////////////////////////////////////////////////////////////////////////

#include "../rejoy.hpp"
#include "../rejoy_private.hpp"
#include "../rejoy_config.h"

///////////////////////////////////////////////////////////////////////////////

#include <dinput.h>

#include <Windows.h>

#include <vector>
#include <assert.h>

///////////////////////////////////////////////////////////////////////////////
// What is even the deal with this
#ifndef DIDFT_OPTIONAL
#define DIDFT_OPTIONAL 0x80000000
#endif

///////////////////////////////////////////////////////////////////////////////

namespace Rejoy {

///////////////////////////////////////////////////////////////////////////////
// Typedefs to work nicer in macros.
typedef LONG Rejoy_axes_t;
typedef BYTE Rejoy_buttons_t;
typedef DWORD Rejoy_povs_t;

///////////////////////////////////////////////////////////////////////////////
// This struct will be our gamepad state format.
struct DInputGamepadState {
    Rejoy_axes_t m_axes[REJOY_MAX_AXES];
    Rejoy_povs_t m_povs[REJOY_MAX_POVS];
    Rejoy_buttons_t m_buttons[REJOY_MAX_BUTTONS];
};

///////////////////////////////////////////////////////////////////////////////

class DInputGamepad : public Rejoy::Gamepad {
    GUID m_guid;
    struct DInputGamepadState m_state;
    char m_name[REJOY_GAMEPAD_NAME_LEN];
    IDirectInputDevice8 *m_device;
    
    // Holds the range of motion of our axes, or 0,0 if we were able to set
    // it to SHRT_MIN, SHRT_MAX.
    // These are applied inside update(), not getAxis()
    struct {
        LONG axis_min, axis_max;
    } m_axis_ranges[REJOY_MAX_AXES];
    
    // TODO: Inherit the hats?
    unsigned short m_num_hats;
    
    bool m_polled;
    
    // Sets the name from a C string. This exists to overload with a wchar_t version.
    inline void setName(const char *name){
        strncpy_s(m_name, REJOY_GAMEPAD_NAME_LEN-1, name, REJOY_GAMEPAD_NAME_LEN-1);
        m_name[REJOY_GAMEPAD_NAME_LEN-1] = '\0';
    }
    
    inline void setName(const wchar_t *name){
        const size_t name_len = wcsnlen_s(name, MAX_PATH);
        const size_t len = WideCharToMultiByte(CP_UTF8, WC_ERR_INVALID_CHARS,
            name, static_cast<int>(name_len), 
            m_name, REJOY_GAMEPAD_NAME_LEN-1,
            NULL, NULL);
        if(len < REJOY_GAMEPAD_NAME_LEN-1)
            m_name[len] = '\0';
        else
            m_name[REJOY_GAMEPAD_NAME_LEN-1] = '\0';
    }
    
    void init(IDirectInput8 *dinput, HWND helper_window, const GUID &guid);
    
public:

    
    template<typename CharT = TCHAR>
    inline DInputGamepad(IDirectInput8 *dinput,
        HWND helper_window,
        const CharT *name,
        const GUID &guid)
      : Rejoy::Gamepad()
      , m_guid(guid)
      , m_device(NULL)
      , m_polled(false)
      , m_num_hats(0) {
        
        setName(name);
        memset(&m_state, 0, sizeof(m_state));
        
        init(dinput, helper_window, guid);
    }
    
    inline DInputGamepad(const DInputGamepad &other)
      : Gamepad(other.m_num_axes, other.m_num_buttons)
      , m_guid(other.m_guid)
      , m_device(other.m_device)
      , m_polled(other.m_polled)
      , m_num_hats(other.m_num_hats){
        
        setName(other.m_name);
        memset(&m_state, 0, sizeof(m_state));
        
        if(m_device){
            m_device->AddRef();
        }
        else{
            assert(m_num_axes == 0);
            assert(m_num_buttons == 0);
            assert(m_num_hats == 0);
        }
    }
    
    ~DInputGamepad(){
        if(m_device != NULL)
            m_device->Release();
    }
    
    virtual short getAxis(unsigned i) const;
    
    virtual bool getButton(unsigned i) const;
    
    virtual const char *name() const { return m_name; }
    
    virtual void update();
};

///////////////////////////////////////////////////////////////////////////////

class DInputDriver : public Rejoy::Driver {
    IDirectInput8 *m_dinput;
    HWND m_helper_window;
    
    std::vector<DInputGamepad> m_gamepads;
    
    bool enumGamepad(const DIDEVICEINSTANCE &dev);
    static BOOL CALLBACK EnumGamepad(const DIDEVICEINSTANCE *dev, void *arg);
    
    void init(void);
    
public:
    
    inline DInputDriver()
      : Driver("DirectInput")
      , m_dinput(NULL)
      , m_helper_window(NULL){
        init();
    }
    
    ~DInputDriver(){
        if(m_dinput != NULL)
            m_dinput->Release();
        
        if(m_helper_window)
            DestroyWindow(m_helper_window);
    }
    
    virtual Gamepad *getGamepad(unsigned i);
};

///////////////////////////////////////////////////////////////////////////////

} // namespace Rejoy

///////////////////////////////////////////////////////////////////////////////

#endif // REJOY_DINPUT_HPP
