// Copyright (c) 2021 Alaskan Emily, Transnat Games
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at http://mozilla.org/MPL/2.0/.

#include "rejoy_evdev.hpp"
#include "../rejoy_private.hpp"
#include "../unix/rejoy_unix_core.h"
#include <unistd.h>
#include <fcntl.h>

///////////////////////////////////////////////////////////////////////////////
// Manual bindings. These are irregular, one-off functions.
extern "C" {

///////////////////////////////////////////////////////////////////////////////
// Gamepads functions
struct Rejoy_Evdev_Gamepad *Rejoy_Evdev_OpenGamepad(int fd);
struct Rejoy_Evdev_Gamepad *Rejoy_Evdev_CopyGamepad(
    const struct Rejoy_Evdev_Gamepad *data);
void Rejoy_Evdev_FinalizeGamepad(struct Rejoy_Evdev_Gamepad *data);
void Rejoy_Evdev_UpdateGamepad(struct Rejoy_Evdev_Gamepad *data, int fd);

///////////////////////////////////////////////////////////////////////////////
// Rust Vec bindings.
void *Rejoy_Evdev_VecAppend(void *vec, void *val);
unsigned Rejoy_Evdev_VecGetLen(const void *vec);
Rejoy::EvdevGamepad *Rejoy_Evdev_VecGetValue(void *vec, unsigned i);
void Rejoy_Evdev_FinalizeVec(void *vec);

///////////////////////////////////////////////////////////////////////////////

} // extern "C

///////////////////////////////////////////////////////////////////////////////

namespace Rejoy {

///////////////////////////////////////////////////////////////////////////////

static inline void DeleteGamepads(void *vec) {
    const unsigned len = Rejoy_Evdev_VecGetLen(vec);
    for(unsigned i = 0; i < len; i++){
        delete Rejoy_Evdev_VecGetValue(vec, i);
    }
    Rejoy_Evdev_FinalizeVec(vec);
}

///////////////////////////////////////////////////////////////////////////////

EvdevGamepad::EvdevGamepad(const EvdevGamepad &other)
  : UnixGamepad(other)
  , m_data(Rejoy_Evdev_CopyGamepad(other.m_data)) {}

///////////////////////////////////////////////////////////////////////////////

EvdevGamepad::~EvdevGamepad() {
    Rejoy_Evdev_FinalizeGamepad(m_data);
}

///////////////////////////////////////////////////////////////////////////////

void EvdevGamepad::update() {
    Rejoy_Evdev_UpdateGamepad(m_data, fd());
}

///////////////////////////////////////////////////////////////////////////////

void EvdevDriver::enumerateGamepad(int fd){
    // TODO: This could be handled totally on the Rust side pretty easily.
    if(m_num_gamepads)
        return;
    if(struct Rejoy_Evdev_Gamepad *const data = Rejoy_Evdev_OpenGamepad(fd)){
        EvdevGamepad *const gamepad = new EvdevGamepad(fd, data);
        m_gamepad_list = Rejoy_Evdev_VecAppend(m_gamepad_list, gamepad);
        m_num_gamepads++;
    }
    else{
        // close(fd);
    }
}

///////////////////////////////////////////////////////////////////////////////

EvdevDriver::~EvdevDriver() {
    DeleteGamepads(m_gamepad_list);
}

///////////////////////////////////////////////////////////////////////////////

void EvdevDriver::update() {
    DeleteGamepads(m_gamepad_list);
    m_gamepad_list = NULL;
    m_num_gamepads = 0;
    Rejoy_Unix_IterateGlob("/dev/input/event*",
        O_RDWR | O_NONBLOCK,
        this,
        EvdevDriver::EnumateGamepad);
}

///////////////////////////////////////////////////////////////////////////////

Gamepad *EvdevDriver::getGamepad(unsigned i) {
    if(i < Rejoy_Evdev_VecGetLen(m_gamepad_list)){
        Gamepad *gamepad = Rejoy_Evdev_VecGetValue(m_gamepad_list, i);
        return gamepad;
    }
    else
        return NULL;
}

///////////////////////////////////////////////////////////////////////////////

REJOY_STATIC_INIT(Evdev);

///////////////////////////////////////////////////////////////////////////////

} // namespace Rejoy

///////////////////////////////////////////////////////////////////////////////

