use std::{fmt, iter};

/// Bitmap container type with a fixed size and variable bit length elements.
/// TODO: This is clearly not optimal.

pub trait Bitmap {
    fn get_bit(&self, i: usize) -> u8;
    fn set_bit(&mut self, i: usize, val: u8) -> bool;
    fn as_ptr(&self) -> *const u8;
    fn as_mut_ptr(&mut self) -> *mut u8;
    fn element_size(&self) -> usize;
    fn len(&self) -> usize;
}

#[inline]
pub fn get_slice<B>(b: &B, i: usize) -> u32
where
    B: Bitmap
{
    let mut n: u32 = 0;
    let esize = b.element_size();
    let base = i * esize;
    for e in  0 .. esize {
        n |= (b.get_bit(base + e) as u32) << e;
    }
    n
}

#[inline]
pub fn set_slice<B>(b: &mut B, i: usize, val: u32)
where
    B: Bitmap
{
    let esize = b.element_size();
    let base = i * esize;
    for e in  0 .. esize {
        b.set_bit(base + e, (val >> e) as u8 & 1);
    }
}

#[derive(Clone, Debug, PartialEq)]
pub struct BitmapRef<B>(pub B, pub usize);

impl<B> BitmapRef<B> {
    #[inline]
    pub fn new(that: B) -> Self {
        BitmapRef(that, 0)
    }
}


impl<'a, B> BitmapRef<&'a B>
where
    B: Bitmap
{
    #[inline]
    pub fn get(&self) -> u32 {
        get_slice(self.0, self.1)
    }
    
    #[inline]
    pub fn end(&self) -> bool {
        self.0.len() < self.1
    }
    
    #[inline]
    pub fn step(&mut self, n: usize) {
        self.1 = std::cmp::min(self.1 + n, self.0.len());
    }
}

impl<'a, B> BitmapRef<&'a mut B>
where
    B: Bitmap
{
    #[inline]
    pub fn get(&self) -> u32 {
        get_slice(self.0, self.1)
    }
    
    #[inline]
    pub fn end(&self) -> bool {
        self.0.len() < self.1
    }
    
    #[inline]
    pub fn step(&mut self, n: usize) {
        self.1 = std::cmp::min(self.1 + n, self.0.len());
    }
}

impl<'a, B> BitmapRef<&'a mut B>
where
    B: Bitmap
{
    pub fn set(&mut self, val: u32) {
        set_slice(self.0, self.1, val)
    }
}

impl<'a, B> iter::Iterator for BitmapRef<&'a B>
where
    B: Bitmap
{
    type Item = u32;
    fn next(&mut self) -> Option<Self::Item> {
        if self.1 >= self.0.len() - 1 {
            None
        }
        else{
            let u = self.get();
            self.step(1);
            Some(u)
        }
    }
    #[inline]
    fn count(self) -> usize {
        self.0.len() - self.1
    }
    
    #[inline]
    fn nth(&mut self, n: usize) -> Option<Self::Item> {
        self.step(n);
        self.next()
    }
}

impl<'a, B> iter::Iterator for BitmapRef<&'a mut B>
where
    B: Bitmap
{
    type Item = u32;
    fn next(&mut self) -> Option<Self::Item> {
        if self.1 >= self.0.len() - 1 {
            None
        }
        else{
            let u = self.get();
            self.step(1);
            Some(u)
        }
    }
    #[inline]
    fn count(self) -> usize {
        self.0.len() - self.1
    }
    
    #[inline]
    fn nth(&mut self, n: usize) -> Option<Self::Item> {
        self.step(n);
        self.next()
    }
}


#[derive(Clone, Debug, PartialEq)]
pub struct BitmapIterator<B, T>(BitmapRef<B>, std::marker::PhantomData<T>);

impl<B, T> BitmapIterator<B, T> {
    #[inline]
    pub fn new(that: B) -> Self {
        BitmapIterator(BitmapRef::new(that), std::marker::PhantomData)
    }
}

impl<B, T> std::convert::From<BitmapRef<B>> for BitmapIterator<B, T> {
    #[inline]
    fn from(that: BitmapRef<B>) -> Self {
        BitmapIterator(that, std::marker::PhantomData)
    }
}

impl<'a, B, T> BitmapIterator<&'a B, T>
where
    B: Bitmap,
    T: From<u32>,
{
    #[inline]
    pub fn get_val(&self) -> T {
        self.0.get().into()
    }
}

impl<'a, B, T> BitmapIterator<&'a mut B, T>
where
    B: Bitmap,
    T: From<u32>,
{
    #[inline]
    pub fn get_val(&self) -> T {
        self.0.get().into()
    }
}

impl<'a, B, T> BitmapIterator<&'a mut B, T>
where
    B: Bitmap,
    T: Into<u32>,
{
    #[inline]
    pub fn set_val(&mut self, val: T) {
        self.0.set(val.into());
    }
}

impl <'a, B, T> iter::Iterator for BitmapIterator<&'a B, T>
where
    B: Bitmap,
    T: From<u32>,
{
    type Item = T;
    #[inline]
    fn next(&mut self) -> Option<Self::Item> {
        self.0.next().map(|a| a.into())
    }
    #[inline]
    fn count(self) -> usize {
        self.0.count()
    }
}

impl <'a, B, T> iter::Iterator for BitmapIterator<&'a mut B, T>
where
    B: Bitmap,
    T: From<u32>,
{
    type Item = T;
    #[inline]
    fn next(&mut self) -> Option<Self::Item> {
        self.0.next().map(|a| a.into())
    }
    #[inline]
    fn count(self) -> usize {
        self.0.count()
    }
}

impl fmt::Display for dyn Bitmap {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        let esize = self.element_size();
        f.write_str("{")?;
        for i in 0 .. self.len() {
            f.write_str("0b")?;
            for e in 0 .. esize {
                f.write_str(
                    match self.get_bit(e + (i * esize)) {
                        0 => "0",
                        _ => "1",
                    })?;
            }
            f.write_str(",")?;
        }
        Ok(())
    }
}

macro_rules! bitmap_size {
    ($esize:expr, $count:expr) => { ((($esize * $count) + 7) / 8) as usize }
}

macro_rules! bitmap {
    ($name:ident [ $esize:expr ; $count:expr ] ) => {
        
        #[derive(Clone, PartialEq, Debug)]
        #[repr(transparent)]
        pub struct $name([u8; bitmap_size!($esize, $count)]);

        impl $name {
            #[inline]
            pub fn new() -> Self {
                $name([0; bitmap_size!($esize, $count)])
            }
            #[inline]
            pub fn iter<'a>(&'a self) -> crate::bitmap::BitmapRef<&'a Self> {
                crate::bitmap::BitmapRef(self, 0)
            }
            #[inline]
            pub fn iter_mut<'a>(&'a mut self) -> crate::bitmap::BitmapRef<&'a mut Self> {
                crate::bitmap::BitmapRef(self, 0)
            }
            #[inline]
            pub fn ptr_size() -> usize { bitmap_size!($esize, $count) }
        }
        
        impl Default for $name {
            #[inline]
            fn default() -> Self { $name::new() }
        }
        
        impl crate::bitmap::Bitmap for $name {
            #[inline]
            fn get_bit(&self, i: usize) -> u8 {
                (self.0[i >> 3] >> (i & 7)) & 1
            }
            fn set_bit(&mut self, i: usize, val: u8) -> bool {
                let mask = 1 << (i & 7);
                let old = (self.0[i >> 3] & mask) != 0;
                if val != 0 {
                    self.0[i >> 3] |= mask;
                }
                else{
                    self.0[i >> 3] &= !mask;
                }
                old
            }
            #[inline]
            fn element_size(&self) -> usize { $esize }
            #[inline]
            fn len(&self) -> usize { $count }
            #[inline]
            fn as_ptr(&self) -> *const u8 { self.0.as_ptr() }
            #[inline]
            fn as_mut_ptr(&mut self) -> *mut u8 { self.0.as_mut_ptr() }
        }
        
        impl<'a> std::iter::IntoIterator for &'a $name {
            type Item = u32;
            type IntoIter = crate::bitmap::BitmapRef<Self>;
            fn into_iter(self) -> Self::IntoIter {
                crate::bitmap::BitmapRef(self, 0)
            }
        }
    }
}

