use std::cmp;
use std::ffi::CString;
use std::fs::File;
use std::io::{self, Read};
use std::vec::Vec;
use ioctl::{self, AbsInfo, AbsValue};
use bitmap::{Bitmap, BitmapIterator};
use std::os::unix::io::RawFd as Fd;
use std::os::unix::io::{IntoRawFd, FromRawFd};

// Generated bindings
include!{concat!(env!("OUT_DIR"), "/rejoy_evdev_bind.rs")}

extern "C" {
// For consistency with other drivers, use the same scaling function.
fn Rejoy_ScaleValue(from: i32, to: i32, t: i32) -> i32;
}

macro_rules! btn_ranges {
    ($($name:ident : $start:literal .. $end:literal,)*) => {
        $(
            const $name: std::ops::Range<u16> = std::ops::Range {
                start: $start,
                end: $end,
            };
        )*
        const TOTAL_BTN_COUNT: usize = 0 $( + $end - $start )*;
        #[inline]
        fn for_each_btn<F>(mut op: F)
        where
            F: FnMut(u16)
        {
            $(
                for i in $name {
                    op(i);
                }
            )*
        }
    }
}

// Ranges of useful BTN values.
btn_ranges!(
    BTN_JOYSTICK: 0x120 .. 0x130,
    BTN_GAMEPAD: 0x130 .. 0x13F,
    BTN_WHEEL: 0x150 .. 0x152,
    );

const REJOY_C_TIMEVAL_SIZE: usize = super::REJOY_C_TIMEVAL_SIZE as usize;
const INPUT_EVENT_SIZE: usize = REJOY_C_TIMEVAL_SIZE + 8;

#[derive(Debug)]
#[repr(C)]
struct InputEvent {
    time: [u8; REJOY_C_TIMEVAL_SIZE],
    typ: u16,
    code: u16,
    value: i32,
}

#[derive(Clone, PartialEq, Debug)]
struct AbsPair(AbsValue, AbsInfo);

// TODO: This will be used when we handle SYN_DROPPED.
/*
impl AbsPair {
    #[inline]
    fn update(&mut self, fd: Fd) {
        if let Ok(info) = ioctl::evdev_get_abs(fd, self.0) {
            self.1 = info
        }
    }
    #[inline]
    fn value<'a>(&'a self) -> &'a AbsInfo {
        &self.1
    }
}
*/

#[derive(Copy, Clone, Eq, PartialEq, Ord, PartialOrd, Debug)]
struct Button(u8);

const REJOY_MAX_BUTTONS: usize = super::REJOY_MAX_BUTTONS as usize;
const SOME_BUTTON: Button = Button(2);
const NONE_BUTTON: Button = Button(0);

impl Button {
    #[inline]
    fn present(&self) -> bool {
        (self.0 & 2) != 0
    }
    #[inline]
    fn value(&self) -> bool {
        (self.0 & 1) != 0
    }
    #[inline]
    fn update(&mut self, val: bool) {
        self.0 = if val { self.0 | 1 } else { self.0 & 2 };
    }
}

impl std::convert::From<u32> for Button {
    #[inline]
    fn from(that: u32) -> Self {
        Button(that as u8)
    }
}

impl std::convert::From<Button> for u32 {
    #[inline]
    fn from(that: Button) -> Self {
        that.0 as u32
    }
}

bitmap!{ Buttons [2; TOTAL_BTN_COUNT] }

/*
#[inline]
fn cmp_abs_pair<T>(a: &(AbsValue, T), b: &(AbsValue, T)) -> std::cmp::Ordering {
    AbsValue::cmp(a, b)
}
*/

#[derive(Clone, PartialEq, Debug)]
pub struct Gamepad {
    name: CString,
    axes: Vec<(AbsValue, AbsInfo)>,
    hats: Vec<(AbsValue, AbsInfo)>,
    buttons: Buttons,
}

impl Gamepad {
    fn open(fd: Fd) -> io::Result<Self> {
        let name = CString::new(ioctl::evdev_get_name(fd)?)?;
        let mut hats = Vec::new();
        let mut axes = Vec::new();
        ioctl::AbsValue::for_each(|a|
            if let Ok(info) = ioctl::evdev_get_abs(fd, a) {
                if info.min == info.max {
                    return; // Not an interesting axis.
                }
                if a.is_hat() {
                    hats.push((a, info));
                }
                else if a.is_axis() {
                    axes.push((a, info));
                }
            });
        if hats.is_empty() && axes.is_empty() {
            Err(io::Error::new(io::ErrorKind::Other, "Not a gamepad/joystick"))
        }
        else{
            // Test for buttons.
            let mut buttons = Buttons::new();
            let mut button_iter = buttons.iter_mut();
            if let Ok(button_test) = ioctl::evdev_get_btn(fd) {
                for_each_btn(|i| {
                    let bit = button_test.get_bit(i as usize);
                    let b = if bit != 0 { SOME_BUTTON } else { NONE_BUTTON };
                    button_iter.set(b.into());
                    button_iter.step(1);
                });
            }
            Ok( Gamepad{ name, axes, hats, buttons } )
        }
    }
    fn button_iter<'a>(&'a self) -> BitmapIterator<&'a Buttons, Button> {
        BitmapIterator::<&'a Buttons, Button>::new(&self.buttons)
    }
    fn button_iter_mut<'a>(&'a mut self) -> BitmapIterator<&'a mut Buttons, Button> {
        BitmapIterator::<&'a mut Buttons, Button>::new(&mut self.buttons)
    }
    fn get_button(&self, i: usize) -> bool {
        self.button_iter().filter(|b| b.present()).nth(i).map(|b| b.value()).unwrap_or(false)
    }
    #[inline]
    fn get_num_buttons(&self) -> usize {
        let n = self.button_iter().filter(|b| b.present()).count();
        cmp::min(n, REJOY_MAX_BUTTONS)
    }
    #[inline]
    fn get_axis(&self, i: usize) -> i16 {
        let info = self.axes[i].1;
        unsafe { Rejoy_ScaleValue(info.min, info.max, info.value) as i16 }
    }
    #[inline]
    fn get_num_axes(&self) -> usize { self.axes.len() }
    #[inline]
    fn get_hat(&self, _i: usize) -> u32 {
        0
    }
    #[inline]
    fn get_num_hats(&self) -> usize { self.hats.len() }
    fn update(&mut self, fd: Fd) {
        assert!(std::mem::size_of::<InputEvent>() == INPUT_EVENT_SIZE);
        let mut f = unsafe { File::from_raw_fd(fd) };
        let mut buffer: [u8; INPUT_EVENT_SIZE] = [0; INPUT_EVENT_SIZE];
        
        // Read as many events as possible.
        let mut result = f.read(&mut buffer);
        while let Ok(INPUT_EVENT_SIZE) = result {
            let event_ptr = buffer.as_ptr() as *const InputEvent;
            let event = unsafe{ &*event_ptr };
            match event.typ {
                ioctl::EV_KEY => {
                    let mut button_iter = self.button_iter_mut();
                    for_each_btn(|i| {
                        if i == event.code {
                            let mut b = button_iter.get_val();
                            if !b.present() {
                                if cfg!(debug_assertions) {
                                    eprintln!("Invalid button {} set", i);
                                }
                            }
                            else {
                                b.update(event.value != 0);
                            }
                            button_iter.set_val(b.into());
                        }
                        button_iter.next();
                    });
                },
                ioctl::EV_MSC => {
                    // println!("{:?}", event);
                },
                ioctl::EV_ABS => {
                    let a = ioctl::ABS_VALUE_INT[event.code as usize];
                    if a.is_hat() {
                        if let Some(mut abspair) = self.hats.iter_mut().find(|p| p.0 == a) {
                            abspair.1.value = event.value;
                        }
                    }
                    else if a.is_axis() {
                        if let Some(mut abspair) = self.axes.iter_mut().find(|p| p.0 == a) {
                            abspair.1.value = event.value;
                        }
                    }
                    else if cfg!(debug_assertions) {
                        eprintln!("Unexpected ABS type {}", a);
                    }
                },
                ioctl::EV_SYN => {
                    // TODO: We should do a full re-read of the gamepad's
                    // state when we get SYN_DROPPED.
                },
                _ => {
                    if cfg!(debug_assertions) {
                        eprintln!("Unkown event type {}", event.typ);
                    }
                },
            }
            result = f.read(&mut buffer);
        }
        // Drop the file descriptor again.
        f.into_raw_fd();

        if !cfg!(debug_assertions) {
            return;
        }
        
        match result {
            Ok(len) =>
                if len != INPUT_EVENT_SIZE && len != 0 {
                    eprintln!("Invalid read size {}", len);
                },
            Err(e) =>
                if e.kind() != io::ErrorKind::WouldBlock {
                    eprintln!("{} err: {}", fd, e);
                },
        }
    }
}

#[allow(non_camel_case_types)]
type Rejoy_Evdev_Gamepad = Gamepad;

#[allow(non_snake_case)]
#[no_mangle]
pub extern "C" fn Rejoy_Evdev_OpenGamepad(fd: Fd) -> *mut Gamepad {
    match Gamepad::open(fd) {
        Ok(gamepad) => {
                let gamepad_box = Box::new(gamepad);
                Box::leak(gamepad_box) as *mut Gamepad
            },
        Err(e) => {
                if cfg!(debug_assertions) { eprintln!("Could not open gamepad: {}", e); }
                std::ptr::null_mut()
            },
    }
}

#[allow(non_snake_case)]
#[no_mangle]
pub extern "C" fn Rejoy_Evdev_CopyGamepad(gamepad_ptr: *const Gamepad) -> *mut Gamepad {
    let gamepad = unsafe { (*gamepad_ptr).clone() };
    let gamepad_box = Box::new(gamepad);
    Box::leak(gamepad_box) as *mut Gamepad
}

#[allow(non_snake_case)]
#[no_mangle]
pub unsafe extern "C" fn Rejoy_Evdev_FinalizeGamepad(gamepad_ptr: *mut Gamepad) {
    let _ = Box::from_raw(gamepad_ptr);
}

#[allow(non_snake_case)]
#[no_mangle]
pub unsafe extern "C" fn Rejoy_Evdev_UpdateGamepad(gamepad_ptr: *mut Gamepad, fd: Fd) {
    (*gamepad_ptr).update(fd);
}

