// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.

//! ioctl glue for using evdev.

use ioctl_sys;

use bitmap::Bitmap;

use std::default::Default;
use std::{io, fmt};
use std::os::unix::io::RawFd as Fd;

pub const KEY_MAX: u32 = 0x3FF;
pub const KEY_CNT: usize = KEY_MAX as usize + 1;
#[allow(unused)]
pub const SYN_REPORT: u16 = 0;
#[allow(unused)]
pub const SYN_CONFIG: u16 = 1;
#[allow(unused)]
pub const SYN_DROPPED: u16 = 3;
pub const EV_SYN: u16 = 0;
pub const EV_KEY: u16 = 1;
#[allow(unused)]
pub const EV_REL: u16 = 2;
pub const EV_ABS: u16 = 3;
pub const EV_MSC: u16 = 4;
#[allow(unused)]
pub const EV_SW: u16 = 5;
#[allow(unused)]
pub const EV_LED: u16 = 17;
#[allow(unused)]
pub const EV_FF: u16 = 31;


bitmap!{ Buttons [1; KEY_CNT] }

#[derive(PartialEq, Eq, Clone, PartialOrd, Ord)]
#[allow(non_camel_case_types, unused)]
pub enum AbsAxisDir { x, y, other }
#[derive(PartialEq, Eq, Clone, PartialOrd, Ord)]
#[allow(non_camel_case_types, unused)]
pub enum AbsAxisType { hat, stick, slider, other }

macro_rules! abs_attribute {
    ($func:ident $what:ident $typ:ident $($attr:ident $name:ident,)*) => (
        #[allow(unused)]
        pub fn $func(self) -> bool {
            match self {
                $(
                    AbsValue::$name => $typ::$what == $typ::$attr,
                )*
            }
        }
    )
}

macro_rules! abs_match_hat {
    (hat    ? $t:literal : $f:tt ) => {{ $t }};
    (stick  ? $t:literal : $f:tt ) => {{ $f }};
    (slider ? $t:literal : $f:tt ) => {{ $f }};
    (other  ? $t:literal : $f:tt ) => {{ $f }};
}

macro_rules! abs_match_stick {
    (hat    ? $t:literal : $f:tt ) => {{ $f }};
    (stick  ? $t:literal : $f:tt ) => {{ $t }};
    (slider ? $t:literal : $f:tt ) => {{ $f }};
    (other  ? $t:literal : $f:tt ) => {{ $f }};
}

macro_rules! abs_match_slider {
    (hat    ? $t:literal : $f:tt ) => {{ $f }};
    (stick  ? $t:literal : $f:tt ) => {{ $f }};
    (slider ? $t:literal : $f:tt ) => {{ $t }};
    (other  ? $t:literal : $f:tt ) => {{ $f }};
}

macro_rules! abs_match_other {
    (hat    ? $t:literal : $f:tt ) => {{ $f }};
    (stick  ? $t:literal : $f:tt ) => {{ $f }};
    (slider ? $t:literal : $f:tt ) => {{ $f }};
    (other  ? $t:literal : $f:tt ) => {{ $t }};
}

#[allow(unused)]
macro_rules! abs_match_axis {
    (hat    ? $t:literal : $f:tt ) => {{ $f }};
    (stick  ? $t:literal : $f:tt ) => {{ $t }};
    (slider ? $t:literal : $f:tt ) => {{ $t }};
    (other  ? $t:literal : $f:tt ) => {{ $f }};
}

macro_rules! abs_values {
    
    ($($axis:ident $typ:ident $name:ident,)*) => {
        #[repr(u8)]
        #[allow(non_camel_case_types, unused)]
        #[derive(PartialEq, Eq, Clone, Copy, Debug, Ord, PartialOrd)]
        pub enum AbsValue {
            $($name,)*
        }
        pub const ABS_VALUE_MAX: usize = 0 $( + (AbsValue::$name as usize * 0) + 1)* ;
        #[allow(unused)]
        pub const ABS_NUM_HATS: usize = 0 $( + abs_match_hat!($typ ? 1 : 0) )* ;
        #[allow(unused)]
        pub const ABS_NUM_STICKS: usize = 0 $( + abs_match_stick!($typ ? 1 : 0) )* ;
        #[allow(unused)]
        pub const ABS_NUM_SLIDERS: usize = 0 $( + abs_match_slider!($typ ? 1 : 0) )* ;
        #[allow(unused)]
        pub const ABS_NUM_AXES: usize = ABS_NUM_STICKS + ABS_NUM_SLIDERS;
        #[allow(unused)]
        pub const ABS_NUM_OTHER: usize = 0 $( + abs_match_other!($typ ? 1 : 0) )* ;
        // This kind of sucks but here we are.
        pub const ABS_VALUE_INT: &'static [AbsValue; ABS_VALUE_MAX] = &[
            $(AbsValue::$name,)*
        ];
        impl AbsValue {
            abs_attribute!(is_x x AbsAxisDir $($axis $name,)*);
            abs_attribute!(is_y y AbsAxisDir $($axis $name,)*);
            abs_attribute!(is_stick stick AbsAxisType $($typ $name,)*);
            abs_attribute!(is_hat hat AbsAxisType $($typ $name,)*);
            abs_attribute!(is_slider slider AbsAxisType $($typ $name,)*);

            #[inline] 
            #[allow(unused)]
            pub fn is_axis(self) -> bool {
                self.is_stick() || self.is_slider()
            }
            
            #[inline] 
            #[allow(unused)]
            pub fn for_each<F>(mut op: F)
            where 
                F: FnMut(AbsValue)
            {
                $(op(AbsValue::$name);)*
            }
        }
        
        impl fmt::Display for AbsValue {
            fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
                f.write_str(match *self {
                    $(
                        AbsValue::$name => stringify!($name),
                    )*
                })
            }
        }
    }
}

abs_values!(
    x       stick   AbsX,
    y       stick   AbsY,
    other   stick   AbsZ,
    x       stick   AbsRX,
    y       stick   AbsRY,
    other   stick   AbsRZ,
    other   slider  AbsThrottle,
    x       slider  AbsRudder,
    x       other   AbsWheel,
    other   slider  AbsGas,
    other   slider  AbsBrake,
    x       hat     AbsHat0X,
    y       hat     AbsHat1Y,
    x       hat     AbsHat1X,
    y       hat     AbsHat2Y,
    x       hat     AbsHat2X,
    y       hat     AbsHat3Y,
    x       hat     AbsHat3X,
    other   other   AbsPressure,
    other   other   AbsDistance,
    x       other   AbsTiltX,
    y       other   AbsTiltY,
    other   other   AbsToolWidth,
    other   slider  AbsVolume,
    other   other   AbsMisc,
);

#[repr(C)]
#[derive(Default, Debug, Clone, Copy, Eq, PartialEq)]
/// Data for EVIOCSABS/EVIOCGABS
pub struct AbsInfo {
    /// Current value. Not clamped.
    pub value: i32,
    /// Logical minimum.
    pub min: i32,
    /// Logical maximum.
    pub max: i32,
    /// "Fuzz" value for input filtering.
    pub fuzz: i32,
    /// Deadzone threshold. Not useful for joysticks.
    pub flat: i32,
    /// Resolution for the values
    pub res: i32,
}

#[inline]
fn io_result_fn<T, F>(res: i32, that: T, op: F) -> io::Result<T>
where 
    F: FnOnce(i32) -> bool,
{
    if op(res) {
        Ok(that)
    }
    else{
        Err(std::io::Error::last_os_error())
    }
}

fn io_result<T>(res: i32, that: T) -> io::Result<T> {
    io_result_fn(res, that, |i| i == 0)
}

pub fn evdev_get_abs(fd: Fd, abs: AbsValue) -> io::Result<AbsInfo> {
    let mut info: AbsInfo = Default::default();
    let info_addr: *mut AbsInfo = &mut info;
    let res = unsafe {
        ioctl_sys::ioctl(fd,
            ior!(b'E', 0x40 + abs as u8, std::mem::size_of::<AbsInfo>()).into(),
            info_addr)
    };
    io_result(res, info)
}

const NAME_BUFFER_LEN: usize = 255;

pub fn evdev_get_name(fd: Fd) -> io::Result<String> {
    let mut buffer: [u8; NAME_BUFFER_LEN] = [0; NAME_BUFFER_LEN];
    let buffer_slice = &mut buffer[0 .. NAME_BUFFER_LEN];
    let res = unsafe {
        ioctl_sys::ioctl(fd,
            ior!(b'E', 0x06, NAME_BUFFER_LEN).into(),
            buffer_slice.as_mut_ptr())
    };
    io_result_fn(res, buffer_slice, |i| i > 0).map(|n| {
        String::from_utf8_lossy(&n[0 .. (res as usize) - 1]).into()
    })
}

pub fn evdev_get_btn(fd: Fd) -> io::Result<Buttons> {
    let mut buttons = Buttons::new();
    let res = unsafe {
        get_bits(fd, EV_KEY.into(), buttons.as_mut_ptr(), Buttons::ptr_size())
    };
    io_result_fn(res, buttons, |i| i >= 0)
}

unsafe fn get_bits(fd: Fd, what: u32, bmp: *mut u8, len: usize) -> i32 {
    ioctl_sys::ioctl(fd,
        ioc!(ioctl_sys::READ, b'E', 0x20 + what, len).into(),
        bmp)
}

// TODO: props!
#[allow(unused)]
const EVDEV_NUM_PROPS: usize = 32;

#[allow(unused)]
pub fn evdev_get_props(fd: Fd) -> io::Result<[u8; EVDEV_NUM_PROPS]> {
    let mut buffer: [u8; EVDEV_NUM_PROPS] = [0; EVDEV_NUM_PROPS];
    let buffer_slice = &mut buffer[0 .. EVDEV_NUM_PROPS];
    let res = unsafe {
        ioctl_sys::ioctl(fd,
            ior!(b'E', 0x09, EVDEV_NUM_PROPS).into(),
            buffer_slice.as_mut_ptr())
    };
    io_result(res, buffer)
}

