// Copyright (c) 2019 Alaskan Emily, Transnat Games
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at http://mozilla.org/MPL/2.0/.

#include "rejoy.hpp"
#include "rejoy_private.hpp"

///////////////////////////////////////////////////////////////////////////////

#include <stddef.h>

///////////////////////////////////////////////////////////////////////////////

static const char *rejoy_skip_whitespace(const char *a){
    while(*a == ' ' || *a == '\t' || *a == '\n' || *a == '\r')
        a++;
    return a;
}

///////////////////////////////////////////////////////////////////////////////

#if (defined _MSC_VER) || (defined _WIN32)

#include <string.h>

#define REJOY_STRCASECMP(A, B) \
    (_stricmp(rejoy_skip_whitespace((A)), (B)) == 0)

#elif (defined __unix__)

#include <strings.h>

#define REJOY_STRCASECMP(A, B) \
    (strcasecmp(rejoy_skip_whitespace((A)), (B)) == 0)

#else

static inline char rejoy_to_lower(const char c){
    if(c >= 'A' && c <= 'Z')
        return c + 'a' - 'A';
    else
        return c;
}

static inline bool rejoy_strcasecmp(const char *a, const char *b){
    a = rejoy_skip_whitespace(a);
    while((rejoy_to_lower(*(a)++) == *(b++)){
        if(*a == 0)
            return true;
    }
    return false;
}

#define REJOY_STRCASECMP rejoy_strcasecmp

#endif

///////////////////////////////////////////////////////////////////////////////

static bool rejoy_find_driver(const char **drivers, const char *what){
    if(drivers == NULL)
        return true;
    for(const char **driver_ptr = drivers; *driver_ptr != NULL; driver_ptr++){
        if(REJOY_STRCASECMP(*driver_ptr, what))
            return true;
    }
    return false;
}

///////////////////////////////////////////////////////////////////////////////

namespace Rejoy {

///////////////////////////////////////////////////////////////////////////////
// Forward declares of the init functions.
#define REJOY_DRIVER_INIT(X) \
    void Init ## X(void); \
    void Shutdown ## X(void)

#ifdef REJOY_DRIVER_DINPUT
REJOY_DRIVER_INIT(DInput);
#endif

#ifdef REJOY_DRIVER_XINPUT
REJOY_DRIVER_INIT(XInput);
#endif

#ifdef REJOY_DRIVER_BSD
REJOY_DRIVER_INIT(BSD);
#endif

#ifdef REJOY_DRIVER_EVDEV
REJOY_DRIVER_INIT(Evdev);
#endif

///////////////////////////////////////////////////////////////////////////////
// This is used by DriverList when statically starting up.
static struct DriverList *driver_list = NULL;

///////////////////////////////////////////////////////////////////////////////

void Init(const char **drivers){
    Shutdown();

    // Start again.
#ifdef REJOY_DRIVER_DINPUT
    if(rejoy_find_driver(drivers, "dinput"))
        InitDInput();
#endif
    
#ifdef REJOY_DRIVER_XINPUT
    if(rejoy_find_driver(drivers, "xinput"))
        InitXInput();
#endif

#ifdef REJOY_DRIVER_BSD
    if(rejoy_find_driver(drivers, "bsd") || rejoy_find_driver(drivers, "uhid"))
        InitBSD();
#endif

#ifdef REJOY_DRIVER_EVDEV
    if(rejoy_find_driver(drivers, "evdev") || rejoy_find_driver(drivers, "evdev"))
        InitEvdev();
#endif
}

///////////////////////////////////////////////////////////////////////////////

void Shutdown() {

#ifdef REJOY_DRIVER_DINPUT
    ShutdownDInput();
#endif

#ifdef REJOY_DRIVER_XINPUT
    ShutdownXInput();
#endif

#ifdef REJOY_DRIVER_BSD
    ShutdownBSD();
#endif

#ifdef REJOY_DRIVER_EVDEV
    ShutdownEvdev();
#endif

    driver_list = NULL;

}

///////////////////////////////////////////////////////////////////////////////

Gamepad::~Gamepad(){

}

///////////////////////////////////////////////////////////////////////////////

void Gamepad::update(){

}

///////////////////////////////////////////////////////////////////////////////

void PollForGamepads(void){
    for(struct DriverList *drv = driver_list; drv != NULL; drv = drv->m_next){
        Driver &driver = drv->m_driver;
        driver.update();
    }
}

///////////////////////////////////////////////////////////////////////////////

unsigned GetNumDrivers(void){
    unsigned i = 0;
    for(struct DriverList *drv = driver_list; drv != NULL; drv = drv->m_next){
        i++;
    }
    return i;
}

///////////////////////////////////////////////////////////////////////////////

Driver *GetDriver(unsigned i){
    for(struct DriverList *drv = driver_list; drv != NULL; drv = drv->m_next){
        
        if(i == 0){
            Driver &driver = drv->m_driver;
            return &driver;
        }
        
        i--;
    }
    return NULL;
}

///////////////////////////////////////////////////////////////////////////////

short NullGamepad::getAxis(unsigned i) const {
    (void)i;
    return 0;
}

///////////////////////////////////////////////////////////////////////////////

bool NullGamepad::getButton(unsigned i) const {
    (void)i;
    return false;
}

///////////////////////////////////////////////////////////////////////////////

Driver::~Driver(){

}

///////////////////////////////////////////////////////////////////////////////

void Driver::update(){

}

///////////////////////////////////////////////////////////////////////////////

DriverList::DriverList(Driver &driver)
  : m_driver(driver)
  , m_next(driver_list){
    driver_list = this;
}

///////////////////////////////////////////////////////////////////////////////

} // namespace Rejoy
