// Copyright (c) 2019 Alaskan Emily, Transnat Games
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at http://mozilla.org/MPL/2.0/.

#ifndef REJOY_HPP
#define REJOY_HPP
#pragma once

///////////////////////////////////////////////////////////////////////////////

#include "rejoy_export.h"
#include "rejoy_hat.h"

#include <stddef.h>

///////////////////////////////////////////////////////////////////////////////

namespace Rejoy {

///////////////////////////////////////////////////////////////////////////////
/// @brief Initializes the library
///
/// If this is called more than once, then library is shutdown and then
/// re-initialized automatically (with a new set of drivers if provided).
///
/// @p drivers is a NULL-terminated array of driver names to use. If this
///     is NULL, then all available drivers will be used.
///
/// @sa Shutdown
REJOY_EXPORT void Init(const char **drivers = NULL);

///////////////////////////////////////////////////////////////////////////////
/// @brief Shuts down the library
REJOY_EXPORT void Shutdown();

///////////////////////////////////////////////////////////////////////////////

class REJOY_EXPORT Gamepad {
protected:
    unsigned m_num_axes;
    unsigned m_num_buttons;
    unsigned m_num_hats;
    
    // TODO: Remove the `=0` on num_hats and fix the fallout.
    inline Gamepad(unsigned num_axes, unsigned num_buttons, unsigned num_hats = 0)
      : m_num_axes(num_axes)
      , m_num_buttons(num_buttons)
      , m_num_hats(num_hats){}
    
    inline Gamepad() {}
    
public:
    virtual ~Gamepad();
    
    virtual void update();
    virtual const char *name() const = 0;
    
    inline unsigned getNumAxes(void) const { return m_num_axes; }
    virtual short getAxis(unsigned i) const = 0;
    
    inline unsigned getNumButtons(void) const { return m_num_buttons; }
    virtual bool getButton(unsigned i) const = 0;

    inline unsigned getNumHats(void) const { return m_num_hats; }

    // TODO: make this =0
    virtual unsigned getHat(unsigned i) const { (void)i; return 0; }
};

///////////////////////////////////////////////////////////////////////////////

class Driver {
protected:
    unsigned m_num_gamepads;
    
    Driver(unsigned numGamepads, const char *name)
      : m_num_gamepads(numGamepads)
      , m_name(name){}
    
    Driver(const char *name)
      : m_num_gamepads(0)
      , m_name(name){}
    
public:
    virtual ~Driver();
    
    virtual void update();
    
    inline unsigned getNumGamepads(void) const { return m_num_gamepads; }
    virtual Gamepad *getGamepad(unsigned i) = 0;

    const char *const m_name;
};

///////////////////////////////////////////////////////////////////////////////

REJOY_EXPORT void PollForGamepads(void);

///////////////////////////////////////////////////////////////////////////////

REJOY_EXPORT unsigned GetNumDrivers(void);

///////////////////////////////////////////////////////////////////////////////

REJOY_EXPORT Driver *GetDriver(unsigned i);

///////////////////////////////////////////////////////////////////////////////

} // namespace Rejoy

///////////////////////////////////////////////////////////////////////////////

#endif // REJOY_HPP
