/* Copyright (c) 2019 Alaskan Emily, Transnat Games
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

#ifndef REJOY_C_H
#define REJOY_C_H
#pragma once

/*****************************************************************************/

#include "rejoy_export.h"
#include "rejoy_hat.h"

/*****************************************************************************/
/* C bindings to Rejoy.
 * This was primarily intended for use from Python and Mercury.
 */

#ifdef __cplusplus
extern "C" {
#endif

/*****************************************************************************/

struct Rejoy_Gamepad;
struct Rejoy_Driver;

typedef struct Rejoy_Gamepad *rejoy_gamepad_t;
typedef struct Rejoy_Driver *rejoy_driver_t;

typedef const char *rejoy_str_t;

/*****************************************************************************/
/**
 * @brief Initializes the library.
 *
 * This will call shutdown if the library was already started, and
 * re-initialized.
 *
 * @p drivers is an array of driver names to try to initialized. This should
 * usually be NULL, in which case all available drivers are initialized.
 *
 * @note When dynamically linking on most platforms (Linux, BSD, OS X, and
 * Windows at the very least), calling this is not necessary. The library uses
 * the dynamic linker's initialization features to startup. This does not work
 * on most platforms when using static libraries, however.
 *
 * It is harmless to call this multiple times.
 *
 * @param drivers NULL-terminated array of driver names to try to initialize.
 */
REJOY_EXPORT REJOY_CALL(void) Rejoy_Init(const char **drivers);

/*****************************************************************************/
/**
 * @brief Shuts down the library.
 *
 * It is harmless to call this multiple times, although Rejoy_Init MUST be
 * called at least once before this is called.
 */
REJOY_EXPORT REJOY_CALL(void) Rejoy_Shutdown(void);

/*****************************************************************************/

REJOY_EXPORT REJOY_CALL(void) Rejoy_PollForGamepads(void);

/*****************************************************************************/

REJOY_EXPORT REJOY_CALL(void) Rejoy_GamepadUpdate(
    struct Rejoy_Gamepad *gamepad);

/*****************************************************************************/

REJOY_EXPORT REJOY_CALL(rejoy_str_t) Rejoy_GamepadName(
    const struct Rejoy_Gamepad *gamepad);

/*****************************************************************************/

REJOY_EXPORT REJOY_CALL(unsigned) Rejoy_GamepadGetNumAxes(
    const struct Rejoy_Gamepad *gamepad);

/*****************************************************************************/

REJOY_EXPORT REJOY_CALL(short) Rejoy_GamepadGetAxis(
    const struct Rejoy_Gamepad *gamepad, unsigned i);

/*****************************************************************************/

REJOY_EXPORT REJOY_CALL(unsigned) Rejoy_GamepadGetNumButtons(
    const struct Rejoy_Gamepad *gamepad);

/*****************************************************************************/

REJOY_EXPORT REJOY_CALL(int) Rejoy_GamepadGetButton(
    const struct Rejoy_Gamepad *gamepad, unsigned i);

/*****************************************************************************/

REJOY_EXPORT REJOY_CALL(unsigned) Rejoy_GetNumDrivers(void);

/*****************************************************************************/

REJOY_EXPORT REJOY_CALL(rejoy_driver_t) Rejoy_GetDriver(unsigned i);

/*****************************************************************************/

REJOY_EXPORT REJOY_CALL(void) Rejoy_DriverUpdate(struct Rejoy_Driver *driver);

/*****************************************************************************/

REJOY_EXPORT REJOY_CALL(rejoy_str_t) Rejoy_DriverName(
    const struct Rejoy_Driver *driver);

/*****************************************************************************/

REJOY_EXPORT REJOY_CALL(unsigned) Rejoy_DriverGetNumGamepads(
    const struct Rejoy_Driver *driver);

/*****************************************************************************/

REJOY_EXPORT REJOY_CALL(rejoy_gamepad_t) Rejoy_DriverGetGamepad(
    struct Rejoy_Driver *driver, unsigned i);

/*****************************************************************************/

#ifdef __cplusplus
} // extern "C"
#endif

/*****************************************************************************/

#endif /* REJOY_C_H */
