/* Copyright (c) 2019-2020 Alaskan Emily, Transnat Games
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

#include "rejoy_unix_core.h"

#include <assert.h>
#include <errno.h>

#include <unistd.h>
#include <fcntl.h>
#include <glob.h>
#include <sys/stat.h>

#ifndef GLOB_NOSORT
#pragma warning No definition of GLOB_NOSORT
#define GLOB_NOSORT 0
#endif

#ifndef GLOB_LIMIT
/* This is an extension to POSIX. We do kind of expect BSD's to have it, but
 * it's fine systems don't.
 */
#define GLOB_LIMIT 0
#endif

/* Try to avoid needing a second call to stat, if possible. */
#if (defined GLOB_KEEPSTAT) || (defined REJOY_USE_GLOB_KEEPSTAT)

/* use GLOB_KEEPSTAT and just read from the glob_t */
#define REJOY_GLOB_KEEPSTAT GLOB_KEEPSTAT
#define REJOY_GLOB_STAT(DATA, I, _) \
    ((DATA)->gl_statv[(I)])

#else

/* Do a second stat on the resulting path. */
#define REJOY_GLOB_KEEPSTAT 0
#define REJOY_GLOB_STAT(DATA, I, DST) \
    (stat((DATA)->gl_pathv[(I)], (DST)) == 0 ? (DST) : NULL)

#endif

/*****************************************************************************/

static int rejoy_unix_glob_callback(const char *path, const int status){
    (void)path;
    if(status == ENOENT || status == ENOTDIR)
        return 0;
    else
        return 1;
}

/*****************************************************************************/

void Rejoy_Unix_IterateGlob(const char *glob_path,
    int fd_flags,
    void *arg,
    rejoy_unix_path_callback_t cb){
    
    glob_t glob_data;
    
    assert(glob_path);
    assert(cb);
    
    glob(glob_path,
        GLOB_NOSORT|GLOB_LIMIT|REJOY_GLOB_KEEPSTAT,
        rejoy_unix_glob_callback,
        &glob_data);
    
    if(glob_data.gl_pathc != 0){
        unsigned i;
        struct stat tmp_stat;
        
        /* This might not actually be used by REJOY_GLOB_STAT. */
        (void)tmp_stat;
        
        assert(glob_data.gl_pathv != NULL);
        
        /* Enumerate the joysticks. */
        for(i = 0; glob_data.gl_pathv[i] != NULL; i++){
            const char *path = glob_data.gl_pathv[i];
            const struct stat *st = REJOY_GLOB_STAT(&glob_data, i, &tmp_stat);
            int fd;
            if(st &&
                /* TODO: This might not even need the block check? */
                (S_ISBLK(st->st_mode) || S_ISCHR(st->st_mode)) &&
                (fd = open(path, fd_flags)) >= 0) {
                
                cb(arg, path, fd);
            }
            else{
                printf("Could not open %s (err %i): \n", path, errno);
#define ERRCASE(NAME, DSCR) case NAME: puts( #NAME ": " DSCR); break
                switch(errno){
                    ERRCASE(EACCES, "permission denied");
                    ERRCASE(EINTR, "timed out");
                    ERRCASE(EMFILE, "too many files open");
                    ERRCASE(ENODEV, "invalid device");
                    ERRCASE(ELOOP, "symlink loop");
#ifdef ENOTDIR
                    ERRCASE(ENOTDIR, "invalid directory component");
#endif
#ifdef EOVERFLOW
                    ERRCASE(EOVERFLOW, "file too large");
#endif
#ifdef EFBIG
                    ERRCASE(EFBIG, "file too large");
#endif
                    ERRCASE(EWOULDBLOCK, "opening would block");
                    default: puts("unknown error.");
                }
            }
        }
    }
    globfree(&glob_data);
}

