// Copyright (c) 2020 Alaskan Emily, Transnat Games
//
// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at http://mozilla.org/MPL/2.0/.

#ifndef REJOY_XINPUT_HPP
#define REJOY_XINPUT_HPP
#pragma once

///////////////////////////////////////////////////////////////////////////////

// Setup for including Windows headers
#define WIN32_LEAN_AND_MEAN

///////////////////////////////////////////////////////////////////////////////

#include "rejoy_xinput_core.h"
#include "../rejoy.hpp"
#include "../rejoy_private.hpp"
#include "../rejoy_config.h"

///////////////////////////////////////////////////////////////////////////////

#include <Windows.h>
#include <XInput.h>

#include <vector>
#include <assert.h>

///////////////////////////////////////////////////////////////////////////////

namespace Rejoy {

///////////////////////////////////////////////////////////////////////////////

class XInputGamepad : public Rejoy::Gamepad {
    XINPUT_STATE m_state;
    const unsigned char m_controller;
    
    // Adjusts for missing navigation buttons.
    static inline unsigned NumButtons(bool has_nav){
        return has_nav ?
            REJOY_XINPUT_NUM_BUTTONS :
            REJOY_XINPUT_NUM_BUTTONS_NO_NAV;
    }
    
public:
    inline XInputGamepad(unsigned char n, bool has_nav)
      : Gamepad(REJOY_XINPUT_NUM_AXES, NumButtons(has_nav))
      , m_controller(n) {}
    
    virtual void update(){
        XInputGetState(m_controller, &m_state);
    }
    
    virtual short getAxis(unsigned i) const{
        assert(i < REJOY_XINPUT_NUM_AXES);
        if(REJOY_UNLIKELY(i >= REJOY_XINPUT_NUM_AXES))
            return 0;
        else
            return Rejoy_XInput_GetAxis(&(m_state.Gamepad), i);
    }
    
    virtual bool getButton(unsigned i) const{
        assert(i < REJOY_XINPUT_NUM_BUTTONS);
        if(REJOY_UNLIKELY(i >= REJOY_XINPUT_NUM_BUTTONS))
            return false;
        else
            return !!(m_state.Gamepad.wButtons & rejoy_xinput_button_masks[i]);
    }
    
    virtual unsigned getHat(unsigned i) const{
        assert(i == 0);
        if(REJOY_UNLIKELY(i != 0))
            return 0;
        else
            return Rejoy_XInput_GetHat(&(m_state.Gamepad));
    }
    
    virtual const char *name() const {
        assert(m_controller < 4);
        return rejoy_xinput_controller_names[m_controller];
    }
};

///////////////////////////////////////////////////////////////////////////////

class XInputDriver : public Rejoy::Driver {
    XInputGamepad *m_gamepads[4];
    
public:
    
    inline XInputDriver()
      : Driver(0, "XInput"){
        m_gamepads[0] = NULL;
        m_gamepads[1] = NULL;
        m_gamepads[2] = NULL;
        m_gamepads[3] = NULL;
        XInputEnable(TRUE);
    }
    
    ~XInputDriver(){
        XInputEnable(FALSE);
    }
    
    virtual void update();
    virtual Gamepad *getGamepad(unsigned i);
};

///////////////////////////////////////////////////////////////////////////////

} // namespace Rejoy

///////////////////////////////////////////////////////////////////////////////

#endif // REJOY_XINPUT_HPP
