/* Copyright (c) 2020 Alaskan Emily, Transnat Games
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

#include "rejoy_xinput_core.h"
#include "../rejoy_hat.h"

#define WIN32_LEAN_AND_MEAN
#include <Windows.h>
#include <XInput.h>

/*****************************************************************************/

unsigned Rejoy_XInput_IsConnected(unsigned n){
    XINPUT_STATE unused;
    return (XInputGetState(n, &unused) == ERROR_SUCCESS);
}

/*****************************************************************************/

unsigned Rejoy_XInput_HasNavigation(unsigned n){
    XINPUT_CAPABILITIES caps;
    XInputGetCapabilities(n, XINPUT_FLAG_GAMEPAD, &caps);
    /* TODO: Apparently caps.Gamepad has calibration info? */
    return !(caps.Flags & XINPUT_CAPS_NO_NAVIGATION);
}

/*****************************************************************************/

short Rejoy_XInput_GetAxis(const struct _XINPUT_GAMEPAD *gamepad, unsigned n){
    switch(n){
        default:
#if (defined _MSC_VER)
            __assume(0);
#elif (defined __GNUC__)
            __builtin_unreachable();
#endif
        case 0:
            return gamepad->sThumbLX;
        case 1:
            return gamepad->sThumbLY;
        case 2:
            return gamepad->sThumbRX;
        case 3:
            return gamepad->sThumbRY;
        case 4:
            return REJOY_XINPUT_CONVERT_TRIGGER(gamepad->bLeftTrigger);
        case 5:
            return REJOY_XINPUT_CONVERT_TRIGGER(gamepad->bRightTrigger);
    }
}

/*****************************************************************************/

unsigned Rejoy_XInput_GetHat(const struct _XINPUT_GAMEPAD *gamepad){
    unsigned dir = 0;
    
    /* Or'ing the DIR bit into each of these masks allows us to avoid an extra
     * check at the end. It's not really important, but it's easy.
     */
    if(gamepad->wButtons & XINPUT_GAMEPAD_DPAD_UP)
        dir |= REJOY_HAT_UP;
    else if(gamepad->wButtons & XINPUT_GAMEPAD_DPAD_DOWN)
        dir |= REJOY_HAT_DOWN;

    if(gamepad->wButtons & XINPUT_GAMEPAD_DPAD_LEFT)
        dir |= REJOY_HAT_LEFT;
    else if(gamepad->wButtons & XINPUT_GAMEPAD_DPAD_RIGHT)
        dir |= REJOY_HAT_RIGHT;
    
    return dir;
}

/*****************************************************************************/

const char rejoy_xinput_controller_names[4][16] = {
    "XInput 1",
    "XInput 2",
    "XInput 3",
    "XInput 4"
};

/*****************************************************************************/

const unsigned rejoy_xinput_button_masks[REJOY_XINPUT_NUM_BUTTONS] = {
    XINPUT_GAMEPAD_A,
    XINPUT_GAMEPAD_B,
    XINPUT_GAMEPAD_X,
    XINPUT_GAMEPAD_Y,
    XINPUT_GAMEPAD_LEFT_SHOULDER,
    XINPUT_GAMEPAD_RIGHT_SHOULDER,
    XINPUT_GAMEPAD_LEFT_THUMB,
    XINPUT_GAMEPAD_RIGHT_THUMB,
    XINPUT_GAMEPAD_START,
    XINPUT_GAMEPAD_BACK
};

/*****************************************************************************/
