
package orderedSet;

import java.util.Objects;
import mapping.order.Default;
import mapping.order.Order;
import text.Format;

/**
 * 開始と終了だけで表現される単純な区間。なお、開始点は含み、終了点は含まない。
 * @author mtomono
 */
public class Range<T extends Comparable<T>> implements Cloneable {
    T start;
    T end;
    Order<T> order;
    
    protected Range() {
        
    }
    
    @Override
    public Range clone() {
        return new Range(start, end, order);
    }
    
    @Override
    public boolean equals(Object o) {
        if (!(o instanceof Range)) {
            return false;
        }
        if (o instanceof EmptyRange) {
            return false;
        }
        return order.eq(this.start, ((Range<T>)o).start) && order.eq(this.end, ((Range<T>)o).end);
    }

    @Override
    public int hashCode() {
        int hash = 3;
        hash = 67 * hash + Objects.hashCode(this.start);
        hash = 67 * hash + Objects.hashCode(this.end);
        return hash;
    }
    
    public Range(T start, T end, Order<T> order) {
        assert order.le(start, end);
        this.start = start;
        this.end = end;
        this.order = order;
    }
    
    public Range(T start, T end) {
        this(start, end, Default.order);
    }
    
    public T getStart() {
        return start;
    }
    
    public T getEnd() {
        return end;
    }
    
    public boolean doesContain(T value) {
        return order.le(this.start, value) && order.lt(value, this.end);
    }
    
    public boolean isUpperThan(T value) {
        return order.lt(value, this.start);
    }
    
    public boolean isUpperThan(Range<T> another) {
        if (another instanceof EmptyRange) {
            return true;
        }
        return order.le(another.end, this.start);
    }
    
    public boolean isLowerThan(T value) {
        return order.le(this.end, value);
    }
    
    public boolean isLowerThan(Range<T> another) {
        if (another instanceof EmptyRange) {
            return true;
        }
        return order.le(this.end, another.start);
    }
    
    public boolean doesOverlap(Range<T> another) {
        if (another instanceof EmptyRange) {
            return false;
        }
        return !(this.isLowerThan(another) || this.isUpperThan(another));
    }
    
    public boolean doesContain(Range<T> another) {
        if (another instanceof EmptyRange) {
            return true;
        }
        return order.le(this.start, another.start) && order.le(another.end, this.end);
    }
    
    public boolean doesAdjoin(Range<T> another) {
        if (another instanceof EmptyRange) {
            return false;
        }
        return doesAdjoinAtStart(another) || doesAdjoinAtEnd(another);
    }
    
    public boolean doesAdjoinAtStart(Range<T> another) {
        if (another instanceof EmptyRange) {
            return false;
        }
        return order.eq(this.start, another.end);
    }
    
    public boolean doesAdjoinAtEnd(Range<T> another) {
        if (another instanceof EmptyRange) {
            return false;
        }
        return order.eq(this.end, another.start);
    }
    
    public Range<T> intersect(Range<T> another) {
        if (doesOverlap(another)) {
            return new Range(order.lt(another.start, this.start) ? this.start : another.start, 
                                order.lt(another.end, this.end) ? another.end : this.end, order);
        } else {
            return new EmptyRange<>();
        }
    }
    
    public Range<T> cover(Range<T> another) {
        if (another instanceof EmptyRange) {
            return this;
        } else {
            return new Range(order.lt(another.start, this.start) ? another.start : this.start, 
                                order.lt(another.end, this.end) ? this.end : another.end, order);
        }
    }
    
    public Range<T> getUpper(T value) {
        if (this.isLowerThan(value)) {
            return new EmptyRange<>();
        } else {
            return new Range<>(order.le(this.start, value) ? value : this.start, this.end, order);
        }
    }
    
    public Range<T> getUpper(Range<T> another) {
        if (another instanceof EmptyRange) {
            return this;
        }
        return getUpper(another.end);
    }

    public Range<T> getLower(T value) {
        if (order.le(value, this.start)) {  //バリは要らないので
            return new EmptyRange<>();
        } else {
            return new Range<>(this.start, order.lt(this.end, value)?this.end:value, order);
        }
    }
    
    public Range<T> getLower(Range<T> another) {
        if (another instanceof EmptyRange) {
            return this;
        }
        return getLower(another.start);
    }
    
    @Override
    public String toString() {
        return Format.<T>range(this.start, this.end, null);
    }
}
