/*
 * Copyright (C) 2021, 2022 mio <stigma@disroot.org>
 *
 * Permission to use, copy, modify, and/or distribute this software for any
 * purpose with or without fee is herby granted.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
 * SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
 * OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PEFORMANCE OF THIS SOFTWARE.
 */


/**
 * An incomplete single-file INI parser for D.
 *
 * The API should be similar to python's configparse module.  Internally it
 * uses the standard D associative array.
 *
 * Example:
 * ---
 * import configparser;
 *
 * auto config = new ConfigParser();
 * // no sections initially
 * assert(config.sections.length == 0);
 * // Section names are case-sensitive
 * conf.addSection("Default");
 * // option names (pythonpath) are case-insensitive (converted to lowercase)
 * conf.set("Default", "pythonpath", "/usr/bin/python3");
 * ---
 *
 * License: 0BSD
 * Version: 0.2
 * History:
 *      0.2 Add .getBool()
 *      0.1 Initial release
 */
module mlib.configparser;

private
{
    import std.conv : ConvException;
    import std.stdio : File;
}

public class DuplicateSectionException : Exception
{
    private string m_section;

    this(string section)
    {
        string msg = "Section " ~ section ~ " already exists.";
        m_section = section;
        super(msg);
    }

    string section()
    {
        return m_section;
    }
}

public class NoSectionException : Exception
{
    private string m_section;

    this(string section)
    {
        string msg = "Section '" ~ section ~ "' does not exist.";
        m_section = section;
        super(msg);
    }

    string section()
    {
        return m_section;
    }
}

public class NoOptionException : Exception
{
    private string m_section;
    private string m_option;

    this(string section, string option)
    {
        string msg = "Section '" ~ section ~ "' does not have option '" ~
            option ~ "'.";
        m_section = section;
        m_option = option;
        super(msg);
    }

    string section() { return m_section; }
    string option() { return m_option; }
}

/**
 * The main configuration parser.
 */
public class ConfigParser
{
    private char[] m_delimiters;
    private char[] m_commentPrefixes;

    /** current section for parsing */
    private string m_currentSection;
    private string[string][string] m_sections;

    /**
     * Creates a new instance of ConfigParser.
     */
    this(char[] delimiters = ['=', ':'], char[] commentPrefixes = ['#', ';'])
    {
        m_delimiters = delimiters;
        m_commentPrefixes = commentPrefixes;
    }

    /**
     * Return an array containing the available sections.
     */
    string[] sections()
    {
        return m_sections.keys();
    }

    ///
    unittest
    {
        auto conf = new ConfigParser();

        assert(0 == conf.sections().length);

        conf.addSection("Section");

        assert(1 == conf.sections().length);
    }

    /**
     * Add a section named `section` to the instance.
     *
     * Throws:
     *   - DuplicateSectionError if a section by the given name already
     * exists.
     */
    void addSection(string section)
    {
        if (section in m_sections)
            throw new DuplicateSectionException(section);
        m_sections[section] = null;
    }

    ///
    unittest
    {
        import std.exception : assertNotThrown, assertThrown;

        auto conf = new ConfigParser();

        /* doesn't yet exist */
        assertNotThrown!DuplicateSectionException(conf.addSection("sample"));
        /* already exists */
        assertThrown!DuplicateSectionException(conf.addSection("sample"));
    }

    /**
     * Indicates whether the named `section` is present in the configuration.
     *
     * Params:
     *   section = The section to check for in the configuration.
     *
     * Returns: `true` if the section exists, `false` otherwise.
     */
    bool hasSection(string section)
    {
        auto exists = (section in m_sections);
        return (exists !is null);
    }

    ///
    unittest
    {
        auto conf = new ConfigParser();
        conf.addSection("nExt");
        assert(true == conf.hasSection("nExt"), "Close the world.");
        assert(false == conf.hasSection("world"), "Open the nExt.");
    }

    string[] options(string section)
    {
        if (false == this.hasSection(section))
            throw new NoSectionException(section);
        return m_sections[section].keys();
    }

    ///
    unittest
    {
        import std.exception : assertNotThrown, assertThrown;

        auto conf = new ConfigParser();

        conf.addSection("Settings");

        assertNotThrown!NoSectionException(conf.options("Settings"));
        assertThrown!NoSectionException(conf.options("void"));

        string[] options = conf.options("Settings");
        assert(0 == options.length, "More keys than we need");
    }

    bool hasOption(string section, string option)
    {
        if (false == this.hasSection(section))
            return false;

        auto exists = (option in m_sections[section]);
        return (exists !is null);
    }
    /*
    string[] read(string[] filenames)
    {
        return null;
    }*/

    void read(string filename)
    {
        import std.file : FileException, exists, isFile;

        if (false == exists(filename)) {
            throw new FileException(filename);
        }

        if (false == isFile(filename)) {
            throw new FileException(filename);
        }

        File file = File(filename, "r");
        scope(exit) { file.close(); }
        read(file, false);
    }

    ///
    unittest
    {
        import std.file : remove;
        import std.stdio : File;

        auto configFile = File("test.conf", "w+");
        configFile.writeln("[Section 1]");
        configFile.writeln("key=value");
        configFile.writeln("\n[Section 2]");
        configFile.writeln("key2 = value");
        configFile.close();

        auto conf = new ConfigParser();
        conf.read("test.conf");

        assert(2 == conf.sections.length, "Incorrect Sections length");
        assert(true == conf.hasSection("Section 1"),
               "Config file doesn't have Section 1");
        assert(true == conf.hasOption("Section 1", "key"),
               "Config file doesn't have 'key' in 'Section 1'");

        remove("test.conf");
    }

    /**
     * Parse a config file.
     *
     * Params:
     *   file = Reference to the file from which to read.
     *   close = Close the file when finished parsing.
     */
    void read(ref File file, bool close = true)
    {
        import std.array : array;
        import std.algorithm.searching : canFind;
        import std.string : strip;

        scope(exit) { if (close) file.close(); }

        string[] lines = file.byLineCopy.array;

        for (auto i = 0; i < lines.length; i++) {
            string line = lines[i].strip();

            if (line == "")
                continue;

            if ('[' == lines[i][0]) {
                parseSectionHeader(lines[i]);
            } else if (false == canFind(m_commentPrefixes, lines[i][0])) {
                parseLine(lines[i]);
            }
            /* ignore comments */
        }
    }

    // void readString(string str)
    // {
    // }

    /**
     * Get an `option` value for the named `section`.
     *
     * Params:
     *   section = The section to look for the given `option`.
     *   option = The option to return the value of
     *   fallback = Fallback value if the `option` is not found. Can be null.
     *
     * Returns:
     *   - The value for `option` if it is found.
     *   - `null` if the `option` is not found and `fallback` is not provided.
     *   - `fallback` if the `option` is not found and `fallback` is provided.
     *
     * Throws:
     *   - NoSectionException if the `section` does not exist and no fallback is provided.
     *   - NoOptionException if the `option` does not exist and no fallback is provided.
     */
    string get(string section, string option)
    {
        if (false == this.hasSection(section))
            throw new NoSectionException(section);

        if (false == this.hasOption(section, option))
            throw new NoOptionException(section, option);

        return m_sections[section][option];
    }

    ///
    unittest
    {
        import std.exception : assertThrown;

        auto conf = new ConfigParser();
        conf.addSection("Section");
        conf.set("Section", "option", "value");

        assert(conf.get("Section", "option") == "value");
        assertThrown!NoSectionException(conf.get("section", "option"));
        assertThrown!NoOptionException(conf.get("Section", "void"));
    }

    /// Ditto
    string get(string section, string option, string fallback)
    {
        string res = fallback;

        try {
            res = get(section, option);
        } catch (NoSectionException e) {
            return res;
        } catch (NoOptionException e) {
            return res;
        }

        return res;
    }

    ///
    unittest
    {
        import std.exception : assertThrown;

        auto conf = new ConfigParser();
        conf.addSection("Section");
        conf.set("Section", "option", "value");

        assert("value" == conf.get("Section", "option"));
        assert("fallback" == conf.get("section", "option", "fallback"));
        assert("fallback" == conf.get("Section", "void", "fallback"));

        /* can use null for fallback */
        assert(null == conf.get("section", "option", null));
        assert(null == conf.get("Section", "void", null));
    }

    /**
     * A convenience method which casts the value of `option` in `section`
     * to an integer.
     *
     * Params:
     *   section = The section to look for the given `option`.
     *   option = The option to return the value for.
     *   fallback = The fallback value to use if `option` isn't found.
     *
     * Returns:
     *
     *
     * Throws:
     *   - NoSectionFoundException if `section` doesn't exist.
     *   - NoOptionFoundException if the `section` doesn't contain `option`.
     *   - ConvException if it failed to parse the value to an int.
     *   - ConvOverflowException if the value would overflow an int.
     *
     * See_Also: get()
     */
    int getInt(string section, string option)
    {
        import std.conv : parse;

        string res;

        res = get(section, option);

        return parse!int(res);
    }

    /// Ditto
    int getInt(string section, string option, int fallback)
    {
        int res = fallback;

        try {
            res = getInt(section, option);
        } catch (Exception e) {
            return res;
        }

        return res;
    }

    /*
    double getDouble(string section, string option)
    {
    }

    double getDouble(string section, string option, double fallback)
    {
    }

    float getFloat(string section, string option)
    {
    }

    float getFloat(string section, string option, float fallback)
    {
    }*/

    /**
     * A convenience method which coerces the $(I option) in the
     * specified $(I section) to a boolean value.
     * 
     * Note that the accepted values for the option are "1", "yes",
     * "true", and "on", which cause this method to return `true`, and
     * "0", "no", "false", and "off", which cause it to return `false`.
     * 
     * These string values are checked in a case-insensitive manner.
     * 
     * Params:
     *   section = The section to look for the given option.
     *   option = The option to return the value for.
     *   fallback = The fallback value to use if the option was not found.
     * 
     * Throws:
     *   - NoSectionFoundException if `section` doesn't exist.
     *   - NoOptionFoundException if the `section` doesn't contain `option`.
     *   - ConvException if any other value was found.
     */
    bool getBool(string section, string option)
    {
        import std.string : toLower;

        string value = get(section, option);
        
        switch (value.toLower)
        {
        case "1":
        case "yes":
        case "true":
        case "on":
            return true;
        case "0":
        case "no":
        case "false":
        case "off":
            return false;
        default:
            throw new ConvException("No valid boolean value found");
        }
    }

    /// Ditto
    bool getBool(string section, string option, bool fallback)
    {
        try {
            return getBool(section, option);
        } catch (Exception e) {
            return fallback;
        }
    }

    /*
    string[string] items(string section)
    {
    }*/

    /**
     * Remove the specified `option` from the specified `section`.
     *
     * Params:
     *   section = The section to remove from.
     *   option = The option to remove from section.
     *
     * Retruns:
     *   `true` if option existed, false otherwise.
     *
     * Throws:
     *  - NoSectionException if the specified section doesn't exist.
     */
    bool removeOption(string section, string option)
    {
        if ((section in m_sections) is null) {
            throw new NoSectionException(section);
        }

        if (option in m_sections[section]) {
            m_sections[section].remove(option);
            return true;
        }

        return false;
    }

    ///
    unittest
    {
        import std.exception : assertThrown;

        auto conf = new ConfigParser();
        conf.addSection("Default");
        conf.set("Default", "exists", "true");

        assertThrown!NoSectionException(conf.removeOption("void", "false"));
        assert(false == conf.removeOption("Default", "void"));
        assert(true == conf.removeOption("Default", "exists"));
    }

    /**
     * Remove the specified `section` from the config.
     *
     * Params:
     *   section = The section to remove.
     *
     * Returns:
     *   `true` if the section existed, `false` otherwise.
     */
    bool removeSection(string section)
    {
        if (section in m_sections) {
            m_sections.remove(section);
            return true;
        }
        return false;
    }

    ///
    unittest
    {
        auto conf = new ConfigParser();
        conf.addSection("Exists");
        assert(false == conf.removeSection("DoesNotExist"));
        assert(true == conf.removeSection("Exists"));
    }

    void set(string section, string option, string value)
    {
        if (false == this.hasSection(section))
            throw new NoSectionException(section);

        m_sections[section][option] = value;
    }

    ///
    unittest
    {
        import std.exception : assertThrown;

        auto conf = new ConfigParser();

        assertThrown!NoSectionException(conf.set("Section", "option",
            "value"));

        conf.addSection("Section");
        conf.set("Section", "option", "value");
        assert(conf.get("Section", "option") == "value");
    }

    // void write(ref File file)
    // {
    // }

  private:

    void parseSectionHeader(ref string line)
    {
        import std.array : appender, assocArray;

        auto sectionHeader = appender!string;
        /* presume that the last character is ] */
        sectionHeader.reserve(line.length - 1);
        string popped = line[1 .. $];

        foreach(c; popped) {
            if (c != ']')
                sectionHeader.put(c);
            else
                break;
        }

        version (DigitalMars) {
            m_currentSection = sectionHeader[];
        } else {
            /* LDC / GNU */
            m_currentSection = sectionHeader.data;
        }

        try {
            this.addSection(m_currentSection);
        } catch (DuplicateSectionException e) {
            /* haven't checked what python would do */
            /* perhaps we should just merge the settings? */
            return;
        }
    }

    void parseLine(ref string line)
    {
      import std.string : indexOfAny, strip;

        ptrdiff_t idx = line.indexOfAny(m_delimiters);
        if (-1 == idx) return;
        string option = line[0 .. idx].dup.strip;
        string value = line[idx + 1 .. $].dup.strip;

        m_sections[m_currentSection][option] = value;
    }
}
